/* Plagiat gallery creation tool for Jakub Steiner's O.R.I.G.I.N.A.L 
 * photo gallery.
 * Copyright (C) 2005 Robert Staudinger <robert.staudinger@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <time.h>
#include <glade/glade.h>
#include "pgt-mainwin.h"
#include "pgt-calendar.h"
#include "pgt-galleries.h"
#include "pgt-utils.h"

struct _PgtMainwin {
	GObject parent;
	gboolean is_disposed;
	GladeXML *xml;
	GtkWidget *win;
	GtkWidget *maintable;
	GtkWidget *gallerycombo;
	GtkWidget *gallerybtn;
	GtkWidget *albumentry;
	GtkWidget *albumbtn;
	GtkWidget *datelabel;
	GtkWidget *albumtext;
	GtkWidget *sourcebtn;
	GtkWidget *options[num_options];
	GtkWidget *progress;
	GtkWidget *prefsbtn;
	GtkWidget *closebtn;
	GtkWidget *convertbtn;
};

struct _PgtMainwinClass {
	GObjectClass parent;
	/* signals */
	void (* gallery_changed) 	(PgtMainwin *self);
	void (* manage_galleries) 	(PgtMainwin *self);
	void (* album_name_changed) 	(PgtMainwin *self);
	void (* description_changed) 	(PgtMainwin *self);
	void (* photo_dir_changed) 	(PgtMainwin *self);
	void (* option_changed) 	(PgtMainwin *self);
	void (* process) 		(PgtMainwin *self);
	void (* delete) 		(PgtMainwin *self);
};

/* signals */
enum {
	GALLERY_CHANGED,
	MANAGE_GALLERIES,
	ALBUM_NAME_CHANGED,
	DESCRIPTION_CHANGED,
	PHOTO_DIR_CHANGED,
	OPTION_CHANGED,
	PROCESS,
	DELETE,
	_LAST_SIGNAL
};

/* properties */
enum {
	_PROP_0,
	PROP_GALLERY_STORE,
	_NUM_PROPS
};

static guint 	     pgt_mainwin_signals[_LAST_SIGNAL] = { 0 };
static GObjectClass *pgt_mainwin_parent_class = NULL;


static void
gallery_changed_cb (PgtMainwin  *self, 
		    GtkComboBox *widget)
{
	g_signal_emit (self, pgt_mainwin_signals[GALLERY_CHANGED], 0);
}

static void
manage_galleries_cb (PgtMainwin    *self, 
		     GtkToolButton *widget)
{
	g_signal_emit (self, pgt_mainwin_signals[MANAGE_GALLERIES], 0);
}

static void
select_date_cb (PgtMainwin    *self, 
		GtkToolButton *widget)
{
	guint year;
	guint month;
	guint day;
	gchar *date = NULL;
	
	PgtCalendar *calendar = pgt_calendar_new ();
	gtk_dialog_run (GTK_DIALOG (calendar));

	pgt_calendar_get_date (calendar, &year, &month, &day);
	date = g_strdup_printf ("%02d.%02d.%d", day, month, year);
	gtk_label_set_text (GTK_LABEL (self->datelabel), date);
	g_free (date);
	date = NULL;

	gtk_widget_destroy (GTK_WIDGET (calendar));
}

static void
album_name_changed_cb (PgtMainwin      *self, 
		       GtkCellEditable *widget)
{
	g_signal_emit (self, pgt_mainwin_signals[ALBUM_NAME_CHANGED], 0);
}

static void
photo_dir_changed_cb (PgtMainwin     *self, 
		      GtkFileChooser *widget)
{
	g_signal_emit (self, pgt_mainwin_signals[PHOTO_DIR_CHANGED], 0);
}

static void
option_changed_cb (PgtMainwin      *self, 
		   GtkToggleButton *widget)
{
	g_signal_emit (self, pgt_mainwin_signals[OPTION_CHANGED], 0);
}

static void
process_cb (PgtMainwin *self, 
	    GtkButton  *widget)
{
	g_signal_emit (self, pgt_mainwin_signals[PROCESS], 0);
}

static void
close_cb (PgtMainwin *self, 
	  GtkButton  *widget)
{
	if (self->win) {
		gtk_widget_destroy (self->win);
		self->win = NULL;
	}
}

static void
delete_cb (PgtMainwin *self,
	   GtkWidget  *widget,
	   GdkEvent   *event)
{
	g_signal_emit (self, pgt_mainwin_signals[DELETE], 0);
}

/* no prefs yet 
static void
manage_prefs_cb (PgtMainwin    *self, 
		 GtkToolButton *widget)
{
}
*/


static void
instance_init (PgtMainwin *self)
{
	GtkWidget *gallerybox = NULL;
	GtkWidget *albumbox = NULL;
	GtkWidget *titlecontainer = NULL;
	GtkWidget *titleimg = NULL;
	GtkWidget *titlelabel = NULL;
	GtkCellRenderer *renderer = NULL;
	GtkStyle *style = NULL;
	GDate date;
	gchar *datestr = NULL;

	self->is_disposed = FALSE;


	if (pgt_runs_from_srcdir ()) {
		self->xml = glade_xml_new ("../data/mainwin.glade", NULL, NULL); 
	}
	else {
		self->xml = glade_xml_new (PLAGIAT_DATADIR "/mainwin.glade", NULL, NULL);
	}
	self->win = glade_xml_get_widget (self->xml, "mainwin");
	
	titleimg = glade_xml_get_widget (self->xml, "titleimg");
	if (pgt_runs_from_srcdir ()) {
		gtk_image_set_from_file (GTK_IMAGE (titleimg), "../data/logo.png");
	}
	else {
		gtk_image_set_from_file (GTK_IMAGE (titleimg), PLAGIAT_PIXMAPDIR "/logo.png");
	}

	titlecontainer = glade_xml_get_widget (self->xml, "titlecontainer");
	titlelabel = glade_xml_get_widget (self->xml, "titlelabel");
	style = gtk_widget_get_style (titlelabel);
	gtk_widget_modify_bg (titlecontainer, GTK_STATE_NORMAL, &style->bg[GTK_STATE_SELECTED]);
	gtk_widget_modify_text (titlelabel, GTK_STATE_NORMAL, &style->text[GTK_STATE_SELECTED]); 

	self->maintable = glade_xml_get_widget (self->xml, "maintable");

	gallerybox = glade_xml_get_widget (self->xml, "gallerybox"); 
	self->gallerycombo = glade_xml_get_widget (self->xml, "gallerycombo");
	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (self->gallerycombo), renderer, TRUE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (self->gallerycombo),
				       renderer, "text", 0);

	self->gallerybtn = (GtkWidget *) gtk_tool_button_new_from_stock (GTK_STOCK_ADD);
	gtk_box_pack_start (GTK_BOX (gallerybox), GTK_WIDGET (self->gallerybtn), FALSE, FALSE, 0);
/* TODO enable when galleries work 
	gtk_widget_set_sensitive (self->gallerybtn, FALSE);
*/
	self->albumentry = glade_xml_get_widget (self->xml, "albumentry");

	self->datelabel = glade_xml_get_widget (self->xml, "datelabel");
	g_date_set_time (&date, time (NULL));
	datestr = g_strdup_printf ("%02d.%02d.%d", 
				   g_date_get_day (&date),
				   g_date_get_month (&date),
				   g_date_get_year (&date));
	gtk_label_set_text (GTK_LABEL (self->datelabel), datestr);
	g_free (datestr);

	albumbox = glade_xml_get_widget (self->xml, "albumbox"); 
	self->albumbtn = (GtkWidget *) gtk_tool_button_new_from_stock (GTK_STOCK_INDEX);
	gtk_box_pack_start (GTK_BOX (albumbox), GTK_WIDGET (self->albumbtn), FALSE, FALSE, 0);

	self->albumtext = glade_xml_get_widget (self->xml, "albumtext"); 
	self->sourcebtn = glade_xml_get_widget (self->xml, "sourcebtn");
	gtk_file_chooser_set_action (GTK_FILE_CHOOSER (self->sourcebtn), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

	self->options[option_lq] = glade_xml_get_widget (self->xml, "lqcheck");
	self->options[option_mq] = glade_xml_get_widget (self->xml, "mqcheck");
	self->options[option_hq] = glade_xml_get_widget (self->xml, "hqcheck");
	self->options[option_zip] = glade_xml_get_widget (self->xml, "zipcheck");
	self->options[option_comment] = glade_xml_get_widget (self->xml, "commentcheck");

	self->progress = glade_xml_get_widget (self->xml, "progress");

/* no prefs for now 
	prefscontainer = glade_xml_get_widget (self->xml, "prefscontainer");
	self->prefsbtn = (GtkWidget *) gtk_tool_button_new_from_stock (GTK_STOCK_PREFERENCES);
	gtk_container_add (GTK_CONTAINER (prefscontainer), self->prefsbtn);
*/
	self->closebtn = glade_xml_get_widget (self->xml, "closebtn");
	self->convertbtn = glade_xml_get_widget (self->xml, "convertbtn");


	/* connect signals */	
	g_signal_connect_swapped (self->gallerycombo, "changed", 
				  G_CALLBACK (gallery_changed_cb), self);
	g_signal_connect_swapped (self->albumentry, "editing-done", 
				  G_CALLBACK (album_name_changed_cb), self);
	g_signal_connect_swapped (GTK_FILE_CHOOSER (self->sourcebtn), "file-activated", 
				  G_CALLBACK (photo_dir_changed_cb), self);
	g_signal_connect_swapped (self->options[option_lq], "toggled", 
				  G_CALLBACK (option_changed_cb), self);
	g_signal_connect_swapped (self->options[option_mq], "toggled", 
				  G_CALLBACK (option_changed_cb), self);
	g_signal_connect_swapped (self->options[option_hq], "toggled", 
				  G_CALLBACK (option_changed_cb), self);
	g_signal_connect_swapped (self->options[option_zip], "toggled", 
				  G_CALLBACK (option_changed_cb), self);
	g_signal_connect_swapped (self->options[option_comment], "toggled", 
				  G_CALLBACK (option_changed_cb), self);
	g_signal_connect_swapped (self->convertbtn, "clicked", 
				  G_CALLBACK (process_cb), self);
	g_signal_connect_swapped (self->closebtn, "clicked", 
				  G_CALLBACK (close_cb), self);
	g_signal_connect_swapped (self->win, "delete-event", 
				  G_CALLBACK (delete_cb), self);

	g_signal_connect_swapped (self->gallerybtn, "clicked", 
				  G_CALLBACK (manage_galleries_cb), self);
	g_signal_connect_swapped (self->albumbtn, "clicked", 
				  G_CALLBACK (select_date_cb), self);
/*
	g_signal_connect_swapped (self->prefsbtn, "clicked", 
				  G_CALLBACK (manage_prefs_cb), self);
*/
}

static void
instance_dispose (GObject *instance)
{
	PgtMainwin *self = PGT_MAINWIN (instance);

	if (self->is_disposed)
		return;

	self->is_disposed = TRUE;

	pgt_mainwin_parent_class->dispose (G_OBJECT (self));
}

/* TODO hook destroy and call gtk_main_quit() */

static void
set_property (GObject      *object,
	      guint         prop_id,
	      GValue const *value,
	      GParamSpec   *pspec)
{
	PgtMainwin *self = PGT_MAINWIN (object);
	GtkTreeModel *gallerystore = NULL;
	gpointer *p = NULL;

	switch (prop_id) {
	case PROP_GALLERY_STORE:
		p = g_value_get_object (value);
		if (p) {
			gallerystore = GTK_TREE_MODEL (p);
			gtk_combo_box_set_model (GTK_COMBO_BOX (self->gallerycombo), gallerystore);
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
get_property (GObject    *object,
	      guint       prop_id,
	      GValue     *value,
	      GParamSpec *pspec)
{
	PgtMainwin *self = PGT_MAINWIN (object);

	switch (prop_id) {
	case PROP_GALLERY_STORE:
		g_value_set_pointer (value, gtk_combo_box_get_model (GTK_COMBO_BOX (self->gallerycombo)));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
class_init (PgtMainwinClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	/* hook gobject vfuncs */
	gobject_class->dispose = instance_dispose;

	pgt_mainwin_parent_class = (GObjectClass*) g_type_class_peek_parent (klass);

	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	pgt_mainwin_signals [GALLERY_CHANGED] =
		g_signal_new (
			"gallery-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtMainwinClass, gallery_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_mainwin_signals [MANAGE_GALLERIES] =
		g_signal_new (
			"manage-galleries",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtMainwinClass, gallery_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_mainwin_signals [ALBUM_NAME_CHANGED] =
		g_signal_new (
			"album-name-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtMainwinClass, album_name_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_mainwin_signals [DESCRIPTION_CHANGED] =
		g_signal_new (
			"description-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtMainwinClass, description_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_mainwin_signals [PHOTO_DIR_CHANGED] =
		g_signal_new (
			"photo-dir-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtMainwinClass, photo_dir_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_mainwin_signals [OPTION_CHANGED] =
		g_signal_new (
			"option-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtMainwinClass, option_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_mainwin_signals [PROCESS] =
		g_signal_new (
			"process",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtMainwinClass, process),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_mainwin_signals [DELETE] =
		g_signal_new (
			"delete",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtMainwinClass, delete),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);


	g_object_class_install_property (gobject_class,
		PROP_GALLERY_STORE,
		g_param_spec_object ("gallery-store", "Gallery Store",
			"GtkListStore for galleries", GTK_TYPE_TREE_MODEL,
			(GParamFlags)(G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY)));
}

GType
pgt_mainwin_get_gtype (void)
{
        static GType type = 0;
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (PgtMainwinClass),
                        NULL,           /* base_init */
                        NULL,           /* base_finalize */
                        (GClassInitFunc) class_init,
                        NULL,           /* class_finalize */
                        NULL,           /* class_data */
                        sizeof (PgtMainwin),
                        0,              /* n_preallocs */
                        (GInstanceInitFunc) instance_init,
                };
                type = g_type_register_static (G_TYPE_OBJECT, "PgtMainwin", &info, (GTypeFlags)0);
        }
        return type;
}

PgtMainwin*   
pgt_mainwin_new (GtkTreeModel *gallerystore)
{
	return PGT_MAINWIN (g_object_new (PGT_TYPE_MAINWIN, 
					  "gallery-store", gallerystore, 
					  NULL));
}

GtkWidget* 
pgt_mainwin_get_window (PgtMainwin *self)
{
	return self->win;
}

gchar const * 
pgt_mainwin_get_gallery_name (PgtMainwin *self)
{
	/* TODO */
	return "";
}

gchar const * 
pgt_mainwin_get_album_name (PgtMainwin *self)
{
	return gtk_entry_get_text (GTK_ENTRY (self->albumentry));
}

gchar const * 
pgt_mainwin_get_album_date (PgtMainwin *self)
{
	return gtk_label_get_text (GTK_LABEL (self->datelabel));
}

gchar const * 
pgt_mainwin_get_album_desc (PgtMainwin *self)
{
	GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (self->albumtext));
	GtkTextIter start, end;

	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	return gtk_text_buffer_get_text (buffer, &start, &end, TRUE);
}

gchar* 
pgt_mainwin_get_photo_dir (PgtMainwin *self)
{
	GtkFileChooser *chooser = GTK_FILE_CHOOSER (self->sourcebtn);
	return gtk_file_chooser_get_filename (chooser);
}

gboolean   
pgt_mainwin_get_option (PgtMainwin *self, 
		     PgtConvertOption option)
{
	GtkToggleButton *btn = NULL;

	btn = GTK_TOGGLE_BUTTON (self->options[option]);
	return gtk_toggle_button_get_active (btn);
}

void	   
pgt_mainwin_set_progress_value (PgtMainwin *self, 
				gfloat val)
{
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (self->progress), val);
}

void	   
pgt_mainwin_set_progress_text (PgtMainwin *self, 
			       gchar const *text)
{
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (self->progress), text);
}
