#!/usr/bin/env python
###################################################################
#  NumExpr - Fast numerical array expression evaluator for NumPy.
#
#      License: MIT
#      Author:  See AUTHORS.txt
#
#  See LICENSE.txt and LICENSES/*.txt for details about copyright and
#  rights to use.
####################################################################

import sys, os, os.path as op, io
# import setuptools
from setuptools import setup, find_packages, Extension
import platform
import numpy as np

if sys.version_info < (3, 6):
    raise RuntimeError("NumExpr requires Python 3.6 or greater")

with open('requirements.txt') as f:
    requirements = f.read().splitlines()

with io.open('README.rst', encoding='utf-8') as f:
    LONG_DESCRIPTION = f.read()

major_ver = 2
minor_ver = 8
nano_ver = 0
branch = ''

version = '%d.%d.%d%s' % (major_ver, minor_ver, nano_ver, branch)
with open('numexpr/version.py', 'w') as fh:
    fh.write('# THIS FILE IS GENERATED BY `SETUP.PY`\n')
    fh.write("version = '%s'\n" % version)
    try:
        import numpy
        fh.write("numpy_build_version = '%s'\n" % numpy.__version__)
    except ImportError:
        pass

lib_dirs = []
inc_dirs = [np.get_include(), op.join('framestream')]
libs = []  # Pre-built libraries ONLY, like python36.so
clibs = []
def_macros = []
sources = ['numexpr/interpreter.cpp',
           'numexpr/module.cpp',
           'numexpr/numexpr_object.cpp']
extra_cflags = []
extra_link_args = []

if platform.uname().system == 'Windows':
    extra_cflags = ['/O2']
    extra_link_args = []
    sources.append('numexpr/win32/pthread.c')
else:
    extra_cflags = []
    extra_link_args = []

def parse_site_cfg():
    """
    Parses `site.cfg`, if it exists, to determine the location of Intel oneAPI MKL.

    To compile NumExpr with MKL (VML) support, typically you need to copy the 
    provided `site.cfg.example` to `site.cfg` and then edit the paths in the 
    configuration lines for `include_dirs` and `library_dirs` paths to point 
    to the appropriate directories on your machine.
    """
    import configparser
    site = configparser.ConfigParser()
    if not op.isfile('site.cfg'):
        return

    site.read('site.cfg')

    if 'mkl' in site.sections():
        inc_dirs.extend(
            site['mkl']['include_dirs'].split(os.pathsep))
        lib_dirs.extend(
            site['mkl']['library_dirs'].split(os.pathsep))
        libs.extend(
            site['mkl']['libraries'].split(os.pathsep))
        def_macros.append(('USE_VML', None))
        

def setup_package():

    parse_site_cfg()

    numexpr_extension = Extension('numexpr.interpreter',
        include_dirs=inc_dirs,
        define_macros=def_macros,
        sources=sources,
        library_dirs=lib_dirs,
        libraries=libs,
        extra_compile_args=extra_cflags,
        extra_link_args=extra_link_args,)


    metadata = dict(
                    name = 'numexpr',
                    version = version,
                    description = 'Fast numerical expression evaluator for NumPy',
                    author = 'David M. Cooke, Francesc Alted, and others',
                    author_email = 'david.m.cooke@gmail.com, faltet@gmail.com',
                    maintainer = 'Robert A. McLeod',
                    maintainer_email = 'robbmcleod@gmail.com',
                    url = 'https://github.com/pydata/numexpr',
                    long_description = LONG_DESCRIPTION,
                    license = 'MIT',
                    packages = find_packages(),
                    install_requires = requirements,
                    setup_requires = requirements,
                    extras_require = {
                        },
                    libraries = clibs,
                    ext_modules = [
                            numexpr_extension
                        ],
                    zip_safe = False,
                    classifiers = [
                            'Development Status :: 6 - Mature',

                            'Intended Audience :: Financial and Insurance Industry',
                            'Intended Audience :: Science/Research',

                            'License :: OSI Approved :: MIT License',
                    
                            'Programming Language :: Python :: 3',
                            'Programming Language :: Python :: 3.6',
							'Programming Language :: Python :: 3.7',
                            'Programming Language :: Python :: 3.8',
                            'Programming Language :: Python :: 3.9',
                            'Programming Language :: Python :: 3.10',
                            # OS
                            'Operating System :: Microsoft :: Windows',
                            'Operating System :: POSIX',
                            'Operating System :: MacOS',
                        ],
                    
    )
    setup(**metadata)


if __name__ == '__main__':
    setup_package()
