/*
 * Mausezahn - A fast versatile traffic generator
 * Copyright (C) 2008-2010 Herbert Haas
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 as published by the 
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more 
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with 
 * this program; if not, see http://www.gnu.org/licenses/gpl-2.0.html
 * 
*/


/////////////////////////////////////////////////////////////////////////////////
//
// Contains various tools to parse user input in a safe and simple way:
// 
// mz_strcmp ......... Matches a string or a prefix of it with given min-length
//                     Example usage: User CLI input
//                     
// mz_tok ............ Decomposes a string into tokens and maps them to args
//                     Example usage: IPv6-addresses, user input for MPLS-tags


#include "mz.h"


// Compares user-provided string with a specified string.
// 
// Return value:
//  
//   0  if at least min characters match 
//   1  if at least one character of usr does NOT match the corresponding character in str.
//             
// Note: Case-insensitive!
// Goal: Should be more practical and secure than strcmp (and related)
int mz_strcmp(char* usr_orig, char* str_orig, int min)
{
   int i, same=0, usrlen, max;
   char usr[80], str[80];
   
   usrlen = strlen(usr_orig);
   max = strlen(str_orig);
   
   strncpy(usr, usr_orig, 80);
   strncpy(str, str_orig, 80);
   
   // User provided not enough or too many chars
   if ((usrlen<min) || (usrlen>max)) return 1;
   
   // now check how many bytes really match
   for (i=0; i<usrlen; i++)
     {
	if (strncasecmp(&usr[i], &str[i], 1)==0)
	  {
	     same++;
	  }
     }
   
   if (same<usrlen) return 1;
   
   return 0;
}





// PURPOSE:
// 
//  Maps an arbitrary number of tokens from 'str' which are separated by 
//  a character 'delim' into provided arguments.
// 
// USAGE EXAMPLE:
// 
//  char str[]="Am:Dam:Des";
//  char t1[64], t2[64], t3[64], t4[64];
//  
//  mz_tok (str, ":", 4, t1, t2, t3, t4)
// 
//  => t1="Am", t2="Dam", t3="Des", t4=NULL
// 
// NOTE: 
// 
//   1. If the delimiter symbol occurs twice without gap, it is interpreted
//      as 'fill-up' command. To avoid ambiguities this may only occur once.
//      See the IPv6 address format shortcuts as similar example.     
//    
//   2. If there are less tokens than allowed, the arguments are filled up
//      in order, while the remaining are casted to NULL:
// 
//   3. str must be smaller than 4096 bytes!
//   
//   4. To mitigate buffer overflow problems, the maximum token size is 
//      currently limited to 64 bytes. Therefore it is recommended to
//      allocate 64 bytes for each argument.
//       
// RETURN VALUE: Number of returned tokens or -1 upon error

int mz_tok(char * str, char * delim, int anz, ...)
{
   
   va_list ap;
   int i=0, n=0, len, llen, rlen, ltok=0, rtok=0;
   char *d, *l, *r, *token, *saveptr, *arg;
   char str2[4096], delim2[4]="", delim3[4]="";;

   if (strlen(delim)!=1) return -1; // delim must contain a single character!
   strncpy(str2, str, 4095);        // protect the original str from strtok => operate on a copy only
   len = strlen(str2);

   // Check if some tokens are omitted (::)
   strncpy(delim2, delim, 1); strncat(delim2, delim, 1);  // create the double-delim
   strncpy(delim3, delim2, 2); strncat(delim3, delim, 1); // create the double-delim
   if (strstr(str2, delim3)!=NULL) return -1;             // Error: ':::' occured!
   
   if ( (d=strstr(str2, delim2))!=NULL ) // delim2 ('::') found
     {
	// Check 3 cases: "::Sat:Sun", "Mon::Sat:Sun", and "Mon:Tue::"
	if (strlen(d)>2) // '::' is not at the end of str2
	  {
	     r=d+2;  // r points to beginning of right string
	     if (strstr(r, delim2)!=NULL) return -1; // Error: '::' occurs more than once!
	     rtok++; // there is at least one token in the right string
	     rlen = strlen(r);
	     for(i=0;i<rlen;i++) if (strncmp(r++,delim,1)==0) rtok++;
	  }
	else
	  rlen = 0;
	
	if (rlen<(len-2)) // '::' is not at the beginning of str2
	  {
	     l=d-1;  // l points to end of left string 
	     ltok++;
	     llen = len - rlen - 2;
	     for(i=0;i<llen;i++) if (strncmp(l--,delim,1)==0) ltok++;
	  }
	//printf("ltok=%i, rtok=%i\n",ltok,rtok);
	if ((ltok+rtok)>anz) return -1; // More tokens than arguments ('::' here leads to ambiguous mapping)
     }
   else
     ltok=len+1; // makes subsequent algorithm to ignore exception handling
   

   
   rtok=anz-rtok; 
   va_start(ap, anz);

   token = strtok_r(str2, delim, &saveptr);
   if (token==NULL) { va_end(ap); return n; }
   
   for(i=0; i<anz; i++)
     {
	arg = va_arg(ap, char *);
	if ( (token==NULL) ||  // less tokens than arguments => assign NULL to the remaining arguments!
	     ((i>=ltok) && (i<rtok)))
	  {
	     arg[0] = 0x00;
	  }
	else // we still have tokens...
	  {
	     n++;
	     strncpy(arg, token, 64);
	     token = strtok_r(NULL, delim, &saveptr);
	  }
     }
   
   va_end(ap);
   return n;
}



// Returns a nice string with default and current value of a given variable
//
// EXAMPLE:
//  
//    char mystring[256];
//    mz_def16 ("20 seconds", pd->max_age, mystring)
//    
int mz_def16 (char *def, u_int16_t val, char *str256)
{
   str256[0]=0x00;
   sprintf(str256, "The default value is %s. The current value is %u (0x%04x).", def, val, val);
   return 0;
}


