/*
 * Mausezahn - A fast versatile traffic generator
 * Copyright (C) 2008 Herbert Haas
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 as published by the 
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more 
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with 
 * this program; if not, see http://www.gnu.org/licenses/gpl-2.0.html
 * 
*/





///////////////////////////////////////////////////////////////////////////////////////////
//
//  Contains various tools for hex-based conversions and manipulation of bytestrings
//  
//  str2hex_mac
//  str2hex
//  num2hex
//  bs2str
//  getbytes
//  str2ip32
//  srr2ip32_rev
//  type2str
//  
////////////////////////////////////////////////////////////////////////////////////////////


#include "mz.h"



// converts MAC address specified in str into u_int8_t array
void str2hex_mac(char* str, u_int8_t *addr)
{
   char *hs;
   int curval,i;
   char tmp[32];

   strcpy(tmp,str); // necessary because strtok cannot operate on fixed strings
   
   hs=(char*)strtok(tmp,"-:., ");

   for (i=0;i<6;i++)
     {
	curval=strtol(hs,NULL,16);
	addr[i]=(u_int8_t) curval;
	hs=(char*)strtok(NULL,"-:., ");

	if ( (hs == NULL ) && (i!=5) )
	     { 
		fprintf(stderr, " You did not provide a valid MAC address; remaining bytes will be zero.\n");
		i=6; 
	     }
     }
   // NOTE: does not care for '\0' termination as it expects addr to be defined
   //       as addr[6] which is automatically terminated properly.
}



//
// Converts ascii hex values (string) into integer array
// For example "1a 00:00-2f" will be converted to {26, 0, 0, 47}
//
int str2hex(char* str, u_int8_t *hp)
{
   char *hs;
   int curval,i;

   if (strlen(str)==0) return 0;
   
   char tmp[8192]=""; //for very long payloads 
     
   strcpy(tmp,str); // necessary because strtok cannot operate on fixed strings
   
   hs=(char*)strtok(tmp,"-:., ");

   i=0;
   do
     {
	curval=strtol(hs,NULL,16);
	hp[i]=(u_int8_t) curval;
	i++;
     }
   while ((hs=(char*)strtok(NULL,"-:., "))!= NULL);
   //hp[i]='\0'; // termination not necessary
   return i; // return the length of the array
}



// Converts ascii numbers (string) into integer array
// Every byte can be specified as integers {0..255}
// For example "192.16.1.1" will be converted to {C0, 10, 01, 01} 
int num2hex(char* str, u_int8_t *hp)
{
   char *hs;
   int i;
   u_int8_t curval;
   
   if (strlen(str)==0) return 0;
   
   char tmp[8192]=""; //for very long payloads 
     
   strcpy(tmp,str); // necessary because strtok cannot operate on fixed strings
   
   hs=(char*)strtok(tmp,"-:., ");

   i=0;
   do
     {
	curval = (u_int8_t) str2int(hs);
	hp[i] = curval;
	i++;
     }
   while ((hs=(char*)strtok(NULL,"-:., "))!= NULL);
   //hp[i]='\0'; // termination not necessary

   return i;
}



// Convert array of integers into string of hex
// E.g. {0,1,10} => "00-01-0A"
// Useful for verification messages. 
int bs2str(u_int8_t *bs, char* str, int len)
{
   int i;
   char t[4];
   
   str[0]='\0';
   
   for (i=0; i<len; i++)
     {
	if (bs[i]<16) strcat(str,"0"); // enforce two hex digits (e.g. "0a")
	
	sprintf(t,"%x:",bs[i]);
	strcat(str,t);
     }
   str[strlen(str)-1]='\0'; //remove the last "-"
   return 1;
}


// Extract contiguous sequence of bytes from an array
// NOTE: first element has number 1 !!!
int getbytes(u_int8_t *source,
	     u_int8_t *target, 
	     int from,
	     int to)
  
{
   int i;

   // Check wrong arguments
   if  (from<1) 
     {
	return -1;
     }
   
   // copy bytes
   for (i=0; i<(to-from+1); i++)
       {
	  target[i]=source[from-1+i];
       }
   
   return 1;
}


// Converts an  IP address given in 'dotted decimal' into an unsigned 32-bit integer
// Example: "192.168.0.1" => 3232235521
u_int32_t str2ip32 (char* str)
{
	u_int32_t ip = 0;
	unsigned int a,b,c,d;
	int r;
	
	// check whether str really contains an IP address
	if (strlen(str)<3) return 0;
	if (str==NULL) return 0;
	
	if ((r=sscanf(str,"%i.%i.%i.%i",&a,&b,&c,&d))==0) return 0;
	if (r==EOF) return 0;
		
	/* or an alternative method...
	// these are the four bytes of a dotted decimal notation IP address:
	a = (unsigned int) strtol(strtok(str,"."), (char **)NULL, 10);
	b = (unsigned int) strtol(strtok(NULL,"."), (char **)NULL, 10);
	c = (unsigned int) strtol(strtok(NULL,"."), (char **)NULL, 10);
	d = (unsigned int) strtol(strtok(NULL,"."), (char **)NULL, 10);
	*/
	
	if ((a>255)||(b>255)||(c>255)||(d>255)) return 0;
	
	ip = d + 256*c + 256*256*b + 256*256*256*a;
      
	//check with: 
	//printf("str2ip32 got 4 bytes: %i %i %i %i\n",a,b,c,d); 
	//printf("str2ip32 returned %u\n",ip);
	
	return ip;
}


// Converts an  IP address given in 'dotted decimal' into an unsigned 32-bit integer
// This version does the same as str2ip32() but in 'network byte order'
u_int32_t str2ip32_rev (char* str)
{
	u_int32_t ip = 0;
	unsigned int a,b,c,d;
	int r;
	
	// check whether str really contains an IP address
	if (strlen(str)<3) return 0;
	if (str==NULL) return 0;
	
	if ((r=sscanf(str,"%i.%i.%i.%i",&a,&b,&c,&d))==0) return 0;
	if (r==EOF) return 0;
		
	/* or an alternative method...
	// these are the four bytes of a dotted decimal notation IP address:
	a = (unsigned int) strtol(strtok(str,"."), (char **)NULL, 10);
	b = (unsigned int) strtol(strtok(NULL,"."), (char **)NULL, 10);
	c = (unsigned int) strtol(strtok(NULL,"."), (char **)NULL, 10);
	d = (unsigned int) strtol(strtok(NULL,"."), (char **)NULL, 10);
	*/
	
	if ((a>255)||(b>255)||(c>255)||(d>255)) return 0;
	
	ip = a + b*256 + c*256*256 + d*256*256*256;
      
	//check with: 
	//printf("str2ip32 got 4 bytes: %i %i %i %i\n",a,b,c,d); 
	//printf("str2ip32 returned %u\n",ip);
	
	return ip;
}


// Converts a 2-byte value (e. g. a EtherType field)
// into a nice string using hex notation.
// Useful for verification messages.
// Example: type2str (tx.eth_type, msg) may result in msg="08:00"
// Return value: how many hex digits have been found.
int type2str(u_int16_t type, char *str)
{
   char hex[8];
   int i=0;
   
   (void) sprintf (hex, "%x",type);
   i=strlen(hex);
     
   switch (i)
     {
      case 1:
	str[0]='0';
	str[1]='0';
	str[2]=':';
	str[3]='0';
	str[4]=hex[0];
	str[5]='\0';
	break;
      case 2:
      	str[0]='0';
	str[1]='0';
	str[2]=':';
	str[3]=hex[0];
	str[4]=hex[1];
	str[5]='\0';
	break;
      case 3:
      	str[0]='0';
	str[1]=hex[0];
	str[2]=':';
	str[3]=hex[1];
	str[4]=hex[2];
	str[5]='\0';
	break;
      case 4:
	str[0]=hex[0];
	str[1]=hex[1];
	str[2]=':';
	str[3]=hex[2];
	str[4]=hex[3];
	str[5]='\0';
	break;
	
     }
   return i;
}

