# -*- coding: utf-8; Mode: Python; indent-tabs-mode: nil; tab-width: 4 -*-
#
# Copyright (C) 2006, 2007, 2009 Canonical Ltd.
# Written by Colin Watson <cjwatson@ubuntu.com>.
# Copyright (C) 2007-2010 Mario Limonciello
#
# This file is part of Ubiquity.
#
# Ubiquity is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Ubiquity is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ubiquity.  If not, see <http://www.gnu.org/licenses/>.

from ubiquity.plugin import *
from mythbuntu_common.installer import MythPageGtk
from ubiquity import install_misc
from mythbuntu_common.vnc import VNCHandler
from ubiquity import misc
import os
import subprocess
import shutil
import time

NAME = 'myth-installtype'
HIDDEN = 'webcam'
AFTER = ['usersetup', None]
WEIGHT = 12

class PageGtk(MythPageGtk):
    plugin_title = 'ubiquity/text/installtype_heading_label'

    def __init__(self, controller, *args, **kwargs):
        self.ui_file='mythbuntu_stepCustomInstallType'
        MythPageGtk.__init__(self,controller,*args,**kwargs)

    def set_installtype(self,type):
        """Preseeds the type of custom install"""
        #if type == "Set Top Box":
        #    self.stb.set_active(True)
        if type == "Frontend":
            self.fe.set_active(True)
        elif type == "Slave Backend":
            self.slave_be.set_active(True)
        elif type == "Master Backend":
            self.master_be.set_active(True)
        elif type == "Slave Backend/Frontend":
            self.slave_be_fe.set_active(True)
        else:
            self.master_be_fe.set_active(True)

    def get_installtype(self):
        """Returns the current custom installation type"""
        if self.master_be_fe.get_active():
            return "Master Backend/Frontend"
        elif self.slave_be_fe.get_active():
            return "Slave Backend/Frontend"
        elif self.master_be.get_active():
            return "Master Backend"
        elif self.slave_be.get_active():
            return "Slave Backend"
        elif self.fe.get_active():
            return "Frontend"
        elif self.stb.get_active():
            return "Set Top Box"

class Page(Plugin):
    def prepare(self):
        #we don't want all the gstreamer stuff standard ubuntu brings in
        #flash is nice though
        more_nonfree = misc.create_bool(self.db.get('ubiquity/use_nonfree'))
        if more_nonfree:
            self.preseed('ubiquity/nonfree_package', 'flashplugin-installer')

        self.questions = ['install_type']
        questions = []
        for question in self.questions:
            answer = self.db.get('mythbuntu/' + question)
            if answer != '':
                self.ui.set_installtype(answer)
            questions.append('^mythbuntu/' + question)
        return (['/usr/share/ubiquity/ask-mythbuntu','type'], questions)

    def ok_handler(self):
        self.preseed('mythbuntu/' + self.questions[0],self.ui.get_installtype())

        admin = self.db.get('passwd/user-password')
        self.preseed('mythtv/mysql_admin_password', admin)

        Plugin.ok_handler(self)

class Install(InstallPlugin):
    def process_package_removals(self):
        packages=set()
        ## system role
        if 'Backend' not in self.type:
            packages.add('mythtv-backend')
            packages.add('php5-common')      #causes mythweb to be removed
            packages.add('libaprutil1')      #causes apache2 to be removed
        if 'Slave' in self.type or self.type == 'Frontend':
            packages.add('ntp')              #causes mythtv-backend-master to go
            packages.add('mythtv-database')
            packages.add('mysql-server-core-5.6')
        if 'Frontend' not in self.type:
            packages.add('mythtv-frontend')
        ## services that are installed by default
        for service in ['samba','openssh-server']:
            if not misc.create_bool(self.db.get('mythbuntu/' + service)):
                packages.add(service)
        if len(packages) >= 0:
            #recursively remove to make sure we get plugins and services that
            #aren't necessary anymore
            install_misc.record_removed(packages,True)

    def setup_common(self):
        #Always generate a new password
        #If the user has a security key, that trumps anyway (via config.xml)
        new_pass_caller = subprocess.Popen(
            ['pwgen', '-s', '8'], stdout=subprocess.PIPE,
            universal_newlines=True)
        answer = new_pass_caller.communicate()[0].split()[0]
        install_misc.set_debconf(self.target, 'mythtv/mysql_mythtv_password', answer)

        os.remove(self.target + '/etc/mythtv/config.xml')
        install_misc.reconfigure(self.target, 'mythtv-common')

    def setup_roles(self):
        passwd = self.progress.get('mythtv/mysql_admin_password')
        user = self.progress.get('passwd/username')

        if 'Master' in self.type:
            #Before beginning, set the initial root sql pass to the user pass
            for key in [ 'mythtv/mysql_admin_password',
                     'mysql-server/root_password',
                     'mysql-server/root_password_again' ]:
                install_misc.set_debconf(self.target, key, passwd)

            #sync mysql trees before reconfigure if we launched setup
            if self.launched:
                shutil.rmtree('/target/var/lib/mysql')
                shutil.copytree('/var/lib/mysql','/target/var/lib')

            #Setup sql server
            install_misc.reconfigure(self.target, 'mysql-server-5.6')

            #upstart managed chroot job, 
            #we need stuff working INSIDE the chroot.
            subprocess.call(['chroot', self.target, 'systemctl', 'stop', 'mysql'])
            subprocess.call(['chroot', self.target, '/etc/init.d/mysql', 'start'])
            while not os.path.exists(os.path.join(self.target, 'run', 'mysqld', 'mysqld.sock')):
                time.sleep(1)

            #build database
            install_misc.reconfigure(self.target, 'mythtv-database')

            #Cleanup
            subprocess.call(['chroot', self.target, '/etc/init.d/mysql', 'stop'])

            #Mythweb
            install_misc.set_debconf(self.target, 'mythweb/enable', 'true')
            install_misc.set_debconf(self.target, 'mythweb/username', user)
            install_misc.set_debconf(self.target, 'mythweb/password', passwd)
            install_misc.reconfigure(self.target, 'mythweb')

        #Normally this would be in myth-services, but we need a way to see the passwd,
        #and this is the last place we can see it.
        if 'Frontend' in self.type:
            if misc.create_bool(self.progress.get('mythbuntu/x11vnc')):
                vnc=VNCHandler()
                bytepass=passwd.encode('utf-8')
                vnc.create_password(bytepass)
                directory = self.target + '/home/' + user + '/.vnc'
                if not os.path.exists(directory):
                    os.makedirs(directory)
                shutil.move('/root/.vnc/passwd', directory + '/passwd')
                install_misc.record_installed(['x11vnc'])
            if misc.create_bool(self.db.get('ubiquity/use_nonfree')):
                install_misc.chroot_setup(self.target)
                install_misc.chrex(self.target, '/usr/share/doc/libdvdread4/install-css.sh')
                install_misc.chroot_cleanup(self.target)

    def install(self, target, progress, *args, **kwargs):
        self.target = target
        self.progress = progress

        self.progress.info('ubiquity/install/mythbuntu')
        self.type = self.progress.get('mythbuntu/install_type')
        self.launched = misc.create_bool(progress.get('mythbuntu/setup-launched'))

        self.setup_common()
        self.setup_roles()
        self.process_package_removals()

        return InstallPlugin.install(self, target, progress, *args, **kwargs)
