# -*- coding: utf-8; Mode: Python; indent-tabs-mode: nil; tab-width: 4 -*-
#
# Copyright (C) 2006, 2007, 2009 Canonical Ltd.
# Written by Colin Watson <cjwatson@ubuntu.com>.
# Copyright (C) 2007-2010 Mario Limonciello
#
# This file is part of Ubiquity.
#
# Ubiquity is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Ubiquity is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ubiquity.  If not, see <http://www.gnu.org/licenses/>.

import debconf
from ubiquity.plugin import *
from mythbuntu_common.installer import *
from mythbuntu_common.mysql import MySQLHandler
from ubiquity import install_misc
import os

NAME = 'myth-passwords'
AFTER = 'myth-drivers'
WEIGHT = 10

class PageGtk(MythPageGtk):
    plugin_title = 'ubiquity/text/masterinfo_heading_label'

    def __init__(self, controller, *args, **kwargs):
        self.ui_file = 'mythbuntu_stepPasswords'
        MythPageGtk.__init__(self, controller, *args, **kwargs)
        self.mysql=MySQLHandler()

    def set_type(self,type):
        """Prevents the user from going forward initially because of the
           type that was selected"""
        if "Master" not in type:
            self.controller.allow_go_forward(False)

    def do_connection_test(self,widget):
        """Tests to make sure that the backend is accessible"""
        result = self.mysql.do_connection_test(self.get_key())
        if not result:
            result = "Success"
        self.connection_results.set_text(result)
        self.controller.allow_go_forward(True)
        self.connection_results_label.show()

    def set_key(self,value):
        """Preseeds a security key"""
        self.security_entry.set_text(value)

    def get_key(self):
        return self.security_entry.get_text()

class Page(Plugin):
    def prepare(self):
        questions = []

        answer = self.db.get('mythtv/mysql_mythtv_user')
        if answer.isdigit():
            self.ui.set_key(answer)
        questions.append('^mythtv/mysql_mythtv_user')

        if 'UBIQUITY_AUTOMATIC' not in os.environ:
            #if we are a Master type, we'll skip this page
            type = self.db.get('mythbuntu/install_type')
            if 'Master' in type:
                os.environ['UBIQUITY_AUTOMATIC'] = "2"
                #regrab the key in case it was supposed preseeded
                key = self.ui.get_key()
                self.preseed('mythtv/mysql_mythtv_user', key)
            else:
                self.ui.set_type(type)

        return (['/usr/share/ubiquity/ask-mythbuntu','passwords'], questions)

    def ok_handler(self):
        #security key
        key = self.ui.get_key()
        self.preseed('mythtv/mysql_mythtv_user', key)

        Plugin.ok_handler(self)

    def cleanup(self):
        #Clear out our skipping if we did it only because of Master
        if 'UBIQUITY_AUTOMATIC' in os.environ and os.environ['UBIQUITY_AUTOMATIC'] == "2":
            del os.environ['UBIQUITY_AUTOMATIC']

        Plugin.cleanup(self)

class Install(InstallPlugin):
    def install(self, target, progress, *args, **kwargs):
        user = progress.get('passwd/username')
        type = progress.get('mythbuntu/install_type')
        key  = progress.get('mythtv/mysql_mythtv_user')

        #Create a .mythtv directory
        home_mythtv_dir = target + '/home/' + user + '/.mythtv'
        if not os.path.isdir(home_mythtv_dir):
            #in case someone made a symlink or file for the directory
            if os.path.islink(home_mythtv_dir) or os.path.exists(home_mythtv_dir):
                os.remove(home_mythtv_dir)
            os.makedirs(home_mythtv_dir)

        #Remove mysql.txt from home directory if it's there, then make one
        sql_txt= home_mythtv_dir + '/mysql.txt'
        if os.path.islink(sql_txt) or os.path.exists(sql_txt):
            os.remove(sql_txt)
        try:
            os.symlink('/etc/mythtv/mysql.txt', sql_txt)
        except OSError:
            #on a live disk there is a chance this was a broken link
            #depending on what the user did in the livefs
            pass

        #If they set a security key, try to copy it in
        try:
             import pwd
             home_dir = pwd.getpwnam("ubuntu").pw_dir
        except KeyError:
            home_dir = ''
        if key.isdigit() and os.path.exists(os.path.join(home_dir,'.mythtv','config.xml')):
            import shutil
            shutil.copy(os.path.join(home_dir,'.mythtv','config.xml'),
                        os.path.join(home_mythtv_dir,'config.xml'))

        #mythtv.desktop autostart
        if 'Frontend' in type:
            config_dir = target + '/home/' + user + '/.config'
            autostart_dir =  config_dir + '/autostart'
            autostart_link = autostart_dir + '/mythtv.desktop'
            if not os.path.isdir(config_dir):
                os.makedirs(config_dir)
            if not os.path.isdir(autostart_dir):
                os.makedirs(autostart_dir)
            elif os.path.islink(autostart_link) or os.path.exists(autostart_link):
                os.remove(autostart_link)
            try:
                os.symlink('/usr/share/applications/mythtv.desktop',autostart_link)
            except OSError:
                #on a live disk, this will appear a broken link, but it works
                pass

        #prep user for stuff we did
        install_misc.chrex(target, 'adduser', user, 'mythtv')
        install_misc.chrex(target, 'adduser', user, 'video')
        install_misc.chrex(target, 'chown','1000:1000','-R', os.path.join('/home',user))

        return InstallPlugin.install(self, target, progress, *args, **kwargs)
