# 
# generates C code to read/write XML data
#

from generator import *


def get_header(orig_file):
    text="""/* Automatically generated by autocode.py */

#include "%s"

#include <glib.h>
#include <string.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include "myx_xml_aux_functions.h"

"""%orig_file
    return text


class C_XMLCommon:
    def __init__(self, group):
        self._group= group
        self._indentation="  "
        
    def indent(self, s, depth=1):
        return self._indentation*depth+s

    def get_entity_name(self, variable):
        return variable["name"]
    
    def is_property(self, variable):
        return "as:property" in variable["options"]

    def make_subst_dict(self, variable, for_array, extra_dict={}):
        d={
        "group_name":self._group["name"],
        "entity":self.get_entity_name(variable),
        "content":["obj->%(name)s","obj->%(name)s[i]"][for_array!=0]%variable
        }
        d.update(extra_dict)
        return d

    def generate_code(self):
        code= self.generate()
        head= "\n".join(self.get_head())+"\n"
        body= code+"\n"
        return head, body


class C_XMLOutputGenerator(OutputGenerator,C_XMLCommon):
    """
    Supported options:
        
        element options:
            as:property
            
    """
    def __init__(self, group, others):
        OutputGenerator.__init__(self, group, others)
        C_XMLCommon.__init__(self, group)
        
        self._root_function= []
        self._aux_functions= []

    def process_int_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        if self.is_property(variable):
            lines= [
            'NewProp_int_content(node, NULL, "%(entity)s", %(content)s);'%d
            ]
        else:
            lines= [
            'NewTextChild_int_content(node, NULL, "%(entity)s", %(content)s);'%d
            ]

        return lines

    def process_str_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        if self.is_property(variable):
            lines= [
            'xmlNewProp(node, "%(entity)s", %(content)s);'%d
            ]
        else:
            lines= [
            'xmlNewTextChild(node, NULL, "%(entity)s", %(content)s);'%d
            ]            
        return lines

    def process_fp_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        if self.is_property(variable):
            lines= [
            'NewProp_double_content(node, NULL, "%(entity)s", %(content)s);'%d
            ]
        else:
            lines= [
            'NewTextChild_double_content(node, NULL, "%(entity)s", %(content)s);'%d
            ]
        return lines

    process_enum_variable= process_int_variable

    def process_struct_variable(self, variable, depth, for_array=0):
        # if this is just a reference, don't store it
        if "ref:parent" in variable["options"]:
            return []

        d= {
        "group_name":self._group["name"],
        "entity":self.find_struct_by_type(variable["type"])["entity"],
        "content":["obj->%(name)s","obj->%(name)s+i"][for_array]%variable
        }
        lines= [
        'store_%(group_name)s_%(entity)s(doc, node, %(content)s);'%d
        ]
        return lines

    def process_custom_variable(self, variable, depth, for_array=0):
        raise NotImplementedError
        return []

    def process_variable(self, variable, depth, for_array=0):
        lines= OutputGenerator.process_variable(self, variable, 0, for_array)

        ll= []
        for l in lines:
            ll.append(self.indent(l, depth))
        return ll

    def process_array(self, count, array, depth):
        lines=[
        self.indent("for (i= 0; i < obj->%(name)s; i++)"%count, depth),
        self.indent("{", depth)
        ]
        lines+= OutputGenerator.process_array(self, count, array, depth+1)
        lines+=[
        self.indent("}", depth)
        ]
        return lines


    def process_struct(self, struct, depth=0):
        d= {
        "group_name":self._group["name"],
        "entity":struct["entity"],
	"type":struct["typedef"]
        }

        lines=[]
        lines+=[
        "static int store_%(group_name)s_%(entity)s(xmlDocPtr doc, xmlNodePtr parent, %(type)s *obj)" % d,
        "{",
        self.indent("xmlNodePtr node;"),
        self.indent("unsigned int i;"),
        "",
        self.indent('node= xmlNewTextChild(parent, NULL, "%(entity)s", NULL);'%d),
        ""
        ]
        aa=OutputGenerator.process_struct(self, struct, depth)
        lines+= aa
        lines+=[
        self.indent("return 0;"),
        "}"
        ]

        self._head.append(lines[0]+";")

        tmp= "\n".join(lines)
        if "root" in struct["options"]:
            tmp+= "\n\n"+self.process_root_struct(struct)

        return tmp


    def process_root_struct(self, struct):
        d= {
        "group_name":self._group["name"],
        "entity":struct["entity"],
        "type":struct["typedef"]
        }
        self._prototypes.append("int myx_%(group_name)s_store(const char *filename, %(type)s *obj);"%d)

        tmpl="""
int myx_%(group_name)s_store(const char *filename, %(type)s *obj)
{
  xmlDocPtr doc;
  xmlNodePtr parent;

  g_return_val_if_fail(obj, -1);
  g_return_val_if_fail(filename, -1);

  doc = xmlNewDoc("1.0");

  parent = doc->children = xmlNewDocRawNode(doc, NULL, "%(group_name)s", NULL);

  if (xmlSaveFormatFile(filename, doc, 1) == -1)
  {
    xmlFreeDoc(doc);
    return -1;
  }

  xmlFreeDoc(doc);
  return 0;
}"""
        return tmpl % d



class C_XMLInputGenerator(InputGenerator,C_XMLCommon):
    def __init__(self, group, others):
        InputGenerator.__init__(self, group, others)
        C_XMLCommon.__init__(self, group)

        self._root_function= []
        self._aux_functions= []


    def process_int_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        if self.is_property(variable):
            lines= [
            '%(content)s= atoi_and_free(prop);'%d
            ]
        else:
            lines= [
            '%(content)s= atoi_and_free(xmlNodeListGetString(doc, cur->children, 1));'%d
            ]
        return lines
    
    def process_str_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        if self.is_property(variable):
            lines= [
            '%(content)s= prop;'%d
            ]
        else:
            lines= [
            '%(content)s= xmlNodeListGetString(doc, cur->children, 1);'%d
            ]            
        return lines

    def process_fp_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        if self.is_property(variable):
            lines= [
            '%(content)s= atof_and_free(prop);'%d
            ]
        else:
            lines= [
            '%(content)s= atof_and_free(xmlNodeListGetString(doc, cur->children, 1));'%d
            ]
        return lines

    
    process_enum_variable= process_int_variable
    
    def get_refparent_variable(self, struct):
        for s in struct["elements"]:
            if "ref:parent" in s["options"]:
                return s["name"]
        return None


    def process_struct_variable(self, variable, depth, for_array=0):
        type_struct=self.find_struct_by_type(variable["type"])
        d= {
        "group_name":self._group["name"],
        "entity":type_struct["entity"],
        "content":["obj->%(name)s","obj->%(name)s+i"][for_array]%variable
        }
        if "ref:parent" in variable["options"]:
            # in case of references, we don't do anything here and
            # instead leave the code in the containing struct to set
            # the reference for us (see [*])
            return []

        lines= [
        'read_%(group_name)s_%(entity)s(doc, cur, %(content)s);'%d
        ]

        # [*] check if this struct contains a variable that's a reference 
        # to this
        var= self.get_refparent_variable(type_struct)
        if var:
            d["refvar"]= var
            lines+=[
            "(%(content)s)->%(refvar)s= obj;"%d
            ]
        return lines

    def process_custom_variable(self, variable, depth, for_array=0):
        raise NotImplementedError
        return []

    def process_variable(self, variable, depth, for_array=0):
        lines= InputGenerator.process_variable(self, variable, 0, for_array)

        if lines:
            ll= []
            for l in lines:
                ll.append(self.indent(l, depth+1))
            return [
            self.indent('if (!xmlStrcmp(cur->name, "%s"))' % self.get_entity_name(variable), depth),
            self.indent(['{','{//'][for_array], depth) # the {// is a hack so that process_array can work ok
            ] + ll + [
            self.indent('}', depth)
            ]
        else:
            return []

    def process_array(self, count, array, depth):
        d=array.copy()
        d["count_name"]= count["name"]
        d["base_type"]= self.array_variable_to_plain(array)["type"]
        lines= InputGenerator.process_array(self, count, array, depth)
        for i in range(len(lines)):
            if lines[i].find("{//")>=0:
                # insert the array resize code in the placeholder
                lines[i:i+1]= [
                self.indent("{", depth),
                self.indent("i= obj->%(name)s; obj->%(name)s++;"%count, depth+1),
                self.indent("obj->%(name)s= g_realloc(obj->%(name)s, obj->%(count_name)s * sizeof(%(base_type)s));"%d, depth+1)
                ]

        return lines

    def process_struct(self, struct, depth=0): # override
        d= {
        "group_name":self._group["name"],
        "entity":struct["entity"],
        "type":struct["typedef"]
        }

        lines=[]
        lines+=[
        "static int read_%(group_name)s_%(entity)s(xmlDocPtr doc, xmlNodePtr node, %(type)s *obj)" % d,
        "{"
        ]
        self._head.append(lines[0]+";")
        lines_vars=[
        self.indent("xmlNodePtr cur;"),
        self.indent("unsigned int i;"),
	self.indent("memset(obj, 0, sizeof(%(type)s));"%d),
        ]

        arrays, variables= self.split_arrays(struct)

        lines_body=[]

        prop_flag=0
        elems=[]
        # get all properties
        for elem in variables:
            if "as:property" in elem["options"]:
                if not prop_flag:
                    prop_flag=1
                    lines_vars+=[self.indent("xmlChar *prop;")]

                lines_body+=[
                self.indent('prop= xmlGetProp(node, "%s");'%elem["name"]),
                self.indent('if (prop)'),
                self.indent('{')
                ]
                lines_body+= self.process_variable(elem, depth+2)
                lines_body+=[
                self.indent('xmlFree(prop);',depth+2),
                self.indent('}')
                ]
            else:
                elems.append(elem)

        lines_body.append("")

        # dump code to traverse list of nodes
        lines_body+= [
        self.indent("for (cur= node->xmlChildrenNode; cur != NULL; cur= cur->next)"),
        self.indent("{")
        ]
        for elem in elems:
            lines_body+= self.process_variable(elem, depth+2)
        for count,elem in arrays:
            lines_body+= self.process_array(count, elem, depth+2)
        lines_body+=[
        self.indent("}")
        ]

        lines+= lines_vars + lines_body
        lines+=[
        self.indent("return 0;"),
        "}"
        ]

        tmp= "\n".join(lines)

        if "root" in struct["options"]:
            tmp+= "\n\n"+self.process_root_struct(struct)

        return tmp


    def process_root_struct(self, struct):
        d= {
        "group_name":self._group["name"],
        "entity":struct["entity"],
        "type":struct["typedef"]
        }
        self._prototypes.append("%(type)s *myx_%(group_name)s_load(const char *filename, MYX_LIB_ERROR *error_code);"%d)

        tmpl="""
%(type)s *myx_%(group_name)s_load(const char *filename, MYX_LIB_ERROR *error_code)
{
  xmlDocPtr doc;
  xmlNodePtr parent;
  %(type)s *obj;

  *error_code = MYX_NO_ERROR;

  if (!file_exists(filename))  
  {
    *error_code = MYX_ERROR_CANT_OPEN_FILE;
    return NULL;
  }
  if (!(doc= xmlParseFile(filename)))
  {
    *error_code = MYX_XML_PARSE_ERROR;
    return NULL;
  }

  parent = xmlDocGetRootElement(doc);
  if (parent == NULL) 
  {
    *error_code = MYX_XML_EMPTY_DOCUMENT;
    xmlFreeDoc(doc);
    return NULL;
  }

  obj= g_new0(%(type)s, 1);

  if (read_%(group_name)s_%(entity)s(doc, parent, obj) < 0)
  {
    myx_%(group_name)s_free(obj);
    xmlFreeDoc(doc);
    return NULL;
  }

  xmlFreeDoc(doc);

  return obj;
}
"""
        return tmpl % d

    
    
class C_XMLDestroyerGenerator(DestroyerGenerator,C_XMLCommon):
    def __init__(self, group, others):
        DestroyerGenerator.__init__(self, group, others)
        C_XMLCommon.__init__(self, group)

    def process_int_variable(self, variable, depth, for_array=0):
        return []

    def process_str_variable(self, variable, depth, for_array=0):
        d= self.make_subst_dict(variable, for_array)
        lines=['if (%(content)s) xmlFree(%(content)s);'%d]
        return lines

    def process_fp_variable(self, variable, depth, for_array=0):
        return []

    def process_enum_variable(self, variable, depth, for_array=0):
        return []

    def process_struct_variable(self, variable, depth, for_array=0):
        type_struct=self.find_struct_by_type(variable["type"])
        d= {
        "group_name":self._group["name"],
        "entity":type_struct["entity"],
        "content":["obj->%(name)s","obj->%(name)s+i"][for_array]%variable
        }
        if "ref:parent" in variable["options"]:
            # in case of references, we don't do anything here and
            # instead leave the code in the containing struct to set
            # the reference for us (see [*])
            return []

        lines= [
        'free_%(group_name)s_%(entity)s(%(content)s);'%d
        ]
        return lines

    def process_custom_variable(self, variable, depth, for_array=0):
        raise NotImplementedError
        return []

    def process_variable(self, variable, depth, for_array=0):
        lines= DestroyerGenerator.process_variable(self, variable, 0, for_array)

        if lines:
            ll= []
            for l in lines:
                ll.append(self.indent(l, depth))
            lines=ll
        return lines

    def process_array(self, count, array, depth):
        lines=[
        self.indent("for (i= 0; i < obj->%(name)s; i++)"%count, depth),
        self.indent('{')
        ]
        lines+=DestroyerGenerator.process_array(self, count, array, depth+1)
        lines+=[
        self.indent('}'),
        self.indent('g_free(obj->%(name)s);'%array)
        ]
        return lines

    def process_struct(self, struct, depth=0): # override
        d= {
        "group_name":self._group["name"],
        "entity":struct["entity"],
        "type":struct["typedef"]
        }

        lines=[]
        if "root" in struct["options"]:
            lines+=[
            "int myx_%(group_name)s_free(%(type)s *obj)" % d,
            "{"
            ]
            self._prototypes.append(lines[0]+";")
        else:
            lines+=[
            "static int free_%(group_name)s_%(entity)s(%(type)s *obj)" % d,
            "{"
            ]

        self._head.append(lines[0]+";")

        arrays, variables= self.split_arrays(struct)

        if arrays:
            lines.append(self.indent("unsigned int i;"))

        for elem in variables:
            lines+= self.process_variable(elem, depth+1)
        for count,elem in arrays:
            lines+= self.process_array(count, elem, depth+1)

        lines+=[
        self.indent("return 0;"),
        "}"
        ]

        tmp= "\n".join(lines)

        return tmp
