/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */



#include "MGCellRenderer.h"


MGCellRendererTristate::MGCellRendererTristate(const ImageList &images)
:
  Glib::ObjectBase(typeid(MGCellRendererTristate)),
  Gtk::CellRenderer(),
  _image_list(images),
  _property_activatable(*this, "activatable", true),
  _property_state(*this, "state", OFF)
{
  property_mode()= Gtk::CELL_RENDERER_MODE_ACTIVATABLE;
  property_xpad()= 2;
  property_ypad()= 2;
}


MGCellRendererTristate::~MGCellRendererTristate()
{
}


Glib::PropertyProxy<bool> MGCellRendererTristate::property_activatable()
{
  return _property_activatable.get_proxy();
}


Glib::PropertyProxy<MGCellRendererTristate::State> MGCellRendererTristate::property_state()
{
  return _property_state.get_proxy();
}


SigC::Signal1<void, const Glib::ustring&>& MGCellRendererTristate::signal_toggled()
{
  return _signal_toggled;
}


void MGCellRendererTristate::get_size_vfunc(Gtk::Widget&,
                                            const Gdk::Rectangle* cell_area,
                                            int* x_offset, int* y_offset,
                                            int* width, int* height)
{
#define TOGGLE_WIDTH 12

  const int calc_width= property_xpad() * 2 + TOGGLE_WIDTH;
  const int calc_height= property_ypad() * 2 + TOGGLE_WIDTH;

  if (width)
    *width = calc_width;

  if (height)
    *height = calc_height;

  if (cell_area)
  {
    if (x_offset)
    {
      *x_offset= int(property_xalign() * (cell_area->get_width() - calc_width));
      *x_offset= std::max(0, *x_offset);
    }

    if (y_offset)
    {
      *y_offset= int(property_yalign() * (cell_area->get_height() - calc_height));
      *y_offset= std::max(0, *y_offset);
    }
  }
}


void MGCellRendererTristate::render_vfunc(const Glib::RefPtr<Gdk::Window>& window,
                                        Gtk::Widget& widget,
                                        const Gdk::Rectangle&,
                                        const Gdk::Rectangle& cell_area,
                                        const Gdk::Rectangle&,
                                        Gtk::CellRendererState flags)
{
  const unsigned int cell_xpad = property_xpad();
  const unsigned int cell_ypad = property_ypad();

  int x_offset = 0, y_offset = 0, width = 0, height = 0;
  get_size(widget, cell_area, x_offset, y_offset, width, height);
  
  width  -= cell_xpad * 2;
  height -= cell_ypad * 2;
  
  if (width <= 0 || height <= 0)
    return;

  Gtk::StateType state = Gtk::STATE_INSENSITIVE;

  if (_property_activatable)
    state = Gtk::STATE_NORMAL;
  
  if ((flags & Gtk::CELL_RENDERER_SELECTED) != 0)
    state = (widget.has_focus()) ? Gtk::STATE_SELECTED : Gtk::STATE_ACTIVE;
  
  
  Glib::RefPtr<Gdk::Pixmap> image;

  switch (_property_state)
  {
  case OFF: image= _image_list[0]; break;
  case MAYBE: image= _image_list[1]; break;
  case ON: image= _image_list[2]; break;
  default: g_assert_not_reached(); break;
  }

  int iw, ih;
  image->get_size(iw, ih);
  
  window->draw_drawable(widget.get_style()->get_fg_gc(state),
                        image, 
                        0, 0,
                        cell_area.get_x() + x_offset + cell_xpad + (width - iw)/2,
                        cell_area.get_y() + y_offset + cell_ypad + (height - ih)/2);
}


bool MGCellRendererTristate::activate_vfunc(GdkEvent*,
                                          Gtk::Widget&,
                                          const Glib::ustring& path,
                                          const Gdk::Rectangle&,
                                          const Gdk::Rectangle&,
                                          Gtk::CellRendererState)
{
  if (_property_activatable)
  {
    _signal_toggled(path);
    return true;
  }

  return false;
}

