unit Grt;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

interface

uses
  gnugettext, TntSystem,
  Windows, SysUtils, Classes, TntClasses, Forms, TntSysUtils, Messages,
  myx_public_interface,
  myx_grt_public_interface, myx_grt_builtin_module_public_interface,
  UniCodeConsole, SyncObjs, AuxFuncs;

type
  TGrt = class;
  TGrtEngine = class;

  GrtLoaderState =
  (
    GrtLsNotInitialized,
    GrtLsInitialized,
    GrtLsInitializeFailed,
    GrtLsModulesLoaded
  );

  GrtValueType =
  (
    GrtAnyValue = Ord(MYX_ANY_VALUE),
    GrtIntValue = Ord(MYX_INT_VALUE),
    GrtRealValue = Ord(MYX_REAL_VALUE),
    GrtStringValue = Ord(MYX_STRING_VALUE),
    GrtListValue = Ord(MYX_LIST_VALUE),
    GrtDictValue = Ord(MYX_DICT_VALUE)
  );

  GrtValueCallbackReason =
  (
    GrtVcrDelete = Ord(MYX_GVCR_DELETE),
    GrtVcrDictItemChange = Ord(MYX_GVCR_DICT_ITEM_CHANGE)
  );

  TGrtProcessOutput = procedure(Text: WideString) of object;
  TGrtProcessMessages = procedure(Messages: TMYX_GRT_MSGS) of object;
  TGrtPerformInput = function(Description: WideString): WideString of object;
  TGrtPerformStatusQuery = function: integer of object;
  TGrtOnShutDown = procedure(Grt: TGrt) of object;
  //TGrtModuleExecuter = function(ModuleName: WideString; FunctionName: WideString; arguments: Pointer): Pointer of object;

  IGrtValueCallbackListener = interface
    function ValueChange(Value: Pointer; Reason: GrtValueCallbackReason): Integer;
  end;

  IGrtModuleImplementor = interface
    function GrtExecuteModuleFunction(ModuleName: WideString; FunctionName: WideString; arguments: Pointer): Pointer;
  end;

  TGrt = class
  protected
    GrtSyncEvent: TEvent;
    FGrtEngine: TGrtEngine;    

    function CreateGrtEngine: TGrtEngine; virtual;

    function GetJavaLoaderState: GrtLoaderState;
    function GetLuaLoaderState: GrtLoaderState;
    function GetBuiltinLoaderState: GrtLoaderState;
    function GetPhpLoaderState: GrtLoaderState;
    function GetUiLoaderState: GrtLoaderState;

    procedure OutputCommandLine(Text: WideString);

    function GrtPGrt: Pointer;

    function GetGlobal(const Path: WideString): Pointer;
    procedure SetGlobal(const Path: WideString; NewValue: Pointer);
    function GetGlobalAsString(const Path: WideString): WideString;
    procedure SetGlobalAsString(const Path: WideString; NewValue: WideString);
    function GetGlobalAsInt(const Path: WideString): Integer;
    procedure SetGlobalAsInt(const Path: WideString; NewValue: Integer);
    function GetGlobalAsReal(const Path: WideString): Double;
    procedure SetGlobalAsReal(const Path: WideString; NewValue: Double);
    function GetGlobalRef(const Path: WideString): Pointer;
    procedure SetGlobalRef(const Path: WideString; NewValue: Pointer);

    function GetListItem(const List: Pointer; const I: Integer): Pointer;
    procedure SetListItem(const List: Pointer; const I: Integer; NewValue: Pointer);
    function GetListString(const List: Pointer; const I: Integer): WideString;
    procedure SetListString(const List: Pointer; const I: Integer; NewValue: WideString);
    function GetListRefItem(const List: Pointer; const I: Integer): Pointer;
    procedure SetListRefItem(const List: Pointer; const I: Integer; NewValue: Pointer);
    function GetListItemByObjectName(const List: Pointer; const Name: WideString): Pointer;
    function GetListRefValueByObjectName(const List: Pointer; const Name: WideString): Pointer;

    function GetDictItem(const Dict: Pointer; const Key: WideString): Pointer;
    procedure SetDictItem(const Dict: Pointer; const Key: WideString; NewValue: Pointer);
    function GetDictString(const Dict: Pointer; const Key: WideString): WideString;
    procedure SetDictString(const Dict: Pointer; const Key: WideString; NewValue: WideString);
    function GetDictInt(const Dict: Pointer; const Key: WideString): Integer;
    procedure SetDictInt(const Dict: Pointer; const Key: WideString; NewValue: Integer);
    function GetDictReal(const Dict: Pointer; const Key: WideString): Double;
    procedure SetDictReal(const Dict: Pointer; const Key: WideString; NewValue: Double);
    function GetDictRef(const Dict: Pointer; const Key: WideString): Pointer;
    procedure SetDictRef(const Dict: Pointer; const Key: WideString; NewValue: Pointer);
    function GetDictKey(const Dict: Pointer; const I: Integer): WideString;

    function GetDictStructName(const Dict: Pointer): WideString;
    procedure SetDictStructName(const Dict: Pointer; StructName: WideString);

    function GetValueInt(const Value: Pointer): Integer;
    function GetValueString(const Value: Pointer): WideString;

    function GetOutputBuffer: WideString;
    procedure SetOutputBuffer(OutputBuffer: WideString);

    function GetShellVar(Name: WideString): Pointer;
    procedure SetShellVar(Name: WideString; Value: Pointer);
  private
    FOnShutDown: TGrtOnShutDown;
    FStructIconsList: TTntStringList;

    FConsole: TUniCodeConsole;

    FVerbose: Boolean;
    FRemoteDebug: Boolean;
    FJvmLibrary: WideString;
  public
    constructor Create(Console: TUniCodeConsole);
    destructor Destroy; override;

    property PGrt: Pointer read GrtPGrt;
    property Console: TUniCodeConsole read FConsole write FConsole;

    property OnShutDown: TGrtOnShutDown read FOnShutDown write FOnShutDown;

    // Options
    property Verbose: Boolean read FVerbose write FVerbose;
    property RemoteDebug: Boolean read FRemoteDebug write FRemoteDebug;
    property JvmLibrary: WideString read FJvmLibrary write FJvmLibrary;

    // Loaders
    property JavaLoaderState: GrtLoaderState read GetJavaLoaderState;
    property LuaLoaderState: GrtLoaderState read GetLuaLoaderState;
    property BuiltinLoaderState: GrtLoaderState read GetBuiltinLoaderState;
    property PhpLoaderState: GrtLoaderState read GetPhpLoaderState;
    property UiLoaderState: GrtLoaderState read GetUiLoaderState;

    // Access values
    property Global[const Path: WideString]: Pointer read GetGlobal write SetGlobal;
    property GlobalAsString[const Path: WideString]: WideString read GetGlobalAsString write SetGlobalAsString;
    property GlobalAsInt[const Path: WideString]: Integer read GetGlobalAsInt write SetGlobalAsInt;
    property GlobalAsReal[const Path: WideString]: Double read GetGlobalAsReal write SetGlobalAsReal;
    property GlobalRef[const Path: WideString]: Pointer read GetGlobalRef write SetGlobalRef;
    procedure GlobalDel(const Path: WideString; const Key: WideString);

    // Access lists
    property ListItem[const List: Pointer; const I: Integer]: Pointer read GetListItem write SetListItem;
    property ListString[const List: Pointer; const I: Integer]: WideString read GetListString write SetListString;
    property ListRefItem[const List: Pointer; const I: Integer]: Pointer read GetListRefItem write SetListRefItem;
    property ListItemByObjectName[const List: Pointer; const Name: WideString]: Pointer read GetListItemByObjectName;
    property ListRefValueByObjectName[const List: Pointer; const Name: WideString]: Pointer read GetListRefValueByObjectName;

    function ListCount(List: Pointer): Integer;
    procedure ListInsert(List: Pointer; Index: Integer; Value: Pointer; IncreaseRefCount: Boolean = True);
    procedure ListAdd(List: Pointer; Value: Pointer; IncreaseRefCount: Boolean = True);
    procedure ListAddString(List: Pointer; Name: WideString);
    function ListDel(List: Pointer; Index: Integer): Boolean; overload;
    function ListDel(List: Pointer; Value: Pointer): Boolean; overload;
    function ListDelString(List: Pointer; Name: WideString): Boolean;
    function ListDelObject(List: Pointer; Name: WideString): Boolean;
    function ListNew(const ContentType: GrtValueType; const ContentStructName: WideString = ''): Pointer;
    procedure ListClear(List: Pointer);

    // Access dicts
    property DictItem[const Dict: Pointer; const Key: WideString]: Pointer read GetDictItem write SetDictItem;
    property DictString[const Dict: Pointer; const Key: WideString]: WideString read GetDictString write SetDictString;
    property DictInt[const Dict: Pointer; const Key: WideString]: Integer read GetDictInt write SetDictInt;
    property DictReal[const Dict: Pointer; const Key: WideString]: Double read GetDictReal write SetDictReal;
    property DictRef[const Dict: Pointer; const Key: WideString]: Pointer read GetDictRef write SetDictRef;
    property DictStructName[const Dict: Pointer]: WideString read GetDictStructName write SetDictStructName;
    property DictKey[const Dict: Pointer; const I: Integer]: WideString read GetDictKey;
    function DictNew(const StructName: WideString): Pointer;
    function DictNewTyped(const ContentType: GrtValueType;
      const ContentStructName: WideString): Pointer;
    function DictItemCount(const Dict: Pointer): Integer;
    procedure DictDel(const Dict: Pointer; const Key: WideString);

    function ObjectNew(const StructName: WideString;
      Name: WideString; Id: WideString; OwnerId: WideString): Pointer;

    // Simple Values
    property ValueInt[const Value: Pointer]: Integer read GetValueInt;
    property ValueString[const Value: Pointer]: WideString read GetValueString;
    function ValueFromString(S: WideString): Pointer;
    function ValueFromInt(I: Integer): Pointer;

    procedure ValueRetain(Value: Pointer);
    procedure ValueRelease(Value: Pointer);
    function ValueType(Value: Pointer): GrtValueType;
    function ValueDuplicate(Value: Pointer): Pointer;
    function ValueLoadFromFile(FileName: WideString): Pointer;
    procedure ValueSaveToFile(FileName: WideString; Value: Pointer);
    procedure ValueListenerAdd(Value: Pointer; CallbackObject: IGrtValueCallbackListener);
    procedure ValueListenerRemove(Value: Pointer; CallbackObject: IGrtValueCallbackListener);
    function ValueAsLuaCode(Value: Pointer; Indention: Integer): WideString;
    function ValueDiffMake(Source: Pointer; Target: Pointer): Pointer;
    function ValueDiffApply(Value: Pointer; Diff: Pointer): Pointer;
    function ValueReference(Value: Pointer): Pointer; overload;
    function ValueReference(Id: WideString): Pointer; overload;

    function GlobalSetRoot(Value: Pointer): Integer;

    // Structs / Members

    function GetStructCaption(StructName: WideString): WideString;
    function GetStructMemberCount(StructName: WideString): Integer;
    function GetStructMemberName(StructName: WideString; Index: Integer): WideString;
    function GetStructMemberType(StructName: WideString; Index: Integer): GrtValueType;
    function GetStructMemberStructName(StructName: WideString; MemberName: WideString): WideString;
    function GetStructMemberContentType(StructName: WideString; Index: Integer): GrtValueType;
    function GetStructMemberContentStructName(StructName: WideString; MemberName: WideString): WideString; overload;
    function GetStructMemberContentStructName(StructName: WideString; Index: Integer): WideString; overload;
    function StructExists(StructName: WideString): Boolean;
    function StructInheritsFrom(StructName: WideString; ParentStructName: WideString): Boolean;

    // Functions
    function ExecuteModalFunction(ModulName: WideString;
      FunctionName: WideString;
      FunctionArguments: array of const;
      ProcessOutputFunction: TGrtProcessOutput = nil;
      ProcessMessagesFunction: TGrtProcessMessages = nil;
      AllowNullAsResult: Boolean = False;
      SearchParent: Boolean = True;
      TimeOutMS: Integer = -1;
      PerformInputFunction: TGrtPerformInput = nil;
      PerformStatusQueryFunction: TGrtPerformStatusQuery = nil): Pointer;
    procedure ExecuteModalProcedure(ModulName: WideString;
      FunctionName: WideString;
      FunctionArguments: array of const;
      ProcessOutputFunction: TGrtProcessOutput = nil;
      ProcessMessagesFunction: TGrtProcessMessages = nil;
      AllowNullAsResult: Boolean = True;
      SearchParent: Boolean = True;
      TimeOutMS: Integer = -1;
      PerformInputFunction: TGrtPerformInput = nil;
      PerformStatusQueryFunction: TGrtPerformStatusQuery = nil);

    function GetGlobalAsParam(Path: WideString): WideString;

    function ExecuteModalShellFunction(Cmd: WideString;
      ProcessOutputFunction: TGrtProcessOutput = nil;
      TimeOutMS: Integer = -1): MYX_GRT_SHELL_COMMAND;
    procedure ExecuteModalScript(FileName: WideString;
      ProcessOutputFunction: TGrtProcessOutput = nil;
      TimeOutMS: Integer = -1);

    procedure AddDelphiModule(ModuleName: WideString; ModuleFunctions: WideString;
      Implementor: IGrtModuleImplementor);
    procedure RemoveDelphiModule(ModuleName: WideString;
      Implementor: IGrtModuleImplementor);

    function FunctionSuccess(Value: Pointer): Pointer;
    function FunctionError(ErrorString: WideString; ErrorDetails: WideString = ''): Pointer;

    property OutputBuffer: WideString read GetOutputBuffer write SetOutputBuffer;

    // Caches
    property StructIconsList: TTntStringList read FStructIconsList;

    // Helper functions
    function BuildGrtParamList(Params: array of const): Pointer;
    function FormatGrtMessagesAsString(Msgs: TMYX_GRT_MSGS): WideString;

    procedure ScriptInputRead;

    property ShellVar[Name: WideString]: Pointer read GetShellVar write SetShellVar;
  end;

  PCallbackEntry = ^TCallbackEntry;
  TCallbackEntry = record
    Type_: Integer;
    TextForOutput: WideString;
    Messages: TMYX_GRT_MSGS;
  end;

  TGrtEngine = class(TThread)
    constructor Create(CreateSuspended: Boolean; Grt: TGrt); virtual;
    destructor Destroy; override;
  protected
    FPGrt: Pointer;

    function ApplicationHook(var Message: TMessage): Boolean; virtual;
    procedure Execute; override;

    procedure InitializeGrt;
    procedure InitializeLoadersAndModules; virtual;

    procedure FinalizeGrt;
    procedure FinalizeLoadersAndModules; virtual;

    function GetPGrt: Pointer;

    function GetExecutionFinished: Boolean;
    function GetExecutionTime: TDateTime;

    procedure DoOutputText(S: WideString);
    procedure DoProcessMessages(Messages: TMYX_GRT_MSGS);
    procedure DoGrtInputText;
    procedure DoStatusQuery;

    function InitDelphiLoader(Error: PMYX_GRT_ERROR): Pointer;

    procedure ExecuteDelphiFunction;

    procedure OutputModuleStatus(LoaderName: WideString; LoadedModuleCount: Integer; Error: MYX_GRT_ERROR);
  private
    FGrt: TGrt;

    FFunctionSynchronizer,
    FVclDataSynchronizer: TCriticalSection;
    FFunctionStartedEvent,
    FFunctionFinishedEvent: TEvent;
    FExecutionFinished: Boolean;
    FModuleName,
    FFunctionName: WideString;
    FSearchParent: Boolean;
    FFunctionArgument: Pointer;
    FPError: PMYX_GRT_ERROR;
    FResult: Pointer;
    FShellCmd: WideString;
    FShellResult: MYX_GRT_SHELL_COMMAND;
    FStartTime,
    FEndTime: TDateTime;

    FDelphiModuleName,
    FDelphiModuleFunctionName: WideString;
    FDelphiModuleImplementor: IGrtModuleImplementor;
    FDelphiModuleFunctionArgument,
    FDelphiModuleFunctionResult: Pointer;

    FProcessOutputFunction: TGrtProcessOutput;
    FTextForOutput: WideString;
    FStatus: Integer;

    FProcessMessagesFunction: TGrtProcessMessages;

    FPerformInputFunction: TGrtPerformInput;
    FPerformStatusQueryFunction: TGrtPerformStatusQuery;
    FTextInput: WideString;

    FJavaLoaderState,
    FLuaLoaderState,
    FBuiltinLoaderState,
    FPhpLoaderState,
    FUiLoaderState: GrtLoaderState;
    FDelphiGrtMessages: TMYX_GRT_MSGS;
    FInputBuffer: Array [0 .. 160] of Char;
    FOutputBuffer: WideString;
    FNativeLoader: Pointer;
    FCallbackEntries: TThreadList;

    procedure ClearCallbackEntries;
    procedure AddCallbackEntry(AType: Integer; OutputText: WideString; Messages: PMYX_GRT_MSGS);
    procedure DoChangeCallback;
  public
    property PGrt: Pointer read GetPGrt;
    property ExecutionFinished: Boolean read GetExecutionFinished;
    property ExecutionTime: TDateTime read GetExecutionTime;
    property OutputBuffer: WideString read FOutputBuffer write FOutputBuffer;

    property DelphiGrtMessages: TMYX_GRT_MSGS read FDelphiGrtMessages;

    function ExecuteModalFunction(ModulName: WideString;
      FunctionName: WideString;
      FunctionArgument: Pointer; Error: PMYX_GRT_ERROR;
      ProcessOutputFunction: TGrtProcessOutput;
      ProcessMessagesFunction: TGrtProcessMessages;
      SearchParent: Boolean;
      TimeOutMS: Integer = -1;
      PerformInputFunction: TGrtPerformInput = nil;
      PerformStatusQueryFunction: TGrtPerformStatusQuery = nil): Pointer;
    function ExecuteModalShellFunction(Cmd: WideString;
      ProcessOutputFunction: TGrtProcessOutput;
      TimeOutMS: Integer = -1): MYX_GRT_SHELL_COMMAND;

    procedure OutputText(S: WideString);
    function ProcessMessages(PMessages: PMYX_GRT_MSGS): Integer;
    function GrtInputText(Caption: WideString; options: MYX_GRT_INPUT_OPTIONS; Text: PPointer): integer;
    function GrtStatusQuery: Integer;

    procedure AddDelphiModule(ModuleName: WideString; ModuleFunctions: WideString;
      Implementor: IGrtModuleImplementor);
    procedure RemoveDelphiModule(ModuleName: WideString;
      Implementor: IGrtModuleImplementor);
  end;

  EGrtError = class(Exception)
  private
    FErrorNumber: Integer;
    FDescription: WideString;
  public
    constructor Create(ErrorNumber: Integer; Description: WideString);

    property ErrorNumber: Integer read FErrorNumber;
    property Description: WideString read FDescription;
  end;

function GetListMemberCount(Grt: Pointer; StructName: WideString; OnlyCheck: Boolean): Integer;
function GetListMember(Grt: Pointer; StructName: WideString; Index: Integer): Pointer;
function FormatGrtMessagesAsString(Msgs: TMYX_GRT_MSGS): WideString;
function GetGrtFunctionArgumentAsString(Argument: Pointer): WideString;

function RuntimeEnvironment: TGrt;

function GetRuntimeEnvironment: TGrt;
procedure SetRuntimeEnvironment(Grt: TGrt);

procedure InitThreads;

//----------------------------------------------------------------------------------------------------------------------

implementation

var
  InternalGrtInstance: TGrt;

const
  WM_PROCESS_CALLBACK = WM_APP + 10001;

//----------------------------------------------------------------------------------------------------------------------

procedure InitThreads;

begin
  myx_grt_init_threads();
end;

//----------------------------------------------------------------------------------------------------------------------

function RuntimeEnvironment: TGrt;

begin
  if (InternalGrtInstance = nil) then
    InternalGrtInstance := TGrt.Create(nil);

  Result := InternalGrtInstance;
end;

// -----------------------------------------------------------------------------

function GetRuntimeEnvironment: TGrt;

begin
  Result := InternalGrtInstance;
end;

// -----------------------------------------------------------------------------

procedure SetRuntimeEnvironment(Grt: TGrt);

begin
  InternalGrtInstance := Grt;
end;

// -----------------------------------------------------------------------------

function GrtValueCallback(grt: Pointer; value: Pointer; reason: MYX_GRT_VALUE_CALLBACK_REASON; user_data: Pointer): Integer; cdecl;

var
  Obj: IGrtValueCallbackListener;

begin
  Obj := IGrtValueCallbackListener(user_data);

  Result := Obj.ValueChange(value, GrtValueCallbackReason(reason));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure ProcessGrtOutput(text: PChar; userdata: Pointer) cdecl;

var
  GrtEngine: TGrtEngine;
  S: WideString;

begin
  GrtEngine := userdata;
  S := UTF8Decode(text);

  GrtEngine.OutputText(S);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure ProcessGrtMessages(PMessages: PMYX_GRT_MSGS; userdata: Pointer) cdecl;

var
  GrtEngine: TGrtEngine;

begin
  GrtEngine := userdata;

  GrtEngine.ProcessMessages(PMessages);
end;

//----------------------------------------------------------------------------------------------------------------------

function ProcessInput(caption: PChar; options: MYX_GRT_INPUT_OPTIONS; text: PPointer; user_data: Pointer): Integer cdecl;

var
  GrtEngine: TGrtEngine;
  S: WideString;

begin
  GrtEngine := user_data;
  S := UTF8Decode(caption);

  Result := GrtEngine.GrtInputText(S, options, text);
end;

//----------------------------------------------------------------------------------------------------------------------

function ProcessStatusQuery(user_data: Pointer): Integer cdecl;

var
  GrtEngine: TGrtEngine;

begin
  GrtEngine := user_data;

  Result := GrtEngine.GrtStatusQuery();
end;

//----------------------------------------------------------------------------------------------------------------------

constructor TGrt.Create(Console: TUniCodeConsole);

var
  TimeStart: TDateTime;
  TimeOut: Boolean;
  TimeOutInterval: TDateTime;
  WaitResult: Cardinal;
  WaitHandle: THandle;
  I: Integer;
  InitScriptFile: WideString;

begin
  inherited Create;

  FStructIconsList := TTntStringList.Create;

  FOnShutDown := nil;

  I := 1;

  // get startup arguments from command line
  Verbose := False;
  RemoteDebug := False;
  JvmLibrary := '';
  InitScriptFile := '';

  while (I <= WideParamCount) do
  begin
    if (WideSameText(WideParamStr(I), '-verbose')) then
      FVerbose := True
    else
      if (WideSameText(WideParamStr(I), '-debug')) then
        FRemoteDebug := True
      else
        if (WideSameText(WideParamStr(i), '-jvm')) and
          (i < WideParamCount) then
        begin
          FJvmLibrary := WideParamStr(i + 1);
          inc(i);
        end
        else
          if (WideSameText(WideParamStr(i), '-initscript')) and (i < ParamCount) then
          begin
            InitScriptFile := WideParamStr(i + 1);
            inc(i);
          end;

    inc(I);
  end;

  FConsole := Console;

  GrtSyncEvent := TEvent.Create(nil, False, False, '');

  FGrtEngine := CreateGrtEngine;
  FGrtEngine.FreeOnTerminate := False;

  // Wait for 20 sek
  TimeOut := False;
  TimeStart := Now;
  TimeOutInterval := (1 / 86400) * 20;
  WaitHandle := GrtSyncEvent.Handle;

  while (not (TimeOut)) do
  begin
    // Wait for the initialisation to finish but every
    // 100 milliseonds check if the timeout value has been reached.
    // Process any incomming message while we wait.
    WaitResult := MsgWaitForMultipleObjects(
      1, WaitHandle, false, 100, QS_ALLEVENTS);
    if WaitResult = WAIT_OBJECT_0 then
      Break;

    Application.ProcessMessages;

    if (Now - TimeStart > TimeOutInterval) then
      TimeOut := True;
  end;

  if (TimeOut) then
    raise Exception.Create(_('Could not initialize the GRT Environment. ' +
      'A timeout occured during the initalization.'));

  GrtSyncEvent.Free;
  GrtSyncEvent := nil;

  if (InitScriptFile <> '') and (FGrtEngine.PGrt <> nil) then
    myx_grt_lua_shell_run_file(FGrtEngine.PGrt, InitScriptFile, 1);
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TGrt.Destroy;

var
  I: Integer;

begin
  if (Assigned(FOnShutDown)) then
    FOnShutDown(self);

  //myx_grt_value_release(myx_grt_get_root(PGrt));

  // Free FGrtStructIconsList
  for I := 0 to FStructIconsList.Count - 1 do
    FStructIconsList.Objects[I].Free;
  FStructIconsList.Free;

  FGrtEngine.Terminate;
  FGrtEngine.WaitFor;
  FGrtEngine.Free;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.CreateGrtEngine: TGrtEngine;

begin
  Result := TGrtEngine.Create(False, self);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetJavaLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FJavaLoaderState;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetLuaLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FLuaLoaderState;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetBuiltinLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FBuiltinLoaderState;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetPhpLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FUiLoaderState;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetUiLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FUiLoaderState;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.OutputCommandLine(Text: WideString);

begin
  if Assigned(FConsole) then
    FConsole.AddOutput(Text + #13#10);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GrtPGrt: Pointer;

begin
  Result := FGrtEngine.PGrt;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ExecuteModalFunction(ModulName: WideString; FunctionName: WideString; FunctionArguments: array of const;
  ProcessOutputFunction: TGrtProcessOutput; ProcessMessagesFunction: TGrtProcessMessages; AllowNullAsResult: Boolean;
  SearchParent: Boolean; TimeOutMS: Integer;
  PerformInputFunction: TGrtPerformInput;
  PerformStatusQueryFunction: TGrtPerformStatusQuery): Pointer;

var
  Args,
  ResultWrapper: Pointer;
  Error: MYX_GRT_ERROR;
  FunctionErrorString: WideString;

begin
  Result := nil;

  Args := BuildGrtParamList(FunctionArguments);

  //Execute the function
  ResultWrapper := FGrtEngine.ExecuteModalFunction(ModulName,
    FunctionName,
    Args, @Error,
    ProcessOutputFunction,
    ProcessMessagesFunction,
    SearchParent,
    TimeOutMS,
    PerformInputFunction,
    PerformStatusQueryFunction);

  // Free argument list
  ValueRelease(Args);

  //Check the result for error keys
  FunctionErrorString := myx_grt_function_check_error(ResultWrapper,
    Ord(AllowNullAsResult));

  //Show error
  if (Error <> MYX_GRT_NO_ERROR) or
    (FunctionErrorString <> '') then
  begin
    if (ResultWrapper <> nil) then
      myx_grt_value_release(ResultWrapper);

    raise EGrtError.Create(Ord(Error),
      ModulName + '.' + FunctionName + ' :' + FunctionErrorString);
  end;

  //If there was no error we have a value key
  //Return the contents of the key as result
  if (ResultWrapper <> nil) then
  begin
    // retain real result value
    Result := myx_grt_dict_item_get_value(ResultWrapper, 'value');
    myx_grt_value_retain(Result);

    // release result wrapper dict
    myx_grt_value_release(ResultWrapper);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ExecuteModalProcedure(ModulName: WideString; FunctionName: WideString; FunctionArguments: array of const;
  ProcessOutputFunction: TGrtProcessOutput; ProcessMessagesFunction: TGrtProcessMessages; AllowNullAsResult: Boolean;
  SearchParent: Boolean; TimeOutMS: Integer;
  PerformInputFunction: TGrtPerformInput;
  PerformStatusQueryFunction: TGrtPerformStatusQuery);

var
  Res: Pointer;

begin
  Res := ExecuteModalFunction(ModulName, FunctionName, FunctionArguments,
    ProcessOutputFunction, ProcessMessagesFunction, AllowNullAsResult,
    SearchParent, TimeOutMS,
    PerformInputFunction,
    PerformStatusQueryFunction);

  if (Res <> nil) then
    myx_grt_value_release(Res);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ExecuteModalShellFunction(Cmd: WideString; ProcessOutputFunction: TGrtProcessOutput;
  TimeOutMS: Integer): MYX_GRT_SHELL_COMMAND;

begin
  Result := FGrtEngine.ExecuteModalShellFunction(Cmd, ProcessOutputFunction, TimeOutMS);
end;

procedure TGrt.ExecuteModalScript(FileName: WideString;
  ProcessOutputFunction: TGrtProcessOutput = nil;
  TimeOutMS: Integer = -1);

var
  Script: TTntStringList;

begin
  if (WideFileExists(FileName)) then
  begin
    Script := TTntStringList.Create;
    try
      Script.LoadFromFile(FileName);

      FGrtEngine.ExecuteModalShellFunction(Script.Text,
        ProcessOutputFunction, TimeOutMS);
    finally
      Script.Free;
    end;
  end
  else
    ShowModalDialog(_('Script file not found'),
      Format(_('The script file %s cannot be found.'), [FileName]),
      myx_mtError);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.BuildGrtParamList(Params: array of const): Pointer;

var
  ParamList: Pointer;
  Param: Pointer;
  I: Integer;

begin
  ParamList := myx_grt_list_new(MYX_ANY_VALUE, '');

  for I := 0 to High(Params) do
  begin
    with Params[I] do
      case VType of
        vtInteger:
          Param := myx_grt_value_from_int(VInteger);
        vtExtended:
          Param := myx_grt_value_from_real(VExtended^);
        vtWideString:
          Param := myx_grt_value_from_string(WideString(VWideString));
        vtString:
          Param := myx_grt_value_from_string(VString^);
        vtAnsiString:
          Param := myx_grt_value_from_string(string(VAnsiString));
        vtChar:
          Param := myx_grt_value_from_string(VChar);
        vtPChar:
          Param := myx_grt_value_from_string(VPChar);
        vtPointer:
          Param := VPointer;
      else
        raise EInOutError.Create(_('BuildGrtParamList called with unsupported parameter type.'));
      end;

    myx_grt_list_item_add(ParamList, Param);

    if (Params[I].VType <> vtPointer) then
      ValueRelease(Param);
  end;

  Result := ParamList;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.AddDelphiModule(ModuleName: WideString; ModuleFunctions: WideString;
  Implementor: IGrtModuleImplementor);

begin
  FGrtEngine.AddDelphiModule(ModuleName, ModuleFunctions, Implementor);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.RemoveDelphiModule(ModuleName: WideString;
  Implementor: IGrtModuleImplementor);

begin
  FGrtEngine.RemoveDelphiModule(ModuleName, Implementor);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.FunctionSuccess(Value: Pointer): Pointer;

begin
  Result := DictNew('');

  DictItem[Result, 'value'] := Value;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.FunctionError(ErrorString: WideString; ErrorDetails: WideString): Pointer;

begin
  Result := DictNew('');

  DictItem[Result, 'error'] := ValueFromString(ErrorString);
  if (ErrorDetails <> '') then
    DictItem[Result, 'detail'] := ValueFromString(ErrorDetails);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetGlobal(const Path: WideString): Pointer;

begin
  Result := myx_grt_dict_item_get_by_path(PGrt, myx_grt_get_root(PGrt), Path);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetGlobal(const Path: WideString; NewValue: Pointer);

begin
  if (myx_grt_dict_item_set_by_path(myx_grt_get_root(PGrt), Path, NewValue) <> 0) then
    raise Exception.Create(Format(_('The value %s cannot be set.'), [Path]));
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetGlobalAsString(const Path: WideString): WideString;

var
  PValue: Pointer;

begin
  PValue := Global[Path];

  if (PValue <> nil) then
  begin
    if (myx_grt_value_get_type(PValue) <> MYX_STRING_VALUE) then
      raise Exception.Create(Format(
        _('The value %s is not a string value.'),
        [Path]));

    Result := myx_grt_value_as_string(PValue);
  end
  else
    Result := '';
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetGlobalAsString(const Path: WideString; NewValue: WideString);

begin
  Global[Path] := myx_grt_value_from_string(NewValue);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetGlobalAsInt(const Path: WideString): Integer;

var
  PValue: Pointer;

begin
  PValue := Global[Path];

  if (PValue <> nil) then
  begin
    if (myx_grt_value_get_type(PValue) <> MYX_INT_VALUE) then
      raise Exception.Create(Format(
        _('The value %s is not a int value.'),
        [Path]));

    Result := myx_grt_value_as_int(PValue);
  end
  else
    Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetGlobalAsInt(const Path: WideString; NewValue: Integer);

begin
  Global[Path] := myx_grt_value_from_int(NewValue);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetGlobalAsReal(const Path: WideString): Double;

var
  PValue: Pointer;

begin
  PValue := Global[Path];

  if (PValue <> nil) then
  begin
    if (myx_grt_value_get_type(PValue) <> MYX_INT_VALUE) then
      raise Exception.Create(Format(
        _('The value %s is not a real value.'),
        [Path]));

    Result := myx_grt_value_as_real(PValue);
  end
  else
    Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetGlobalAsReal(const Path: WideString; NewValue: Double);

begin
  Global[Path] := myx_grt_value_from_real(NewValue);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetGlobalRef(const Path: WideString): Pointer;

var
  Id: WideString;

begin
  Result := nil;

  Id := GetGlobalAsString(Path);
  if (Id <> '') then
    Result := myx_grt_reference_cache_lookup(PGrt, Id);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetGlobalRef(const Path: WideString; NewValue: Pointer);

begin
  Global[Path] := DictItem[NewValue, '_id'];
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.GlobalDel(const Path: WideString; const Key: WideString);

var
  Value: Pointer;

begin
  Value := myx_grt_dict_item_get_by_path(PGrt, myx_grt_get_root(PGrt), Path);

  if (ValueType(Value) = GrtDictValue) then
    myx_grt_dict_item_del(Value, Key)
  else
    if (ValueType(Value) = GrtListValue) then
      myx_grt_list_item_del(Value, StrToInt(Key));
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueType(Value: Pointer): GrtValueType;

begin
  Result := GrtValueType(myx_grt_value_get_type(Value));
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueDuplicate(Value: Pointer): Pointer;

begin
  Result := myx_grt_value_dup(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueLoadFromFile(FileName: WideString): Pointer;

begin
  Result := myx_grt_retrieve_from_file(PGrt, FileName);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ValueSaveToFile(FileName: WideString; Value: Pointer);

var
  Error: MYX_GRT_ERROR;

begin
  Error := myx_grt_store_to_file(PGrt, Value, FileName);

  if (Error <> MYX_GRT_NO_ERROR) then
    raise Exception.CreateFmt(
      _('The following error occured. Error Nr. %d'), [Ord(Error)]);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ValueListenerAdd(Value: Pointer; CallbackObject: IGrtValueCallbackListener);
begin
  myx_grt_value_listener_add(PGrt, Value, Pointer(CallbackObject), @GrtValueCallback);

  // Add a reference because a pointer to the interfaces is store in the GRT
  CallbackObject._AddRef;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ValueListenerRemove(Value: Pointer; CallbackObject: IGrtValueCallbackListener);

begin
  myx_grt_value_listener_remove(Value, Pointer(CallbackObject), @GrtValueCallback);

  // Release refcount since the interface ref is now no longer stored in the GRT
  CallbackObject._Release;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GlobalSetRoot(Value: Pointer): Integer;

begin
  Result := Ord(myx_grt_set_root(PGrt, Value));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ValueRetain(Value: Pointer);

begin
  myx_grt_value_retain(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ValueRelease(Value: Pointer);

begin
  myx_grt_value_release(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetStructMemberCount(StructName: WideString): Integer;

var
  Struct: Pointer;

begin
  Struct := myx_grt_struct_get(PGrt, StructName);

  if (Struct <> nil) then
    Result := myx_grt_struct_get_member_count_total(PGrt, Struct)
  else
    Result := -1;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetStructMemberName(StructName: WideString; Index: Integer): WideString;

var
  Struct,
  Member: Pointer;

begin
  Result := '';

  Struct := myx_grt_struct_get(PGrt, StructName);

  if (Struct <> nil) then
  begin
    Member := myx_grt_struct_get_member_by_index_total(PGrt, Struct, Index);

    if (Member <> nil) then
      Result := myx_grt_struct_get_member_name(Member);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetStructMemberType(StructName: WideString; Index: Integer): GrtValueType;

var
  Struct,
  Member: Pointer;

begin
  Result := GrtAnyValue;

  Struct := myx_grt_struct_get(PGrt, StructName);

  if (Struct <> nil) then
  begin
    Member := myx_grt_struct_get_member_by_index_total(PGrt, Struct, Index);

    if (Member <> nil) then
      Result := GrtValueType(myx_grt_struct_member_get_type(Member));
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetStructMemberContentType(StructName: WideString; Index: Integer): GrtValueType;

var
  Struct,
  Member: Pointer;

begin
  Result := GrtAnyValue;

  Struct := myx_grt_struct_get(PGrt, StructName);

  if (Struct <> nil) then
  begin
    Member := myx_grt_struct_get_member_by_index_total(PGrt, Struct, Index);

    if (Member <> nil) then
      Result := GrtValueType(myx_grt_struct_member_get_content_type(Member));
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetStructMemberStructName(StructName: WideString; MemberName: WideString): WideString;

var
  Struct,
  Member: Pointer;

begin
  Result := '';

  Struct := myx_grt_struct_get(PGrt, StructName);

  if (Struct <> nil) then
  begin
    Member := myx_grt_struct_get_member_by_name(PGrt, Struct, MemberName, 1);

    if (Member <> nil) then
      Result := myx_grt_struct_member_get_struct_name(Member);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetStructMemberContentStructName(StructName: WideString; MemberName: WideString): WideString;

var
  Struct,
  Member: Pointer;

begin
  Result := '';

  Struct := myx_grt_struct_get(PGrt, StructName);

  if (Struct <> nil) then
  begin
    Member := myx_grt_struct_get_member_by_name(PGrt, Struct, MemberName, 1);

    if (Member <> nil) then
      Result := myx_grt_struct_member_get_content_struct_name(Member);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetStructMemberContentStructName(StructName: WideString; Index: Integer): WideString;

var
  Struct,
  Member: Pointer;

begin
  Result := '';

  Struct := myx_grt_struct_get(PGrt, StructName);

  if (Struct <> nil) then
  begin
    Member := myx_grt_struct_get_member_by_index_total(PGrt, Struct, Index);

    if (Member <> nil) then
      Result := myx_grt_struct_member_get_content_struct_name(Member);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetStructCaption(StructName: WideString): WideString;

var
  Struct: PMYX_GRT_STRUCT;
  InheritedCaption: Integer;

begin
  Struct := myx_grt_struct_get(PGrt, StructName);

  Result := myx_grt_struct_get_caption(PGrt, Struct, @InheritedCaption);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.StructExists(StructName: WideString): Boolean;

begin
  Result := (myx_grt_struct_get(PGrt, StructName) <> nil);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.StructInheritsFrom(StructName: WideString; ParentStructName: WideString): Boolean;

begin
  Result := (myx_grt_struct_inherits_from(PGrt, StructName, ParentStructName) = 1);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.FormatGrtMessagesAsString(Msgs: TMYX_GRT_MSGS): WideString;

begin
  Result := FormatGrtMessagesAsString(Msgs);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ListCount(List: Pointer): Integer;

begin
  Result := myx_grt_list_item_count(List);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetListItem(const List: Pointer; const I: Integer): Pointer;

begin
  Result := myx_grt_list_item_get(List, I);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetListItem(const List: Pointer; const I: Integer; NewValue: Pointer);

begin
  myx_grt_list_item_set(List, I, NewValue);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetListString(const List: Pointer; const I: Integer): WideString;

begin
  Result := myx_grt_list_item_get_as_string(List, I);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetListString(const List: Pointer; const I: Integer; NewValue: WideString);

var
  Value: Pointer;
  OrgValue: Pointer;
  ValueType: MYX_GRT_VALUE_TYPE;
  
begin
  // Check what the original value's type is and auto-convert the given string to a native type if possible.
  OrgValue := myx_grt_list_item_get(List, I);
  ValueType := myx_grt_value_get_type(OrgValue);
  case ValueType of
    MYX_INT_VALUE:
      try
        Value := myx_grt_value_from_int(StrToInt(NewValue));
      except
        raise Exception.Create(Format(_('The string %s cannot be converted to an integer value.'), [NewValue]));
      end;
    MYX_REAL_VALUE:
      try
        Value := myx_grt_value_from_real(StrToFloat(NewValue));
      except
        raise Exception.Create(Format(_('The string %s cannot be converted to a float value.'), [NewValue]));
      end;
  else
    // Everything else.
    Value := myx_grt_value_from_string(NewValue);
  end;
  myx_grt_list_item_set(List, I, Value);
  myx_grt_value_release(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetListRefItem(const List: Pointer; const I: Integer): Pointer;

begin
  Result := myx_grt_list_item_get_reference_value(PGrt, List, I);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetListRefItem(const List: Pointer; const I: Integer; NewValue: Pointer);

var
  Value: Pointer;

begin
  Value := myx_grt_value_from_string(DictString[NewValue, '_id']);

  myx_grt_list_item_set(List, I, Value);

  myx_grt_value_release(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetListItemByObjectName(const List: Pointer; const Name: WideString): Pointer;

begin
  Result := myx_grt_list_item_get_by_object_name(List, Name);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetListRefValueByObjectName(const List: Pointer; const Name: WideString): Pointer;

begin
  Result := myx_grt_list_item_get_reference_value_by_object_name(PGrt, List, Name);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ListAdd(List: Pointer; Value: Pointer; IncreaseRefCount: Boolean);

begin
  myx_grt_list_item_add(List, Value);

  // as myx_grt_list_item_add increases refcount,
  // release it one time if it should not be increased
  if (not (IncreaseRefCount)) then
    myx_grt_value_release(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ListAddString(List: Pointer; Name: WideString);

var
  Value: Pointer;

begin
  Value := ValueFromString(Name);

  ListAdd(List, Value, False);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ListInsert(List: Pointer; Index: Integer; Value: Pointer; IncreaseRefCount: Boolean);

begin
  myx_grt_list_item_insert(List, Index, Value);

  // as myx_grt_list_item_insert increases refcount,
  // release it one time if it should not be increased
  if (not (IncreaseRefCount)) then
    myx_grt_value_release(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ListDel(List: Pointer; Index: Integer): Boolean;

begin
  Result := (myx_grt_list_item_del(List, Index) = 0);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ListDel(List: Pointer; Value: Pointer): Boolean;

begin
  Result := (myx_grt_list_item_del_value(List, Value) = 0);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ListDelString(List: Pointer; Name: WideString): Boolean;

begin
  Result := (myx_grt_list_item_del_as_string(List, Name) = 0);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ListDelObject(List: Pointer; Name: WideString): Boolean;

begin
  Result := (myx_grt_list_del_by_object_name(List, Name) = 0);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ListNew(const ContentType: GrtValueType;
  const ContentStructName: WideString = ''): Pointer;

begin
  Result := myx_grt_list_new(MYX_GRT_VALUE_TYPE(ContentType),
    ContentStructName);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ListClear(List: Pointer);

begin
  myx_grt_list_clear(List);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetDictItem(const Dict: Pointer; const Key: WideString): Pointer;

begin
  Result := myx_grt_dict_item_get_value(Dict, Key);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetDictItem(const Dict: Pointer; const Key: WideString; NewValue: Pointer);

begin
  myx_grt_dict_item_set_value(Dict, Key, NewValue);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetDictString(const Dict: Pointer; const Key: WideString): WideString;

begin
  if (Dict <> nil) then
    Result := myx_grt_dict_item_get_as_string(Dict, Key)
  else
    Result := '';
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetDictString(const Dict: Pointer; const Key: WideString; NewValue: WideString);

var
  Value: Pointer;
  OrgValue: Pointer;
  ValueType: MYX_GRT_VALUE_TYPE;

begin
  // Check what the original value's type is and auto-convert the given string to a native type if possible.
  OrgValue := myx_grt_dict_item_get_value(Dict, Key);
  ValueType := myx_grt_value_get_type(OrgValue);
  case ValueType of
    MYX_INT_VALUE:
      try
        Value := myx_grt_value_from_int(StrToInt(NewValue));
      except
        raise Exception.Create(Format(_('The string %s cannot be converted to an integer value.'), [NewValue]));
      end;
    MYX_REAL_VALUE:
      try
        Value := myx_grt_value_from_real(StrToFloat(NewValue));
      except
        raise Exception.Create(Format(_('The string %s cannot be converted to a float value.'), [NewValue]));
      end;
  else
    // Everything else.
    Value := myx_grt_value_from_string(NewValue);
  end;

  myx_grt_dict_item_set_value(Dict, Key, Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetDictInt(const Dict: Pointer; const Key: WideString): Integer;

begin
  Result := myx_grt_dict_item_get_as_int(Dict, Key);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetDictInt(const Dict: Pointer; const Key: WideString; NewValue: Integer);

begin
  myx_grt_dict_item_set_value_from_int(Dict, Key, NewValue);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetDictReal(const Dict: Pointer; const Key: WideString): Double;

begin
  Result := myx_grt_dict_item_get_as_real(Dict, Key);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetDictReal(const Dict: Pointer; const Key: WideString; NewValue: Double);

begin
  myx_grt_dict_item_set_value_from_real(Dict, Key, NewValue);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetDictRef(const Dict: Pointer; const Key: WideString): Pointer;

begin
  Result := myx_grt_dict_item_get_reference_value(FGrtEngine.PGrt, Dict, Key);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetDictRef(const Dict: Pointer; const Key: WideString; NewValue: Pointer);

begin
  myx_grt_dict_item_set_value_from_string(Dict, Key,
    DictString[NewValue, '_id']);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetDictStructName(const Dict: Pointer): WideString;

begin
  Result := myx_grt_dict_struct_get_name(Dict);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetDictStructName(const Dict: Pointer; StructName: WideString);

begin
  myx_grt_dict_struct_set_name(PGrt, Dict, StructName);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.DictNew(const StructName: WideString): Pointer;

begin
  Result := myx_grt_dict_new(PGrt, StructName);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.DictNewTyped(const ContentType: GrtValueType;
  const ContentStructName: WideString): Pointer;

begin
  Result := myx_grt_dict_new_typed(MYX_GRT_VALUE_TYPE(ContentType),
    ContentStructName);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ObjectNew(const StructName: WideString;
  Name: WideString; Id: WideString; OwnerId: WideString): Pointer;

begin
  Result := myx_grt_dict_new_obj(PGrt, StructName, Name, Id, OwnerId);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetValueInt(const Value: Pointer): Integer;

begin
  Result := myx_grt_value_as_int(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetValueString(const Value: Pointer): WideString;

begin
  Result := myx_grt_value_as_string(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueFromString(S: WideString): Pointer;

begin
  Result := myx_grt_value_from_string(S);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueFromInt(I: Integer): Pointer;

begin
  Result := myx_grt_value_from_int(I);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueAsLuaCode(Value: Pointer; Indention: Integer): WideString;

begin
  Result := myx_grt_value_as_lua_code(Value, Indention);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueDiffMake(Source: Pointer; Target: Pointer): Pointer;

begin
  Result := myx_grt_value_diff_make(PGrt, Source, Target);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueDiffApply(Value: Pointer; Diff: Pointer): Pointer;

begin
  Result := myx_grt_value_diff_apply(PGrt, Value, Diff);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueReference(Value: Pointer): Pointer;

begin
  Result := myx_grt_reference_cache_lookup(PGrt,
    ValueString[Value]);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.ValueReference(Id: WideString): Pointer;

begin
  Result := myx_grt_reference_cache_lookup(PGrt,
    PWideChar(Id));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.ScriptInputRead;

begin
  if (GrtSyncEvent <> nil) then
    GrtSyncEvent.SetEvent;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetOutputBuffer: WideString;

begin
  Result := FGrtEngine.OutputBuffer;

  FGrtEngine.OutputBuffer := '';
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetOutputBuffer(OutputBuffer: WideString);

begin
  FGrtEngine.OutputBuffer := OutputBuffer;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetShellVar(Name: WideString): Pointer;

begin
  Result := myx_grt_lua_shell_get_global_var(PGrt, Name);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.SetShellVar(Name: WideString; Value: Pointer);

begin
  myx_grt_lua_shell_set_global_var(PGrt, Name, Value);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetGlobalAsParam(Path: WideString): WideString;

begin
  Result := 'global::' + Path;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.DictItemCount(const Dict: Pointer): Integer;

begin
  Result := myx_grt_dict_item_count(Dict);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrt.DictDel(const Dict: Pointer; const Key: WideString);

begin
  myx_grt_dict_item_del(Dict, Key);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrt.GetDictKey(const Dict: Pointer; const I: Integer): WideString;

begin
  Result := myx_grt_dict_item_key_by_index(Dict, I);
end;


//----------------------------------------------------------------------------------------------------------------------
// Grt Engine
//----------------------------------------------------------------------------------------------------------------------

constructor TGrtEngine.Create(CreateSuspended: Boolean; Grt: TGrt);

begin
  inherited Create(CreateSuspended);

  FGrt := Grt;
  FPGrt := nil;

  FCallbackEntries := TThreadList.Create;

  FJavaLoaderState := GrtLsNotInitialized;
  FLuaLoaderState := GrtLsNotInitialized;
  FBuiltinLoaderState := GrtLsNotInitialized;
  FPhpLoaderState := GrtLsNotInitialized;
  FUiLoaderState := GrtLsNotInitialized;

  FFunctionSynchronizer := TCriticalSection.Create;
  FFunctionStartedEvent := TEvent.Create(nil, False, False, '');
  FFunctionFinishedEvent := TEvent.Create(nil, False, False, '');

  FVclDataSynchronizer := TCriticalSection.Create;

  FDelphiGrtMessages := TMYX_GRT_MSGS.Create;
  Application.HookMainWindow(ApplicationHook);
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TGrtEngine.Destroy;

begin
  Application.UnhookMainWindow(ApplicationHook);

  ClearCallbackEntries;
  FCallbackEntries.Free;
  
  if FPGrt <> nil then
    myx_grt_finalize(FPGrt);

  FFunctionFinishedEvent.Free;
  FFunctionStartedEvent.Free;
  FFunctionSynchronizer.Free;

  FVclDataSynchronizer.Free;

  FDelphiGrtMessages.Free;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.Execute;

var
  FunctionStartedWaitResult: TWaitResult;
  ErrorDetails: TMYX_STRINGLIST;
  GrtOptions: Integer;

begin
  GrtOptions := 0;
  if (FGrt.Verbose) then
    GrtOptions := GrtOptions + Ord(MYX_GRT_VERBOSE);
  if (FGrt.RemoteDebug) then
    GrtOptions := GrtOptions + Ord(MYX_GRT_REMOTE_DEBUG);

  FPGrt := myx_grt_initialize(GrtOptions);

  InitializeGrt;

  while not Terminated do
  begin
    FunctionStartedWaitResult := FFunctionStartedEvent.WaitFor(1000);

    if FunctionStartedWaitResult = wrSignaled then
    begin
      FStartTime := Now;

      try
        // if the function name is set, it is a GrtFunction
        if (FFunctionName <> '') then
          FResult := myx_grt_function_get_and_call(FPGrt,
            FModuleName, FFunctionName, Ord(FSearchParent),
            FFunctionArgument, FPError)
        else
          // otherwise its a shell function
          FShellResult := myx_grt_lua_shell_execute(FPGrt, FShellCmd);
      except
        on X: Exception do
        begin
          ErrorDetails := TMYX_STRINGLIST.Create;
          try
            ErrorDetails.strings.Text := Format(
              _('The call of the function %s:%s cause an exception.'),
              [FModuleName, FFunctionName]);

            myx_grt_messages_stack_add(FPGrt, 1, X.Message,
              ErrorDetails.get_record_pointer, 1, -1);

            myx_grt_messages_stack_flush(FPGrt, 0);
          finally
            ErrorDetails.Free;
          end;
        end;
      end;

      FEndTime := Now;

      FFunctionFinishedEvent.SetEvent;
    end;
  end;
  FinalizeGrt;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.FinalizeGrt;

begin
  FGrt.ScriptInputRead;

  FinalizeLoadersAndModules;
end;

procedure TGrtEngine.FinalizeLoadersAndModules;

begin
  FGrt.ScriptInputRead;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.InitializeGrt;

var
  Error: MYX_GRT_ERROR;
  I: integer;

begin
  myx_grt_catch_glib_messages(FPGrt, 1);

  // Register print output function
  myx_grt_set_output_callback(FPGrt, self, @ProcessGrtOutput);

  // Register message processing function
  myx_grt_set_message_callback(FPGrt, self, @ProcessGrtMessages);

  // Register input function
  myx_grt_set_input_callback(FPGrt, self, @ProcessInput);

  // Register status query function
  myx_grt_set_status_query_callback(FPGrt, self, @ProcessStatusQuery);

  myx_grt_shell_print_welcome(FPGrt);

  OutputText('');

  // -------------------------------------------------
  // Load Structs

  I := myx_grt_scan_for_structs(FPGrt, './xml', @Error);
  if (Error <> MYX_GRT_NO_ERROR) then
    OutputText('  ' + Format(
      _('Error while loading struct definitions (%d).') + #13#10,
      [Ord(Error)]))
  else
    if (I = 1) then
      OutputText(
        _('Registered 1 struct definition file.') + #13#10)
    else
      OutputText(Format(
        _('Registered %d struct definition files.') + #13#10,
        [I]));

  InitializeLoadersAndModules;


  OutputText(#13#10);

  //Init lua shell
  myx_grt_init_lua_shell(FPGrt);

  FGrt.GrtSyncEvent.SetEvent;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.InitializeLoadersAndModules;

var
  Loader: Pointer;
  Error: MYX_GRT_ERROR;
  I: integer;

begin
  // -------------------------------------------------
  // Initialized Loaders

  // Init Delphi loader
  OutputText(_('Initializing native loader...') + #13#10);
  FUiLoaderState := GrtLsInitializeFailed;
  FNativeLoader := InitDelphiLoader(@Error);
  if (FNativeLoader <> nil) then
  begin
    if (myx_grt_register_module_loader(FPGrt, FNativeLoader) <> MYX_GRT_NO_ERROR) then
      OutputText('  ' + _('Could not register Delphi modules.') + #13#10)
    else
      FUiLoaderState := GrtLsInitialized;
  end
  else
    OutputText('  ' + Format(
      _('Error initializing Delphi modules (%d)') + #13#10,
      [Ord(error)]));

  // Init Java loader
  if (WideDirectoryExists(GetApplDir + 'java')) then
  begin
    OutputText(_('Initializing Java loader...') + #13#10);
    if (FGrt.Verbose) then
      OutputText(#13#10);

    FJavaLoaderState := GrtLsInitializeFailed;

    Loader := myx_java_init_loader(FPGrt, 'java', @Error, FGrt.JvmLibrary, './');

    if (Loader <> nil) then
    begin
      FJavaLoaderState := GrtLsInitialized;

      if (myx_grt_register_module_loader(FPGrt, Loader) <> MYX_GRT_NO_ERROR) then
        OutputText('  ' + _('Could not register Java module loader.') + #13#10)
      else
        FJavaLoaderState := GrtLsInitialized;
    end
    else
      OutputText('  ' + Format(
        _('Error initializing Java module loader (%d). ' +
          'Please check the .\java and .\java\lib directories.') + #13#10,
        [Ord(error)]));
  end;

  // Init PHP loader
  if (WideDirectoryExists(GetApplDir + 'php')) then
  begin
    OutputText(_('Initializing PHP loader...') + #13#10);
    FPhpLoaderState := GrtLsInitializeFailed;

    Loader := myx_php_init_loader(FPGrt, @Error);
    if (Loader <> nil) then
    begin
      if (myx_grt_register_module_loader(FPGrt, Loader) <> MYX_GRT_NO_ERROR) then
        OutputText('  ' + _('Could not register PHP module loader.') + #13#10)
      else
        FPhpLoaderState := GrtLsInitialized;
    end
    else
      OutputText('  ' + Format(
        _('Error initializing PHP module loader (%d)') + #13#10,
        [Ord(error)]));
  end;

  // Init lua loader
  OutputText(_('Initializing Lua loader...') + #13#10);
  FLuaLoaderState := GrtLsInitializeFailed;
  Loader := myx_lua_init_loader(FPGrt, @Error, './lua');
  if (Loader <> nil) then
  begin
    if (myx_grt_register_module_loader(FPGrt, Loader) <> MYX_GRT_NO_ERROR) then
      OutputText('  ' + _('Could not register Lua module loader.') + #13#10)
    else
      FLuaLoaderState := GrtLsInitialized;
  end
  else
    OutputText('  ' + Format(
      _('Error initializing Lua module loader (%d)') + #13#10,
      [Ord(error)]));

  // -------------------------------------------------
  // Load modules

  //Load builtin modules
  FBuiltinLoaderState := GrtLsInitializeFailed;
  if (myx_register_builtin_grt_module_base(FPGrt) <> nil) and
    (myx_register_builtin_grt_module_reverse_engineer_mysql(FPGrt) <> nil) and
    (myx_register_builtin_grt_module_transformation_mysql(FPGrt) <> nil) then
  begin
    FBuiltinLoaderState := GrtLsModulesLoaded;

    OutputModuleStatus('builtin', 3, MYX_GRT_NO_ERROR);
  end;

  // Java modules
  if (FJavaLoaderState = GrtLsInitialized) then
  begin
    if (FGrt.Verbose) then
      OutputText(#13#10);

    //Scan for Java plugins
    I := myx_grt_scan_for_modules(FPGrt, './java/com/mysql/grt/modules', @Error);
    OutputModuleStatus('Java', I, Error);

    FJavaLoaderState := GrtLsModulesLoaded;
  end;

  // Php modules
  if (FPhpLoaderState = GrtLsInitialized) then
  begin
    //Scan for PHP plugins
    I := myx_grt_scan_for_modules(FPGrt, './php/modules', @Error);
    OutputModuleStatus('PHP', I, Error);
  end;

  // Lua modules
  if (FLuaLoaderState = GrtLsInitialized) and
    (WideDirectoryExists(GetApplDir + 'lua')) then
  begin
    //Scan for Lua plugins
    I := myx_grt_scan_for_modules(FPGrt, './lua', @Error);
    OutputModuleStatus('Lua', I, Error);
  end;
end;


//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.OutputModuleStatus(LoaderName: WideString; LoadedModuleCount: Integer; Error: MYX_GRT_ERROR);

begin
  if (Error <> MYX_GRT_NO_ERROR) then
    OutputText(Format(
      _('Error while loading %s modules (%d).') + #13#10,
      [LoaderName, Ord(Error)]))
  else
    if (LoadedModuleCount = 1) then
      OutputText(Format(
        _('Registered 1 %s module.') + #13#10, [LoaderName]))
    else
      if (LoadedModuleCount > 0) then
        OutputText(Format(
          _('Registered %d %s modules.') + #13#10, [LoadedModuleCount, LoaderName]))
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.GetPGrt: Pointer;

begin
  Result := FPGrt;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.ExecuteModalFunction(ModulName: WideString;
  FunctionName: WideString;
  FunctionArgument: Pointer; Error: PMYX_GRT_ERROR;
  ProcessOutputFunction: TGrtProcessOutput;
  ProcessMessagesFunction: TGrtProcessMessages;
  SearchParent: Boolean;
  TimeOutMS: Integer;
  PerformInputFunction: TGrtPerformInput;
  PerformStatusQueryFunction: TGrtPerformStatusQuery): Pointer;

const
  MaxTimeOut = 5000; // Maximum number of seconds a single GRT call is allowed to take if no explizit timeout was given.
  
var
  StartTime: DWORD;
  TimeOut: Boolean;
  WaitHandle: THandle;
  WaitResult: DWORD;

begin
  FResult := nil;

  // Allow only one function at a time
  FFunctionSynchronizer.Acquire;
  try
    FExecutionFinished := False;
    FModuleName := ModulName;
    FFunctionName := FunctionName;
    FSearchParent := SearchParent;
    FFunctionArgument := FunctionArgument;
    FPError := Error;

    // Set callback functions
    FProcessOutputFunction := ProcessOutputFunction;
    FProcessMessagesFunction := ProcessMessagesFunction;
    FPerformInputFunction := PerformInputFunction;
    FPerformStatusQueryFunction := PerformStatusQueryFunction;

    // Trigger the function execution
    FFunctionStartedEvent.SetEvent;

    StartTime := GetTickCount;
    TimeOut := False;

    WaitHandle := FFunctionFinishedEvent.Handle;
    while (not (TimeOut)) do
    begin
      // Wait for the function to finish but every 100 milliseonds check if the timeout value has been reached.
      // Process any incomming message while we wait.
      WaitResult := MsgWaitForMultipleObjects(1, WaitHandle, False, 100, QS_ALLEVENTS);
      if WaitResult = WAIT_OBJECT_0 then
        Break;

      Application.ProcessMessages;

      if (TimeOutMS > -1) then
      begin
        TimeOut := Integer (GetTickCount - StartTime) > TimeOutMS;

        if (TimeOut) then
          if (ShowModalDialog(_('Function Timeout Occured'),
            Format(_('The executed function %s:%s has not returned ' +
            'after %d seconds. Do you want to wait for another ' +
            '%d seconds?'),
            [FModuleName, FFunctionName, TimeOutMS, TimeOutMS]),
              myx_mtInformation, _('Yes') + #13#10 + _('No')) = 2) then
          begin
            raise Exception.Create(
              Format(_('The function %s:%s did not complete in %d seconds.'),
                [FModuleName, FFunctionName, TimeOutMS]));
          end
          else
            StartTime := GetTickCount;
      end
      else
      begin
        // If no timeout was given and the application is said to terminate then check for a maximum execution time
        // and go out if this is exceeded.
        if Application.Terminated and (GetTickCount - StartTime > MaxTimeOut) then
          Break;
      end;
    end;
  finally
    FProcessOutputFunction := nil;
    FProcessMessagesFunction := nil;
    FPerformInputFunction := nil;
    FPerformStatusQueryFunction := nil;

    FFunctionSynchronizer.Release;

    FFunctionName := '';
  end;

  Result := FResult;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.ExecuteModalShellFunction(Cmd: WideString; ProcessOutputFunction: TGrtProcessOutput;
  TimeOutMS: Integer): MYX_GRT_SHELL_COMMAND;

var
  StartTime: TDateTime;
  TimeOut: Boolean;
  TimeOutTime: TDateTime;
  WaitHandle: THandle;
  WaitResult: DWORD;

begin
  FShellResult := MYX_GRT_SHELL_COMMAND_UNKNOWN;

  // Allow only one function at a time
  FFunctionSynchronizer.Acquire;
  try
    FShellCmd := Cmd;

    // Set output function
    FProcessOutputFunction := ProcessOutputFunction;

    // Trigger the function execution
    FFunctionStartedEvent.SetEvent;

    StartTime := Now;
    TimeOut := False;
    TimeOutTime := 0;
    if (TimeOutMS > -1) then
      TimeOutTime := TimeOutMS / (24 * 60 * 60 * 1000);

    WaitHandle := FFunctionFinishedEvent.Handle;
    while (not (TimeOut)) do
    begin
      // Wait for the function to finish but every 100 milliseonds check if the timeout value has been reached.
      // Process any incomming message while we wait.
      WaitResult := MsgWaitForMultipleObjects(1, WaitHandle, false, 100, QS_ALLEVENTS);
      if WaitResult = WAIT_OBJECT_0 then
        Break;

      Application.ProcessMessages;

      if (TimeOutMS > -1) then
      begin
        TimeOut := (Now - StartTime > TimeOutTime);

        if (TimeOut) then
          if (ShowModalDialog(_('Function Timeout Occured'),
            Format(_('The executed function %s:%s has not returned ' +
            'after %s minutes. Do you want to wait for another ' +
            '%s minutes?'),
            [FModuleName,
            FFunctionName,
              FormatDateTime('n:sss', TimeOutTime),
              FormatDateTime('n:sss', TimeOutTime)]),
              myx_mtInformation, _('Yes') + #13#10 + _('No')) = 2) then
          begin
            raise Exception.Create(
              Format(_('The function %s:%s ' +
              'did not complete in %s minutes.'),
              [FModuleName,
              FFunctionName,
                FormatDateTime('n:sss', TimeOutTime)]));
          end
          else
            StartTime := Now;
      end;
    end;
  finally
    FProcessOutputFunction := nil;

    FFunctionSynchronizer.Release;
  end;

  Result := FShellResult;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.GetExecutionFinished: Boolean;

begin
  Result := FExecutionFinished;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.GetExecutionTime: TDateTime;

begin
  Result := FEndTime - FStartTime;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.OutputText(S: WideString);

begin
  AddCallbackEntry(0, S, nil);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.DoOutputText(S: WideString);

begin
  if (Assigned(FProcessOutputFunction)) then
    FProcessOutputFunction(S)
  else
  begin
    if Assigned(FGrt.FConsole) then
    begin
      {if (Copy(S, Length(S), 1) <> #10) then
        S := S + #13#10
      else
        if (Trim(S) = '') then
          Exit;}
        
      FGrt.Console.AddOutput(S);
      FGrt.Console.Invalidate;
    end
    else
      FOutputBuffer := FOutputBuffer + S;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.ProcessMessages(PMessages: PMYX_GRT_MSGS): Integer;

begin
  AddCallbackEntry(1, '', PMessages);
  Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.DoProcessMessages(Messages: TMYX_GRT_MSGS);

begin
  if (Assigned(FProcessMessagesFunction)) then
    FProcessMessagesFunction(Messages)
  else
  begin
    // print the message
    DoOutputText(FormatGrtMessagesAsString(Messages));
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.GrtInputText(Caption: WideString; options: MYX_GRT_INPUT_OPTIONS; Text: PPointer): integer;

var
  Input: String;

begin
  FVclDataSynchronizer.Acquire;
  try
    FTextForOutput := Caption;
    Synchronize(DoGrtInputText);

    if (Assigned(FPerformInputFunction)) then
      Input := UTF8Encode(FTextInput)
    else
    begin
      // Wait till user input is finished
      FGrt.GrtSyncEvent := TEvent.Create(nil, False, False, '');
      try
        FGrt.GrtSyncEvent.WaitFor(10000000);

        Input := UTF8Encode(FGrt.Console.ConsoleCommand);
        Input := Copy(Input, 0, Length(Input) - 2);
      finally
        FGrt.Console.ReadScriptInput := False;
        FGrt.GrtSyncEvent.Free;
        FGrt.GrtSyncEvent := nil;
      end;
    end;

    StrLCopy(FInputBuffer, PChar(Input), 160);

    Text^ := @FInputBuffer;

    Result := 0;
  finally
    FVclDataSynchronizer.Release;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.DoGrtInputText;

begin
  if (Assigned(FPerformInputFunction)) then
    FTextInput := FPerformInputFunction(FTextForOutput)
  else
  begin
    FGrt.Console.ReadScriptInput := True;
    FGrt.Console.ConsolePrompt := FTextForOutput;
    FGrt.Console.Content.DeleteLine(FGrt.Console.Content.Count - 1);
    FGrt.Console.PrepareNextConsoleCommand;
  end
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.GrtStatusQuery: Integer;

begin
  Synchronize(DoStatusQuery);

  Result := FStatus;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.DoStatusQuery;

begin
  if (Assigned(FPerformStatusQueryFunction)) then
    FStatus := FPerformStatusQueryFunction
  else
    FStatus := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.ExecuteDelphiFunction;

begin
  FDelphiModuleFunctionResult :=
    FDelphiModuleImplementor.GrtExecuteModuleFunction(
    FDelphiModuleName, FDelphiModuleFunctionName, FDelphiModuleFunctionArgument);
end;

//----------------------------------------------------------------------------------------------------------------------

function DelphiGrtCallFunction(func: PMYX_GRT_FUNCTION; argument: Pointer; retval: PPointer): MYX_GRT_ERROR cdecl;

var
  GrtEngine: TGrtEngine;

begin
  Result := MYX_GRT_NO_ERROR;

  GrtEngine := func.module.priv;

  retval^ := nil;

  try
    GrtEngine.FDelphiModuleImplementor := IGrtModuleImplementor(func.priv);
    GrtEngine.FDelphiModuleName := UTF8Decode(func.module.name);
    GrtEngine.FDelphiModuleFunctionName := UTF8Decode(func.name);
    GrtEngine.FDelphiModuleFunctionArgument := argument;

    // Call function synchronized
    GrtEngine.Synchronize(GrtEngine.ExecuteDelphiFunction);

    GrtEngine.FDelphiModuleImplementor := nil;

    retval^ := GrtEngine.FDelphiModuleFunctionResult;
  except
    on x: Exception do
    begin
      // Log error
      GrtEngine.DelphiGrtMessages.msgs.Add(
        TMYX_GRT_MSG.Create(1, x.Message, -1, nil));
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtEngine.InitDelphiLoader(Error: PMYX_GRT_ERROR): Pointer;

begin
  Result := myx_grt_module_loader_create(FPGrt,
    MYX_DELPHI_MODULE_TYPE, nil,
    nil, @DelphiGrtCallFunction,
    self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.AddDelphiModule(ModuleName: WideString; ModuleFunctions: WideString;
  Implementor: IGrtModuleImplementor);

var
  PModule: PMYX_GRT_MODULE;
  I: Integer;
  FuncList: TTntStringList;
  Impl: Pointer;

begin
  PModule := myx_grt_module_create(FNativeLoader, ModuleName, '', '', self);

  FuncList := TTntStringList.Create;
  try
    FuncList.Text := ModuleFunctions;

    Impl := Pointer(Implementor);

    for I := 0 to FuncList.Count - 1 do
      myx_grt_module_add_function(PModule, FuncList[I], '', '',
        Impl);

    // Add a reference because a pointer to the interfaces is store in the GRT
    Implementor._AddRef;

    myx_grt_add_module(PGrt, PModule);
  finally
    FuncList.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngine.RemoveDelphiModule(ModuleName: WideString;
  Implementor: IGrtModuleImplementor);

begin
  // Release refcount since the interface ref is now no longer stored in the GRT
  Implementor._Release;
end;

//----------------------------------------------------------------------------------------------------------------------

constructor EGrtError.Create(ErrorNumber: Integer; Description: WideString);

begin
  FErrorNumber := ErrorNumber;
  FDescription := Description;

  if (FErrorNumber <> 0) then
    Message := FDescription + ' Error Nr.' + IntToStr(FErrorNumber)
  else
    Message := FDescription;
end;

//----------------------------------------------------------------------------------------------------------------------

function GetListMemberCount(Grt: Pointer; StructName: WideString; OnlyCheck: Boolean): Integer;

var
  PStruct,
    PMember: Pointer;
  i,
    count: integer;

begin
  PStruct := myx_grt_struct_get(Grt, StructName);

  count := 0;
  for i := 0 to myx_grt_struct_get_member_count_total_excluding_struct(
    Grt, PStruct, 'db.DatabaseObject') - 1 do
  begin
    PMember := myx_grt_struct_get_member_by_index_total(
      Grt, PStruct, i);

    if (myx_grt_struct_member_get_type(PMember) = MYX_LIST_VALUE) and
      (myx_grt_struct_member_get_content_type(PMember) = MYX_DICT_VALUE) then
    begin
      inc(count);

      if (OnlyCheck) then
        break;
    end;
  end;

  Result := count;
end;

//----------------------------------------------------------------------------------------------------------------------

function GetListMember(Grt: Pointer; StructName: WideString; Index: Integer): Pointer;

var
  PStruct: Pointer;
  i,
  Count: integer;

begin
  Result := nil;
  PStruct := myx_grt_struct_get(Grt, StructName);

  count := 0;
  for i := 0 to myx_grt_struct_get_member_count_total(Grt, PStruct) - 1 do
  begin
    Result := myx_grt_struct_get_member_by_index_total(
      Grt, PStruct, i);

    if (myx_grt_struct_member_get_type(Result) = MYX_LIST_VALUE) and
      (myx_grt_struct_member_get_content_type(Result) = MYX_DICT_VALUE) then
    begin
      inc(Count);

      if (Index = Count - 1) then
        break;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function FormatGrtMessagesAsString(Msgs: TMYX_GRT_MSGS): WideString;

var
  S: WideString;
  I,
  J: Integer;

begin
  S := '';

  for I := 0 to Msgs.msgs.Count - 1 do
  begin
    // do not log progress messages
    if (Msgs.msgs[I].msg_type = 2) then
      Continue;

    if (Msgs.msgs[I].msg_type = 1) then
      S := S + 'ERROR: ';

    S := S +
      Tnt_WideStringReplace(
      Tnt_WideStringReplace(
      Msgs.msgs[I].msg, #13#10, #10, [rfReplaceAll], False),
      #10, #13#10, [rfReplaceAll], False) +
      #13#10;

    if (Msgs.msgs[I].msg_detail <> nil) then
    begin
      for j := 0 to Msgs.msgs[I].msg_detail.strings.Count - 1 do
      begin
        S := S +
          Tnt_WideStringReplace(
          Tnt_WideStringReplace(
          Msgs.msgs[I].msg_detail.strings[J], #13#10, #10, [rfReplaceAll], False),
          #10, #13#10, [rfReplaceAll], False) +
          #13#10;
      end;
    end;
  end;

  Result := S;
end;

//----------------------------------------------------------------------------------------------------------------------

function GetGrtFunctionArgumentAsString(Argument: Pointer): WideString;

var
  ValueType: MYX_GRT_VALUE_TYPE;
  Value: Pointer;

begin
  Result := '';

  ValueType := myx_grt_value_get_type(Argument);

  if (ValueType = MYX_STRING_VALUE) then
    Result := myx_grt_value_as_string(Argument)
  else
    if (ValueType = MYX_LIST_VALUE) then
    begin
      Value := myx_grt_list_item_get(Argument, 0);
      ValueType := myx_grt_value_get_type(Value);

      if (ValueType = MYX_STRING_VALUE) then
        Result := myx_grt_value_as_string(Value)
    end;

  if (ValueType <> MYX_STRING_VALUE) then
    raise Exception.Create('A wrong argument was passed to the function.');
end;

// -----------------------------------------------------------------------------

function TGrtEngine.ApplicationHook(var Message: TMessage): Boolean;

begin
  Result := False;
  case Message.Msg of
    WM_PROCESS_CALLBACK:
      begin
        DoChangeCallback;
        Result := True;
      end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngine.ClearCallbackEntries;

var
  I: Integer;
  Entry: PCallbackEntry;

begin
  with FCallbackEntries, LockList do
  try
    for I := 0 to Count - 1 do
    begin
      Entry := Items[I];
      Entry.Messages.Free;
      Dispose(Entry);
    end;
    Clear;
  finally
    UnlockList;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngine.AddCallbackEntry(AType: Integer; OutputText: WideString; Messages: PMYX_GRT_MSGS);

var
  Entry: PCallbackEntry;

begin
  with FCallbackEntries, LockList do
  try
    New(Entry);
    Entry.Type_ := AType;
    Entry.TextForOutput := OutputText;
    Entry.Messages := TMYX_GRT_MSGS.Create(Messages);
    FCallbackEntries.Add(Entry);
  finally
    UnlockList;
  end;

  PostMessage(Application.Handle, WM_PROCESS_CALLBACK, 0, 0);
end;

// -----------------------------------------------------------------------------

procedure TGrtEngine.DoChangeCallback;

var
  Entry: PCallbackEntry;

begin
  repeat
    with FCallbackEntries, LockList do
    begin
      if Count > 0 then
      begin
        Entry := First;
        Remove(Entry);
      end
      else
        Entry := nil;
      UnlockList;
    end;

    if Entry = nil then
      Break;

    case Entry.Type_ of
      0:
        DoOutputText(Entry.TextForOutput);
      1:
       begin
          DoProcessMessages(Entry.Messages);
          Entry.Messages.Free;
       end;
    end;
    Dispose(Entry);
  until False;
end;

// -----------------------------------------------------------------------------

end.

