/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file  myx_grt_module_messaging.c
 * @brief GRT module messaging related functions.
 *
 * See also: <a href="../grt.html#ModuleMessaging">Module Messaging</a>
 */

#include "myx_grt_private.h"

#include <errno.h>

# include <time.h>

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
# define MSG_DONTWAIT 0
#else
# include <sys/types.h>
# include <sys/time.h>
# include <sys/select.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>
# include <sys/fcntl.h>
# include <netdb.h>
# include <unistd.h>
# define closesocket close
#endif



int myx_grt_setup_messaging(MYX_GRT *grt)
{
  struct sockaddr_in addr;

  grt->comm_server_sock= socket(PF_INET, SOCK_STREAM, 0);

  for (;;) 
  {
    grt->comm_server_port= 2000+rand()%60000;
  
    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(grt->comm_server_port);
    memset(&addr.sin_addr, 0, sizeof(addr.sin_addr));
  
    if (bind(grt->comm_server_sock, (struct sockaddr*)&addr, sizeof(addr)) < 0)
    {
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
      if (errno == WSAEADDRINUSE)
        continue;
#else
      if (errno == EADDRINUSE) // retry with another port
        continue;
#endif


      g_error("could not bind socket: %s", g_strerror(errno));
      closesocket(grt->comm_server_sock);
      return -1;
    }
    else
      break;
  }

  if (listen(grt->comm_server_sock, 5) < 0)
  {
    g_error("could not listen on socket: %s", g_strerror(errno));
    closesocket(grt->comm_server_sock);
    return -1;
  }

  return 0;
}


/** 
 ****************************************************************************
 * @brief Prepares the GRT to call a module that will perform messaging.
 * 
 * @param grt - the GRT environment. There should be no other messaging sessions
 *   enabled in the GRT.
 *
 * @return 0 if ok, -1 on error.
 *****************************************************************************/
int myx_grt_setup_module_messaging(MYX_GRT *grt)
{
  g_return_val_if_fail(grt->comm_cookie==NULL, -1);
  
  srand(time(NULL));

  grt->comm_cookie= g_strdup_printf("%i%p", rand(), grt);
  grt->comm_sock= -1;
  
  return 0;
}

/** 
 ****************************************************************************
 * @brief Cleans up module messaging related state in the GRT.
 *
 * Should be called after a module that performs messaging is called.
 * 
 * @param grt - the GRT environment
 *
 * @return 0 
 *****************************************************************************/
int myx_grt_cleanup_module_messaging(MYX_GRT *grt)
{
  g_free(grt->comm_cookie);
  grt->comm_cookie= NULL;

  closesocket(grt->comm_sock);
  grt->comm_sock= -1;

  return 0;
}

/** 
 ****************************************************************************
 * @brief Returns the file descriptor used for the messaging socket.
 *
 * Use this to retrieve a file descriptor suitable for use with select(),
 * to verify whether a module has connected to the socket and 
 * myx_grt_check_module_connected() should be called.
 * 
 * You don't need this function unless you want to poll for several file
 * descriptors at once or other specific uses.
 * 
 * @param grt
 *
 * @return A file descriptor.
 *****************************************************************************/
MYX_SOCKET myx_grt_get_messaging_fd(MYX_GRT *grt)
{
  return grt->comm_server_sock;
}

/** 
 ****************************************************************************
 * @brief Returns the file descriptor used in the current messaging session.
 *
 * The returned file descriptor is a socket with an established connection
 * to the module. 
 * 
 * You usually don't need this function and should use 
 * myx_grt_send_module_message() and myx_grt_check_module_message() to 
 * send and receive data from the moduole.
 * 
 * @param grt  
 *
 * @return The file descriptor or -1 if there's no current messaging session.
 *****************************************************************************/
MYX_SOCKET myx_grt_get_messaging_module_fd(MYX_GRT *grt)
{
  return grt->comm_sock;
}

/** 
 ****************************************************************************
 * @brief Checks whether the module function has already connected to the GRT.
 * 
 * This should be called by the main program after the
 * myx_grt_setup_module_messaging() and the module function call has been
 * issued (in a secondary thread). If it returns 1, you should be able
 * to communicate with the module using myx_grt_send_module_message() and
 * myx_grt_check_module_message()
 *
 * @param grt - grt environment where myx_grt_setup_module_messaging has been
 * called before.
 *
 * @return 0 if the module still has not connected
 * @return 1 if the module has connected
 * @return -1 if there was an error
 *****************************************************************************/
int myx_grt_check_module_connected(MYX_GRT *grt)
{
  fd_set rfd;
  MYX_SOCKET rc;
  struct timeval timeout;
  struct sockaddr_in addr;
  int addrlen;
  char *msg;
  
  g_return_val_if_fail(grt->comm_cookie!=NULL, -1);
  
  if (grt->comm_sock >= 0)
    return 1;

  FD_ZERO(&rfd);
  FD_SET(grt->comm_server_sock, &rfd);
  
  timeout.tv_sec= 0;
  timeout.tv_usec= 0;

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  if ((rc= select(0, &rfd, NULL, NULL, &timeout)) < 0)
    return -1;
#else
  if ((rc= select(grt->comm_server_sock+1, &rfd, NULL, NULL, &timeout)) < 0)
    return -1;
#endif

  if (rc == 0)
    return 0;
  
  if ((grt->comm_sock= accept(grt->comm_server_sock, (struct sockaddr*)&addr, &addrlen)) < 0)
    return 0;

  msg= NULL;
  // wait for auth cookie, timeout in 10seconds
  if (myx_grt_check_module_message(grt, &msg, 10000) < 0
      || !msg || strcmp(msg, grt->comm_cookie)!=0)
  {
    g_free(msg);
    closesocket(grt->comm_sock);
    grt->comm_sock= -1;
    return 0;
  }

  return 1;
}

/** 
 ****************************************************************************
 * @brief Checks for messages from the currently active module function call.
 *
 * @param grt 
 * @param message output parameter where the message will be placed (or NULL)
 *    Should be a zero-terminated UTF-8 string.
 * @param timeout number of milliseconds to wait for the message. If -1 it will
 *    block until a message is received (not recommended).
 *
 * @return -1 on error, 0 on success.
 *****************************************************************************/
int myx_grt_check_module_message(MYX_GRT *grt, char **message, int timeout_ms)
{
  fd_set rfd;
  int rc;
  struct timeval timeout;
  char *tmp;
  int tmp_pos;
  int tmp_size= 32;

  g_return_val_if_fail(grt->comm_sock >= 0, -1);

  FD_ZERO(&rfd);
  FD_SET(grt->comm_server_sock, &rfd);
  
  *message= NULL;

  if (timeout_ms >= 0)
  {
    timeout.tv_sec= timeout_ms/1000;
    timeout.tv_usec= (timeout_ms%1000)*10;

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
    if ((rc= select(0, &rfd, NULL, NULL, &timeout)) < 0)
      return -1;
#else
    if ((rc= select(grt->comm_sock+1, &rfd, NULL, NULL, &timeout)) < 0)
      return -1;
#endif

    if (rc == 0)
      return 0;
  }
  tmp_pos= 0;
  tmp= g_malloc(tmp_size);
  for (;;)
  {
    int c;
    if (tmp_pos == tmp_size)
    {
      tmp_size+= 32;
      tmp= g_realloc(tmp, tmp_size);
    }
    c= recv(grt->comm_sock, tmp+tmp_pos, 1, 0);
    if (c <= 0)
    {
      rc= -1;
      break;
    }
    if (tmp[tmp_pos] == 0)
    {
      rc= 0;
      break;
    }
    tmp_pos+= c;
  }

  *message= tmp;

  return rc;
}


/** 
 ****************************************************************************
 * @brief Sends a message to the current active module function call.
 * 
 * @param grt 
 * @param message Message string to be passed to the module.
 *      Should be a zero terminated UTF-8 string.
 *
 * @return 0 if ok, -1 on error.
 *****************************************************************************/
int myx_grt_send_module_message(MYX_GRT *grt, const char *message)
{
  g_return_val_if_fail(grt->comm_sock >= 0, -1);

  if (send(grt->comm_sock, message, (int)strlen(message)+1, 0) < 0)
    return -1;
  return 0;
}


