//
//  MWLayerTree.m
//  MySQL Workbench
//
//  Created by Alfredo Kojima on 05/8/3.
//  Copyright 2005 MySQL AB. All rights reserved.
//

#import "MWLayerTree.h"

@implementation MWElementTreeNode

- (id)initWithGRT:(MYX_GRT*)grt
          element:(MYX_GRT_VALUE*)element
{
  self= [super init];
  if (self)
  {
    MYX_GRT_VALUE *object;
//    _grt= grt;
    _element= element;
    
    object= element;
    if (strcmp(myx_grt_dict_struct_get_name(element), "db.workbench.TableElement")==0)
    {
      object= myx_grt_bridge_dict_item_get_value(element, "table", 0);
      if (object)
        _object= myx_grt_reference_cache_lookup(grt, myx_grt_value_as_string(object));
    }
    else if (strcmp(myx_grt_dict_struct_get_name(element), "db.workbench.ViewElement")==0)
    {
      object= myx_grt_bridge_dict_item_get_value(element, "view", 0);
      if (object)
        _object= myx_grt_reference_cache_lookup(grt, myx_grt_value_as_string(object));
    }
    else if (strcmp(myx_grt_dict_struct_get_name(element), "db.workbench.RoutinesElement")==0)
    {
      object= myx_grt_bridge_dict_item_get_value(element, "routineGroup", 0);
      if (object)
        _object= myx_grt_reference_cache_lookup(grt, myx_grt_value_as_string(object));
    }
    _description= [[NSString stringWithUTF8String:myx_grt_bridge_value_as_string(myx_grt_bridge_dict_item_get_value(_object?:element, "name", 0))] retain];
  }
  return self;
}


- (void)setDescription:(NSString*)description
{
  myx_grt_bridge_dict_item_set_value(_object?:_element, "name", 
                                     myx_grt_value_from_string([description UTF8String]), 0);
  [_description release];
  _description= [description retain];
}


- (NSString*)description
{
  return _description;
}

- (void)dealloc
{
  [_description release];
  [super dealloc];
}

- (MYX_GRT_VALUE*)value
{
  return _element;
}
@end



@implementation MWLayerTreeNode

- (id)initWithGRT:(MYX_GRT*)grt layer:(MYX_GRT_VALUE*)value
{
  self= [super init];
  if (self)
  {
    _grt= grt;
    // do not retain this
    _layer= value;        

    _children= nil;
    _elements= nil;

    [self update];

    [self scanSublayers];
    [self scanElements];
  }
  return self;
}

- (void)dealloc
{
  [_description release];
  [_elements release];
  [_children release];
  [super dealloc];
}


- (void)setDescription:(NSString*)description
{
  myx_grt_bridge_dict_item_set_value(_layer, "name", 
                                     myx_grt_value_from_string([description UTF8String]), 0);
  [_description release];
  _description= [description retain];
}


- (NSString*)description
{
  return _description;
}

- (NSMutableArray*)children
{
  return _children;
}


- (NSArray*)elements
{
  return _elements;
}


- (void)update
{
  [_description release];
  _description= [[NSString stringWithUTF8String:myx_grt_bridge_value_as_string(myx_grt_bridge_dict_item_get_value(_layer,"name",0))] retain];
}


- (void)scanSublayers
{
  unsigned int i, c;
  NSMutableArray *newList= [NSMutableArray array];
  MYX_GRT_VALUE *layers= myx_grt_bridge_dict_item_get_value(_layer, "subLayers", 0);
  MYX_GRT_VALUE *layer;
  
  c= myx_grt_bridge_list_item_count(layers);
  for (i= 0; i < c; i++)
  {
    MWLayerTreeNode *node;
    layer= myx_grt_reference_cache_lookup(_grt, myx_grt_bridge_value_as_string(myx_grt_bridge_list_item_get(layers, i, 0)));
    
    node= [[MWLayerTreeNode alloc] initWithGRT:_grt layer:layer];
    [node scanSublayers];
    [newList addObject:node];
    [node release];
  }
  [_children release];
  _children= [newList retain];
}


- (void)scanElements
{
  MYX_GRT_VALUE *elements= myx_grt_bridge_dict_item_get_value(_layer, "elements", 0);
  unsigned int i, c;
  c= myx_grt_bridge_list_item_count(elements);
  
  [_elements release];
  _elements= [[NSMutableArray arrayWithCapacity:c] retain];
  
  for (i= 0; i < c; i++)
  {
    MYX_GRT_VALUE *elem= myx_grt_bridge_list_item_get(elements, i, 0);
    elem= myx_grt_reference_cache_lookup(_grt, myx_grt_bridge_value_as_string(elem));
    MWElementTreeNode *node= [[MWElementTreeNode alloc] initWithGRT:_grt element:elem];
    [_elements addObject:node];
    [node release];
  }
}  

- (MYX_GRT_VALUE*)value
{
  return _layer;
}

@end


@implementation MWLayerDataSource

- (id)initWithGRT:(MYX_GRT*)grt
{
  self= [super init];
  if (self)
  {
    _grt= grt;

    _layerTable= g_hash_table_new(g_direct_hash,g_direct_equal);
  }
  return self;
}


- (void)dealloc 
{
  g_hash_table_destroy(_layerTable);
  [_rootLayer release];
  [_path release];
  [super dealloc];
}


- (void)handleElementAdd:(MYX_GRT_VALUE*)value
{
  MYX_GRT_VALUE *layer= myx_grt_reference_cache_lookup(_grt, myx_grt_dict_item_get_as_string(value, "layer"));
  MWLayerTreeNode *node;
  
  node= g_hash_table_lookup(_layerTable, layer);
  [node scanElements];
}


- (void)handleElementChange:(MYX_GRT_VALUE*)value
{
  MYX_GRT_VALUE *layer= myx_grt_reference_cache_lookup(_grt, myx_grt_dict_item_get_as_string(value, "layer"));
  MWLayerTreeNode *node;
  
  node= g_hash_table_lookup(_layerTable, layer);
  [node scanElements];  
}

- (void)handleElementDelete:(MYX_GRT_VALUE*)value
{
  MYX_GRT_VALUE *layer= myx_grt_reference_cache_lookup(_grt, myx_grt_dict_item_get_as_string(value, "layer"));
  MWLayerTreeNode *node;
  
  node= g_hash_table_lookup(_layerTable, layer);
  [node scanElements];  
}


- (MWLayerTreeNode*)findParentLayerOf:(MYX_GRT_VALUE*)layer
                              inLayer:(MYX_GRT_VALUE*)root
  
{
  MYX_GRT_VALUE *subLayers= myx_grt_bridge_dict_item_get_value(root, "subLayers", 0);
  unsigned int i, c= myx_grt_bridge_list_item_count(subLayers);
  for (i= 0; i < c; i++)
  {
    MYX_GRT_VALUE *subLayer= myx_grt_reference_cache_lookup(_grt, 
                                                            myx_grt_bridge_value_as_string(myx_grt_bridge_list_item_get(subLayers, i, 0)));
    MWLayerTreeNode *node;
    
    if (subLayer == layer)
      return g_hash_table_lookup(_layerTable, root);
    
    node= [self findParentLayerOf:layer inLayer:subLayer];
    if (node)
      return node;
  }
  return nil;
}


- (MWLayerTreeNode*)handleLayerAdd:(MYX_GRT_VALUE*)value
{
  MWLayerTreeNode *node= [[MWLayerTreeNode alloc] initWithGRT:_grt layer:value];
  MWLayerTreeNode *parent;
  
  g_hash_table_insert(_layerTable, value, node);

  if (_rootLayer)
    parent= [self findParentLayerOf:value inLayer:[_rootLayer value]];
  else
    parent= nil;
  if (!parent)
  {
    _rootLayer= node;
  }
  else
  {
    [[parent children] addObject:node];
    [node release];
  }
  return node;
}


- (void)handleLayerDelete:(MYX_GRT_VALUE*)value
{
  MWLayerTreeNode *node, *parent;
  NSLog(@"DEL LAYER");
  node= g_hash_table_lookup(_layerTable, value);
  parent= [self findParentLayerOf:value inLayer:[_rootLayer value]];
  
  if (!parent)
    [[_rootLayer children] removeObject:node];
  else
    [[parent children] removeObject:node];
  
  g_hash_table_remove(_layerTable, value);
}


- (void)handleLayerChange:(MYX_GRT_VALUE*)value
{
  MWLayerTreeNode *node;
  
  node= g_hash_table_lookup(_layerTable, value);
  if (node)
    [node update];
}


- (void)handleLayerMove:(MYX_GRT_VALUE*)value
{
}


- (NSMutableArray*)rescanLayersFromList:(MYX_GRT_VALUE*)layers
{
  unsigned int i, c;
  NSMutableArray *newList= [NSMutableArray array];
  MYX_GRT_VALUE *layer;
  
  c= myx_grt_bridge_list_item_count(layers);
  for (i= 0; i < c; i++)
  {
    MWLayerTreeNode *node;
    layer= myx_grt_bridge_list_item_get(layers, i, 0);
    
    node= [[MWLayerTreeNode alloc] initWithGRT:_grt layer:layer];
    [node scanSublayers];
    [newList addObject:node];
    [node release];
  }
  return newList;
}


- (void)setRootLayerPath:(NSString*)path
{
  /*
  _path= [path retain];
 
  [_layers release];
  
  MYX_GRT_VALUE *layers= myx_grt_dict_item_get_by_path(myx_grt_get_root(_grt), 
                                                        [_path UTF8String]);
  if (layers)
    _layers= [self rescanLayersFromList:layers];
  else
    NSLog(@"no layers object");
  [_layers retain];
   */
}


- (int)outlineView:(NSOutlineView *)outlineView numberOfChildrenOfItem:(id)item
{
  if (!item)
    item= _rootLayer;
  
  if ([item isKindOfClass:[MWLayerTreeNode class]])
    return [[item children] count] + [[item elements] count];
  else
    return 0;
}


- (id)outlineView:(NSOutlineView *)outlineView objectValueForTableColumn:(NSTableColumn *)tableColumn byItem:(id)item
{
  if ([[tableColumn identifier] isEqualTo:@"description"])
    return [item description];
  return nil;
}


- (void)outlineView:(NSOutlineView *)outlineView
     setObjectValue:(id)object
     forTableColumn:(NSTableColumn *)tableColumn 
             byItem:(id)item
{
  if ([[tableColumn identifier] isEqualTo:@"description"])
  {
    return [item setDescription:object];
  }
}


- (BOOL)outlineView:(NSOutlineView *)outlineView isItemExpandable:(id)item
{
  return item==nil || [item isKindOfClass:[MWLayerTreeNode class]];
}


- (id)outlineView:(NSOutlineView *)outlineView child:(int)index ofItem:(id)parentItem
{
  if (!parentItem)
    parentItem= _rootLayer;

  unsigned int count= [[parentItem children] count];
  if (index < count)
    return [[parentItem children] objectAtIndex:index];
  else
    return [[parentItem elements] objectAtIndex:index-count];
  return nil;
}


@end

