#include <math.h>
#include <list>
#include <algorithm>
#include <set>
#include "GraphRenderer.h"

// equals for doubles
// algorithm by D.E.Knuth (TAOCP Vol.3 Sec. 4.2.2)
static bool eq(double d1, double d2)
{
  static int delta = 1;

  long long i1 = * (long long*) &d1;
  long long i2 = * (long long*) &d2;

  if(i1 < 0) 
  {
    i1 = -i1;//0x8000000000000000 - i1;
  }
  if(i2 < 0) 
  {
    i2 = -i2;//0x8000000000000000 - i2;
  }

  long long v = i1 - i2;
  if(v < 0)
  {
    v = -v;
  }

  return (v < delta);
}

double GraphNode::distance(const GraphNode& n1, const GraphNode& n2)
{
  double xdist, left, right, leftw;
  double ydist, top, bottom, toph;

  if(n1._left < n2._left) 
  {
    left = n1._left;
    leftw = n1._width;
    right = n2._left;
  } 
  else
  {
    left = n2._left;
    leftw = n2._width;
    right = n1._left;
  }

  xdist = right - left - leftw;
  if(xdist <= 0) 
  {
    xdist = 1;
  }

  if(n1._top < n2._top) 
  {
    top = n1._top;
    toph = n1._height;
    bottom = n2._top;
  } 
  else
  {
    top = n2._top;
    toph = n2._height;
    bottom = n1._top;
  }

  ydist = bottom - top - toph;
  if(ydist <= 0) 
  {
    ydist = 1;
  }

  return sqrt(xdist*xdist + ydist*ydist);

  //double dx = n1.centerx() - n2.centerx();
  //double dy = n1.centery() - n2.centery();
  //return sqrt(dx*dx + dy*dy);
}

bool operator == (const GraphNode& n1, const GraphNode& n2) 
{
  return (n1._left == n2._left) && (n1._top == n2._top) && (n1._width == n2._width) && (n1._height == n2._height); 
}

GraphRenderer::GraphRenderer() : _length(50)
{
}

GraphRenderer::~GraphRenderer()
{
}

void GraphRenderer::get_neighbours(const GraphNode& node, GraphNodeRefList& to) const
{
  for(GraphEdgeList::const_iterator it= _alledges.begin(); it != _alledges.end(); it++)
  {
    GraphEdge e= *it;
    if(e.contains(node))
    {
      to.push_back(&e.get_other(node));
    }
  }
}

bool GraphRenderer::is_focus_node(const GraphNode& node, const GraphNodeRefList& neighbours) const
{
  for(GraphNodeRefList::const_iterator it= neighbours.begin(); it != neighbours.end(); it++)
  {
    GraphNode& neinode= **it;
    for(GraphEdgeList::const_iterator edgeit= _alledges.begin(); edgeit != _alledges.end(); edgeit++)
    {
      GraphEdge e= *edgeit;
      if(e.contains(neinode) && !e.contains(node))
      {
        return false;
      }
    }
  }
  return true;
}

int GraphRenderer::get_delta(const GraphNode& node, GraphNode::Coord c) const
{
  GraphNodeRefList neighbours;
  get_neighbours(node, neighbours);

  int cc= node.coord(c);
  bool node_is_focus= is_in_focus_list(node);

  // sum for neighbours
  double nei_sum= 0;
  for(GraphNodeRefList::const_iterator it= neighbours.begin(); it != neighbours.end(); it++)
  {
    const GraphNode& nei= **it;
    double d= GraphNode::distance(node, nei);
    if(eq(d, 0.))
    {
      continue;
    }
    double v= -(d - _length)*(cc - nei.coord(c));
    v /= d;
    double k= (node_is_focus || is_in_focus_list(nei)) ? GraphRenderer::K1F : GraphRenderer::K1N;
    v /= k;
    nei_sum += v;
  }

  // sum for all nodes
  double all_sum= 0;
  for(GraphNodeRefList::const_iterator it= _allnodes.begin(); it != _allnodes.end(); it++) 
  {
    const GraphNode& nei= **it;
    if(nei == node)
    {
      continue;
    }
    double d= GraphNode::distance(node, nei);
    if(eq(d, 0.))
    {
      continue;
    }
    double v = cc - nei.coord(c);
    v *= GraphRenderer::K2;
    v /= d;
    v /= d;
    all_sum += v;
  }

  // sum for focus nodes
  double fcs_sum= 0;
  for(GraphNodeRefList::const_iterator it= _focusnodes.begin(); it != _focusnodes.end(); it++)
  {
    const GraphNode& fn= **it;
    double d= GraphNode::distance(node, fn);
    if(fn == node)
    {
      continue;
    }
    if(eq(d, 0.))
    {
      continue;
    }
    double v = cc - fn.coord(c);
    v *= GraphRenderer::K3;
    v /= d;
    v /= d;
    fcs_sum += v;
  }

  double t= 5.; // threshold
  double sum= nei_sum + all_sum + fcs_sum;

  if(sum >= t)
    return (int)t;
  if(sum <= -t)
    return (int)-t;
  return 0;
}

void GraphRenderer::add_node(GraphNode& node) 
{ 
  _allnodes.push_back(&node); 
  _framenodes.push_back(&node); 
  
  GraphNodeRefList neighbours;
  get_neighbours(node, neighbours);

  if(is_focus_node(node, neighbours)) 
  {
    _focusnodes.push_back(&node);
  }
}

void GraphRenderer::add_edge(GraphEdge& edge) 
{ 
  _alledges.push_back(edge); 
}

void GraphRenderer::recalc_outer_rect()
{
  _left = INT_MAX;
  _top = INT_MAX;
  _right = INT_MIN;
  _bottom = INT_MIN;

  for(GraphRenderer::GraphNodeRefList::iterator it= _allnodes.begin(); it != _allnodes.end(); it++) 
  {
    GraphNode& node = **it;
    int left = node.left();
    int top = node.top();
    int width = node.width();
    int height = node.height();
    
    if(_left > left) {
      _left = left;
    } 
    if(_right < left+width) {
      _right = left+width;
    }
    if(_top > top) {
      _top = top;
    } 
    if(_bottom < top+height) {
      _bottom = top+height;
    } 
  }
}

void GraphRenderer::recalc_positions()
{
  CoordSet cs;

  for(GraphRenderer::GraphNodeRefList::iterator it= _allnodes.begin(); it != _allnodes.end(); it++) 
  {
    GraphNode& node = **it;
    node.setnewleft(node.left() + get_delta(node, GraphNode::CX));
    node.setnewtop(node.top() + get_delta(node, GraphNode::CY));
    std::pair<CoordSet::iterator, bool> pr = cs.insert(CoordPair(node.newleft(), node.newtop()));
    while(!pr.second) 
    {
      node.setnewleft(node.newleft() + 1);
      node.setnewtop(node.newtop() + 1);
      pr = cs.insert(CoordPair(node.newleft(), node.newtop()));
    }
    //node.apply();
  }

  for(GraphRenderer::GraphNodeRefList::iterator it= _allnodes.begin(); it != _allnodes.end(); it++) 
  {
    GraphNode& node = **it;
    node.apply();
  }
}

void GraphRenderer::rotate_point(int *x, int *y, double matrix[2][2])
{
  *x = (int)( *x*matrix[0][0] + *y*matrix[0][1] );
  *y = (int)( *x*matrix[1][0] + *y*matrix[1][1] );
}

void GraphRenderer::move(int dx, int dy)
{
  for(GraphRenderer::GraphNodeRefList::iterator it= _allnodes.begin(); it != _allnodes.end(); it++) 
  {
    GraphNode& node = **it;
    node.setnewleft(node.left() + dx);
    node.setnewtop(node.top() + dy);
    node.apply();
  }
}

void GraphRenderer::rotate()
{
  static double pi = 3.1415926535;
  //static double step = pi/120.;
  static double step = pi/12000.;
  static double rm1[2][2] = { { cos(step), sin(step) }, { -sin(step), cos(step) } };  // +step rotation matrix
  static double rm2[2][2] = { { cos(step), -sin(step) }, { sin(step), cos(step) } };  // -step rotation matrix

  int curmsd = 0, posmsd = 0, negmsd = 0;
  int yzero = (_top + _bottom)/2;
  int xzero = (_left + _right)/2;

  for(GraphRenderer::GraphNodeRefList::iterator it= _allnodes.begin(); it != _allnodes.end(); it++) 
  {
    GraphNode& node = **it;
    int x = node.centerx() - xzero;
    int y = node.centery() - yzero;
    int x1 = x;
    int y1 = y;
    int x2 = x;
    int y2 = y;

    rotate_point(&x1, &y1, rm1);
    rotate_point(&x2, &y2, rm2);

    curmsd += y*y;
    posmsd += y1*y1;
    negmsd += y2*y2;
  }

  //if((curmsd < posmsd) && (curmsd < negmsd)) 
  //{
  //  return;
  //}
  
  for(GraphRenderer::GraphNodeRefList::iterator it= _allnodes.begin(); it != _allnodes.end(); it++) 
  {
    GraphNode& node = **it;
    //int x = node.left() - xzero;
    //int y = node.top() - yzero;
    int x = node.centerx() - xzero;
    int y = node.centery() - yzero;
    rotate_point(&x, &y, (posmsd < negmsd) ? rm1 : rm2);
    node.setnewleft(x - node.width()/2 + xzero);
    node.setnewtop(y - node.height()/2 + yzero);
    node.apply();
  }
}

void GraphRenderer::recalc()
{
  recalc_length();
  recalc_positions();
  rotate();
  recalc_outer_rect();
}

void GraphRenderer::get_outer_rect(int *left, int *top, int *right, int *bottom)
{
  *left = _left;
  *top = _top;
  *right = _right;
  *bottom = _bottom;
}

void GraphRenderer::recalc_length()
{
  _density_ratio = 0.;
  for(GraphRenderer::GraphNodeRefList::iterator it= _allnodes.begin(); it != _allnodes.end(); it++) 
  {
    GraphNode& node = **it;
    double w = node.width();
    double h = node.height();
    _density_ratio += w*h;
  }
  _density_ratio /= (_right - _left)*(_bottom - _top);
  _length = (int) (1000.*_density_ratio*_density_ratio);
}
