unit AdminServerHealth;

interface

uses
  gnugettext, Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, StdCtrls, ExtCtrls, StrUtils, Contnrs, Clipbrd,
  Series, InstanceSections, TntClasses, TntForms,
  ApplicationDataModule, AuxFuncs, Translations,
  myx_admin_public_interface, myx_util_public_interface,
  myx_public_interface,
  AdditionalClasses,
  AdminServerHealthLineGraph,
  AdminServerHealthBarGraph, ImgList, Menus,
  PNGImage, MySQLConnection,
  Options, MyxError, TntExtCtrls,
  AdminServerHealthGraphSettings, TntMenus, TntStdCtrls, TntComCtrls;

type
  TVarListId = class (TObject)
    mysql_id: WideString;

    constructor Create(mysql_id: WideString);
  end;

  TAdminServerHealthForm = class(TInstanceSectionForm)
    ServerStatusPnl: TTntPanel;
    HealthPageControl: TTntPageControl;
    ServerStatusTabSheet: TTabSheet;
    ServerVariablesTabSheet: TTabSheet;
    ServerStatusMainPnl: TTntPanel;
    ServerStatusTreeView: TTntTreeView;
    StatusVarSplitter: TTntSplitter ;
    ServerStatusListView: TTntListView;
    ServerVariablesMainPnl: TTntPanel;
    ServerVarSplitter: TTntSplitter ;
    ServerVariablesTreeView: TTntTreeView;
    VariablesListView: TTntListView;
    GraphRefreshTmr: TTimer;
    VarImageList: TImageList;
    VariablesPopupMenu: TTntPopupMenu;
    CopyVariablestoClipboardMI: TTntMenuItem;
    CopyselectedVariableMI: TTntMenuItem;
    HeaderPnl: TTntPanel;
    MemoryHealthBevel: TTntBevel;
    HeaderDescriptionLbl: TTntLabel;
    HeaderImg: TTntImage;
    HeaderLbl: TTntLabel;
    BottomPnl: TTntPanel;
    RefreshBtn: TTntButton;
    Panel1: TTntPanel;
    Button1: TTntButton;
    CopyselectedVariableNameMI: TTntMenuItem;
    PageControlPopupMenu: TTntPopupMenu;
    AddaPageMI: TTntMenuItem;
    AddaGroupMI: TTntMenuItem;
    DeletePageMI: TTntMenuItem;
    N1: TTntMenuItem;
    GroupBoxPopupMenu: TTntPopupMenu;
    AddaGraphMI: TTntMenuItem;
    DeleteGroupMI: TTntMenuItem;
    N2: TTntMenuItem;
    GraphPopupMenu: TTntPopupMenu;
    EditGraphMI: TTntMenuItem;
    N3: TTntMenuItem;
    DeleteGraphMI: TTntMenuItem;
    AddaGraph2MI: TTntMenuItem;
    DeleteGroup2MI: TTntMenuItem;
    AddaGroup2MI: TTntMenuItem;
    AddaGroup3MI: TTntMenuItem;
    RestoreDefaultsMI: TTntMenuItem;
    RestoreDefaults2MI: TTntMenuItem;
    N4: TTntMenuItem;
    N5: TTntMenuItem;
    N6: TTntMenuItem;
    N7: TTntMenuItem;
    RestoreDefaults3MI: TTntMenuItem;
    DeletePage2MI: TTntMenuItem;
    DeletePage3MI: TTntMenuItem;
    AddPage2MI: TTntMenuItem;
    AddaPage3MI: TTntMenuItem;
    MoveGraphUpMI: TTntMenuItem;
    MoveGraphDownMI: TTntMenuItem;
    N8: TTntMenuItem;
    N9: TTntMenuItem;
    GraphPopupMenuImgs: TImageList;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ServerStatusPnlResize(Sender: TObject);
    procedure HealthPageControlChange(Sender: TObject);
    procedure RefreshBtnClick(Sender: TObject);
    procedure BuildVariablesTreeViews;
    procedure RefreshVars(Treeview: TTntTreeView; var CurrentVarGroup: Pointer;
      ListView: TTntListView; Vars: TMYX_VARIABLES;
      VarListing: TMYX_VARIABLES_LISTING; var NotListedVars: TTntStringList);

    procedure FetchStatusVars(Sender: TObject);
    procedure FetchedStatusVars(Sender: TObject);
    procedure RefreshStatusVars;
    procedure FetchServerVars(Sender: TObject);
    procedure FetchedServerVars(Sender: TObject);
    procedure RefreshServerVars;
    procedure ServerStatusTreeViewChange(Sender: TObject; Node: TTreeNode);
    procedure ServerVariablesTreeViewChange(Sender: TObject;
      Node: TTreeNode);

    procedure FetchStatusVarsTimer(Sender: TObject);
    procedure FetchedStatusVarsTimer(Sender: TObject);
    procedure GraphRefreshTmrTimer(Sender: TObject);
    function GetVarIndex(vars: TMYX_VARIABLES;
      VarName: WideString): integer;
    procedure PrepareGraphFormulars;

    procedure LoadGraphs;
    procedure SaveGraphs;
    procedure BuildGraphs;
    procedure AddPage(page: TMYX_HEALTH_PAGE;
      var PageTabSheet: TTabSheet; var ScrollBox: TTntScrollBox);
    procedure AddGraphToGroupBox(GroupBox: TTntGroupBox;
      graph: TMYX_HEALTH_GRAPH; var current_widget_ypos: integer);

    procedure CopyVariablestoClipboardMIClick(Sender: TObject);
    procedure VariablesListViewDblClick(Sender: TObject);
    procedure CopyselectedVariableMIClick(Sender: TObject);
    procedure CopyselectedVariableNameMIClick(Sender: TObject);

    procedure Disconnected(var Message: TMessage); message WM_Disconnected;
    procedure Reconnected(var Message: TMessage); message WM_Reconnected;

    procedure AddaPageMIClick(Sender: TObject);
    procedure DeletePageMIClick(Sender: TObject);
    procedure AddaGroupMIClick(Sender: TObject);
    procedure EditGraphMIClick(Sender: TObject);
    procedure ServerStatusListViewMouseDown(Sender: TObject;
      Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure AddaGraphMIClick(Sender: TObject);

    procedure OptionsChanged(var Message: TMessage); message WM_OptionsChanged;
    procedure DeleteGraphMIClick(Sender: TObject);
    procedure DeleteGraph(Graph: TControl);
    procedure DeleteGroupMIClick(Sender: TObject);
    procedure DeleteGroupBox(GroupBox: TTntGroupBox);
    procedure RestoreDefaultsMIClick(Sender: TObject);
    procedure DeletePage(pos: integer; AskBeforeDelete: Boolean = True);
    procedure PageControlPopupMenuPopup(Sender: TObject);
    procedure MoveGraphUpMIClick(Sender: TObject);
    procedure MoveGraph(Frame: TTntFrame; MoveUp: Boolean);
    procedure MoveGraphDownMIClick(Sender: TObject);
    procedure ListViewDeletion(Sender: TObject; Item: TListItem);
    procedure MenuDrawItem(Sender: TObject; ACanvas: TCanvas; ARect: TRect; State: TOwnerDrawState);
    procedure MenuMeasureItem(Sender: TObject; ACanvas: TCanvas; var Width, Height: Integer);
  private
    server_vars,
    status_vars,
    status_vars_timer,
    status_vars_timer_old: TMYX_VARIABLES;

    StatusVarListing,
    ServerVarListing: TMYX_VARIABLES_LISTING;

    NotListedStatusVars,
    NotListedServerVars: TTntStringList;

    CurrentStatusVarGroup,
    CurrentServerVarGroup: Pointer;

    StatusVar_PMySQL: Pointer;

    GraphFrameBundleList: TObjectList;

    ClosingApplication: Boolean;

    HeaderPNGImg: TPNGObject;

    AdminServerHealthGraphSettingsForm: TAdminServerHealthGraphSettingsForm;

    GraphControlCount: integer;
  public
    HealthPages: TMYX_HEALTH_PAGES;

    RefreshSystemVarsInFormulas: Boolean;
  end;

const
  GroupBoxWidth = 535;
  GroupBoxVSpace = 10;

var
  AdminServerHealthForm: TAdminServerHealthForm;

implementation

{$R *.dfm}

uses
  PNGTools;
  
constructor TVarListId.Create(mysql_id: WideString);
begin
  self.mysql_id:=mysql_id;
end;

procedure TAdminServerHealthForm.FormCreate(Sender: TObject);
begin
  InitForm(self);
  
  DockedPanel:=ServerStatusPnl;

  server_vars:=nil;
  status_vars:=nil;
  status_vars_timer:=nil;
  status_vars_timer_old:=nil;

  StatusVarListing:=nil;
  ServerVarListing:=nil;

  CurrentStatusVarGroup:=nil;
  CurrentServerVarGroup:=nil;

  NotListedStatusVars:=nil;
  NotListedServerVars:=nil;

  StatusVar_PMySQL:=nil;

  GraphControlCount:=0;

  AdminServerHealthGraphSettingsForm:=nil;

  ClosingApplication:=False;

  HealthPages:=nil;

  RefreshSystemVarsInFormulas:=False;

  HeaderPNGImg:=LoadPNGImageFromResource('health', HeaderImg);

  BuildVariablesTreeViews;

  ServerStatusTreeView.FullExpand;
  ServerVariablesTreeView.FullExpand;

  if(MYXCommonOptions.XPStyleEnabled)then
  begin
    StatusVarSplitter.Color:=clWhite;
    ServerVarSplitter.Color:=clWhite;
  end;

  GraphFrameBundleList:=TObjectList.Create;

  //Paint ProgressGraphs for the first time
  //ServerStatusPnlResize(self);

  if(MySQLConn.Connected)then
  begin
    BuildGraphs;

    GraphRefreshTmr.Enabled:=True;
  end
  else
  begin
    DisableEnableControls(ServerStatusTabSheet, False);
    DisableEnableControls(ServerVariablesTabSheet, False);
  end;

  HealthPageControl.ActivePageIndex:=0;
  HealthPageControlChange(self);
end;

procedure TAdminServerHealthForm.FormDestroy(Sender: TObject);

begin
  HeaderPNGImg.Free;
  AdminServerHealthGraphSettingsForm.Free;
  GraphFrameBundleList.Free;

  server_vars.Free;
  status_vars.Free;
  status_vars_timer.Free;
  status_vars_timer_old.Free;

  StatusVarListing.Free;
  ServerVarListing.Free;

  NotListedStatusVars.Free;
  NotListedServerVars.Free;

  if(StatusVar_PMySQL<>nil)then
    myx_mysql_close(StatusVar_PMySQL);

  if Assigned(HealthPages) then
  begin
    myx_save_health_pages(HealthPages.get_record_pointer,
      MYXCommonOptions.UserDataDir+'mysqladmin_health.xml');

    HealthPages.Free;
  end;
end;

procedure TAdminServerHealthForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  if(AdminServerHealthGraphSettingsForm<>nil)then
    AdminServerHealthGraphSettingsForm.Close;
    
  ClosingApplication:=True;
end;

procedure TAdminServerHealthForm.ServerStatusPnlResize(
  Sender: TObject);
var i: integer;
begin
  for i:=0 to GraphFrameBundleList.Count-1 do
    if(TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame is TAdminServerHealthLineGraphFrame)then
      TAdminServerHealthLineGraphFrame(TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame).FrameResize(self);
end;

procedure TAdminServerHealthForm.HealthPageControlChange(
  Sender: TObject);
begin
  HeaderPnl.Parent:=HealthPageControl.ActivePage;
  HeaderPnl.Align:=alTop;
  HeaderPnl.Visible:=True;

  if(HealthPageControl.ActivePage=ServerStatusTabSheet)then
  begin
    HeaderLbl.Caption:=_('Status Variables');
    HeaderDescriptionLbl.Caption:=_('Current status variables of the server.');
  end
  else if(HealthPageControl.ActivePage=ServerVariablesTabSheet)then
  begin
    HeaderLbl.Caption:=_('System Variables');
    HeaderDescriptionLbl.Caption:=_('Overview of the system variables.');
  end
  else
  begin
    if(HealthPageControl.ActivePageIndex<=HealthPages.pages.Count)then
    begin
      HeaderLbl.Caption:=HealthPages.pages[HealthPageControl.ActivePageIndex].caption;
      HeaderDescriptionLbl.Caption:=HealthPages.pages[HealthPageControl.ActivePageIndex].description;
    end;
  end;

  ServerStatusPnlResize(self);
end;

procedure TAdminServerHealthForm.RefreshBtnClick(Sender: TObject);
begin
  if(HealthPageControl.ActivePage=ServerStatusTabSheet)then
  begin
    FreeAndNil(status_vars);
    RefreshStatusVars;
  end
  else if(HealthPageControl.ActivePage=ServerVariablesTabSheet)then
  begin
    FreeAndNil(server_vars);
    RefreshServerVars;
  end;
end;

procedure TAdminServerHealthForm.BuildVariablesTreeViews;
var PVarListing: PMYX_VARIABLES_LISTING;
  error: MYX_ADMIN_LIB_ERROR;
  i, j: integer;
  GroupNode: TTntTreeNode;
begin
  PVarListing:=myx_get_variables_listing(
    MYXCommonOptions.XMLDir+'mysqladmin_status_variables.xml', @error);
  if(error<>MYX_ADMIN_NO_ERROR)then
    raise EMyxError.Create(_('The status variables definition could '+
      'not be loaded from the XML file.')+#13#10+
      _('Error Nr.')+IntToStr(Ord(error)));
  try
    StatusVarListing:=TMYX_VARIABLES_LISTING.create(PVarListing);
  finally
    myx_free_variables_listing(PVarListing);
  end;

  ServerStatusTreeView.Items.BeginUpdate;
  try
    ServerStatusTreeView.Items.Clear;

    for i:=0 to StatusVarListing.groups.Count-1 do
    begin
      GroupNode:=AddTreeViewChildNode(ServerStatusTreeView, nil,
        StatusVarListing.groups[i].name, 4, StatusVarListing.groups[i]);

      for j:=0 to StatusVarListing.groups[i].subgroups.Count-1 do
      begin
        AddTreeViewChildNode(ServerStatusTreeView, GroupNode,
          StatusVarListing.groups[i].subgroups[j].name, 4,
          StatusVarListing.groups[i].subgroups[j]);
      end;
    end;
  finally
    ServerStatusTreeView.Items.EndUpdate;
  end;


  PVarListing:=myx_get_variables_listing(
   MYXCommonOptions.XMLDir+'mysqladmin_system_variables.xml', @error);
  if(error<>MYX_ADMIN_NO_ERROR)then
    raise EMyxError.Create(_('The status variables definition could '+
      'not be loaded from the XML file.') + #13#10+
      _('Error Nr.')+IntToStr(Ord(error)));
  try
    ServerVarListing:=TMYX_VARIABLES_LISTING.create(PVarListing);
  finally
    myx_free_variables_listing(PVarListing);
  end;

  ServerVariablesTreeView.Items.BeginUpdate;
  try
    ServerVariablesTreeView.Items.Clear;

    for i:=0 to ServerVarListing.groups.Count-1 do
    begin
      GroupNode:=AddTreeViewChildNode(ServerVariablesTreeView, nil,
        ServerVarListing.groups[i].name, 4, ServerVarListing.groups[i]);

      for j:=0 to ServerVarListing.groups[i].subgroups.Count-1 do
      begin
        AddTreeViewChildNode(ServerVariablesTreeView, GroupNode,
          ServerVarListing.groups[i].subgroups[j].name, 4,
          ServerVarListing.groups[i].subgroups[j]);
      end;
    end;
  finally
    ServerVariablesTreeView.Items.EndUpdate;
  end;
end;

procedure TAdminServerHealthForm.RefreshVars(Treeview: TTntTreeView;
  var CurrentVarGroup: Pointer; ListView: TTntListView; Vars: TMYX_VARIABLES;
  VarListing: TMYX_VARIABLES_LISTING; var NotListedVars: TTntStringList);
var i, j, k, l: integer;
  Item: TListItem;
  found: Boolean;
  VarGrp: TMYX_VARIABLES_GROUP;
begin
  if(VarListing=nil)then
    Exit;

  //get all vars that are not listed
  if(NotListedVars=nil)then
  begin
    NotListedVars:=TTntStringList.Create;

    for k:=0 to Vars.variables.Count-1 do
    begin
      found:=False;

      //search for a description of Vars.variables[k].name
      for i:=0 to VarListing.groups.Count-1 do
      begin
        //compare with all variables of this group
        for j:=0 to VarListing.groups[i].variables.Count-1 do
        begin
          if(CompareText(VarListing.groups[i].variables[j].mysql_id,
            Vars.variables[k].name)=0)then
          begin
            found:=True;
            break;
          end;
        end;

        //compare with all variables in all subgroups of this group
        if(Not(found))then
        begin
          for j:=0 to VarListing.groups[i].subgroups.Count-1 do
          begin
            for l:=0 to VarListing.groups[i].subgroups[j].variables.Count-1 do
              if(CompareText(VarListing.groups[i].subgroups[j].variables[l].mysql_id,
                Vars.variables[k].name)=0)then
              begin
                found:=True;
                break;
              end;

            if(found)then
              break;
          end;
        end;

        if(found)then
          break;
      end;

      if(Not(found))then
        NotListedVars.Add(Vars.variables[k].name);
    end;

    if(NotListedVars.Text<>'')then
    begin
      VarGrp:=TMYX_VARIABLES_GROUP.create(VarListing.groups.Count,
        'New Variables', '');
      VarListing.groups.Add(VarGrp);

      for i:=0 to NotListedVars.Count-1 do
        VarGrp.variables.Add(TMYX_VARIABLE_ELEMENT.create(
          NotListedVars[i], 'New Variable', 0));

      AddTreeViewChildNode(Treeview, nil,
        'New Variables', 4, VarGrp);
    end;
  end;


  ListView.Items.BeginUpdate;
  try
    if(Treeview.Selected<>nil)then
      if(Treeview.Selected.Data<>nil)then
      begin
        if(CurrentVarGroup<>Treeview.Selected.Data)then
        begin
          if(TObject(Treeview.Selected.Data) is TMYX_VARIABLES_GROUP)then
          begin
            ListView.Items.Clear;

            for i:=0 to Vars.variables.Count-1 do
              for j:=0 to TMYX_VARIABLES_GROUP(Treeview.Selected.Data).variables.Count-1 do
                if(Vars.variables[i].name=
                  TMYX_VARIABLES_GROUP(Treeview.Selected.Data).variables[j].mysql_id)then
                begin
                  Item:=AddListViewItem(ListView, nil,
                    Vars.variables[i].name,
                    TMYX_VARIABLES_GROUP(Treeview.Selected.Data).variables[j].editable,
                    TVarListId.Create(Vars.variables[i].name));

                  Item.SubItems.Add(Vars.variables[i].value);
                  Item.SubItems.Add(_T('MySQLVariables',
                    TMYX_VARIABLES_GROUP(Treeview.Selected.Data).variables[j].desc_id,
                    ''));
                end;

            CurrentVarGroup:=Treeview.Selected.Data;
          end
          else if(TObject(Treeview.Selected.Data) is TMYX_VARIABLES_SUBGROUP)then
          begin
            if(CurrentVarGroup<>Treeview.Selected.Data)then
            begin
              ListView.Items.Clear;

              for i:=0 to Vars.variables.Count-1 do
                for j:=0 to TMYX_VARIABLES_SUBGROUP(Treeview.Selected.Data).variables.Count-1 do
                  if(Vars.variables[i].name=
                    TMYX_VARIABLES_SUBGROUP(Treeview.Selected.Data).variables[j].mysql_id)then
                  begin
                    Item:=AddListViewItem(ListView, nil,
                      Vars.variables[i].name, TMYX_VARIABLES_SUBGROUP(Treeview.Selected.Data).variables[j].editable,
                      TVarListId.Create(Vars.variables[i].name));

                    Item.SubItems.Add(Vars.variables[i].value);
                    Item.SubItems.Add(_T('MySQLVariables',
                      TMYX_VARIABLES_SUBGROUP(Treeview.Selected.Data).variables[j].desc_id,
                      ''));
                  end;

              CurrentVarGroup:=Treeview.Selected.Data;
            end;
          end;
        end
        else
        begin
          //Current Group has NOT changed, so simply refresh Values
          for i:=0 to Vars.variables.Count-1 do
            for j:=0 to ListView.Items.Count-1 do
              if(ListView.Items[j].Data<>nil)then
                if(TObject(ListView.Items[j].Data) is TVarListId)then
                  if(Vars.variables[i].name=
                    TVarListId(ListView.Items[j].Data).mysql_id)then
                  begin
                    if(ListView.Items[j].SubItems.Count>0)then
                      ListView.Items[j].SubItems[0]:=Vars.variables[i].value;
                  end;

        end;
      end;
  finally
    ListView.Items.EndUpdate;
  end;
end;

procedure TAdminServerHealthForm.FetchStatusVars(Sender: TObject);
var pvars: PMYX_VARIABLES;
begin
  pvars:=myx_get_status_variables(MySQLConn.MySQL);
  if(pvars=nil)then
    raise EMyxError.Create(_('Error while fetching Status Variables.'));
  try
    status_vars:=TMYX_VARIABLES.create(pvars);
  finally
    myx_free_variables(pvars);
  end;
end;

procedure TAdminServerHealthForm.FetchedStatusVars(Sender: TObject);
begin
  RefreshStatusVars;
end;

procedure TAdminServerHealthForm.RefreshStatusVars;
begin
  if(Not(MySQLConn.Connected))then
    Exit;

  if(status_vars=nil)then
    MySQLConn.FetchData(KindOfData_StatusVars,
      FetchStatusVars, FetchedStatusVars,
      nil, _('Fetching Status Variables ...'))
  else
    RefreshVars(ServerStatusTreeView, CurrentStatusVarGroup,
      ServerStatusListView, status_vars,
      StatusVarListing, NotListedStatusVars);
end;

procedure TAdminServerHealthForm.FetchServerVars(Sender: TObject);
var pvars: PMYX_VARIABLES;
begin
  pvars:=myx_get_server_variables(MySQLConn.MySQL);
  if(pvars=nil)then
    raise EMyxError.Create(_('Error while fetching Server Variables.'));
  try
    server_vars:=TMYX_VARIABLES.create(pvars);
  finally
    myx_free_variables(pvars);
  end;
end;

procedure TAdminServerHealthForm.FetchedServerVars(Sender: TObject);
begin
  RefreshServerVars;
end;

procedure TAdminServerHealthForm.RefreshServerVars;
begin
  if(Not(MySQLConn.Connected))then
    Exit;

  if(server_vars=nil)then
    MySQLConn.FetchData(KindOfData_ServerVars,
      FetchServerVars, FetchedServerVars,
      nil, _('Fetching System Variables ...'))
  else
  begin
    RefreshVars(ServerVariablesTreeView, CurrentServerVarGroup,
      VariablesListView, server_vars,
      ServerVarListing, NotListedServerVars);

    RefreshSystemVarsInFormulas:=True;
  end;
end;

procedure TAdminServerHealthForm.ServerStatusTreeViewChange(
  Sender: TObject; Node: TTreeNode);
begin
  if(status_vars=nil)then
    RefreshStatusVars
  else
    RefreshVars(ServerStatusTreeView,
      CurrentStatusVarGroup, ServerStatusListView, status_vars,
      StatusVarListing, NotListedStatusVars);
end;

procedure TAdminServerHealthForm.ServerVariablesTreeViewChange(
  Sender: TObject; Node: TTreeNode);
begin
  if(server_vars=nil)then
    RefreshServerVars
  else
    RefreshVars(ServerVariablesTreeView,
      CurrentServerVarGroup, VariablesListView, server_vars,
      ServerVarListing, NotListedServerVars);
end;

procedure TAdminServerHealthForm.CopyVariablestoClipboardMIClick(
  Sender: TObject);
var i: integer;
  txt: WideString;
  ListView: TTntListView;
begin
  if(HealthPageControl.ActivePage=ServerStatusTabSheet)then
    ListView:=ServerStatusListView
  else
    ListView:=VariablesListView;

  with ListView do
  begin
    txt:='';
    for i:=0 to Items.Count-1 do
    begin
      txt:=txt+StringAlignLeft(Items[i].Caption, 32);
      if(Items[i].SubItems.Count>0)then
        txt:=txt+Items[i].SubItems[0];
      txt:=txt+#13#10;
    end;

    Clipboard.AsText:=txt;
  end;
end;

procedure TAdminServerHealthForm.CopyselectedVariableMIClick(
  Sender: TObject);
var
  txt: WideString;
  ListView: TTntListView;
begin
  if(HealthPageControl.ActivePage=ServerStatusTabSheet)then
    ListView:=ServerStatusListView
  else
    ListView:=VariablesListView;

  if(ListView.Selected<>nil)then
  begin
    txt:=txt+StringAlignLeft(ListView.Selected.Caption, 32);
    if(ListView.Selected.SubItems.Count>0)then
      txt:=txt+ListView.Selected.SubItems[0];

    Clipboard.AsText:=txt;
  end;
end;

procedure TAdminServerHealthForm.CopyselectedVariableNameMIClick(
  Sender: TObject);
var
  txt: WideString;
  ListView: TTntListView;
begin
  if(HealthPageControl.ActivePage=ServerStatusTabSheet)then
    ListView:=ServerStatusListView
  else
    ListView:=VariablesListView;

  if(ListView.Selected<>nil)then
  begin
    txt:=txt+ListView.Selected.Caption;

    Clipboard.AsText:=txt;
  end;
end;

procedure TAdminServerHealthForm.VariablesListViewDblClick(
  Sender: TObject);
var  ListView: TTntListView;
  value: WideString;
begin
  if(HealthPageControl.ActivePage=ServerStatusTabSheet)then
    ListView:=ServerStatusListView
  else
    ListView:=VariablesListView;

  if(ListView.Selected<>nil)then
    if(ListView.Selected.ImageIndex=1)and
      (ListView.Selected.Data<>nil)then
    begin
      if(ListView.Selected.SubItems.Count>0)then
        value:=ListView.Selected.SubItems[0]
      else
        value:='';

      if(ShowModalEditDialog(_('Set Variable'),
        _('Please enter the new value of the variable.'),
        myx_mtEdit, _('Set')+#13#10+_('Abort'),
        True, TVarListId(ListView.Selected.Data).mysql_id+':',
        value)=1)then
      begin
        if(myx_set_variable(MySQLConn.MySQL,
          TVarListId(ListView.Selected.Data).mysql_id,
          value)<>0)then
        begin
          raise EMyxSQLError.Create(
            Format(_('Variable %s cannot be set.'), [TVarListId(ListView.Selected.Data).mysql_id]),
            myx_mysql_errno(MySQLConn.MySQL),
            myx_mysql_error(MySQLConn.MySQL));
        end
        else
          RefreshBtnClick(self);
      end;
    end;
end;

procedure TAdminServerHealthForm.FetchStatusVarsTimer(Sender: TObject);
var pvars: PMYX_VARIABLES;
  MySQLErrorNr: Integer;
begin
  if(StatusVar_PMySQL=nil)then
  begin
    StatusVar_PMySQL:=myx_mysql_init();
    if(StatusVar_PMySQL=nil)then
      raise EMyxError.Create(_('Error while allocating memory for MySQL Struct.'));

    if(myx_connect_to_instance(
      MySQLConn.User_Connection.get_record_pointer,
      StatusVar_PMySQL)<>0)then
    begin
      StatusVar_PMySQL:=nil;
      Exit;
    end;
  end;

  pvars:=myx_get_status_variables(StatusVar_PMySQL);
  if(pvars=nil)then
  begin
    MySQLErrorNr:=myx_mysql_errno(StatusVar_PMySQL);

    if(MySQLErrorNr=CR_SERVER_GONE_ERROR)or
      (MySQLErrorNr=CR_SERVER_LOST)or
      (MySQLErrorNr=CR_CONN_HOST_ERROR)then
    begin
      MySQLConn.Disconnect;
      StatusVar_PMySQL:=nil;
    end;

    TFetchDataThread(Sender).StatusBarText:=
      format(_('Error fetching Status Variables (%s)'),
             [IntToStr(MySQLErrorNr)]);
    Exit;
  end;

  try
    status_vars_timer:=TMYX_VARIABLES.create(pvars);
  finally
    myx_free_variables(pvars);
  end;
end;

procedure TAdminServerHealthForm.FetchedStatusVarsTimer(Sender: TObject);
var i: integer;
  val, MaxVal: Double;
  Pcompiled_expr,
  Pcompiled_MaxExpr: PMYX_COMPILED_EXPRESSION;
  expr_error: MYX_EXPRESSION_ERROR;
  PVars, POldVars: PMYX_VARIABLES;
  graph: TMYX_HEALTH_GRAPH;
begin
  if(HealthPages=nil)or(ClosingApplication)or(Not(MySQLConn.Connected))then
    Exit;

  if(status_vars_timer<>nil)and(status_vars_timer_old<>nil)then
  begin
    PVars:=status_vars_timer.get_record_pointer;
    POldVars:=status_vars_timer_old.get_record_pointer;

    if(RefreshSystemVarsInFormulas)then
    begin
      PrepareGraphFormulars;
      RefreshSystemVarsInFormulas:=False;
    end;

    for i:=0 to GraphFrameBundleList.Count-1 do
    begin
      graph:=TGraphFrameBundle(GraphFrameBundleList[i]).Graph;

      Pcompiled_expr:=myx_compile_expression(
        TGraphFrameBundle(GraphFrameBundleList[i]).value_formular,
        @expr_error);
      if(expr_error=MYX_EXPRESSION_SYNTAX_ERROR)then
      begin
        SetStatusText(format(_('Expression error in graph %s'),[graph.graph_caption])+': '+
          TGraphFrameBundle(GraphFrameBundleList[i]).value_formular);
        continue;
      end;

      if(TGraphFrameBundle(GraphFrameBundleList[i]).max_formular<>'')then
      begin
        Pcompiled_MaxExpr:=myx_compile_expression(
          TGraphFrameBundle(GraphFrameBundleList[i]).max_formular,
          @expr_error);
        if(expr_error=MYX_EXPRESSION_SYNTAX_ERROR)then
        begin
          SetStatusText(format(_('Expression error in graph %s'),[graph.graph_caption])+': '+
            TGraphFrameBundle(GraphFrameBundleList[i]).max_formular);
          continue;
        end;
      end
      else
        Pcompiled_MaxExpr:=nil;

      try
        val:=Abs(myx_eval_expression(Pcompiled_expr, POldVars, PVars, @expr_error));
        if(expr_error=MYX_EXPRESSION_DIVISION_BY_ZERO)then
          val:=0
        else if(expr_error=MYX_EXPRESSION_SYNTAX_ERROR)then
        begin
          SetStatusText(format(_('Expression error in graph %s'),[graph.graph_caption])+': '+
            TGraphFrameBundle(GraphFrameBundleList[i]).value_formular);
          continue;
        end;
      finally
        myx_free_expression(Pcompiled_expr);
      end;

      if(Pcompiled_MaxExpr<>nil)then
      begin
        try
          MaxVal:=myx_eval_expression(Pcompiled_MaxExpr, POldVars, PVars, @expr_error);
          if(expr_error=MYX_EXPRESSION_DIVISION_BY_ZERO)then
            val:=0
          else if(expr_error=MYX_EXPRESSION_SYNTAX_ERROR)then
          begin
            SetStatusText(format(_('Expression error in graph %s'),[graph.graph_caption])+': '+
              TGraphFrameBundle(GraphFrameBundleList[i]).max_formular);
            continue;
          end;
        finally
          myx_free_expression(Pcompiled_MaxExpr);
        end;
      end
      else
        MaxVal:=graph.max;

      //Add new value to graph
      if(graph.graphtype=MYX_LINE_GRAPH)then
      begin
        TAdminServerHealthLineGraphFrame(
          TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame
          ).SetValue(val);

        TAdminServerHealthLineGraphFrame(
          TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame
          ).SetMaxValue(MaxVal);
      end
      else if(graph.graphtype=MYX_BAR_GRAPH)then
      begin
        TAdminServerHealthBarGraphFrame(
          TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame
          ).SetMaxValue(MaxVal);

        TAdminServerHealthBarGraphFrame(
          TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame
          ).SetValue(val);
      end;
    end;
  end
  else
  begin
    PrepareGraphFormulars;

    //Initialize graphs with 0 values
    for i:=0 to GraphFrameBundleList.Count-1 do
    begin
      if(TGraphFrameBundle(GraphFrameBundleList[i]).Graph.graphtype=
        MYX_LINE_GRAPH)then
      begin
        TAdminServerHealthLineGraphFrame(
          TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame
          ).SetValue(0);
      end
      else if(TGraphFrameBundle(GraphFrameBundleList[i]).Graph.graphtype=
        MYX_BAR_GRAPH)then
      begin
        TAdminServerHealthBarGraphFrame(
          TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame
          ).SetValue(0);
      end;
    end;
  end;

  if(status_vars_timer_old<>nil)then
    status_vars_timer_old.Free;

  status_vars_timer_old:=status_vars_timer;
  status_vars_timer:=nil;
end;

procedure TAdminServerHealthForm.PrepareGraphFormulars;
var i, j: integer;
begin
  if(status_vars_timer=nil)or(server_vars=nil)then
    Exit;

  for i:=0 to GraphFrameBundleList.Count-1 do
  begin
    TGraphFrameBundle(GraphFrameBundleList[i]).value_formular:=
      TGraphFrameBundle(GraphFrameBundleList[i]).graph.value_formula;
    TGraphFrameBundle(GraphFrameBundleList[i]).max_formular:=
      TGraphFrameBundle(GraphFrameBundleList[i]).graph.max_formula;
  end;

  //Replace all status variable names with index numbers
  for i:=0 to status_vars_timer.variables.Count-1 do
    for j:=0 to GraphFrameBundleList.Count-1 do
    begin
      TGraphFrameBundle(GraphFrameBundleList[j]).value_formular:=
        AnsiReplaceText(TGraphFrameBundle(GraphFrameBundleList[j]).value_formular,
        '['+status_vars_timer.variables[i].name+']',
        '['+IntToStr(i)+']');

      TGraphFrameBundle(GraphFrameBundleList[j]).max_formular:=
        AnsiReplaceText(TGraphFrameBundle(GraphFrameBundleList[j]).max_formular,
        '['+status_vars_timer.variables[i].name+']',
        '['+IntToStr(i)+']');
    end;

  //Replace all system variable names with values
  for i:=0 to server_vars.variables.Count-1 do
    for j:=0 to GraphFrameBundleList.Count-1 do
    begin
      TGraphFrameBundle(GraphFrameBundleList[j]).value_formular:=
        AnsiReplaceText(TGraphFrameBundle(GraphFrameBundleList[j]).value_formular,
        '['+server_vars.variables[i].name+']',
        server_vars.variables[i].value);

      TGraphFrameBundle(GraphFrameBundleList[j]).max_formular:=
        AnsiReplaceText(TGraphFrameBundle(GraphFrameBundleList[j]).max_formular,
        '['+server_vars.variables[i].name+']',
        server_vars.variables[i].value);
    end;
end;

procedure TAdminServerHealthForm.GraphRefreshTmrTimer(Sender: TObject);
begin
  if(ClosingApplication)then
    Exit;

  //Get new status vars every time
  // TODO: Use only one thread that awakes every second and collects the data.
  if(status_vars_timer<>nil)then
  begin
    status_vars_timer.Free;
    status_vars_timer:=nil;
  end;

  MySQLConn.FetchData(KindOfData_StatusVars,
    FetchStatusVarsTimer, FetchedStatusVarsTimer,
    nil, '');
end;

function TAdminServerHealthForm.GetVarIndex(vars: TMYX_VARIABLES;
  VarName: WideString): integer;
var i, VarPos: integer;
begin
  VarPos:=0;

  for i:=0 to vars.variables.Count-1 do
    if(CompareText(vars.variables[i].Name, VarName)=0)then
    begin
      VarPos:=i;
      break;
    end;

  Result:=VarPos;
end;

procedure TAdminServerHealthForm.LoadGraphs;
var phealth_pages: PMYX_HEALTH_PAGES;
  error: MYX_ADMIN_LIB_ERROR;
begin
  if(Not(FileExists(MYXCommonOptions.UserDataDir+'mysqladmin_health.xml')))then
    CopyDiskFile(MYXCommonOptions.XMLDir+'mysqladmin_health.xml',
      MYXCommonOptions.UserDataDir+'mysqladmin_health.xml', False);

  phealth_pages:=myx_read_in_health_pages(
    MYXCommonOptions.UserDataDir+'mysqladmin_health.xml',
    @error);
  if(phealth_pages=nil)or(error<>MYX_ADMIN_NO_ERROR)then
    raise EMyxAdminLibError.Create(
      Format(_('Error while loading the health graphs from %s.'),
        [MYXCommonOptions.UserDataDir+'mysqladmin_health.xml']),
      Ord(error),
      MYXCommonOptions.XMLDir+'mysqladmin_health.xml');

  try
    HealthPages.Free;
    HealthPages:=TMYX_HEALTH_PAGES.Create(phealth_pages);
  finally
    myx_free_health_pages(phealth_pages);
  end;
end;

procedure TAdminServerHealthForm.SaveGraphs;
begin
  if(HealthPages<>nil)then
    myx_save_health_pages(HealthPages.get_record_pointer,
      MYXCommonOptions.UserDataDir+'mysqladmin_health.xml');
end;

procedure TAdminServerHealthForm.BuildGraphs;
var i, j, k: integer;
  PageTabSheet: TTabSheet;
  ScrollBox: TTntScrollBox;
  GroupBox: TTntGroupBox;
  page: TMYX_HEALTH_PAGE;
  current_widget_ypos,
  current_group_height,
  current_group_ypos: integer;
begin
  LoadGraphs;

  if(HealthPages<>nil)then
  begin
    for i:=0 to HealthPages.pages.Count-1 do
    begin
      page:=HealthPages.pages[i];

      PageTabSheet:=nil;
      ScrollBox:=nil;
      AddPage(page, PageTabSheet, ScrollBox);

      current_group_ypos:=8;

      for j:=0 to page.groups.Count-1 do
      begin
        GroupBox:=TTntGroupBox.Create(ScrollBox);
        GroupBox.Parent:=ScrollBox;
        GroupBox.Name:='GBox'+IntToStr(GraphControlCount);
        GroupBox.Caption:=page.groups[j].caption;
        GroupBox.Left:=12;
        GroupBox.Top:=current_group_ypos;
        GroupBox.Width:=GroupBoxWidth;
        GroupBox.Anchors:=[akLeft, akTop, akRight];
        GroupBox.PopupMenu:=GroupBoxPopupMenu;
        inc(GraphControlCount);

        current_group_height:=25;

        current_widget_ypos:=22;
        for k:=0 to page.groups[j].graphs.Count-1 do
        begin
          AddGraphToGroupBox(GroupBox, page.groups[j].graphs[k],
            current_widget_ypos);

          current_group_height:=current_widget_ypos;
        end;

        GroupBox.Height:=current_group_height;

        inc(current_group_ypos, GroupBox.Height+GroupBoxVSpace);
      end;
    end;

    //Now display Pages
    for i:=0 to HealthPageControl.PageCount-1 do
      HealthPageControl.Pages[i].TabVisible:=True;
  end;

  ServerStatusTabSheet.PageIndex:=HealthPageControl.PageCount-1;
  ServerVariablesTabSheet.PageIndex:=HealthPageControl.PageCount-1;
end;

procedure TAdminServerHealthForm.AddPage(page: TMYX_HEALTH_PAGE;
  var PageTabSheet: TTabSheet; var ScrollBox: TTntScrollBox);
begin
  PageTabSheet:=TTabSheet.Create(self);
  PageTabSheet.Name:='HeathPage'+IntToStr(GraphControlCount);
  PageTabSheet.Caption:=page.caption;
  PageTabSheet.TabVisible:=False;
  PageTabSheet.PageControl:=HealthPageControl;
  PageTabSheet.Width:=563;
  PageTabSheet.Height:=412;
  inc(GraphControlCount);

  ScrollBox:=TTntScrollBox.Create(PageTabSheet);
  ScrollBox.Parent:=PageTabSheet;
  ScrollBox.Name:='SB'+IntToStr(GraphControlCount);
  ScrollBox.Top:=46;
  ScrollBox.Left:=0;
  ScrollBox.Height:=PageTabSheet.Height-ScrollBox.Top;
  ScrollBox.Width:=PageTabSheet.Width;
  ScrollBox.Anchors:=[akLeft, akTop, akRight, akBottom];
  //ScrollBox.Align:=alClient;
  ScrollBox.BorderStyle:=bsNone;
  if(MYXCommonOptions.XPStyleEnabled)then
    ScrollBox.Color:=clWhite;
  ScrollBox.VertScrollBar.Smooth:=True;
  ScrollBox.VertScrollBar.Tracking:=True;
  ScrollBox.AutoScroll:=True;
  ScrollBox.Width:=PageTabSheet.Width;
  inc(GraphControlCount);
end;

procedure TAdminServerHealthForm.AddGraphToGroupBox(GroupBox: TTntGroupBox;
  graph: TMYX_HEALTH_GRAPH; var current_widget_ypos: integer);
var LineGraph: TAdminServerHealthLineGraphFrame;
  BarGraph: TAdminServerHealthBarGraphFrame;
begin
  if(graph.graphtype=MYX_LINE_GRAPH)then
  begin
    LineGraph:=TAdminServerHealthLineGraphFrame.Create(
      GroupBox,
      graph,
      graph.value_caption,
      graph.min,
      graph.max,
      graph.value_unit);
    LineGraph.Parent:=GroupBox;
    LineGraph.Name:='HealthGraph_'+IntToStr(GraphControlCount);
    LineGraph.Left:=16;
    LineGraph.Top:=current_widget_ypos;
    LineGraph.Width:=GroupBox.Width-30; //504;
    LineGraph.Anchors:=[akLeft, akTop, akRight];
    LineGraph.PopupMenu:=GraphPopupMenu;
    LineGraph.OnDblClick:=EditGraphMIClick;
    inc(GraphControlCount);

    GraphFrameBundleList.Add(TGraphFrameBundle.Create(
      graph, LineGraph));

    Inc(current_widget_ypos, LineGraph.Height+10);
  end
  else if(graph.graphtype=MYX_BAR_GRAPH)then
  begin
    BarGraph:=TAdminServerHealthbarGraphFrame.Create(
      GroupBox,
      graph.graph_caption,
      graph.display_graph_caption,
      graph.value_caption,
      graph.max_caption,
      graph.min,
      graph.max,
      graph.value_unit);
    BarGraph.Parent:=GroupBox;
    BarGraph.Name:='HealthGraph_'+IntToStr(GraphControlCount);
    BarGraph.Left:=16;
    BarGraph.Top:=current_widget_ypos;
    BarGraph.Anchors:=[akLeft, akTop, akRight];
    BarGraph.PopupMenu:=GraphPopupMenu;
    BarGraph.OnDblClick:=EditGraphMIClick;
    inc(GraphControlCount);

    GraphFrameBundleList.Add(TGraphFrameBundle.Create(
      graph, BarGraph));

    Inc(current_widget_ypos, 19+10);
  end;
end;

procedure TAdminServerHealthForm.Disconnected(var Message: TMessage);
begin
  if Assigned(StatusVar_PMySQL)then
    myx_mysql_close(StatusVar_PMySQL);
  StatusVar_PMySQL := nil;

  DisableEnablePages(HealthPageControl, False);

  GraphRefreshTmr.Enabled:=False;
end;

procedure TAdminServerHealthForm.Reconnected(var Message: TMessage);
begin
  DisableEnablePages(HealthPageControl, True);

  if(HealthPages=nil)then
    BuildGraphs;

  if(server_vars=nil)then
    FetchServerVars(self);

  if(status_vars=nil)then
    FetchStatusVars(self);

  GraphRefreshTmr.Enabled:=True;

  HealthPageControl.ActivePageIndex:=0;
  HealthPageControlChange(self);
end;

procedure TAdminServerHealthForm.AddaPageMIClick(Sender: TObject);
var Caption, Description: Widestring;
  pos: integer;
  page: TMYX_HEALTH_PAGE;
  PageTabSheet: TTabSheet;
  ScrollBox: TTntScrollBox;
begin
  if(ShowModalEditDialog(_('Add a Page'),
    _('Enter the caption of the page to add.'),
    myx_mtEdit, _('Continue')+#13#10+_('Cancel'),
    True, _('Page Caption:'), Caption)=1)then
  begin
    if(ShowModalEditDialog(_('Page Description'),
      format(_('Enter the page description of the new %s page'),[Caption]),
      myx_mtEdit, _('Create Page')+#13#10+_('Cancel'),
      True, _('Page Description:'), Description)=1)then
    begin
      pos:=HealthPageControl.ActivePageIndex+1;
      if(pos>HealthPages.pages.Count-1)then
        pos:=HealthPages.pages.Count-1;
      if(pos<0)then
        pos:=0;

      page:=TMYX_HEALTH_PAGE.create(Caption, Caption,
        Description, Description,
        pos);
      HealthPages.pages.Insert(pos, page);


      PageTabSheet:=nil;
      ScrollBox:=nil;
      AddPage(page, PageTabSheet, ScrollBox);

      PageTabSheet.PageIndex:=pos;
      PageTabSheet.TabVisible:=True;

      HealthPageControl.ActivePage:=PageTabSheet;

      HealthPageControlChange(self);

      SaveGraphs;
    end;
  end;
end;

procedure TAdminServerHealthForm.DeletePage(pos: integer; AskBeforeDelete: Boolean);
var TabSheet: TTabSheet;
  i, j, k: integer;
  Res: integer;
begin
  if(pos<HealthPageControl.PageCount-2)and
    (pos<HealthPages.pages.Count)then
  begin
    if(AskBeforeDelete)then
      Res:=ShowModalDialog('Delete Page?',
        'Are you sure you want to delete the selected page '+
        HealthPages.pages[pos].caption+'?',
        myx_mtConfirmation, 'Yes'#13#10'No')
    else
      Res:=1;

    if(Res=1)then
    begin
      GraphRefreshTmr.Enabled:=False;

      HealthPageControl.ActivePageIndex:=pos+1;

      //Hide Tab since it cannot be removed
      TabSheet:=HealthPageControl.Pages[pos];
      //TabSheet.TabVisible:=False;
      TabSheet.PageIndex:=HealthPageControl.PageCount-1;

      while(TabSheet.ComponentCount>0)do
        DeleteComponentWithChildren(TabSheet.Components[0]);

      for i:=0 to HealthPages.pages[pos].groups.Count-1 do
        for j:=0 to HealthPages.pages[pos].groups[i].graphs.Count-1 do
          for k:=0 to GraphFrameBundleList.Count-1 do
            if(TGraphFrameBundle(GraphFrameBundleList[k]).Graph=
              HealthPages.pages[pos].groups[i].graphs[j])then
            begin
              GraphFrameBundleList.Delete(k);
              break;
            end;

      HealthPages.pages.Delete(pos);

      HealthPageControlChange(self);

      TabSheet.Free;

      GraphRefreshTmr.Enabled:=True;

      SaveGraphs;
    end;
  end;
end;

procedure TAdminServerHealthForm.DeletePageMIClick(Sender: TObject);
begin
  DeletePage(HealthPageControl.ActivePageIndex);
end;

procedure TAdminServerHealthForm.AddaGroupMIClick(Sender: TObject);
var Caption: WideString;
  pos: integer;
  group: TMYX_HEALTH_GROUP;
  GroupBox: TTntGroupBox;
  ScrollBox: TTntScrollBox;
  current_group_ypos: integer;
  i: integer;
begin
  if(HealthPageControl.ActivePage.Controls[0] is TTntScrollBox)then
    ScrollBox:=TTntScrollBox(HealthPageControl.ActivePage.Controls[0])
  else
    Exit;

  if(ShowModalEditDialog('Add a Group',
    'Enter the caption of the group to add.',
    myx_mtEdit, 'Create Group'#13#10'Cancel',
    True, 'Group Caption:', Caption)=1)then
  begin
    pos:=HealthPages.pages[HealthPageControl.ActivePageIndex].groups.Count;

    group:=TMYX_HEALTH_GROUP.create(Caption, Caption, pos+1);

    HealthPages.pages[HealthPageControl.ActivePageIndex].groups.Add(group);

    current_group_ypos:=0;

    for i:=0 to ScrollBox.ComponentCount-1 do
      if(ScrollBox.Components[i] is TTntGroupBox)then
        if(TTntGroupBox(ScrollBox.Components[i]).Top+
          TTntGroupBox(ScrollBox.Components[i]).Height+GroupBoxVSpace>
            current_group_ypos)then
          current_group_ypos:=TTntGroupBox(ScrollBox.Components[i]).Top+
            TTntGroupBox(ScrollBox.Components[i]).Height+GroupBoxVSpace;

    GroupBox:=TTntGroupBox.Create(ScrollBox);
    GroupBox.Parent:=ScrollBox;
    GroupBox.Name:='GBox'+IntToStr(GraphControlCount);
    GroupBox.Caption:=Caption;
    GroupBox.Left:=12;
    GroupBox.Top:=current_group_ypos;
    GroupBox.Width:=ScrollBox.Width-28;
    GroupBox.Anchors:=[akLeft, akTop, akRight];
    GroupBox.PopupMenu:=GroupBoxPopupMenu;
    GroupBox.Height:=25;
    inc(GraphControlCount);

    SaveGraphs;
  end;
end;

procedure TAdminServerHealthForm.EditGraphMIClick(Sender: TObject);
var Frame: TTntFrame;
  i: integer;
begin
  if(Sender is TTntFrame)then
    Frame:=TTntFrame(Sender)
  else if(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent.InheritsFrom(TTntFrame))then
    Frame:=TTntFrame(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent)
  else
    Frame:=nil;

  if(Frame<>nil)then
  begin
    if(AdminServerHealthGraphSettingsForm=nil)then
      AdminServerHealthGraphSettingsForm:=
        TAdminServerHealthGraphSettingsForm.Create(self);

    for i:=0 to GraphFrameBundleList.Count-1 do
      if(TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame=Frame)then
      begin
        AdminServerHealthGraphSettingsForm.SetGraph(
          TGraphFrameBundle(GraphFrameBundleList[i]));
        break;
      end;

    AdminServerHealthGraphSettingsForm.Show;
  end;
end;

procedure TAdminServerHealthForm.ServerStatusListViewMouseDown(
  Sender: TObject; Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
begin
  //TTntListView(Sender).BeginDrag(False, 5);
end;

procedure TAdminServerHealthForm.AddaGraphMIClick(Sender: TObject);
var GroupBox: TTntGroupBox;
begin
  if(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent is TTntGroupBox)then
    GroupBox:=TTntGroupBox(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent)
  else if(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent.InheritsFrom(TTntFrame))then
    GroupBox:=TTntGroupBox(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent.Owner)
  else
    Exit;

  if(AdminServerHealthGraphSettingsForm=nil)then
    AdminServerHealthGraphSettingsForm:=
      TAdminServerHealthGraphSettingsForm.Create(self);

  AdminServerHealthGraphSettingsForm.NewGraph(GroupBox);

  AdminServerHealthGraphSettingsForm.Show;
  AdminServerHealthGraphSettingsForm.ActiveControl:=
    AdminServerHealthGraphSettingsForm.GraphTypeCBox;
end;

procedure TAdminServerHealthForm.OptionsChanged(var Message: TMessage);
begin
  //
end;


procedure TAdminServerHealthForm.DeleteGraphMIClick(Sender: TObject);
begin
  if(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent is TAdminServerHealthBarGraphFrame)or
    (TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent is TAdminServerHealthLineGraphFrame)then
    DeleteGraph(TControl(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent));
end;

procedure TAdminServerHealthForm.DeleteGraph(Graph: TControl);
var GBox: TTntGroupBox;
  i, j, GHeight: integer;
begin
  GBox:=TTntGroupBox(Graph.Parent);

  GraphRefreshTmr.Enabled:=False;

  GHeight:=Graph.Height;

  for i:=0 to GBox.ControlCount-1 do
  begin
    if(GBox.Controls[i]<>Graph)then
      if(TControl(GBox.Components[i]).Top>Graph.Top)then
        TControl(GBox.Components[i]).Top:=
          TControl(GBox.Components[i]).Top-(GHeight+10);
  end;

  //Free Graph and GraphFrameBundleList Item
  for i:=0 to GraphFrameBundleList.Count-1 do
    if(TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame=Graph)then
    begin
      for j:=0 to HealthPages.pages[HealthPageControl.ActivePageIndex].groups.Count-1 do
      begin
        if(HealthPages.pages[HealthPageControl.ActivePageIndex].groups[j].graphs.IndexOf(
          TGraphFrameBundle(GraphFrameBundleList[i]).Graph)>=0)then
          HealthPages.pages[HealthPageControl.ActivePageIndex].groups[j].graphs.Remove(
            TGraphFrameBundle(GraphFrameBundleList[i]).Graph);
      end;

      GraphFrameBundleList.Delete(i);
      break;
    end;

  GBox.Height:=GBox.Height-(GHeight+10);

  for i:=0 to GBox.Parent.ControlCount-1 do
    if(GBox.Parent.Controls[i].Top>GBox.Top)then
      GBox.Parent.Controls[i].Top:=GBox.Parent.Controls[i].Top-(GHeight+10);

  Graph.Free;

  GraphRefreshTmr.Enabled:=True;

  SaveGraphs;
end;

procedure TAdminServerHealthForm.DeleteGroupMIClick(Sender: TObject);
begin
  if(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent is TTntGroupBox)then
    DeleteGroupBox(TTntGroupBox(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent))
  else if(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent is TAdminServerHealthBarGraphFrame)or
    (TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent is TAdminServerHealthLineGraphFrame)then
    DeleteGroupBox(TTntGroupBox(TControl(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent).Parent))

end;

procedure TAdminServerHealthForm.DeleteGroupBox(GroupBox: TTntGroupBox);
var i, GroupNr: integer;
begin
  i:=0;
  while(i<GroupBox.ControlCount)do
  begin
    if(GroupBox.Controls[i] is TAdminServerHealthBarGraphFrame)or
      (GroupBox.Controls[i] is TAdminServerHealthLineGraphFrame)then
      DeleteGraph(GroupBox.Controls[i])
    else
      inc(i);
  end;

  for i:=0 to GroupBox.Parent.ControlCount-1 do
    if(GroupBox.Parent.Controls[i].Top>GroupBox.Top)then
      GroupBox.Parent.Controls[i].Top:=GroupBox.Parent.Controls[i].Top-
        (GroupBox.Height+GroupBoxVSpace);

  //delete group from HealthPages
  GroupNr:=0;
  for i:=0 to GroupBox.Parent.ControlCount-1 do
    if(GroupBox=GroupBox.Parent.Controls[i])then
    begin
      GroupNr:=i;
      break;
    end;

  if(GroupNr<HealthPages.pages[HealthPageControl.ActivePageIndex].groups.Count)then
    HealthPages.pages[
      HealthPageControl.ActivePageIndex].groups.Delete(GroupNr);

  GroupBox.Free;

  SaveGraphs;
end;


procedure TAdminServerHealthForm.RestoreDefaultsMIClick(Sender: TObject);
var i, index: integer;
begin
  index:=HealthPageControl.PageCount-2;
  for i:=0 to index do
    DeletePage(0, False);

  //Restore original file
  CopyDiskFile(MYXCommonOptions.XMLDir+'mysqladmin_health.xml',
    MYXCommonOptions.UserDataDir+'mysqladmin_health.xml', False);


  status_vars_timer_old.Free;
  status_vars_timer_old:=nil;

  BuildGraphs;

  HealthPageControl.ActivePageIndex:=0;
  HealthPageControlChange(self);
end;

procedure TAdminServerHealthForm.PageControlPopupMenuPopup(
  Sender: TObject);
begin
  if(HealthPageControl.ActivePage=ServerStatusTabSheet)or
    (HealthPageControl.ActivePage=ServerVariablesTabSheet)then
  begin
    AddaGroupMI.Enabled:=False;
    RestoreDefaults3MI.Enabled:=False;
  end
  else
  begin
    AddaGroupMI.Enabled:=True;
    RestoreDefaults3MI.Enabled:=True;
  end;
end;

procedure TAdminServerHealthForm.MoveGraphUpMIClick(Sender: TObject);
begin
  if(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent.InheritsFrom(TTntFrame))then
    MoveGraph(TTntFrame(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent),
    True);
end;

procedure TAdminServerHealthForm.MoveGraphDownMIClick(Sender: TObject);
begin
  if(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent.InheritsFrom(TTntFrame))then
    MoveGraph(TTntFrame(TTntPopupMenu(TTntMenuItem(Sender).GetParentMenu).PopupComponent),
    False);
end;

procedure TAdminServerHealthForm.MoveGraph(Frame: TTntFrame; MoveUp: Boolean);
var
  GroupBox: TTntGroupBox;
  i, FlipIndex, FlipPos: integer;
  Graph, OtherGraph: TMYX_HEALTH_GRAPH;
begin
  if(Frame.Parent is TTntGroupBox)then
  begin
    GroupBox:=TTntGroupBox(Frame.Parent);
    if(GroupBox<>nil)then
    begin
      FlipIndex:=-1;
      FlipPos:=1000*Ord(MoveUp);

      for i:=0 to GroupBox.ControlCount-1 do
        if(GroupBox.Controls[i]<>Frame)and
          (GroupBox.Controls[i].Top<Frame.Top)and
          (MoveUp)then
        begin
          if(FlipPos>GroupBox.Controls[i].Top)then
          begin
            FlipPos:=GroupBox.Controls[i].Top;
            FlipIndex:=i;
          end;
        end
        else if(GroupBox.Controls[i]<>Frame)and
          (GroupBox.Controls[i].Top>Frame.Top)and
          (Not(MoveUp))then
        begin
          if(FlipPos<GroupBox.Controls[i].Top)then
          begin
            FlipPos:=GroupBox.Controls[i].Top;
            FlipIndex:=i;
          end;
        end;

      if(FlipIndex>=0)then
      begin
        GroupBox.Controls[FlipIndex].Top:=Frame.Top;
        Frame.Top:=FlipPos;

        Graph:=nil;
        OtherGraph:=nil;

        for i:=0 to GraphFrameBundleList.Count-1 do
          if(TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame=
            Frame)then
          begin
            Graph:=TGraphFrameBundle(GraphFrameBundleList[i]).Graph;
            break;
          end;

        for i:=0 to GraphFrameBundleList.Count-1 do
          if(TGraphFrameBundle(GraphFrameBundleList[i]).GraphFrame=
            GroupBox.Controls[FlipIndex])then
          begin
            OtherGraph:=TGraphFrameBundle(GraphFrameBundleList[i]).Graph;
            break;
          end;

        if(Graph<>nil)and(OtherGraph<>nil)then
        begin
          FlipPos:=Graph.pos;
          Graph.pos:=OtherGraph.pos;
          OtherGraph.pos:=FlipPos;
        end;

        SaveGraphs;
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServerHealthForm.ListViewDeletion(Sender: TObject; Item: TListItem);

begin
  TObject(Item.Data).Free;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServerHealthForm.MenuMeasureItem(Sender: TObject; ACanvas: TCanvas; var Width, Height: Integer);

var
  Size: TSize;
  Item: TTntMenuItem;

begin
  if Sender is TTntMenuItem then
  begin
    Item := Sender as TTntMenuItem;
    ACanvas.Font := Font;

    if Item.IsLine then
    begin
      Width := 10; // This will actually have no effect, because other entries are much wider.
      Height := 6;
    end
    else
    begin
      GetTextExtentPoint32W(ACanvas.Handle, PWideChar(Item.Caption), Length(Item.Caption), Size);

      // Border around each entry.
      Width := Size.cx + 4;
      Height := Size.cy + 6;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TAdminServerHealthForm.MenuDrawItem(Sender: TObject; ACanvas: TCanvas; ARect: TRect; State: TOwnerDrawState);

var
  Item: TTntMenuItem;

begin
  if Sender is TTntMenuItem then
  begin
    Item := Sender as TTntMenuItem;
    ACanvas.Font := Font;

    if Item.IsLine then
    begin
      // A menu separator.
      ACanvas.Pen.Color := clBtnShadow;
      ACanvas.MoveTo(ARect.Left + 2, (ARect.Bottom + ARect.Top) div 2);
      ACanvas.LineTo(ARect.Right - 2, (ARect.Bottom + ARect.Top) div 2);
    end
    else
    begin
      // Top level items have an invisible parent, so have to check the parent of the parent.
      if (Item.Parent.Parent = nil) and not (Item.GetParentMenu is TPopupMenu) then
      begin
        if [odHotLight, odSelected] * State <> [] then
          ACanvas.Brush.Color := clHighlight
        else
          ACanvas.Brush.Color := clBtnFace;
      end;
      ACanvas.FillRect(ARect);
      Inc(ARect.Left, 8);
      SetBKMode(ACanvas.Handle, TRANSPARENT);
      Windows.DrawTextW(ACanvas.Handle, PWideChar(Item.Caption), Length(Item.Caption), ARect, DT_LEFT + DT_SINGLELINE +
        DT_HIDEPREFIX + DT_VCENTER);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

end.
