/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

                          uilocalsocket.cpp  -  description
                             -------------------
    begin                : Wed Mar 26 2003
    copyright            : (C) 2003 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>	/* waitpid () */
#include <sys/wait.h>	/* waitpid () */
#include <unistd.h>	/* fnctl () */
#include <fcntl.h>	/* fnctl () */

#include <sys/socket.h>
#include <sys/un.h>

#include "mutella.h"
#include "structures.h"

#include "controller.h"
#include "mui.h"
#include "property.h"
#include "preferences.h"
#include "mprintf.h"
#include "event.h"
#include "messages.h"
#include "lineinput.h"
#include "conversions.h"
#include "common.h"
#include "gnumarkedfiles.h"
#include "asyncsocket.h"
#include "uitextmode.h"
#include "uilocalsocket.h"

class MPrintfNocolor : public MPrintfMore
{
public:
	MPrintfNocolor() : MPrintfMore(-1, -1) {
		m_bBreak = false;
	}
protected:
	virtual void PrePrint(CString& s){
		// search for '??' combinations which are suposed to define 'style'
		int nStart = 0;
		int nPos;
		int nPos1;
		int n,i;
		int nNameLen;
		while ( -1 != (nPos = s.find("??",nStart)) )
		{
			CString sStyle = s.substr(nPos+2,16); // style names are normally short
			bool bFound = false;
			nPos1 = -1;
			// unfortunately sStyle.find('>'); wont work
			n = sStyle.length();
			for(i = 0; i<n; ++i)
				if (sStyle[i] == '>')
				{
					nPos1 = i;
					break;
				}
				else if (sStyle[i]<'0' || sStyle[i]>'z' || (sStyle[i]>'9' && sStyle[i]<'A'))
				{
					// non-alphanum
					break;
				}
			if (nPos1>=0)
			{
				//found '>' -- we have to cut out ??...>
				CString sTmp = s.substr(nPos+2+nPos1+1); // '+1' is for spacer after
				s.cut(nPos); // cut without memory realloc
				nStart = s.length();
				s += sTmp;
			}
			else
			{
				// '??' case -- just get rid of it
				CString sTmp = s.substr(nPos+2);
				s.cut(nPos); // cut without memory realloc
				nStart = s.length();
				s += sTmp;
			}
		}
	}
};


class MUILSockPriv : public MUITextModePriv {
public:
	MUILSockPriv(MUILocalSocket* pMaster, MController* pCtrl);
	bool ProcessLine(const CString& sInput, CString& sOutput);
	//
	virtual bool attach() { return true; }
	virtual bool init() { return true; }
	virtual void detach() {}
	virtual void ui_loop() {}
	virtual void stop(bool bForce) { ASSERT(m_pMaster); m_pMaster->StopUnlocked(); }
	virtual bool vcom_color(char * arg) { return false; }
	virtual bool vcom_system(char * arg);
	virtual bool is_set_permitted(const CString& sProp);
	//
	void SetCallingSocket(int hSocket) { m_hSocket = hSocket; }
protected:
	MPrintfNocolor  m_printf;
	MUILocalSocket* m_pMaster;
	int m_hSocket;
};

class MUILSocketCommunicate : public MAsyncSocket {
public:      
	MUILSocketCommunicate(MUILocalSocket* pMaster) {
		m_pMaster = pMaster;
		//m_sOut = "Welcome to MUILocalSocket\n";
	}
	virtual void OnReceive(int nErrorCode) {
		char pBuff[1024];
		int nRecvd = Receive(pBuff, 1023);
		if (nRecvd <= 0)
		{
			Close();
			m_pMaster->RemoveSocket(this);
			return;
		}
		ASSERT(nRecvd < 1024);
		pBuff[nRecvd] = '\0';
		m_sIn += pBuff;
		int nPos;
		CString sLine;
		CString sOut;
		while ( (nPos = m_sIn.find('\n'))>= 0 )
		{
			sLine = m_sIn.substr(0, nPos);
			m_sIn = m_sIn.substr(nPos+1);
			if ( (nPos=sLine.find('\r')) >= 0 )
				sLine.cut(nPos);
			m_pMaster->GetMutex()->lock();
			ASSERT(m_pMaster->GetInterpreter());
			m_pMaster->GetInterpreter()->SetCallingSocket(m_hSocket);
			m_pMaster->GetInterpreter()->ProcessLine(sLine, sOut);
			m_pMaster->GetMutex()->unlock();
		}
		SendText(sOut);
	}
	virtual void OnSend(int nErrorCode) {
		if (m_sOut.empty())
		{
			ModifySelectFlags(0, FD_WRITE);
			return;
		}
		int nSent = Send(m_sOut.c_str(), m_sOut.size());
		if (nSent <= 0)
		{
			Close();
			m_pMaster->RemoveSocket(this);
			return;
		}
		m_sOut = m_sOut.substr(nSent);
		if (m_sOut.empty())
			ModifySelectFlags(0, FD_WRITE);
	}
	void SendText(const CString& s) {
		m_sOut += s;
		if (m_sOut.size())
			ModifySelectFlags(FD_WRITE, 0);
	}
protected:
	MUILocalSocket* m_pMaster;
	CString m_sOut;
	CString m_sIn;
private:
	MUILSocketCommunicate();
};

class MUILSocketListen : public MAsyncSocket {
public:
	MUILSocketListen(MUILocalSocket* pMaster) {
		m_pMaster = pMaster;
	}
	virtual void OnAccept(int nErrorCode) {
		ASSERT(m_pMaster);
		MUILSocketCommunicate* pSock = new MUILSocketCommunicate(m_pMaster);
		if (!Accept(*pSock) || !m_pMaster->IsEnabled() )
			delete pSock;
		else
		{
			pSock->ModifySelectFlags(FD_WRITE|FD_READ, 0);
			m_pMaster->AddSocket(pSock);
		}
		MAsyncSocket::OnAccept(nErrorCode);
	}
protected:
	MUILocalSocket* m_pMaster;
};

//////////////////////////////////////////////////////////////////////////////////////////////////
MUILSockPriv::MUILSockPriv(MUILocalSocket* pMaster, MController* pCtrl) : MUITextModePriv(pCtrl), m_pMaster(pMaster)
{
	m_pOutput = &m_printf;
}

bool MUILSockPriv::ProcessLine(const CString& sInput, CString& sOutput)
{
	//cout << "Socket command: " << sInput << endl;
	if (sInput.empty())
		return true;
	bool bResult = execute_line(sInput);
	sOutput += m_printf.GetBuffer();
	m_printf.Discard();
	return bResult;
}

bool MUILSockPriv::vcom_system(char * arg)
{
	ASSERT(m_pMaster);
	if (m_pMaster->IsExecEnabled())
	{
		// backup stdout
		int hOldStdout;
		if ((hOldStdout = dup (STDOUT_FILENO)) == -1) {
			m_printf.print ("error: failed to backup stdout\n");
			return false;
		}
		// redirect stdout to a socket
		if (dup2(m_hSocket, STDOUT_FILENO) == -1) {
			m_printf.print ("error: failed to redirect stdout\n");
			return false;
		}
		
		bool bRes = MUITextModePriv::vcom_system(arg);
		
		// restore everything
		if (dup2(hOldStdout, STDOUT_FILENO) == -1) {
			m_printf.print ("error: failed to restore stdout\n");
		}
		return bRes;
	}
	m_printf.print("error: command is not permitted\n");
	return false;
}

bool MUILSockPriv::is_set_permitted(const CString& sProp)
{
	ASSERT(m_pMaster);
	if (sProp == "SocketEnableExec")
		return m_pMaster->IsExecEnabled();
	return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////
MUILocalSocket::MUILocalSocket()
{
	m_pListen = NULL;
	m_pCtrl = NULL;
	m_pPriv = NULL;
}

MUILocalSocket::~MUILocalSocket()
{
	m_mutex.lock();
	delete m_pListen;
	while (!m_listCommunicate.empty())
	{
		delete m_listCommunicate.front();
		m_listCommunicate.pop_front();
	}
	delete m_pPriv;
	m_mutex.unlock();
}

bool MUILocalSocket::Attach(MController* pC)
{
	m_pCtrl = pC;
	ASSERT(m_pCtrl);
	MPropertyContainer* pPC = m_pCtrl->GetPropertyContainer();
	MProperty* pP;
	pPC->AddSection("SocketUI");
	pP = pPC->AddProperty("SocketEnable",        &m_bEnable,              true);
	pP->SetPropertyDescription("Enables the 'socket' UI",
		"Enables/disables the unix sockets based text mode user interface");
	pP = pPC->AddProperty("SocketPath",          m_szSocketPath,          128,  "~/.mutella/socket");
	pP->SetPropertyDescription("Path to the socket",
		"Sets the path to the unix socket used by the SocketUI");
	pP = pPC->AddProperty("SocketEnableExec",    &m_bEnableExec,          false);
	pP->SetPropertyDescription("Enables external commands execution",
		"Enables/disables execution of the external commands ('system' command or pipe symbol) "
		"in the socket interface. This option cannot be modified via the socket interface");
	pPC->SetCurrentSection(NULL);
	return true;
}

bool MUILocalSocket::Init()
{
	ASSERT(m_pCtrl);
	ASSERT(m_pPriv == NULL);
	m_pPriv = new MUILSockPriv(this, m_pCtrl);
	//
	int nFD = socket(PF_UNIX, SOCK_STREAM, 0);
	if (nFD < 0)
		return false;
	CString sPath = ExpandPath(m_szSocketPath);
	sockaddr_un sa;
	sa.sun_family = AF_UNIX;
	strncpy(sa.sun_path, sPath.c_str(), 107);
	unlink(sa.sun_path); // free the path in case mutella crashed previously
	if (bind(nFD, (sockaddr*) &sa, sizeof(sa)))
	{
		close(nFD);
		return false;
	}

	m_pListen = new MUILSocketListen(this);
	m_pListen->Attach(nFD);
	m_pListen->AsyncSelect(FD_ACCEPT /*= FD_READ | FD_WRITE  | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/);
	m_pListen->Listen();

	m_sListenSockPath = sa.sun_path;
	
	return true;
}

void MUILocalSocket::Detach()
{
	m_pCtrl = NULL;
}

void MUILocalSocket::Do()
{
	m_mutex.lock();
	m_waitStop.wait(&m_mutex);
	m_mutex.unlock();
}

void MUILocalSocket::Stop()
{
	m_mutex.lock();
	StopUnlocked();
	m_mutex.unlock();
}

void MUILocalSocket::StopUnlocked()
{
	ASSERT(m_pListen);
	m_pListen->Close();
	if (!m_sListenSockPath.empty())
		unlink(m_sListenSockPath.c_str());
	for (list<MUILSocketCommunicate*>::iterator it = m_listCommunicate.begin(); it != m_listCommunicate.end(); ++it)
		(*it)->Close();

	m_waitStop.wakeAll();
}

void MUILocalSocket::AddSocket(MUILSocketCommunicate* pSock)
{
	m_mutex.lock();
	m_listCommunicate.push_back(pSock);
	m_mutex.unlock();
}

void MUILocalSocket::RemoveSocket(MUILSocketCommunicate* pSock)
{
	m_mutex.lock();
	for (list<MUILSocketCommunicate*>::iterator it = m_listCommunicate.begin(); it != m_listCommunicate.end(); ++it)
		if (*it == pSock)
		{
			m_listCommunicate.erase(it);
			break;
		}
	delete pSock;
	m_mutex.unlock();
}

