/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnuupload.cpp  -  File upload controller class
 
 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"

#include "event.h"
#include "messages.h"
#include "gnushare.h"
#include "gnudirector.h"

#include "gnuupload.h"
#include "conversions.h"
#include "common.h"
#include "property.h"
#include "preferences.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "asyncfile.h"

#define DEF_UPLOAD_BUFFER 0x00004000

//////////////////////////////////////////////////////////////////////////////
class MUploadFile : public MAsyncFile
{
public:
	MUploadFile(MGnuUpload* pUpload);
	virtual ~MUploadFile();
	virtual void OnSuccessDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState);
	virtual void OnErrorDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState);
	void DetachUpload();
protected:
	MGnuUpload* m_pBuddy;
};

class MUploadBuffer : public MAFBufferTrivial
{
public:
	MUploadBuffer(int nBufferSize = 16384) : MAFBufferTrivial(nBufferSize) {
		m_nContains = 0;
		m_nOffset = 0;
	}
	// TODO: mutex-protect it all
	inline char* data(){return m_pBuffer + m_nOffset;}
	inline char* space(){ if (remains()>0) return m_pBuffer + m_nContains; return NULL; }
	inline int   contains(){return m_nContains - m_nOffset;}
	inline int   size(){return m_nSize - m_nOffset;}
	inline int   remains(){return m_nSize - m_nContains;}
	inline bool  increment(int nBy){ASSERT(nBy>=0); ASSERT(m_nContains>=0); m_nContains += nBy; ASSERT(m_nContains <= m_nSize); return true;}
	inline bool  isFull(){return m_nContains == m_nSize;}
	inline bool  isEmpty(){return m_nContains==0;}
	inline void  clear(){m_nContains = 0; m_nOffset = 0;}
	inline bool  offset(int nBy) {ASSERT(nBy>=0); ASSERT(m_nOffset>=0); m_nOffset += nBy; ASSERT(m_nOffset <= m_nSize); return true;}
protected:
	int    m_nContains;
	int    m_nOffset;
	//
	~MUploadBuffer(){}
};

////////////////////////////////////////////////////////////////////////////////

MUploadFile::MUploadFile(MGnuUpload* pBuddy) : MAsyncFile(AFM_READ), m_pBuddy(pBuddy)
{
}

MUploadFile::~MUploadFile()
{
}

void MUploadFile::DetachUpload()
{
	m_pBuddy = NULL;
}

void MUploadFile::OnSuccessDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState)
{
	switch (nReqType)
	{
		case AFR_OPEN:
			if (m_pBuddy)
				m_pBuddy->OnFileOpen();
		break;
		case AFR_READ:
		case AFR_READ|AFR_SEEK:
			if (m_pBuddy)
				m_pBuddy->OnFileRead(FileState.nSize, FileState.nReturn, FileState.nErrNo);
		break;
		case AFR_NONE:
			TRACE("MUploadFile::OnSuccess: AFR_NONE - should not happen");
		break;
	}
}

void MUploadFile::OnErrorDelayed(DWORD dwSeqReqID, int nReqType, const SAFileState& FileState)
{
	if (m_pBuddy)
		m_pBuddy->OnFileError(FileState.nErrNo);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////

MGnuUpload::MGnuUpload(MGnuDirector* pComm)
{
	MLock lock(m_mutex);
	//
	m_pDirector  = pComm;
	m_pShare = m_pDirector->GetShare();
	m_pPrefs = m_pDirector->GetPrefs();

	m_sUserAgent = "Unknown";
	

	m_pAFile = NULL;

	m_bKeepAlive = false;

	// Bandwidth
	for(int i = 0; i < 60; i++)
		m_dwAvgBytes[i] = 0;

	m_nSecsUnderLimit = 0;
	m_nSecsDead       = 0;
	m_dwBytes60       = 0;
	m_dwSecBytes      = 0;
    m_dRate           = 0;
	m_dMaxRate        = -1;
	m_bBelowRateLimit = true;
	m_bBufferReady    = false;
	
	if(m_pPrefs->m_dBandwidthTotal > 0)
		m_dMaxRate = 0;
	else
		m_dMaxRate = -1;
	
	m_nSecNum = 0;
	m_nSecPos = 0;

	m_nRequestedRange = -1;

	//
	m_dwID = 0;
}

void MGnuUpload::ResetState(bool bLock)
{             
	MLock lock(m_mutex, bLock);
	// close the file
	if (m_pAFile)
	{
		m_pAFile->DetachUpload();
		m_pAFile->Close();
		delete m_pAFile;
		m_pAFile = NULL;
	}
	// release the buffer
	m_pBuffer=NULL;

	// for the record...
	if (m_bHttpOK)
	{
		// this message is only needed for logging
		// if we intend to use it for anything else we'll have to
		// define a custom event type
		CString s;
		int nRelTime = xtime() - m_nStartTime;
		s.format("start:%s\t pos:%d\t sent:%d\t speed:%sb/s\t host:%s\t client:%s\t file:%s",
				FormatAbsTimeFull(m_nStartTime).c_str(),
				m_nStartOffset,
				m_nBytesCompleted-m_nStartOffset,
				nRelTime>0 ? FormatSize((m_nBytesCompleted-m_nStartOffset)/nRelTime).c_str() : "--.-",
				Ip2Str(m_ipHost).c_str(),
				m_sUserAgent.c_str(),
				m_sFileName.c_str());
		POST_EVENT( MStringEvent(
			ET_MESSAGE,
			ES_MINIMAL,
			s,
			MSG_UPLOAD_FINISHED,
			MSGSRC_UPLOAD
		));
	}

	// indeed reset the state
	if (m_sFileName.empty())
		m_sFileName = "**not yet known**";
	else
		m_sFileName += " **completed**";
	m_nFileLength = 0;
	m_nRequestedRange = -1;

	m_nBytesCompleted = 0;
	m_nFileIndex = -1;
	m_nFileIndexPush = -1;

	m_bUriRequest = false;

	m_nStartOffset = 0;
	m_nStartTime = xtime();
	m_bHttpOK = false; // we have not yet accepted the transfer

	m_nSecsDead       = 0; // reset it here as well

	m_sHandshake.clear();
	m_sTmpHandshake.clear();

	m_nError = UPLERR_NONE;
	StatusUpdate(TRANSFER_CONNECTING);
}

MGnuUpload::~MGnuUpload()
{
	//
	m_mutex.lock();
	ForceDisconnect();

	ResetState(false);
	m_mutex.unlock();
}

/////////////////////////////////////////////////////////////////////////////
// MGnuUpload member functions

LPCSTR SGnuUpload::GetErrorString(int nError)
{
	switch (nError)
	{
		case UPLERR_NONE:
			return "";
		case UPLERR_NODATA:
			return "No Data";
		case UPLERR_SOCKET_EROR:
			return "Socket Error";
		case UPLERR_BAD_PUSH:
			return "Bad Push";
		case UPLERR_BUSY:
			return "Busy";
		case UPLERR_NOT_FOUND:
			return "No File";
		case UPLERR_REM_CANCELED:
			return "Rem. Canceled";
		case UPLERR_CONNECT_FAIL:
			return "Connect Fail";
		case UPLERR_BAD_REQUEST:
			return "Bad Request";
		case UPLERR_NO_RESPONSE:
			return "No Response";
		case UPLERR_TOO_SLOW:
			return "Too Slow";
	}
	return "Unknown";
}

void MGnuUpload::OnConnect(int nErrorCode) 
{
	MLock lock(m_mutex);
	
	if (TRANSFER_PUSH_CONNECTING != m_nStatus)
	{
		MAsyncSocket::OnConnect(nErrorCode);
		return;
	}

	BYTE* ClientID = (BYTE*) m_pDirector->GetClientID();
	char szGuid[40];
	sprintf(szGuid,"%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
				ClientID[0],  ClientID[1],  ClientID[2],  ClientID[3],  ClientID[4],
				ClientID[5],  ClientID[6],  ClientID[7],  ClientID[8],  ClientID[9],
				ClientID[10], ClientID[11], ClientID[12], ClientID[13], ClientID[14],
				ClientID[15]);
	MakeUpper(szGuid);

	CString HttpGiv = "GIV " + DWrdtoStr(m_nFileIndex) + ":" + szGuid + "/" + m_sFileName + "\n\n";

	Send(HttpGiv.c_str(), HttpGiv.length());
	StatusUpdate(TRANSFER_PUSH_CONNECTED);

	m_sHandshake += HttpGiv;
	
	MAsyncSocket::OnConnect(nErrorCode);
}

void MGnuUpload::OnReceive(int nErrorCode)
{
	//TRACE("MGnuUpload::OnReceive");
	MLock lock(m_mutex);
	
	BYTE* pBuff = new BYTE[4096];
	ASSERT(pBuff);
	DWORD dwBuffLength = Receive(pBuff, 4096);
	
	switch (dwBuffLength)
	{
	case 0:
		delete [] pBuff;
		m_nError = UPLERR_NODATA;
		ForceDisconnect();
		return;
	case SOCKET_ERROR:
		delete [] pBuff;
		m_nError = UPLERR_SOCKET_EROR;
		ForceDisconnect();
		return;
	}
	
	if (m_nStatus != TRANSFER_PUSH_CONNECTED &&
		(m_nStatus != TRANSFER_CONNECTING || !m_bKeepAlive ))
	{
		delete [] pBuff;
		MAsyncSocket::OnReceive(nErrorCode);
		return; // ignore everything
	}

	pBuff[dwBuffLength] = 0;
	CString Header((char*)pBuff);
	m_sHandshake += Header;
	m_sTmpHandshake += Header;

	// New Upload
	if(Header.find("\r\n\r\n") >= 0 )
	{
		if( m_sTmpHandshake.find("GET /get/") == 0                  ||
			m_sTmpHandshake.find("GET /uri-res/N2R?urn:sha1:") == 0 )
		{
			// Get Node info
			CString Host;
			UINT    nPort;
			GetPeerName(Host, nPort);
			
			// Set Variables
			Str2Ip(Host,m_ipHost);
			m_nPort = nPort;
			// "false" means that the mutex is locked
			UploadFile(m_sTmpHandshake,false);
			m_sTmpHandshake.clear();
		}
		else
		{
			m_nError = UPLERR_BAD_PUSH;
			ForceDisconnect();
		}
	}

	delete [] pBuff;

	MAsyncSocket::OnReceive(nErrorCode);
}

void MGnuUpload::Send_HttpOK()
{
	CString HttpOK;
	if(m_nBytesCompleted || m_nRequestedRange != m_nFileLength)
		HttpOK =  "HTTP/1.1 206 Partial Content\r\n";
	else
		HttpOK =  "HTTP/1.0 200 OK\r\n";
	HttpOK += "Server: Mutella-";
	HttpOK += VERSION;
	HttpOK += "\r\n";
	HttpOK += "Content-Type: application/binary\r\n";
	if(m_nBytesCompleted || m_nRequestedRange != m_nFileLength)
	{
		HttpOK += "Accept-Ranges: bytes\r\n";
		HttpOK += "Content-range: bytes ";
		HttpOK +=  DWrdtoStr(m_nBytesCompleted) + "-" + DWrdtoStr(m_nRequestedRange - 1) + "/" + DWrdtoStr(m_nFileLength) + "\r\n";
	}

	HttpOK += "Content-Length: " + DWrdtoStr(m_nRequestedRange - m_nBytesCompleted) + "\r\n";

	if (m_bKeepAlive)
		HttpOK += "Connection: Keep-Alive\r\n";

	// X-Filename or X-Gnutella-Content-URN
	if(m_bUriRequest)
		HttpOK += "X-Filename: " + m_sFileName + "\r\n";
	else if (!m_sSha1.empty())
		HttpOK += "X-Gnutella-Content-URN: urn:sha1:" + m_sSha1 + "\r\n";
	
	HttpOK += "\r\n";
	Send(HttpOK.c_str(), HttpOK.length());

	StatusUpdate(TRANSFER_SENDING);
	m_sHandshake += HttpOK;
}

void MGnuUpload::Send_HttpBusy()
{
	CString Http503 =  "HTTP/1.0 503 Server Is Busy\r\n";
			Http503 += "Server: Mutella\r\n";
			Http503 += "Content-type:text/html\r\n\r\n";
			Http503 += "<HTML>\r\n";
			Http503 += "<HEAD><TITLE>503 Server Is Busy</TITLE></HEAD>\r\n";
			Http503 += "<BODY>\r\n";
			Http503 += "<H1>Server Is Busy</H1>\r\n";
			Http503 += "This server's upload max has been met, try again later.\r\n";
			Http503 += "<HR NOSHADE SIZE=1>\r\n";
			Http503 += "Powered by <I><B><A HREF=\"http://mutella.sourceforge.net\">Mutella</A></B></I>.\r\n";
			Http503 += "</BODY>\r\n";
			Http503 += "</HTML>\r\n\r\n";
	Send(Http503.c_str(), Http503.length());

	m_nError = UPLERR_BUSY;
	Close();
	StatusUpdate(TRANSFER_CLOSED);
	m_sHandshake += Http503;
}

void MGnuUpload::Send_HttpNotFound()
{
	CString Http404 =  "HTTP/1.0 404 Not Found\r\n";
			Http404 += "Server: Mutella\r\n";
			Http404 += "Content-type:text/html\r\n\r\n";
			Http404 += "<HTML>\r\n";
			Http404 += "<HEAD><TITLE>404 Not Found</TITLE></HEAD>\r\n";
			Http404 += "<BODY>\r\n";
			Http404 += "<H1>Not Found</H1>\r\n";
			Http404 += "The requested file \"<I>" + m_sFileName + "</I>\" was not found on this server.\r\n";
			Http404 += "<HR NOSHADE SIZE=1>\r\n";
			Http404 += "Powered by <I><B><A HREF=\"http://mutella.sourceforge.net\">Mutella</A></B></I>.\r\n";
			Http404 += "</BODY>\r\n";
			Http404 += "</HTML>\r\n\r\n";
	Send(Http404.c_str(), Http404.length());

	m_nError = UPLERR_NOT_FOUND;
	Close();
	StatusUpdate(TRANSFER_CLOSED);
	m_sHandshake += Http404;
}

void MGnuUpload::StatusUpdate(DWORD Status)
{
	ASSERT(m_mutex.locked());
	m_nSecsDead = 0;

	m_nStatus = Status;
	m_nChangeTime = xtime();
}

void MGnuUpload::ForceDisconnect()
{
	ASSERT(m_mutex.locked());
	
	if(m_hSocket != INVALID_SOCKET)
	{
		AsyncSelect(FD_CLOSE);
		ShutDown(2);
	}
	
	StatusUpdate(TRANSFER_CLOSED);
	Close();
}

void MGnuUpload::OnClose(int nErrorCode)
{
	MLock lock(m_mutex);
	
	if(m_nError == UPLERR_NONE)
		m_nError = UPLERR_REM_CANCELED;
	
	StatusUpdate(TRANSFER_CLOSED);
		
	MAsyncSocket::OnClose(nErrorCode);
}

void MGnuUpload::PushFile()
{
	MLock lock(m_mutex);
	
	StatusUpdate(TRANSFER_PUSH);
	
	// Set file name and size
	if(!m_pShare->GetFileByIndex(m_nFileIndex, m_sFileName, m_sFilePath, false))
	{
		m_nError = UPLERR_NOT_FOUND;
		ForceDisconnect();
		return;
	}
	
	ASSERT(m_pAFile==NULL);
	m_pAFile = new MUploadFile(this);
	m_pAFile->Open(m_sFilePath.c_str());
}

void MGnuUpload::UploadFile(const CString& sHandshake, bool bLock)
{
	//TRACE("MGnuUpload::UploadFile");
	MLock lock(m_mutex, bLock);
	ASSERT(m_mutex.locked());

	// check for black-listed hosts
	if (m_pPrefs->m_bBlackListActive && m_pPrefs->m_ipnetsetBlackList.IsIn(m_ipHost))
	{
		// we dont serve files to bustards
		Send_HttpNotFound();
		return;
	}
	
	// white list is a list to which upload slots are always given...
	if (!m_pPrefs->m_ipnetsetWhiteList.IsIn(m_ipHost))
	{
		// If there's an upload max
		if(m_pPrefs->m_nMaxUploads >= 0 && m_pDirector->CountUploads() >= m_pPrefs->m_nMaxUploads)
		{
			Send_HttpBusy();
			return;
		}

		// If we are already uploading to this host, give a busy signal
		if ( m_pPrefs->m_nMaxPerHostUploads > 0 && m_pDirector->CountUploads(m_ipHost, this) >= m_pPrefs->m_nMaxPerHostUploads )
		{
			Send_HttpBusy();
			return;
		}
	}

	CString sLowHandshake = sHandshake;
	sLowHandshake.make_lower();

	// or, process the request
	if (0 == strncmp(sHandshake.c_str(),"GET /get/",9))
	{
		// we got traditional get request
			
		// Index
		m_nFileIndex = -1;
		CString FirstLine = sLowHandshake.substr(0, sLowHandshake.find("\r\n") );
	
		sscanf( FirstLine.c_str(), "get /get/%ld/", &m_nFileIndex);

		// Filename
		int Begin  = FirstLine.find("/", 9) + 1;
		int	End    = FirstLine.find(" http/", Begin);
		if (End < 0)
			End = FirstLine.length();
		m_sFileName = FirstLine.substr(Begin, End - Begin);
		// Web browser compatibility
		m_sFileName = restore_string(m_sFileName, false);
		//ReplaceSubStr(m_sFileName,"%28", "(");
		//ReplaceSubStr(m_sFileName,"%29", ")");
		//ReplaceSubStr(m_sFileName,"%20", " ");
	}
	else if (0 == strncmp(sHandshake.c_str(),"GET /uri-res/N2R?urn:sha1:",26))
	{
		// we got HUGE request
		int nEnd = sHandshake.find(" HTTP/");
		if (nEnd < 0)
		{
			Send_HttpNotFound();
			return;
		}
		m_sSha1 = StripWhite(sHandshake.substr(26, nEnd-26));
		if (m_sSha1.length() != 32)
		{
			Send_HttpNotFound();
			return;
		}
		// resolve index from Sha1
		m_nFileIndex = m_pShare->GetIndexBySha1(m_sSha1);
		if (m_nFileIndex < 0)
		{
			Send_HttpNotFound();
			return;
		}

		m_bUriRequest = true;
	}
	else
	{
		Send_HttpNotFound();
		return;
	}

	// If we are already uploading the same file too many times
	if ( m_pPrefs->m_nMaxPerFileUploads > 0 && m_pDirector->CountUploads(m_nFileIndex, this) >= m_pPrefs->m_nMaxPerFileUploads )
	{
		Send_HttpBusy();
		return;
	}

    // User-Agent header
	m_sUserAgent = find_header("User-Agent",sHandshake,sLowHandshake,"Unknown");
	if (m_sUserAgent.length() > 25)
	{
		int nPos = m_sUserAgent.find(' ');
		if (nPos>0)
			m_sUserAgent.cut(nPos);
	}

	// Range header
	m_nBytesCompleted = 0;
	CString sRange = find_header("Range",sHandshake,sLowHandshake);
	if(!sRange.empty())
	{
		TRACE2("Range header : ", sRange);
		sRange.make_lower();
		int nDashPos = sRange.find('-');
		CString sRangeEnd;
		if(nDashPos>0)
		{
			sRangeEnd = StripWhite(sRange.substr(nDashPos+1));
			sRange.cut(nDashPos);
			if (!sRangeEnd.empty())
			{
				if(0 >= sscanf(sRangeEnd.c_str(), "%ld", &m_nRequestedRange))
				{
                        		m_nError = UPLERR_BAD_REQUEST; // TODO: send http reply
                        		StatusUpdate(TRANSFER_CLOSED);
                        		Close();
                        		return;
                		}
				if (m_nRequestedRange >= 0)
                        		++m_nRequestedRange; // FIX: in HTTP/1.1 ranges are inclusive!
			}
		}
		if (0 >= sscanf( sRange.c_str(), "bytes=%ld", &m_nBytesCompleted))
		{
			m_nError = UPLERR_BAD_REQUEST; // TODO: send http reply
			StatusUpdate(TRANSFER_CLOSED);
			Close();
			return;
		}
		TRACE4("Detected range request : ", m_nBytesCompleted, " - ", m_nRequestedRange);
	}
	
	// Connection: KeepAlive header
	CString sConnection = find_header("Connection",sHandshake,sLowHandshake);
	sConnection.make_lower();
	if (sConnection.find("keep-alive") >= 0)
		m_bKeepAlive = true;
	// here compex part starts: if it is a push we have file already open, but it would make sence to verify if it is the right file.
	if (m_nStatus != TRANSFER_PUSH_CONNECTING && m_nStatus != TRANSFER_PUSH_CONNECTED)
	{
		// normal transfer
		if(m_nFileIndex == -1 || !m_pShare->GetFileByIndex(m_nFileIndex, m_sFileName, m_sFilePath, m_sSha1.empty()))
		{
			m_nBytesCompleted = -1;
			Send_HttpNotFound();
			return;
		}
		ASSERT(m_pAFile == NULL);
		StatusUpdate(TRANSFER_CONNECTED);
		m_pAFile = new MUploadFile(this);
		m_pAFile->Open(m_sFilePath.c_str());
		return;
	}

	// okm it's a push
	// so the file is already open, lets check if it is the right one
	ASSERT(m_pAFile);
	if (m_nFileIndexPush != m_nFileIndex ||
		!m_pAFile->IsOpen()              )
	{
		TRACE("wrong push");
		m_nBytesCompleted = -1;
		Send_HttpNotFound();
		return;
	}
	//
	StatusUpdate(TRANSFER_CONNECTED);
	// need to call OnFileOpen or do what is done there
	OnFileOpen(false);
}

void MGnuUpload::OnFileOpen(bool bLock)
{
	//TRACE("OnFileOpen");	
	MLock lock(m_mutex, bLock);
	ASSERT(m_mutex.locked());

	if (!m_pAFile)
		return;
	
	// called in two cases: push request and actual upload request.
	// we keep file open after push request.
	
	if (TRANSFER_PUSH == m_nStatus)
	{
		StatusUpdate(TRANSFER_PUSH_CONNECTING);
		m_nFileIndexPush = m_nFileIndex;
		m_nFileLength = m_pAFile->GetSize();
		if(!Connect(Ip2Str(m_ipHost).c_str(), m_nPort))
			if(GetLastError() != EWOULDBLOCK && GetLastError() != EINPROGRESS)
			{
				m_nError = UPLERR_CONNECT_FAIL;
				StatusUpdate(TRANSFER_CLOSED);
				Close();
			}
		return;
	}

	m_nFileLength = m_pAFile->GetSize();

	if (m_nRequestedRange<=0)
		m_nRequestedRange = m_nFileLength;
	else if (m_nRequestedRange > m_nFileLength) // they made a mistake, we'll do what we can
		m_nRequestedRange = m_nFileLength;
	
	if (m_nBytesCompleted < 0 || m_nBytesCompleted>=m_nRequestedRange)
	{
		// shall we send something to the server here, like bad request or something?
		m_nError = UPLERR_BAD_REQUEST;
		StatusUpdate(TRANSFER_CLOSED);
		Close();
		return;
	}
	
	Send_HttpOK();
	m_bHttpOK = true; // we have accepted the transfer

	// time to allocate the memory
	m_pBuffer = new MUploadBuffer(4096); // 4k
	m_pBuffer->Release(); // SmartPtr takes responsibility
	
	m_bBufferReady = false;
	UpdateSelectFlags();
	//TRACE("MGnuUpload::OnFileOpen -- calling PreFetch");
	if(m_nBytesCompleted)
	{
		VERIFY(m_pAFile->ReadSeek(m_nBytesCompleted,SEEK_SET, m_pBuffer,
				min(m_pBuffer->remains(),m_nRequestedRange - m_nBytesCompleted))>=0);
	}
	else
	{
		VERIFY(m_pAFile->Read(m_pBuffer, min(m_pBuffer->remains(),m_nRequestedRange - m_nBytesCompleted))>=0);
	}
	m_nStartOffset = m_nBytesCompleted;
}

void MGnuUpload::OnFileRead(int Requested, int nRead, int nError)
{
	//TRACE("MGnuUpload::OnFileRead");
	
	MLock lock(m_mutex);

	if (!m_pAFile)
		return;

    ASSERT(m_pBuffer->isEmpty());
    ASSERT(nRead > 0);
	m_pBuffer->increment(nRead);
	
	m_bBufferReady = true;
	UpdateSelectFlags();
}

void MGnuUpload::OnFileError(int nError)
{
	//TRACE("OnFileError");
	MLock lock(m_mutex);	
	if ( m_nStatus == TRANSFER_CONNECTED )
	{
		// we are just opening file and for what ever reason it did not work out...
		Send_HttpNotFound();
		// notify everybody who might be intrested
		POST_EVENT( MIntEvent(
			ET_ERROR,
			ES_UNIMPORTANT,
			m_nFileIndex,
			MSG_UPLOAD_FILE_OPEN_FAILED,
			MSGSRC_UPLOAD
		));
	}
	// TODO: set m_nError for other cases correctly
	ForceDisconnect();
}

void MGnuUpload::UpdateSelectFlags()
{
	if (m_bBelowRateLimit && m_bBufferReady)
		ModifySelectFlags(FD_WRITE,0);
	else
		ModifySelectFlags(0,FD_WRITE);
}

void MGnuUpload::BandwidthTimer()
{
	//TRACE("BandwidthTimer");
	MLock lock(m_mutex);
	
	m_nSecNum++;
	if(m_nSecPos >= 60)
		m_nSecPos = 0;

	// Bytes
	m_dwBytes60 -= m_dwAvgBytes[m_nSecPos];
	m_dwAvgBytes[m_nSecPos] = m_dwSecBytes;
	m_dwBytes60 += m_dwSecBytes;
	if (m_nSecNum)
		if (m_nSecNum >= 60)
			m_dRate = m_dwBytes60/60.0;
		else
			m_dRate = m_dwBytes60/m_nSecNum;

	// Mininum Trasfer Speed Verification
	if(TRANSFER_CONNECTING == m_nStatus ||
	   TRANSFER_PUSH_CONNECTING == m_nStatus ||
	   TRANSFER_CONNECTED  == m_nStatus)
	{
		m_nSecsDead++;

		if(m_nSecsDead > 60)
		{
			m_nError = UPLERR_NO_RESPONSE;
			StatusUpdate(TRANSFER_CLOSED);
			Close();
		}
	}
	else if(TRANSFER_PUSH == m_nStatus)
	{
		m_nSecsDead++;

		if(m_nSecsDead > 60)
		{
			m_nError = UPLERR_CONNECT_FAIL;
			StatusUpdate(TRANSFER_CLOSED);
			Close();
		}
	}
	else if(TRANSFER_SENDING == m_nStatus)
	{
		// Check for dead transfer
		if(m_dwBytes60 == 0)
		{
			m_nSecsDead++;

			if(m_nSecsDead > 30)
			{
				m_nError = UPLERR_NO_RESPONSE;
				Close();
				StatusUpdate(TRANSFER_CLOSED);
			}
		}
		else
			m_nSecsDead = 0;

		if(m_pPrefs->m_dMinUpSpeed > 0 && m_nSecNum > 30)
		{
			// Check if its under the bandwidth limit
			if(m_dRate < m_pPrefs->m_dMinUpSpeed)
				m_nSecsUnderLimit++;
			else
				m_nSecsUnderLimit = 0;

			if(m_nSecsUnderLimit > 30)
			{	
				m_nError = UPLERR_TOO_SLOW;
				Close();
				StatusUpdate(TRANSFER_CLOSED);
			}
		}
		
		// bandwidth limits
		if (TRANSFER_SENDING == m_nStatus && !m_bBelowRateLimit)
		{
			if (m_dMaxRate<0 || m_dRate<m_dMaxRate)
			{
				m_bBelowRateLimit = true;
				UpdateSelectFlags();
			}
		}
	}
	else if(TRANSFER_COMPLETED == m_nStatus)
	{
		m_nSecsDead++;
		if(m_nSecsDead > 5 && m_bKeepAlive)
			ResetState(false);
		else if (m_nSecsDead > 15)
		{
			ForceDisconnect();
			m_nStatus = TRANSFER_CLOSED;
		}
	}
	else if(TRANSFER_CLOSED == m_nStatus)
	{
	}

	m_dwSecBytes = 0;
	m_nSecPos++;

	// Check for completion
	//ASSERT(TRANSFER_SENDING != m_nStatus || m_nRequestedRange>0 );
	if (!(TRANSFER_SENDING != m_nStatus || m_nRequestedRange>0 ))
	{
		cout << m_sHandshake << endl;
		ASSERT(0);
	}
	if(TRANSFER_SENDING == m_nStatus && ( m_nRequestedRange <= m_nBytesCompleted ))
	{
		Close();
		StatusUpdate(TRANSFER_COMPLETED);
	}
}

void MGnuUpload::OnSend(int nErrorCode) 
{
	//TRACE("OnSend");
	MLock lock(m_mutex);
	
	ASSERT(m_pAFile);
	ASSERT(m_pBuffer != NULL);
	if (!m_pBuffer->contains())
	{
		TRACE( "WARNING: MGnuUpload::OnSend occured when no data is available");
		TRACE2("         m_nBytesCompleted                      = ", m_nBytesCompleted);
		TRACE2("         m_nFileLength                          = ", m_nFileLength);
		TRACE2("         m_bBufferReady                         = ", m_bBufferReady);
		TRACE2("         m_pAFile->GetPos()                     = ", m_pAFile->GetPos());
		TRACE2("         m_pAFile->GetAsyncPos()                = ", m_pAFile->GetPos());
		TRACE2("         m_pAFile->GetCurrRequest()             = ", m_pAFile->GetCurrRequest());
		TRACE2("         m_pAFile->GetCurrComplRequest()        = ", m_pAFile->GetCurrComplRequest());
		TRACE2("         m_pAFile->GetCurrDelayedComplRequest() = ", m_pAFile->GetCurrDelayedComplRequest());

		return ; // quick and dirty
	}
	if (m_nStatus != TRANSFER_SENDING)
	{
		TRACE("WARNING: MGnuUpload::OnSend occured when m_nStatus != TRANSFER_SENDING");
		return ; // quick and dirty
	}
	ASSERT(m_nRequestedRange>0);
	if(m_nBytesCompleted >= m_nRequestedRange)
	{
		TRACE("WARNING: MGnuUpload::OnSend encounetered m_nBytesCompleted >= m_nRequestedRange situation");
		return ; // quick and dirty
	}
	if (!m_bBufferReady)
	{
		TRACE ("WARNING: MGnuUpload::OnSend happen when m_bBufferReady = false");
		return ; // quick and dirty
	}
		
	// we have what to send -- do it
	int nBytesAvail = m_pBuffer->contains();
	int nBytesSentNow = Send(m_pBuffer->data(), nBytesAvail);
	if (nBytesSentNow <= 0)
	{
		m_pBuffer->clear();
		ForceDisconnect();
		m_nStatus = TRANSFER_CLOSED;
		m_nError = UPLERR_SOCKET_EROR;
		return;
	}
	if (nBytesSentNow == nBytesAvail)
		m_pBuffer->clear();
	else
	{
		VERIFY(m_pBuffer->offset(nBytesSentNow));
	}
	// total count, statistics, etc
	m_dwSecBytes += nBytesSentNow;
	m_nBytesCompleted += nBytesSentNow;
	// limit bandwidth
	if (m_dMaxRate>0 && m_dwSecBytes>m_dMaxRate && m_bBelowRateLimit)
	{
		m_bBelowRateLimit = false;
		UpdateSelectFlags();
	}
	// check for completion
	ASSERT(m_nRequestedRange>0);
	if (m_nBytesCompleted >= m_nRequestedRange)
	{
		 //ForceDisconnect();
		 ModifySelectFlags(0,FD_WRITE);
		 m_nStatus = TRANSFER_COMPLETED;
	}
	else
	if (!m_pBuffer->contains())
	{
		ASSERT(m_pAFile->GetPos()==m_nBytesCompleted);
		m_bBufferReady = false;
		UpdateSelectFlags();
		ASSERT(m_pBuffer->remains()==4096);
		m_pAFile->Read(m_pBuffer, min(m_pBuffer->remains(),m_nRequestedRange - m_nBytesCompleted));
	}
	MAsyncSocket::OnSend(nErrorCode);
}

