/***************************************************************************
                          gnulogcentre.cpp  -  description
                             -------------------
    begin                : Tue Oct 8 2002
    copyright            : (C) 2002 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "mthread.h"
#include "event.h"
#include "messages.h"
#include "conversions.h"
#include "common.h"
#include "controller.h"

#include "gnulogcentre.h"

class MEventQueueLog : public MAsyncEventReceiver {
public:
	MEventQueueLog() : MAsyncEventReceiver(INT_MAX) {}
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		switch (p->GetID())
		{
			case MSG_MUTELLA_EXIT:
			case MSG_REMOTE_ACCESS:
			case MSG_UPLOAD_FINISHED:
			case MSG_CONNECTION_BYE:
#ifdef _DEBUG
			case MSG_CONNECTION_HANDSHAKE_OK:
			case MSG_CONNECTION_HANDSHAKE_FAIL:
#endif //_PACKET_DEBUG
#ifdef _PACKET_DEBUG
			case MSG_PACKET_RECEIVED:
			case MSG_PACKET_SENT:
			case MSG_PACKET_REMOVED_GOOD:
			case MSG_PACKET_REMOVED_BAD:
			case MSG_PACKET_RECEIVED_UNKNOWN:
#endif //_DEBUG
				return true;
		}
		return false;
	}
};


MGnuLogCentre::MGnuLogCentre(MController* pController) : m_pController(pController)
{
	m_pEventQueue = new MEventQueueLog();
	ED().AddReceiver(m_pEventQueue);
}

MGnuLogCentre::~MGnuLogCentre()
{
}

#define ONE_HOUR          (60*60)
#define TWO_HOURS        (120*60)
#define HALF_A_DAY     (12*60*60)
#define ONE_DAY        (24*60*60)
#define TWO_DAYS     (2*24*60*60)
#define HALF_A_WEEK  (7*12*60*60)
#define ONE_WEEK     (7*24*60*60)
#define TWO_WEEKS   (14*24*60*60)
#define THREE_WEEKS (21*24*60*60)
#define FOUR_WEEKS  (28*24*60*60)
#define ONE_MONTH   (30*24*60*60)

void MGnuLogCentre::run()
{
	if (!m_pEventQueue)
		return;
	//
	for(;;)
	{
		if (!m_pEventQueue->Poll())
			continue;
		// now we do have an event!
		TSmartPtr<MEvent> spEvent = m_pEventQueue->SafeFront();
		m_pEventQueue->Pop();
		switch (spEvent->GetID())
		{
			case MSG_MUTELLA_EXIT:
				ED().RemoveReceiver(m_pEventQueue);
				delete m_pEventQueue;
				m_pEventQueue = NULL;
				return;
			// replace this with user-configurable something
			case MSG_REMOTE_ACCESS:
				logEvent(spEvent, "htaccess.log", ONE_MONTH, THREE_WEEKS);
				break;
			case MSG_UPLOAD_FINISHED:
				logEvent(spEvent, "upload.log", TWO_WEEKS, ONE_WEEK);
				break;
			case MSG_CONNECTION_BYE:
				logEvent(spEvent, "bye.log", HALF_A_WEEK, ONE_DAY);
				break;
#ifdef _DEBUG
			case MSG_CONNECTION_HANDSHAKE_OK:
				logEvent(spEvent, "handshake-ok.log", TWO_DAYS, ONE_DAY);
				break;
			case MSG_CONNECTION_HANDSHAKE_FAIL:
				logEvent(spEvent, "handshake-fail.log", HALF_A_DAY, TWO_HOURS);
				break;
#endif //_DEBUG
#ifdef _PACKET_DEBUG
			case MSG_PACKET_RECEIVED:
			case MSG_PACKET_SENT:
			case MSG_PACKET_REMOVED_GOOD:
			case MSG_PACKET_REMOVED_BAD:
			case MSG_PACKET_RECEIVED_UNKNOWN:
				logEvent(spEvent, "packet.log", 15*60, 3*60);
				break;
#endif //_PACKET_DEBUG
		}
	}
}

void MGnuLogCentre::logEvent(MEvent* pEvent, const CString& sFile, int nMaxTime, int nTrimTime)
{
	CString sPath = m_pController->GetRcDirPath() + "/" + sFile;
	//
	FILE* pFile = fopen( sPath.c_str(), "a+");
	if (!pFile)
	{
		TRACE("MGnuLogCentre::logEvent(): Failed to open log file");
		POST_EVENT( MStringEvent(
					ET_ERROR,
					ES_IMPORTANT,
					"Failed to open '" + sFile + "'; error: " + strerror(errno),
					MSG_LOGGING_FAILED,
					MSGSRC_LOGCENTRE
			));
		return;
	}
	fseek(pFile, 0, SEEK_SET);
	char szLine[1024];
	if (fgets(szLine, 1024, pFile))
	{
		time_t line_time = ParseAbsTimeFull(szLine);
		ASSERT(line_time != (time_t)-1);
		int nDiff = pEvent->GetTime() - line_time;
		if (nDiff > nMaxTime) // too old
		{
			// we have to cut the file before it eats the whole disk
			TRACE2("Trimming log file ", sFile);
			CString sPathN = m_pController->GetRcDirPath() + sFile + ".tmp";
			FILE* pFileN = fopen( sPathN.c_str(), "w");
			if (!pFileN)
			{
				TRACE("MGnuLogCentre::logEvent(): Failed to open temporary log file");
				POST_EVENT( MStringEvent(
						ET_ERROR,
						ES_IMPORTANT,
						CString("Failed to open temporary log file; error: ") + strerror(errno),
						MSG_LOGGING_FAILED,
						MSGSRC_LOGCENTRE
				));
				fclose(pFile);
				return;
			}
			// scan for the first valid line
			while(nDiff > nTrimTime)
			{
				if (!fgets(szLine, 1024, pFile))
					break;
				time_t line_time = ParseAbsTimeFull(szLine);
				if (line_time == (time_t)-1)
					continue;
				nDiff = pEvent->GetTime() - line_time;
			}
			do
			{
				fputs(szLine, pFileN);
			} while(fgets(szLine, 1024, pFile));

			fclose(pFile);
			pFile = pFileN;
			DeleteFile(sPath);
			MoveFile(sPathN, sPath);
			TRACE2("Finished trimming log file ", sFile);
		}
	}
	fseek(pFile, 0, SEEK_END);
	fprintf(pFile, "%s\t %s\n", FormatAbsTimeFull(pEvent->GetTime()).c_str(), pEvent->FormatLog().c_str());
	fclose(pFile);
}


