!
!  photometric calibration
!
!  Copyright © 2012-7 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.
!
!  Problems:
!    * calibration frame is not referenced when a frame is referenced to a frame
!    * standard deviation has bad value
!
!  Adds:
!    * Re-design of command line parameters: better defaults to --col-*,
!      rename all mags to be VO-compatible (Vmag,e_Vmag). User must specify
!      only filter/photometry system (or nothing when filter/phsystem is
!      presented in FITS header).
!

program phcal

  use fitsio
  use phsysfits
  use mfits
  use iso_fortran_env

  implicit none

  integer, parameter :: dbl = selected_real_kind(15)

  character(len=4*FLEN_FILENAME) :: line, key, val
  character(len=FLEN_FILENAME), dimension(:), allocatable :: filename, backup, newname
  character(len=FLEN_FILENAME) :: ref = '', cat = '', tratable = '', &
       phsystable = 'photosystems.fits'
  character(len=FLEN_VALUE), dimension(:), allocatable :: qlabels
  character(len=FLEN_VALUE) :: col_ra = FITS_COL_RA, col_dec = FITS_COL_DEC, &
       photsys_ref = '', photsys_instr = ''
  character(len=FLEN_VALUE), dimension(:), allocatable :: col_mag, col_magerr, filters
  logical :: verbose = .false., plog = .false., list = .false.
  logical :: init_area = .false. , init_photsys = .false.
  logical :: cal_manual = .false.
  logical :: advanced = .false.
  real(dbl) :: utol = -1.0/3600.0
  real(dbl) :: area
  real(dbl), dimension(:), allocatable :: ctph_manual,dctph_manual
  integer :: eq,nfile,n,nmag,nmagerr,nfilters,nctph,saper
  character(len=FLEN_KEYWORD), dimension(8) :: keys
  integer :: nqlabels

  keys(1) = FITS_KEY_EXPTIME
  keys(2) = FITS_KEY_AREA
  keys(3) = FITS_KEY_DATEOBS
  keys(4) = FITS_KEY_PHOTSYS
  keys(5) = FITS_KEY_FILTER
  keys(6) = FITS_KEY_LONGITUDE
  keys(7) = FITS_KEY_LATITUDE
  keys(8) = FITS_KEY_TIME

  nqlabels = 7
  qlabels = (/'PHOTRATE','FLUX    ','FNU     ','FLAM    ','MAG     ', &
       'ABMAG   ','STMAG   '/)

  allocate(col_mag(0), col_magerr(0), filters(0))
  allocate(ctph_manual(0), dctph_manual(0))
  allocate(filename(0), backup(0), newname(0))
  nfilters = 0
  nmag = 0
  nmagerr = 0
  nfile = 0
  area = 1
  saper = 0

  do
     read(*,'(a)',end=20) line

     eq = index(line,'=')
     if( eq == 0 ) stop 'Improper control data.'

     key = line(:eq-1)
     val = line(eq+1:)

     if( key == 'VERBOSE' ) then

        read(val,*) verbose

     else if( key == 'PIPELOG' ) then

        read(val,*) plog

     else if( key == 'ADVANCED' ) then

        read(val,*) advanced

     else if( key == 'FITS_KEY_EXPTIME' ) then

        read(val,*) keys(1)

     else if( key == 'FITS_KEY_AREA' ) then

        read(val,*) keys(2)

     else if( key == 'FITS_KEY_DATEOBS' ) then

        read(val,*) keys(3)

     else if( key == 'FITS_KEY_PHOTOSYS' ) then ! remove?

        read(val,*) keys(4)

     else if( key == 'FITS_KEY_FILTER' ) then

        read(val,*) keys(5)

     else if( key == 'FITS_KEY_LONGITUDE' ) then

        read(val,*) keys(6)

     else if( key == 'FITS_KEY_LATITUDE' ) then

        read(val,*) keys(7)

     else if( key == 'FITS_KEY_TIME' ) then

        read(val,*) keys(8)

     else if( key == 'NFILTERS' ) then

        read(val,*) nfilters

        deallocate(filters)
        allocate(filters(nfilters))
        filters = ''

     else if( key == 'FILTERS' ) then

        ! a set of filters
        read(val,*) filters

     else if( key == 'COL_RA' ) then

        read(val,*) col_ra

     else if( key == 'COL_DEC' ) then

        read(val,*) col_dec

     else if( key == 'COL_NMAG' ) then

        read(val,*) nmag

        deallocate(col_mag)
        allocate(col_mag(nmag))
        col_mag = ''

     else if( key == 'COL_MAG' ) then

        ! a set columns
        read(val,*) col_mag

     else if( key == 'COL_NMAGERR' ) then

        read(val,*) nmagerr

        deallocate(col_magerr)
        allocate(col_magerr(nmagerr))
        col_magerr= ''

     else if( key == 'COL_MAGERR' ) then

        read(val,*) col_magerr

     else if( key == 'TOL' ) then

        read(val,*) utol

     else if( key == 'SAPER' ) then

        read(val,*) saper

     else if( key == 'AREA' ) then

        read(val,*) area
        init_area = .true.

     else if( key == 'NQUANTITIES' ) then

        read(val,*) nqlabels

        deallocate(qlabels)
        allocate(qlabels(nqlabels))
        qlabels = ''

     else if( key == 'QUANTITIES' ) then

        read(val,*) qlabels

     else if( key == 'PHOTSYS_REF' ) then

        read(val,*) photsys_ref

     else if( key == 'PHOTSYS_INSTR' ) then

        read(val,*) photsys_instr
        init_photsys = .true.

     else if( key == 'PHSYSTABLE' ) then

        read(val,*) phsystable

     else if( key == 'TRATABLE' ) then

        read(val,*) tratable

     else if( key == 'LIST' ) then

        read(val,*) list

     else if( key == 'NCTPH' ) then

        read(val,*) nctph
        deallocate(ctph_manual,dctph_manual)
        allocate(ctph_manual(nctph),dctph_manual(nctph))
        ctph_manual = 1
        dctph_manual= 0

     else if( key == 'CTPH' ) then

        ! manual calibration
        read(val,*) ctph_manual
        dctph_manual = 0
        cal_manual = .true.

     else if( key == 'CAT' ) then

        ! calibration agains to the catalogue
        read(val,*) cat

     else if( key == 'REF' ) then

        ! calibration agains to the already calibrated frame
        read(val,*) ref

     else if( line(1:5) == 'NFILE' .and. eq > 0) then

        read(val,*) nfile

        deallocate(filename,backup,newname)
        allocate(filename(nfile),backup(nfile),newname(nfile))
        filename = ''
        backup = ''
        newname = ''
        n = 0

     else if( key == 'FILE' ) then

        n = n + 1
        if( n > size(filename) ) stop 'Too many files.'

        read(line(eq+1:),*) filename(n), backup(n), newname(n)

     end if

  end do

20 continue


  if( list ) then

     call listphsys(phsystable)

  else

     if( size(filename) == 0 ) stop 'No frames to process.'

     if( cat /= '' ) then
        call calibrate   ! catcal ?
     else if( ref /= '' ) then
        call framecal
     else if( cal_manual ) then
        call mancal
     else
        stop 'Calibration type unknown.'
     end if

  end if

  deallocate(filename,backup,newname,col_mag,col_magerr,filters,ctph_manual, &
       dctph_manual)

  stop 0


contains

  ! calibration
  subroutine calibrate

    use phsysfits
    use calibre
    use minpacks
    use photoconv
    use jamming
    use sfits
    use fits_fotran

    character(len=FLEN_VALUE) :: catid,photosys_instr,photsys1,photsys2
    character(len=FLEN_VALUE), dimension(:), allocatable :: filter_frames
    real(dbl), allocatable, dimension(:) :: refra, refdec, ctph,dctph, &
         exptime, areas, tol
    real(dbl), allocatable, dimension(:,:) :: tr,tr1,trerr,tr1err,refph,drefph, &
         cts,dcts,dn,ddn,ra,dec,mag,dmag
    real(dbl) :: q
    type(type_phsys) :: phsyscal
    type(photores), dimension(:), allocatable :: phres
    integer, dimension(:,:), allocatable :: pairs
    integer :: status,nfiles,ncat,i

    if( photsys_ref == '' ) &
         stop 'No identificator for reference photometry system.'

    if( nmag == 0 ) &
         stop 'Magnitude column(s) are not specified.'

    if( nmagerr > 0 .and. nmag /= nmagerr ) &
         stop 'Magnitude error column(s) does not match magnitudes itself.'

    if( nmagerr == 0 ) &
         write(error_unit,*) 'Warning: Magnitude error column(s) missing (consider include them).'

    nfiles = size(filename)

    call readcat(cat,(/col_ra,col_dec/),col_mag,col_magerr, &
         refra,refdec,mag,dmag,catid,status)
    ! filters as results of readcat (?!)
    if( status /= 0 ) stop 'Failed to read a catalogue.'
    if( .not. allocated(mag) ) stop 'An empty catalogue.'

    ! reference system
    call phselect(phsystable,photsys_ref,phsyscal)
    call phsyspairs(phsyscal,filters,pairs)

    call readframes(filename,keys,phsyscal%filter,photosys_instr,ra,dec,dn,ddn, &
         utol,exptime,areas,filter_frames,tol,init_area,saper,status)
    if( status /= 0 ) stop 'Failed to read frames.'
    if( size(filter_frames) /= size(filters) ) &
         stop 'Filters and frames dimensions differs.'
    if( init_area ) areas = area

    if( verbose ) then
       write(*,*) 'Filename, filter, exptime [s], area [m2]:'
       do i = 1,size(filename)
          write(*,'(2(a,3x),2(f0.3,1x))') trim(filename(i)),trim(filter_frames(i)), &
               exptime(i), areas(i)
       end do
    end if

    call jamcat(tol, refra, refdec, mag, dmag, ra,dec,dn,ddn,cts,dcts)

    ncat = size(cts,1)
    if( ncat == 0 ) stop 'Joint stars not found.'
    allocate(refph(ncat,size(mag,2)),drefph(ncat,size(mag,2)))

    if( photsys_ref /= '' ) then
       call phsysmagph(phsyscal,filters,pairs,mag(1:ncat,:),dmag(1:ncat,:),refph,drefph)
    else
       ! undefined photosystem, relative rates
       call relmagph(mag(1:ncat,:),dmag(1:ncat,:),refph,drefph)
    end if


    if( tratable/= '' ) then
       call traload(tratable,photsys2,photsys1,tr,trerr,tr1,tr1err,status)
       if( status /= 0 ) stop 'Instrumental to standard conversion table not found.'
       if( photsys2 /= photsys_ref ) &
            stop 'Reference photometry system does not match the conversion table.'
       if( photsys1 /= photsys_instr ) &
            stop 'Instrumental photometry system does not match the conversion table.'
       if( size(tr,1) /= size(filters) ) &
            stop "Amount of filters doesn't correspons to transformation table."
    else
       allocate(tr(size(refph,2),size(refph,2)),tr1(size(refph,2),size(refph,2)), &
            trerr(size(refph,2),size(refph,2)),tr1err(size(refph,2),size(refph,2)))
       tr = 0.0_dbl
       forall(n=1:size(refph,2))
          tr(n,n) = 1.0_dbl
       end forall
       tr1 = tr
    end if

    if( verbose ) then
       write(*,*) 'Photons to counts transformation:'
       do i = 1,size(filters)
          write(*,'(1x,a,6f8.3)') trim(filters(i)),tr(i,:)
       end do
       write(*,*) 'Counts to photons transformation:'
       do i = 1,size(filters)
          write(*,'(1x,a,6f8.3)') trim(filters(i)),tr1(i,:)
       end do
    end if

    do n = 1,nfiles
       q = exptime(n)*areas(n)
       cts(1:ncat,n) = cts(1:ncat,n) / q
       dcts(1:ncat,n) = dcts(1:ncat,n) / sqrt(q)
    end do

    allocate(phres(nfiles),ctph(nfiles),dctph(nfiles))
    do n = 1, nfiles
       call photores_init(phres(n),ncat)
       phres(n)%ra = refra(1:ncat)
       phres(n)%dec = refdec(1:ncat)
    enddo

    if( nfiles == 1 ) then
       call calibr(refph,drefph,cts(1:ncat,:),dcts(1:ncat,:),ctph,dctph,phres,verbose)
    else
       call caliber(pairs,filters,tr,tr1,refph,drefph,cts(1:ncat,:),dcts(1:ncat,:), &
            ctph,dctph,phres,verbose)
    end if

    if( verbose ) then
       if( nfiles > 1 ) then
          write(*,*) 'Relative residuals:'
          write(*,'(5x,tr5,5a10)') filters
          do i = 1, ncat
             write(*,'(i5,5f10.4)') i,(phres(n)%res(i),n=1,size(phres))
          end do
       end if
       write(*,*) 'Final solutions:'
       do i = 1,size(ctph)
          write(*,'(3a,g15.5,a,1pg8.1)') 'ctph(',trim(filters(i)),') =', &
               ctph(i),'+-',dctph(i)
       end do
    end if

    call writecal(filename,backup,newname,keys,advanced,phsystable,filters, &
         catid,photsys_ref,area,init_area,saper,tr1,ctph,dctph,qlabels,phres)

    do i = 1,size(phres)
       call photores_destroy(phres(i))
    end do
    deallocate(phres)

    if( allocated(pairs) ) deallocate(pairs)
    call deallocate_phsyscal(phsyscal)

    deallocate(refra,refdec,refph,drefph,filter_frames,exptime,areas, &
         tol,tr,tr1,trerr,tr1err,cts,dcts,dn,ddn,ra,dec,mag,dmag,ctph,dctph)

  end subroutine calibrate

  subroutine relmagph(mag,dmag,ph,dph)

    use photoconv

    real(dbl), dimension(:,:), intent(in) :: mag,dmag
    real(dbl), dimension(:,:), intent(out) :: ph,dph

    do n = 1, size(mag,2)
       call mag2rate(mag(:,n),dmag(:,n),ph(:,n),dph(:,n))
!       dph(:,n) = sqrt(ph(:,n))
    end do


  end subroutine relmagph


  subroutine framecal

    ! framecal suppose calibration in single filter only

    use jamming
    use calibre
    use sfits

    character(len=FLEN_VALUE) :: catid,photosys_frames
    character(len=FLEN_VALUE), dimension(:), allocatable :: filter_ref,filters_fram
    real(dbl), allocatable, dimension(:) :: refra, refdec, ctph,dctph, &
         exptime, areas, tol
    real(dbl), allocatable, dimension(:,:) :: refph,drefph, &
         cts,dcts,ra,dec
    real(dbl), dimension(1,1), parameter :: tratab = real(1,dbl)
    type(photores), dimension(:), allocatable :: phres
    type(type_phsys) :: phsyscal
    integer :: status,nfiles,ncat,i
    real(dbl) :: q

    nfiles = size(filename)
    status = 0

    nfilters = 1
    allocate(filters_fram(1),filter_ref(1))

    ! reference system
    call phselect(phsystable,photsys_ref,phsyscal)

    call readref(ref,refra,refdec,refph,drefph,filter_ref(1),status)
    if( status /= 0 ) stop 'Failed to read a reference frame.'

    call readframes(filename,keys,phsyscal%filter,photosys_frames,ra,dec,cts,dcts,&
         utol,exptime,areas,filters_fram,tol,init_area,saper,status)
    if( status /= 0 ) stop 'Failed to read frames.'
    if( init_area ) areas = area

    call jamref(tol, refra, refdec, refph, drefph, ra,dec,cts,dcts)

    ncat = size(cts,1)

    do n = 1,nfiles
       q = exptime(n)*areas(n)
       cts(1:ncat,n) = cts(1:ncat,n) / q
       dcts(1:ncat,n) = dcts(1:ncat,n) / sqrt(q)
    end do

    allocate(phres(nfiles),ctph(nfiles),dctph(nfiles))
    do n = 1, nfiles
       call photores_init(phres(n),ncat)
       phres(n)%ra = refra(1:ncat)
       phres(n)%dec = refdec(1:ncat)
    enddo

    call calibr(refph(1:ncat,:),drefph(1:ncat,:),cts(1:ncat,:),dcts(1:ncat,:),&
         ctph,dctph,phres,verbose)

    call writecal(filename,backup,newname,keys,advanced,phsystable,filters_fram,&
         catid,photsys_ref,area,init_area,saper,tratab,ctph,dctph,qlabels,phres)

    deallocate(refra,refdec,refph,drefph,filter_ref,filters_fram,exptime,areas, &
         tol,cts,dcts,ra,dec,ctph,dctph)

    do i = 1,size(phres)
       call photores_destroy(phres(i))
    end do

    deallocate(phres)
    call deallocate_phsyscal(phsyscal)

  end subroutine framecal

  ! manual calibration
  subroutine mancal

    use sfits
    use fits_fotran

    character(len=FLEN_VALUE) :: catid,photsys1,photsys2
    real(dbl), allocatable, dimension(:,:) :: tr,tr1,trerr,tr1err
    integer :: n,status

    if( tratable/= '' ) then
       call traload(tratable,photsys2,photsys1,tr,trerr,tr1,tr1err,status)
       if( status /= 0 ) stop 'Instrumental to standard conversion table not found.'
       if( size(tr,1) /= size(filters) ) &
            stop "Dimensions of transformation table and filters doesn't corresponds."
       if( photsys2 /= photsys_ref ) &
            stop 'Reference photometry system does not match the conversion table.'
       if( photsys1 /= photsys_instr ) &
            stop 'Instrumental photometry system does not match the conversion table.'

    else

       allocate(tr(size(filename),size(filename)),tr1(size(filename),size(filename)), &
            trerr(0,0),tr1err(0,0))
       tr = 0.0_dbl
       forall(n=1:size(tr,1))
          tr(n,n) = 1.0_dbl
       end forall
    end if

    catid = ''

    call writecal(filename,backup,newname,keys,advanced,phsystable,filters, &
         catid,photsys_ref,area,init_area,saper,tr1,ctph_manual, &
         dctph_manual,qlabels)

    deallocate(tr,tr1,trerr,tr1err)

  end subroutine mancal


  subroutine readref(ref,refra,refdec,refph,drefph,filter,status)

    !
    ! WARNING: Only one band is supported !
    !

    real(dbl), parameter :: rad = 57.295779513082322865_dbl

    character(len=*), intent(in) :: ref
    real(dbl), dimension(:), allocatable, intent(out) :: refra,refdec
    real(dbl), dimension(:,:), allocatable, intent(out) :: refph,drefph
    character(len=*), intent(out) :: filter
    integer, intent(in out) :: status

    real(dbl), dimension(:), allocatable :: ra,dec,ph,dph
    integer, dimension(:), allocatable :: id
    character(len=FLEN_COMMENT) :: com
    character(len=FLEN_CARD) :: key
    integer, parameter :: frow = 1, felem = 1
    real(dbl), parameter :: nullval = 0
    real(dbl) :: sep, r, cosd2, ctph, dctph, skystd, phrel, phmin
    logical :: anyf, found
    integer :: nrows,nid,rcol,dcol,pcol,ecol,i,j

    if( status /= 0 ) return

    ! open and move to a table extension
    call ftiopn(15,ref,0,status)

    call ftgkys(15,keys(5),filter,com,status)
    if( status == KEYWORD_NOT_FOUND ) &
         stop 'Filter not found in reference frame. Detected an internal inconsistency.'

    call ftmnhd(15,BINARY_TBL,PHOTOEXTNAME,0,status)
    if( status == BAD_HDU_NUM ) &
         stop 'Failed to find a photometry extension in reference frame.'

    call ftgnrw(15,nrows,status)

    call ftkeyn(FITS_KEY_ANNULUS,1,key,status)
    call ftgkyd(15,key,sep,com,status)

    call ftgkyd(15,FITS_KEY_CTPH,ctph,com,status)
    call ftgkyd(15,FITS_KEY_CTPHERR,dctph,com,status)
    call ftgkyd(15,FITS_KEY_SKYSTD,skystd,com,status)

    call ftgcno(15,.true.,FITS_COL_RA,rcol,status)
    call ftgcno(15,.true.,FITS_COL_DEC,dcol,status)
    call ftgcno(15,.true.,FITS_COL_PHOTRATE,pcol,status)
    call ftgcno(15,.true.,FITS_COL_PHOTRATEERR,ecol,status)
    if( status /= 0 ) goto 666

    allocate(ra(nrows),dec(nrows),ph(nrows),dph(nrows),id(nrows))
    call ftgcvd(15,rcol,frow,felem,nrows,nullval,ra,anyf,status)
    call ftgcvd(15,dcol,frow,felem,nrows,nullval,dec,anyf,status)
    call ftgcvd(15,pcol,frow,felem,nrows,nullval,ph,anyf,status)
    call ftgcvd(15,ecol,frow,felem,nrows,nullval,dph,anyf,status)

    if( status /= 0 ) goto 666


    ! selection of stars suitable as calibration ones

    ! we accept only bright stars under the conditions:
    !   * relative precision is better than one for reference
    phrel = max(dctph / ctph, 1e-3)
    !   * minimal photon rate is larger than sky error within inner aperture
    phmin = skystd * 3.14*sep**2 * 3600**2

!    write(*,*) phrel,phmin

    ! selection of isolated stars
    nid = 0
    do i = 1, nrows
       found = .false.
       cosd2 = cos(dec(i) / rad)**2
       do j = 1, nrows
          if( i /= j ) then
             r = sqrt((ra(i) - ra(j))**2*cosd2 + (dec(i) - dec(j))**2)
             if( r < sep ) then
                found = .true.
                goto 90
             end if
          end if
       end do
90     continue

       ! accept only valid measurements  ...
       if( .not. found .and. ph(i) > 0 .and. dph(i) > 0 ) then

          ! ... and only bright stars, the limit is derived from relative
          ! precision of calibration .. and sky error determination
          !          if( dph(i) / ph(i) < phrel .and.  ph(i) > phmin ) then
          if( .true. ) then
             nid = nid + 1
             id(nid) = i
          end if
       end if
    end do

    allocate(refra(nid),refdec(nid),refph(nid,1),drefph(nid,1))
    refra = ra(1:nid)
    refdec = dec(1:nid)
    refph(:,1) = ph(1:nid)
    drefph(:,1) = dph(1:nid)
    deallocate(ra,dec,ph,dph,id)

666 continue

    call ftclos(15,status)
    call ftrprt('STDERR',status)

    if( allocated(ra) ) deallocate(ra,dec,ph,dph,id)

  end subroutine readref

end program phcal
