#!/bin/bash

# This script creates light curves and other related
# products by using Munipack data for 0716
#
# See detailed description in lctut.html

set -x

# switch-off backups
# export VERSION_CONTROL=off

# data
wget ftp://munipack.physics.muni.cz/pub/munipack/munipack-data-0716.tar.gz
tar zxf munipack-data-0716.tar.gz
cd munipack-data-0716/
chmod u+w *.fits

# preprocessing
munipack dark -o d30.fits d30_*.fits
munipack dark -o d120.fits d120_*.fits
munipack flat -o f_V.fits -dark d30.fits f30_*V.fits
munipack flat -o f_R.fits -dark d30.fits f30_*R.fits
munipack phcorr -dark d120.fits -flat f_V.fits 0716_*V.fits
munipack phcorr -dark d120.fits -flat f_R.fits 0716_*R.fits

# grow, aperture photometry and star detection
munipack find -f 3 0716_*.fits
munipack aphot 0716_*.fits
#munipack gphot 0716_*.fits

# astrometry calibration
munipack cone -c UCAC5 -r 0.2 -- 110.5 71.3
munipack astrometry --maxmatch 20 -c cone.fits 0716_*.fits

# photometry calibration
wget http://munipack.physics.muni.cz/0716+71.lst
wget http://munipack.physics.muni.cz/ucac_jmuc.py
munipack fits --restore 0716+71.lst
munipack cone -c UCAC4 -o cone_ucac4.fits -r 0.2 -- 110.5 71.3
python ucac_jmuc.py cone_ucac4.fits cone_Johnson.fits

for FILTER in V R; do

  # manual
  for A in 0716_*${FILTER}.fits; do
    munipack phcal -C 1 --photsys-ref Johnson -f ${FILTER} \
        -O --mask '\!\1_mancal.\2' $A;
  done

  # mancal, aperture 7, extinction
  for A in 0716_*${FILTER}.fits; do
      munipack phcal \
	--saper 7 -C 1 \
        --photsys-ref Johnson -f ${FILTER}  \
        -O --mask '\!\1_mancal7.\2' \
        $A;
  done

  # mancal, aperture 4, extinction
  for A in 0716_*${FILTER}.fits; do
      munipack phcal \
	--saper 4 -C 1 \
        --photsys-ref Johnson -f ${FILTER}  \
        -O --mask '\!\1_mancal4.\2' \
        $A;
  done

  # mancal, aperture 10, extinction
  for A in 0716_*${FILTER}.fits; do
      munipack phcal \
	--saper 10 -C 1 \
        --photsys-ref Johnson -f ${FILTER}  \
        -O --mask '\!\1_mancal10.\2' \
        $A;
  done

  # catalogue
  for A in 0716_*${FILTER}.fits; do
      munipack phcal \
        --photsys-ref Johnson --area 0.3 \
        -f ${FILTER} --col-mag ${FILTER} --col-magerr ${FILTER}ERR  \
        -c 0716+71.fits \
        -O --mask '\!\1_catcal.\2' \
        $A;
  done

  # reference frame
  for A in 0716_*${FILTER}.fits; do
      munipack phcal \
        --photsys-ref Johnson --area 0.3 \
        -f ${FILTER}  \
        -r 0716_035${FILTER}_catcal.fits -O --mask '\!\1_refcal.\2' $A;
  done

  # ucac
  for A in 0716_*${FILTER}.fits; do
      munipack phcal \
        --photsys-ref Johnson --area 0.3 --col-ra RAJ2000 --col-dec DEJ2000 \
        -f ${FILTER} --col-mag ${FILTER}mag --col-magerr e_${FILTER}mag  \
        -c cone_Johnson.fits \
        -O --mask '\!\1_ucacal.\2' \
        $A;
  done

done

#munipack kombine --rcen 110.47 --dcen 71.34 --width 640 --height 480 -o V.fits 0716_*V.fits
#munipack kombine --rcen 110.47 --dcen 71.34 --width 640 --height 480 -o R.fits 0716_*R.fits


# light curves
for FILTER in V R; do
   for TYPE in man cat ref uca; do
      munipack timeseries 110.473,71.343 110.389,71.322 110.468,71.305 \
          -c MAG,MAGERR --stdout  0716_*${FILTER}_${TYPE}cal.fits | sort -n > ${TYPE}cal_${FILTER}
   done
done

# list calibration in an aperture
for FILTER in V R; do
    for D in 4 7 10; do
	munipack timeseries 110.473,71.343 110.389,71.322 110.468,71.305 \
	 -c MAG,MAGERR --stdout 0716_*${FILTER}_mancal${D}.fits | sort -n > mancal${D}_${FILTER}
    done
done

# list of radius of 90% of flux
for FILTER in V R; do
  ( for A in 0716_*${FILTER}_catcal.fits; do
      JD=$(munipack timeseries --stdout 110.389,71.322 $A | awk '{print $1;}')
      RF90=$(munipack fits --shell -K RADFLX90 ${A}\[PHOTOMETRY\] | awk -F '=' '{print 3600*$2;}')
      echo $JD $RF90
  done ) > rf90_${FILTER}
done

# airmass
for FILTER in V R; do
  ( for A in 0716_*${FILTER}_catcal.fits; do
      JD=$(munipack timeseries --stdout 110.389,71.322 $A | awk '{print $1;}')
      M=$(munipack fits --shell -K AIRMASS ${A}\[PHOTOMETRY\] | awk -F '=' '{print $2;}')
      C=$(munipack fits --shell -K CTPH ${A}\[PHOTOMETRY\] | awk -F '=' '{print $2;}')
      echo $JD $M $C
  done ) | sort -n > airmass_${FILTER}
done

# join extinction files
for FILTER in V R; do
    join airmass_${FILTER} mancal_${FILTER} > ext_${FILTER}
    join airmass_${FILTER} mancal7_${FILTER} > ext7_${FILTER}
done



# plots, V filter
gnuplot <<EOF
set key bottom left
set title "0716+71"
set ylabel "magnitude in V"
set xlabel "JD - 2453409"
set yrange[14.8:13.5]
set term svg dynamic
set output 'lc0716_V.svg'
plot 'mancal_V' u (\$1-2453409):(\$2-\$4+11.8) t "diff(0716+71 - A) + 11.8", 'catcal_V' u (\$1-2453409):(\$2-0.2) t "std.field - 0.2" ls 1 lc 2, 'refcal_V' u (\$1-2453409):2 t "frame" ls 1 lc 3, 'ucacal_V' u (\$1-2453409):(\$2+0.1) t "UCAC5 + 0.2" ls 1 lc 4
EOF


gnuplot <<EOF
set key top left
set title "Calibration star A"
set ylabel "magnitude in V"
set xlabel "JD - 2453409"
set yrange[11.9:11.2]
set term svg dynamic
set output 'comp0716_V.svg'
plot 'mancal_V' u (\$1-2453409):(\$4-\$6+12.65) t "diff(A-B)+12.65", 'catcal_V' u (\$1-2453409):(\$4-0.1) t "std.field - 0.1" ls 1 lc 2, 'refcal_V' u (\$1-2453409):4 t "frame" ls 1 lc 3, 'ucacal_V' u (\$1-2453409):(\$4+0.1) t "UCAC5 + 0.2" ls 1 lc 4
EOF

# plots, R filter
gnuplot <<EOF
set key bottom left
set title "0716+71"
set ylabel "magnitude in R"
set xlabel "JD - 2453409"
set yrange[14.5:13.0]
set term svg dynamic
set output 'lc0716_R.svg'
plot 'mancal_R' u (\$1-2453409):(\$2-\$4+11.7) t "diff(0716+71 - A) + 11.7", 'catcal_R' u (\$1-2453409):(\$2-0.2) t "std.field - 0.2" ls 1 lc 2, 'refcal_R' u (\$1-2453409):2 t "frame" ls 1 lc 3, 'ucacal_R' u (\$1-2453409):(\$2+0.1) t "UCAC5 + 0.2" ls 1 lc 4
EOF


gnuplot <<EOF
set key top left
set title "Calibration star A"
set ylabel "magnitude in R"
set xlabel "JD - 2453409"
set yrange[11.8:10.8]
set term svg dynamic
set output 'comp0716_R.svg'
plot 'mancal_R' u (\$1-2453409):(\$4-\$6+12.5) t "diff(A-B+12.5)", 'catcal_R' u (\$1-2453409):(\$4-0.1) t "std.field - 0.1" ls 1 lc 2, 'refcal_R' u (\$1-2453409):4 t "frame" ls 1 lc 3, 'ucacal_R' u (\$1-2453409):(\$4+0.1) t "UCAC5 + 0.2" ls 1 lc 4
EOF


# extinction (list instrumental magnitudes)
#gnuplot <<EOF
#set key top right
#set title "Extinction on 0716+71 field"
#set ylabel "Instrumental magnitude in V filter"
#set xlabel "airmass"
#set xrange[1:1.8]
#set yrange[16.6:16.2]
#set term svg dynamic
#set output 'ext_V.svg'
#plot 'ext_V' u 2:6 t "total flux", 'ext7_V' u 2:6 t "flux in 7-th aperture" ls 1 lc 2
#EOF
#
#gnuplot <<EOF
#set key top right
#set title "Extinction on 0716+71 field"
#set ylabel "Instrumental magnitude in R filter"
#set xlabel "airmass"
#set xrange[1:1.8]
#set yrange[17:16.5]
#set term svg dynamic
#set output 'ext_R.svg'
#plot 'ext_R' u 2:6 t "total flux", 'ext7_R' u 2:6 t "flux in 7-th aperture" ls 1 lc 2
#EOF

gnuplot <<EOF
set key top right
set title "Extinction on 0716+71 field"
set ylabel "Instrumental magnitude"
set xlabel "airmass"
set xrange[1:1.8]
set yrange[17:16.3]
set term svg dynamic
set output 'ext.svg'
plot 'ext_R' u 2:6 t "R total flux", 'ext7_R' u 2:6 t "R flux in 7-th aperture" ls 1 lc 2, 'ext_V' u 2:6 t "V total flux", 'ext7_V' u 2:6 t "V flux in 7-th aperture" ls 1 lc 2
EOF


# spread
gnuplot <<EOF
set key top left
set title "Radius of 90% of total flux on 0716+71 field"
set ylabel "radius in arcsec"
set xlabel "JD - 2453409"
set yrange[0:15]
set term svg dynamic
set output 'rf90.svg'
plot 'rf90_R' u (\$1-2453409):2 t "R filter", 'rf90_V' u (\$1-2453409):2 t "V filter" ls 1 lc 2
EOF

# aperture affair
gnuplot <<EOF
set key top left
set title "Differential photometry in different apertures"
set ylabel "Instrumental V magnitude difference"
set xlabel "JD - 2453409"
set yrange[-0.4:-1.4]
set term svg dynamic
set output 'apertures.svg'
plot 'mancal_V' u (\$1-2453409):(\$4-\$6) t "total flux", 'mancal7_V' u (\$1-2453409):(\$4-\$6 + 0.1) t "aperture 7-th" ls 1 lc 2, 'mancal4_V' u (\$1-2453409):(\$4-\$6 - 0.1) t "aperture 4-th" ls 1 lc 3, 'mancal10_V' u (\$1-2453409):(\$4-\$6 + 0.2) t "aperture 10-th" ls 1 lc 4
EOF
