!
!  fitsfind
!
!  Copyright © 2013, 2018-9 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.


module fitsfind

  use fitsio
  use iso_fortran_env

  implicit none

contains

  subroutine fits_find_read(filename,fkeys,data,readns,saturation,status)

    character(len=*), intent(in) :: filename
    character(len=*), dimension(:), intent(in) :: fkeys
    real, intent(out) :: readns,saturation
    real, dimension(:,:), allocatable, intent(out) :: data
    integer, intent(in out) :: status

    integer, parameter :: DIM = 2
    integer :: naxis, bitpix
    integer, dimension(DIM) :: naxes
    character(len=FLEN_CARD) :: buf
    logical :: satkey

    if( status /= 0 ) return

    call fits_open_image(15,filename,READONLY,status)
    if( status /= 0 ) goto 666

    call fits_get_img_dim(15,naxis,status)
    if( naxis /= DIM ) then
       write(error_unit,*) 'Error in FIND: Only 2D frames are supported.'
       goto 666
    end if
    call fits_get_img_size(15,naxis,naxes,status)

    call fits_read_key(15,fkeys(1),saturation,buf,status)
    satkey = status == 0
    if( status == KEYWORD_NOT_FOUND ) then
       status = 0
       call fits_get_img_type(15,bitpix,status)
       if( status /= 0 ) goto 666
    end if

    call fits_read_key(15,fkeys(2),readns,buf,status)
    if( status == KEYWORD_NOT_FOUND ) then
       readns = 0
       status = 0
    end if

    if( status /= 0 ) goto 666

    allocate(data(naxes(1),naxes(2)))
    call fits_read_2d(15,data,status)
    if( status /= 0 ) goto 666

    if( .not. satkey ) then
       if( bitpix > 0 ) then
          saturation = 2.0**bitpix - 1
       else
          saturation = 0.99*maxval(data)
       end if
    end if

666 continue

    call fits_close_file(15,status)

    if( status /= 0 ) then
       call fits_report_error('STDERR',status)
       if( allocated(data) ) deallocate(data)
    end if

  end subroutine fits_find_read



  subroutine fits_find_save(filename,output,fkeys,nstar, &
       fwhm,threshold,shrplo,shrphi,rndlo,rndhi, readns, &
       lothresh,  lobad, hibad, hmin, skymod, skyerr, skysig, maxsky, status)

    use oakleaf

    ! results fills new FITS extension

    character(len=*), intent(in) :: filename, output
    character(len=*), dimension(:), intent(in) :: fkeys

    real, intent(in) :: fwhm, threshold, &
         shrplo,shrphi,rndlo,rndhi, lothresh, readns, lobad, hibad, hmin, &
         skymod, skyerr, skysig
    integer, intent(in) :: maxsky, nstar
    integer, intent(in out) :: status

    integer, parameter :: extver = 0
    character(len=FLEN_CARD) :: buf
    character(len=FLEN_VALUE), dimension(5) :: ttype, tform, tunit
    real, dimension(:), allocatable :: xcen,ycen,hstar,round,sharp,ecc,incl
    real :: ecc_mean, incl_mean
    integer :: n, hdutype

    if( status /= 0 ) return

    call fits_open_file(15,filename,output,status)
    if( status /= 0 ) goto 666

    ! look for an older extension
    call fits_movnam_hdu(15,BINARY_TBL,FINDEXTNAME,extver,status)
    if( status == BAD_HDU_NUM ) then
       status = 0
    else if( status == 0 ) then
       ! already presented ? remove it !
       call fits_delete_hdu(15,hdutype,status)
    end if
    if( status /= 0 ) goto 666

    ttype(1) = FITS_COL_X
    ttype(2) = FITS_COL_Y
    ttype(3) = FITS_COL_PEAKRATIO
    ttype(4) = FITS_COL_SHARP
    ttype(5) = FITS_COL_ROUND
    tform = '1E'
    tunit = ''

    call fits_insert_btbl(15,nstar,ttype,tform,tunit,FINDEXTNAME,status)
    call fits_update_key(15,fkeys(1),hibad,5,'[cts] saturation',status)
    call fits_update_key(15,fkeys(2),readns,-7,'[ADU] read noise',status)
    call fits_update_key(15,FITS_KEY_FWHM,fwhm,-2, &
         '[pix] standard FWHM of objects',status)
    call fits_update_key(15,FITS_KEY_THRESHOLD,threshold,-2, &
         'threshold in sigmas above background',status)
    call fits_update_key(15,FITS_KEY_LOWBAD,lobad,-3, &
         '[cts] low good datum',status)
    call fits_update_key(15,FITS_KEY_HIGHBAD,hibad,-3, &
         '[cts] high good datum',status)
    call fits_update_key(15,FITS_KEY_RNDLO,rndlo,-3,'low round',status)
    call fits_update_key(15,FITS_KEY_RNDHI,rndhi,-3,'high round',status)
    call fits_update_key(15,FITS_KEY_SHRPLO,shrplo,-3,'low sharp',status)
    call fits_update_key(15,FITS_KEY_SHRPHI,shrphi,-3,'high sharp',status)

    call fits_write_comment(15,'Star detection parameters:',status)

    write(buf,*) 'Saturation (counts)=',hibad
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Read noise (ADU)=',readns
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Lower threshold (sigma)=',lothresh
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Levels range (counts) =',lobad, '..',hibad
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Hmin (counts) =',hmin
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Round range =',rndlo, '..',rndhi
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Sharp range =',shrplo, '..',shrphi
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Approximate sky value =',skymod,'+-',skyerr
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Estimated sky sigma =',skysig
    call fits_write_comment(15,buf,status)

    write(buf,*) 'Pixels used for sky determination =',maxsky
    call fits_write_comment(15,buf,status)

    allocate(xcen(nstar),ycen(nstar),sharp(nstar),round(nstar),hstar(nstar),&
         ecc(nstar),incl(nstar))
    do n = 1, nstar
       read(3) xcen(n),ycen(n),hstar(n),sharp(n),round(n),ecc(n),incl(n)
    end do

    ! sort arrays by height above lower threshold
    call sorter(xcen,ycen,hstar,sharp,round)

    ! mean eccentricity and inclination
    call rmean(ecc,ecc_mean)
    call rmean(incl,incl_mean)
    call fits_update_key(15,FITS_KEY_ECCENTRICITY,ecc_mean,-2, &
         ' mean eccentricity',status)
    call fits_update_key(15,FITS_KEY_INCLINATION,nint(incl_mean), &
         ' mean inclination',status)

    call fits_write_col(15,1,xcen,status)
    call fits_write_col(15,2,ycen,status)
    call fits_write_col(15,3,hstar,status)
    call fits_write_col(15,4,sharp,status)
    call fits_write_col(15,5,round,status)
!    call fits_write_col(15,6,shape,status)

    deallocate(xcen,ycen,hstar,round,sharp,ecc,incl)

666 continue

    call fits_close_file(15,status)
    call fits_report_error('STDERR',status)

  end subroutine fits_find_save


  subroutine sorter(xcen,ycen,hstar,sharp,round)

    use quicksort

    real, dimension(:),intent(in out) :: xcen,ycen,hstar,sharp,round

    integer, parameter :: rp = selected_real_kind(15)
    real(rp), dimension(:), allocatable :: htmp
    real, dimension(:), allocatable :: tmp
    integer, dimension(:), allocatable :: id, idx
    integer :: i,n,m

    n = size(xcen)
    allocate(tmp(n),htmp(n),id(n),idx(n))
    id = [ (i, i = 1,n) ]
    htmp = hstar

    call qsort(htmp,id)
    ! hstar sorted into low to high order

    ! reverse sort
    m = n + 1
    forall( i = 1:n ) hstar(i) = real(htmp(m - i))
    forall( i = 1:n ) idx(i) = id(m - i)

    tmp = xcen
    xcen = tmp(idx)

    tmp = ycen
    ycen = tmp(idx)

    tmp = sharp
    sharp = tmp(idx)

    tmp = round
    round = tmp(idx)

    deallocate(tmp,htmp,id,idx)

  end subroutine sorter


end module fitsfind
