!
!  fitsaphot
!
!  Copyright © 2013-6, 2018-9 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.

! Parameters by:
! http://stsdas.stsci.edu/cgi-bin/gethelp.cgi?psfmeasure.hlp


module fitsaphot

  use fitsio
  use iso_fortran_env

  implicit none

contains

  subroutine fits_aphot_image(filename,fkeys,data,stderr,saturate,status)

    integer, parameter :: naxis = 2

    character(len=*),intent(in) :: filename
    character(len=*), dimension(:), intent(in) :: fkeys
    real, dimension(:,:), allocatable, intent(out) :: data,stderr
    real, intent(out) :: saturate
    integer, intent(in out) :: status

    integer, parameter :: extver = 0
    character(len=FLEN_COMMENT) :: comment
    integer, dimension(naxis) :: naxes
    integer :: bitpix

    if( status /= 0 ) return

    call fits_open_image(15,filename,READONLY,status)
    call fits_get_img_size(15,naxis,naxes,status)
    if( status /= 0 ) goto 666

    call fits_read_key(15,fkeys(1),saturate,comment,status)
    if( status == KEYWORD_NOT_FOUND ) then
       status = 0
       call fits_get_img_type(15,bitpix,status)
       if( status /= 0 ) goto  666
       if( bitpix > 0 ) then
          saturate = 2.0**bitpix - 1
       else
          saturate = huge(saturate)
       end if
    end if

    allocate(data(naxes(1),naxes(2)),stderr(naxes(1),naxes(2)))
    call fits_read_2d(15,data,status)
    call fits_movnam_hdu(15,IMAGE_HDU,EXT_STDERR,extver,status)
    if( status == 0 ) then
       call fits_read_2d(15,stderr,status)
    else if ( status == BAD_HDU_NUM ) then
       ! if the information about standard errors is not available,
       ! we are continuing with the Poisson component only
       where( data > 0 )
          stderr = sqrt(data)
       elsewhere
          stderr = -1
       end where
       status = 0
    end if

666 continue

    call fits_close_file(15,status)

    if( status /= 0 ) then
       call fits_report_error('STDERR',status)
       if( allocated(data) ) deallocate(data,stderr)
    end if

  end subroutine fits_aphot_image


  subroutine fits_aphot_read(filename,data,stderr,xcens,ycens, &
       lobad,hibad,fwhm,ecc,incl,status)

    integer, parameter :: DIM = 2

    character(len=*),intent(in) :: filename
    real, dimension(:,:), allocatable, intent(out) :: data,stderr
    real, dimension(:), allocatable, intent(out) :: xcens,ycens
    real, intent(out) :: lobad,hibad,fwhm,ecc,incl
    integer, intent(in out) :: status

    integer, parameter :: extver = 0
    integer, dimension(DIM) :: naxes
    character(len=FLEN_COMMENT) :: com
    integer :: naxis,nrows,xcol,ycol,hcol

    if( status /= 0 ) return

    call fits_open_image(15,filename,READONLY,status)
    if( status /= 0 ) goto 666

    call fits_get_img_dim(15,naxis,status)
    if( naxis /= 2 ) then
       write(error_unit,*) 'Error in aphot: Only 2D frames are supported.'
       goto 666
    end if
    call fits_get_img_size(15,naxis,naxes,status)
    if( status /= 0 ) goto 666

    allocate(data(naxes(1),naxes(2)),stderr(naxes(1),naxes(2)))
    call fits_read_2d(15,data,status)

    call fits_movnam_hdu(15,IMAGE_HDU,EXT_STDERR,extver,status)
    if( status == 0 ) then
       call fits_read_2d(15,stderr,status)
    else if ( status == BAD_HDU_NUM ) then
       ! if the information about standard errors is not available,
       ! we continues with Poisson component only
       where( data > 0 )
          stderr = sqrt(data)
       elsewhere
          stderr = -1
       end where
       status = 0
    end if

    if( status /= 0 ) then
       write(error_unit,*) trim(filename),": Failed to read data."
       goto 666
    end if

    call fits_movnam_hdu(15,BINARY_TBL,FINDEXTNAME,extver,status)
    if( status == BAD_HDU_NUM ) then
       write(error_unit,*) "Error: ",trim(FINDEXTNAME)//" extension not found."
       write(error_unit,*) "       Has been stars detected by `munipack find " &
            //trim(filename)//"' ?"
       goto 666
    end if

    call fits_get_num_rows(15,nrows,status)

    call fits_read_key(15,FITS_KEY_LOWBAD,lobad,com,status)
    call fits_read_key(15,FITS_KEY_HIGHBAD,hibad,com,status)
    call fits_read_key(15,FITS_KEY_FWHM,fwhm,com,status)
    call fits_read_key(15,FITS_KEY_ECCENTRICITY,ecc,com,status)
    call fits_read_key(15,FITS_KEY_INCLINATION,incl,com,status)
    if( status /= 0 ) then
       write(error_unit,*) trim(filename),": Required keywords ", &
            trim(FITS_KEY_LOWBAD),", ",trim(FITS_KEY_HIGHBAD),", ", &
            trim(FITS_KEY_FWHM),", ",trim(FITS_KEY_ECCENTRICITY),", or ",&
            trim(FITS_KEY_INCLINATION)," not found."
       goto 666
    end if

    allocate(xcens(nrows),ycens(nrows))

    call fits_get_colnum(15,.true.,FITS_COL_X,xcol,status)
    call fits_get_colnum(15,.true.,FITS_COL_Y,ycol,status)
    call fits_read_col(15,xcol,xcens,status)
    call fits_read_col(15,ycol,ycens,status)

666 continue

    call fits_close_file(15,status)

    if( status /= 0 ) then
       call fits_report_error('STDERR',status)
       if( allocated(data) ) deallocate(data,stderr)
       if( allocated(xcens) ) deallocate(xcens,ycens)
    end if

  end subroutine fits_aphot_read

  subroutine fits_aphot_save(filename, output, hwhm, ecc, incl, raper, ring,  &
       xcens, ycens, apcts,apcts_err,sky,sky_err, status)

    character(len=*), intent(in) :: filename, output
    real, intent(in) :: hwhm, ecc, incl
    real, dimension(:), intent(in) :: raper, ring, xcens, ycens, sky,sky_err
    real, dimension(:,:), intent(in) :: apcts,apcts_err
    integer, intent(in out) :: status

    integer, parameter :: extver = 0, frow = 1, felem = 1, nbegin = 4
    character(len=FLEN_VALUE), dimension(:), allocatable :: ttype, tform, tunit
    character(len=FLEN_VALUE) :: key
    integer :: hdutype,i,j,n


    if( status /= 0 ) return

    call fits_open_file(15,filename,output,status)
    if( status /= 0 ) goto 666

    ! store results to the aperture photometry extension
    call fits_movnam_hdu(15,BINARY_TBL,APEREXTNAME,extver,status)
    if( status == BAD_HDU_NUM ) then
       status = 0
    else
       ! already presented ? remove it !
       call fits_delete_hdu(15,hdutype,status)
    end if
    if( status /= 0 ) goto 666

    n = nbegin + 2*size(raper)
    allocate(ttype(n), tform(n), tunit(n))

    tform = '1D'
    tunit = ''
    ttype(1) = FITS_COL_X
    ttype(2) = FITS_COL_Y
    ttype(3) = FITS_COL_SKY
    ttype(4) = FITS_COL_SKYERR

    do i = 1, size(raper)
       j = nbegin - 1 + 2*i
       write(ttype(j),'(a,i0)') trim(FITS_COL_APCOUNT),i
       write(ttype(j+1),'(a,i0)') trim(FITS_COL_APCOUNTERR),i
    end do

    ! aperture photometry table
    call fits_insert_btbl(15,size(xcens),ttype,tform,tunit,APEREXTNAME,status)

    call fits_write_key(15,FITS_KEY_HWHM,hwhm,-4, &
         '[pix] half width at half of maximum',status)
    call fits_update_key(15,FITS_KEY_ECCENTRICITY,ecc,-2, &
         ' eccentricity',status)
    call fits_update_key(15,FITS_KEY_INCLINATION,nint(incl), &
         ' inclination',status)
    call fits_write_key(15,FITS_KEY_NAPER,size(raper), &
         'Count of apertures',status)
    do i = 1, size(raper)
       call fits_make_keyn(FITS_KEY_APER,i,key,status)
       call fits_write_key(15,key,raper(i),-5,'[pix] aperture radius',status)
    end do

    call fits_write_key(15,trim(FITS_KEY_ANNULUS)//'1',ring(1),-5, &
         '[pix] inner sky annulus radius',status)
    call fits_write_key(15,trim(FITS_KEY_ANNULUS)//'2',ring(2),-5, &
         '[pix] outer sky annulus radius',status)

    call fits_write_col(15,1,xcens,status)
    call fits_write_col(15,2,ycens,status)
    call fits_write_col(15,3,sky,status)
    call fits_write_col(15,4,sky_err,status)

    do i = 1,size(apcts,2)
       j = nbegin-1+2*i
       call fits_write_col(15,j,apcts(:,i),status)
       call fits_write_col(15,j+1,apcts_err(:,i),status)
    end do
    deallocate(ttype,tform,tunit)

666 continue

    call fits_close_file(15,status)
    call fits_report_error('STDERR',status)

  end subroutine fits_aphot_save

  subroutine estim_hwhm(data,xcens,ycens,sky,fwhm,lobad,hibad,hwhm)

    use oakleaf

    real, dimension(:,:), intent(in) :: data
    real, dimension(:), intent(in) :: xcens,ycens,sky
    real, intent(in) :: fwhm,lobad,hibad
    real, intent(out) :: hwhm

    real, dimension(:), allocatable :: xhwhm
    real :: sx,sy,w,sw,w0
    integer :: nx,ny,i,j,l,m,n,i0,j0

    allocate(xhwhm(size(xcens)))

    nx = size(data,1)
    ny = size(data,2)

    n = 0
    m = nint(1.5*fwhm)
    m = nint(fwhm / 2) * 3
    ! neighborhood is 3*hwhm of expected which prefers important
    ! parts of profile

    do l = 1, size(xcens)

       sx = 0
       sy = 0
       sw = 0
       i0 = nint(xcens(l))
       j0 = nint(ycens(l))
       w0 = data(i0,j0) - sky(l)
       if( w0 > 0 .and. sky(l) > 0 ) then
          do i = i0-m,i0+m
             do j = j0-m,j0+m
                if( 0 < i .and. i <= nx .and. 0 < j .and. j <= ny ) then
                   w = data(i,j) - sky(l)
                   if( lobad < data(i,j).and.data(i,j) < hibad .and. w > 0) then
                      sx = sx + w*(i - xcens(l))**2
                      sy = sy + w*(j - ycens(l))**2
                      sw = sw + w
                   end if
                end if
             end do
          end do
          if( sw > 0 .and. sqrt(w0) / w0 < 0.01) then
             ! estimation of hwhm is sensitive on noise in data (w),
             ! we're selecting only bright stars
             n = n + 1
             xhwhm(n) = (sqrt(sx/sw) + sqrt(sy/sw)) / 2
          end if
       end if
    end do

    if( n > 1 ) then
       call rmean(xhwhm(1:n),hwhm,w)
    else
       hwhm = -1
    end if

    deallocate(xhwhm)

  end subroutine estim_hwhm

end module fitsaphot
