!
!  FITSIO definitions - derived by f77.inc and fitsio.h
!
!  Copyright © 2010-20 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.



module fitsio

  use iso_fortran_env

  implicit none

  ! version identification
  character(len=*), parameter, private :: VERSION = "0.5.12"

  character(len=*), parameter :: MUNIPACK_VERSION = &
       "Munipack "//VERSION//", (C)1997-2020 F. Hroch <hroch@physics.muni.cz>"

  character(len=*), parameter :: FITS_VALUE_CREATOR = &
       "Munipack "//VERSION
  character(len=*), parameter :: FITS_COM_CREATOR = &
       "http://munipack.physics.muni.cz"

  ! basic constants (fitsio.h)
  integer, parameter :: FLEN_FILENAME = 1025 ! max length of a filename
  integer, parameter :: FLEN_KEYWORD = 75    ! max length of a keyword (include HIERARCH)
  integer, parameter :: FLEN_CARD = 81       ! length of a FITS header card
  integer, parameter :: FLEN_VALUE = 71      ! max length of a keyword value string
  integer, parameter :: FLEN_COMMENT = 73    ! max length of a keyword comment string
  integer, parameter :: FLEN_ERRMSG = 81     ! max length of a FITSIO error message
  integer, parameter :: FLEN_STATUS = 31     ! max length of a FITSIO status text string


  ! Codes for FITS extension types
  integer, parameter :: &
       IMAGE_HDU = 0, &
       ASCII_TBL = 1, &
       BINARY_TBL = 2, &
       ANY_HDU = 2

  integer, parameter :: &
       READONLY = 0, &
       READWRITE = 1

  ! Codes for FITS table data types
  integer, parameter :: &
       TBIT        =   1, &
       TBYTE       =  11, &
       TLOGICAL    =  14, &
       TSTRING     =  16, &
       TSHORT      =  21, &
       TINT        =  31, &
       TFLOAT      =  42, &
       TDOUBLE     =  82, &
       TCOMPLEX    =  83, &
       TDBLCOMPLEX = 163

  ! Codes for iterator column types
  integer, parameter :: &
       InputCol       = 0, &
       InputOutputCol = 1, &
       OutputCol      = 2

  ! codes for header record class
  integer, parameter :: &
       TYP_STRUC_KEY = 10, &
       TYP_CMPRS_KEY = 20, &
       TYP_SCAL_KEY  = 30, &
       TYP_NULL_KEY  = 40, &
       TYP_DIM_KEY   = 50, &
       TYP_RANG_KEY  = 60, &
       TYP_UNIT_KEY  = 70, &
       TYP_DISP_KEY  = 80, &
       TYP_HDUID_KEY = 90, &
       TYP_CKSUM_KEY = 100,&
       TYP_WCS_KEY   = 110,&
       TYP_REFSYS_KEY= 120,&
       TYP_COMM_KEY  = 130,&
       TYP_CONT_KEY  = 140,&
       TYP_USER_KEY  = 150


  integer, parameter :: FILE_NOT_OPENED = 104
  integer, parameter :: FILE_NOT_CREATED = 105
  integer, parameter :: READ_ERROR = 108
  integer, parameter :: BAD_HDU_NUM = 301 ! HDU number < 1 or > MAXHDU
  integer, parameter :: KEYWORD_NOT_FOUND = 202
  integer, parameter :: COLUMN_NOT_FOUND = 219
  integer, parameter :: NUMERICAL_OVERFLOW = 412
  integer, parameter :: MULTIPLE_MATCH = 237

  character(len=*), parameter :: FINDEXTNAME = 'FIND'
  character(len=*), parameter :: APEREXTNAME = 'APERPHOT'
  character(len=*), parameter :: GROWEXTNAME = 'GROWPHOT'
  character(len=*), parameter :: GROWCURVEXTNAME = 'GROWCURVE'
  character(len=*), parameter :: GROWFUNCEXTNAME = 'GROWFUNC'
  character(len=*), parameter :: GROWDATEXTNAME = 'GROWDATA'
  character(len=*), parameter :: PSFEXTNAME  = 'PSFPHOT'
  character(len=*), parameter :: PHOTOEXTNAME= 'PHOTOMETRY'
  character(len=*), parameter :: MEXTNAMETS =  'TIMESERIES'
  character(len=*), parameter :: MEXTNAMETSC = 'CATALOGUE'
  character(len=*), parameter :: FHDUNAME =    'PHOTOSYS'
  character(len=*), parameter :: FTHDUNAME =   'FOTRAN'
  character(len=*), parameter :: EXT_STDERR =  'STDERR'
  character(len=*), parameter :: EXT_PHRES =   'PHRES'

  character(len=*), parameter :: BEGIN_ASTROMETRY = &
       '=== Astrometric Solution by Munipack ==='
  character(len=*), parameter :: END_ASTROMETRY = &
       '=== End of Astrometric Solution by Munipack ==='
  character(len=*), parameter :: BEGIN_PHOTOCAL = &
       '=== Photometric Calibration by Munipack ==='
  character(len=*), parameter :: END_PHOTOCAL = &
       '=== End of Photometric Calibration by Munipack ==='

  character(len=*), parameter :: FITS_KEY_NAPER = 'NAPER'
  character(len=*), parameter :: FITS_KEY_APER = 'APER'
  character(len=*), parameter :: FITS_KEY_SAPER = 'SAPER'
  character(len=*), parameter :: FITS_KEY_ANNULUS = 'ANNULUS'
  character(len=*), parameter :: FITS_KEY_FWHM = 'FWHM'
  character(len=*), parameter :: FITS_KEY_HWHM = 'HWHM'
  character(len=*), parameter :: FITS_KEY_ECCENTRICITY = 'ECCENTR'
  character(len=*), parameter :: FITS_KEY_INCLINATION = 'INCL'
  character(len=*), parameter :: FITS_KEY_RF90 = 'RADFLX90'
  character(len=*), parameter :: FITS_KEY_EXTINK0 = 'EXTIN_K0'
  character(len=*), parameter :: FITS_KEY_EXTINR = 'EXTIN_R'
  character(len=*), parameter :: FITS_KEY_EXTINREF = 'EXTINREF'
  character(len=*), parameter :: FITS_KEY_THRESHOLD = 'THRESH'
  character(len=*), parameter :: FITS_KEY_LOWBAD = 'LOWBAD'
  character(len=*), parameter :: FITS_KEY_HIGHBAD = 'HIGHBAD'
  character(len=*), parameter :: FITS_KEY_RNDLO = 'RNDLO'
  character(len=*), parameter :: FITS_KEY_RNDHI = 'RNDHI'
  character(len=*), parameter :: FITS_KEY_SHRPLO = 'SHRPLO'
  character(len=*), parameter :: FITS_KEY_SHRPHI = 'SHRPHI'
  character(len=*), parameter :: FITS_KEY_PHOTPLAM = 'PHOTPLAM'
  character(len=*), parameter :: FITS_KEY_PHOTZPT = 'PHOTZPT'
  character(len=*), parameter :: FITS_KEY_PHOTFLAM = 'PHOTFLAM'
  character(len=*), parameter :: FITS_KEY_PHOTBW = 'PHOTBW'
  character(len=*), parameter :: FITS_KEY_CTPH = 'CTPH'
  character(len=*), parameter :: FITS_KEY_CTPHERR = 'CTPHERR'
  character(len=*), parameter :: FITS_KEY_CSPACE = 'CSPACE'
  character(len=*), parameter :: FITS_KEY_REFRAME = 'REFRAME'
  character(len=*), parameter :: FITS_KEY_SATURATE = 'SATURATE'
  character(len=*), parameter :: FITS_KEY_GAIN = 'GAIN'
  character(len=*), parameter :: FITS_KEY_READNS = 'READNS'
  character(len=*), parameter :: FITS_KEY_AREA = 'AREA'
  character(len=*), parameter :: FITS_KEY_EXPTIME = 'EXPTIME'
  character(len=*), parameter :: FITS_KEY_PHOTSYS = 'PHOTSYS'
  character(len=*), parameter :: FITS_KEY_FILTER = 'FILTER'
  character(len=*), parameter :: FITS_KEY_FILTREF = 'FILTREF'
  character(len=*), parameter :: FITS_KEY_OBJECT = 'OBJECT'
  character(len=*), parameter :: FITS_KEY_OBSERVER = 'OBSERVER'
  character(len=*), parameter :: FITS_KEY_ORIGIN = 'ORIGIN'
  character(len=*), parameter :: FITS_KEY_AUTHOR = 'AUTHOR'
  character(len=*), parameter :: FITS_KEY_INSTRUME = 'INSTRUME'
  character(len=*), parameter :: FITS_KEY_TELESCOP = 'TELESCOP'
  character(len=*), parameter :: FITS_KEY_BIBREF = 'BIBREF'
  character(len=*), parameter :: FITS_KEY_LONGITUDE = 'LONGITUD'
  character(len=*), parameter :: FITS_KEY_LATITUDE =  'LATITUDE'
  character(len=*), parameter :: FITS_KEY_EPOCH = 'EPOCH'
  character(len=*), parameter :: FITS_KEY_CREATOR = 'CREATOR'
  character(len=*), parameter :: FITS_KEY_ORIGHDU = 'ORIGHDU'
  character(len=*), parameter :: FITS_KEY_BUNIT = 'BUNIT'
  character(len=*), parameter :: FITS_KEY_SKYMAG = 'SKYMAG'
  character(len=*), parameter :: FITS_KEY_SKYMEAN = 'SKYMEAN'
  character(len=*), parameter :: FITS_KEY_SKYSIG = 'SKYSIG'
  character(len=*), parameter :: FITS_KEY_SKYSTD = 'SKYSTD'
  character(len=*), parameter :: FITS_KEY_IMAGETYP = 'IMAGETYP'
  character(len=*), parameter :: FITS_KEY_DATEOBS = 'DATE-OBS'
  character(len=*), parameter :: FITS_KEY_TIMEOBS = 'TIME-OBS'
  character(len=*), parameter :: FITS_KEY_TEMPERATURE = 'TEMPERAT'
  character(len=*), parameter :: FITS_KEY_AIRMASS = 'AIRMASS'
  character(len=*), parameter :: FITS_KEY_JD = 'JD'
  character(len=*), parameter :: FITS_KEY_HJD = 'HJD'
  character(len=*), parameter :: FITS_KEY_FILENAME = 'FILENAME'
  character(len=*), parameter :: FITS_KEY_PHOTOTYP = 'PHOTOTYP'

  ! definitions of column labels, common
  character(len=*), parameter :: FITS_COL_FILENAME = 'FILENAME'
  character(len=*), parameter :: FITS_COL_TIME = 'TIME'
  character(len=*), parameter :: FITS_COL_X = 'X'
  character(len=*), parameter :: FITS_COL_Y = 'Y'
  character(len=*), parameter :: FITS_COL_RA = 'RAJ2000'
  character(len=*), parameter :: FITS_COL_DEC = 'DEJ2000'
  character(len=*), parameter :: FITS_COL_PMRA = 'pmRA'
  character(len=*), parameter :: FITS_COL_PMDEC = 'pmDE'
  character(len=*), parameter :: FITS_COL_SKY = 'SKY'
  character(len=*), parameter :: FITS_COL_SKYERR = 'SKYERR'
  character(len=*), parameter :: FITS_COL_AZIMUTH = 'AZIMUTH'
  character(len=*), parameter :: FITS_COL_ZENITD = 'ZENITD'
  character(len=*), parameter :: FITS_COL_AIRMASS = 'AIRMASS'
  character(len=*), parameter :: FITS_COL_R = 'R'
  character(len=*), parameter :: FITS_COL_GROW = 'GROWCURVE'
  character(len=*), parameter :: FITS_COL_GROWERR = 'GROWCURVEERR'
  character(len=*), parameter :: FITS_COL_RPROF = 'RADIALPROFILE'
  character(len=*), parameter :: FITS_COL_RESGROW = 'RESGROW'
  character(len=*), parameter :: FITS_COL_GROWFLAG = 'GROWFLAG'

  ! definitions of column labels, find
  character(len=*), parameter :: FITS_COL_PEAKRATIO = 'PEAKRATIO'
  character(len=*), parameter :: FITS_COL_SHARP = 'SHARP'
  character(len=*), parameter :: FITS_COL_ROUND = 'ROUND'

  ! definitions of column labels, general photometry
  character(len=*), parameter :: FITS_COL_COUNT =     'COUNT'
  character(len=*), parameter :: FITS_COL_COUNTERR =  'COUNTERR'
  character(len=*), parameter :: FITS_COL_PHOTON =    'PHOTON'
  character(len=*), parameter :: FITS_COL_PHOTONERR = 'PHOTONERR'
  character(len=*), parameter :: FITS_COL_PHOTRATE =    'PHOTRATE'
  character(len=*), parameter :: FITS_COL_PHOTRATEERR = 'PHOTRATEERR'

  ! definitions of column labels, aperture photometry
  character(len=*), parameter :: FITS_COL_APCOUNT =    'APCOUNT'
  character(len=*), parameter :: FITS_COL_APCOUNTERR = 'APCOUNTERR'

  ! definitions of column labels, growth-curve photometry
  character(len=*), parameter :: FITS_COL_GCOUNT =    'GCOUNT'
  character(len=*), parameter :: FITS_COL_GCOUNTERR = 'GCOUNTERR'

  ! definitions of column labels, L-photometry
  character(len=*), parameter :: FITS_COL_LBCOUNT =    'LBCOUNT'
  character(len=*), parameter :: FITS_COL_LBCOUNTERR = 'LBCOUNTERR'

  ! definitions of column labels, PSF photometry
  character(len=*), parameter :: FITS_COL_PSFCOUNT =   'PSFCOUNT'
  character(len=*), parameter :: FITS_COL_PSFCOUNTERR ='PSFCOUNTERR'
  character(len=*), parameter :: FITS_COL_PSFPEAK =    'PSFPEAK'
  character(len=*), parameter :: FITS_COL_PSFPEAKERR = 'PSFPEAKERR'

  ! definitions of column labels, output photometry quantities
  character(len=*), parameter :: FITS_COL_PHOTNU =     'PHOTNU'
  character(len=*), parameter :: FITS_COL_PHOTNUERR =  'PHOTNUERR'
  character(len=*), parameter :: FITS_COL_PHOTLAM =    'PHOTLAM'
  character(len=*), parameter :: FITS_COL_PHOTLAMERR = 'PHOTLAMERR'
  character(len=*), parameter :: FITS_COL_FLUX =       'FLUX'
  character(len=*), parameter :: FITS_COL_FLUXERR =    'FLUXERR'
  character(len=*), parameter :: FITS_COL_FNU =        'FNU'
  character(len=*), parameter :: FITS_COL_FNUERR =     'FNUERR'
  character(len=*), parameter :: FITS_COL_FLAM =       'FLAM'
  character(len=*), parameter :: FITS_COL_FLAMERR=     'FLAMERR'
  character(len=*), parameter :: FITS_COL_MAG =        'MAG'
  character(len=*), parameter :: FITS_COL_MAGERR =     'MAGERR'
  character(len=*), parameter :: FITS_COL_ABMAG =      'ABMAG'
  character(len=*), parameter :: FITS_COL_ABMAGERR =   'ABMAGERR'
  character(len=*), parameter :: FITS_COL_STMAG =      'STMAG'
  character(len=*), parameter :: FITS_COL_STMAGERR =   'STMAGERR'
  character(len=*), parameter :: FITS_COL_RATE =       'RATE'
  character(len=*), parameter :: FITS_COL_RATEERR =    'RATEERR'

  character(len=*), parameter :: FITS_COL_FILTER =     'FILTER'

  character(len=*), parameter :: FITS_COL_LAMEFF =     'LAM_EFF'
  character(len=*), parameter :: FITS_COL_LAMFWHM =    'LAM_FWHM'
  character(len=*), parameter :: FITS_COL_NUEFF =      'NU_EFF'
  character(len=*), parameter :: FITS_COL_NUFWHM =     'NU_FWHM'
  character(len=*), parameter :: FITS_COL_FNUREF =     'FNU_REF'
  character(len=*), parameter :: FITS_COL_FLAMREF =    'FLAM_REF'

!  character(len=*), parameter :: FITS_COL__ERR = '_ERR'


!!$  interface
!!$
!!$     subroutine ftpcle(unit,colnum,frow,felem,nelements,values,status)
!!$       integer, intent(in) :: unit, colnum, frow,felem,nelements
!!$       real, dimension(:), intent(in) :: values
!!$       integer, intent(out) :: status
!!$     end subroutine ftpcle
!!$
!!$  end interface

  interface
     function ftgkcl(buf)
       integer :: ftgkcl
       character(len=*) :: buf
     end function ftgkcl
  end interface

  interface fits_open_file
     module procedure fits_open_file_canonical, fits_open_file_output
  end interface fits_open_file

  interface fits_read_key
     module procedure fits_read_key_str, fits_read_key_int, &
          fits_read_key_real, fits_read_key_double
  end interface fits_read_key

  interface fits_write_key
     module procedure fits_write_key_str, fits_write_key_int, &
          fits_write_key_real, fits_write_key_double
  end interface fits_write_key

  interface fits_update_key
     module procedure fits_update_key_str, fits_update_key_int, &
          fits_update_key_real, fits_update_key_double
  end interface fits_update_key

  interface fits_read_col
     module procedure fits_read_col_str, fits_read_col_int, &
          fits_read_col_logical, fits_read_col_real, fits_read_col_double
  end interface fits_read_col

  interface fits_write_col
     module procedure fits_write_col_str, fits_write_col_int, &
          fits_write_col_logical, fits_write_col_real, fits_write_col_double
  end interface fits_write_col

  type FITSFILE

     character(len=FLEN_FILENAME) :: filename, scratch, output

  end type FITSFILE

  type(FITSFILE), dimension(49), private :: fitsfiles

  private :: fits_scratch_filename, fits_init_scratch, fits_open_scratch, &
       fits_keep_scratch, init_seed

contains

  ! Open/close procedures which modifies characteristic of
  ! original FITSIO procedures. Some additional helper utilities
  ! are included.

  subroutine fits_copy(from,to)

    character(len=*), intent(in) :: from,to
    integer :: blocksize, status, u,v

    if( from == '' .or. to == '' ) return

    status = 0
    call ftgiou(u,status)
    call ftgiou(v,status)
    call ftopen(u,from,READONLY,blocksize,status)
    blocksize = 1
    call ftinit(v,to,blocksize,status)
    call ftcpfl(u,v,1,1,1,status)
    call ftclos(v,status)
    call ftclos(u,status)
    call ftfiou(u,status)
    call ftfiou(v,status)
    call ftrprt('STDERR',status)

  end subroutine fits_copy


  subroutine fits_unlink(file)

    character(len=*), intent(in) :: file
    logical :: ex

    if( file == '' ) return

    inquire(file=file,exist=ex)
    if( ex ) then
       open(26,file=file)
       close(26,status='DELETE')
    end if

  end subroutine fits_unlink

  subroutine fits_datetime(unit,keys,dateobs,status)

    integer, intent(in) :: unit
    character(len=*), dimension(:), intent(in) :: keys
    character(len=*), intent(out) :: dateobs
    integer, intent(in out) :: status

    character(len=FLEN_VALUE) :: date, time
    character(len=FLEN_COMMENT) :: com
    integer :: status1,status2

    dateobs = ''

    if( status /= 0 ) return

    status1 = 0
    status2 = 0

    com = ''
    date = ''
    time = ''
    call ftgkys(unit,keys(1),date,com,status1)
    call ftgkys(unit,keys(2),time,com,status2)

    if( status1 == 0 .and. status2 == 0 ) then
       ! full date is in dateobs, full time is in timeobs
       ! this time specification is obsolete now (since 2000)
       dateobs = trim(date) // "T" // trim(time)
       status = 0
    else if ( status1 == 0 ) then
       ! both date and time are in one record, the standard way
       dateobs = date
       status = 0
    else
       status = status1
    end if

  end subroutine fits_datetime

  subroutine init_seed

    integer, dimension(:), allocatable :: seed
    integer, dimension(8) :: values
    integer :: n

    call date_and_time(values=values)
    call random_seed(size=n)
    allocate(seed(n))
    seed = 666 ! the devil's seed
    n = min(size(seed),size(values))
    seed(1:n) = values(1:n)
    call random_seed(put=seed)

  end subroutine init_seed

  function fits_scratch_filename(filename,nlen) result(scratch)

    ! A scratch filename is created by characters in reverse-order
    ! of the original filename. It prevents a race-condition for very
    ! long filenames, having their beginning parts omitted.

    character(len=*), intent(in) :: filename
    integer, intent(in) :: nlen
    character(len=nlen) :: scratch

    real :: x
    real, parameter :: s = 10**precision(x)
    integer :: l
    character :: a
    character(len=nlen-(15+precision(x))) :: t
    ! 15 is length of prefix 'scratch_' and others

    ! non-alphabetical characters (/\.-..) are removed from filename
    t = ''
    do l = min(len_trim(filename),len(t)),1,-1
       a = filename(l:l)
       if(  (lle('0',a) .and. lle(a,'9')) .or. &
            (lle('A',a) .and. lle(a,'Z')) .or. &
            (lle('a',a) .and. lle(a,'z')) ) then
          t = trim(t) // a
       end if
    end do

    call random_number(x)
    write(scratch,'(3a,i0,a)') 'scratch_',trim(t),'_',nint(s*x),'.fits'

  end function fits_scratch_filename

  subroutine fits_init_scratch(unit,filename,scratch,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: filename
    character(len=*), intent(out) :: scratch
    integer, intent(in out) :: status

    integer, parameter :: blocksize = 1

    call init_seed
    do
       scratch = fits_scratch_filename(filename,len(scratch))

       status = 0
       call ftinit(unit,scratch,blocksize,status)
       if( status == 0 ) return
       if( status == FILE_NOT_CREATED ) status = 0
       if( status /= 0 ) return
    end do

  end subroutine fits_init_scratch

  subroutine fits_open_scratch(unit,filename,mode,scratch,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: filename
    integer, intent(in) :: mode
    character(len=*), intent(out) :: scratch
    integer, intent(in out) :: status

    integer :: blocksize, u

    call init_seed
    do
       scratch = fits_scratch_filename(filename,len(scratch))

       blocksize = 1
       call ftinit(unit,scratch,blocksize,status)
       if( status == 0 ) then

          call ftgiou(u,status)
          call ftopen(u,filename,READONLY,blocksize,status)
          call ftcpfl(u,unit,1,1,1,status)
          call ftclos(u,status)
          call ftclos(unit,status)
          call ftfiou(u,status)
          ! the file is re-opened, there is a risk of race-condition,
          ! but our approach enables use of specified access mode
          call ftopen(unit,scratch,mode,blocksize,status)
          return

       else if( status == FILE_NOT_CREATED ) then
          status = 0
       else
          return
       end if

    end do

  end subroutine fits_open_scratch

  subroutine fits_keep_scratch(unit,filename,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: filename
    integer, intent(in out) :: status

    integer, parameter :: blocksize = 1
    integer :: u

    call fits_unlink(filename)
    call ftgiou(u,status)
    call ftinit(u,filename,blocksize,status)
    call ftcpfl(unit,u,1,1,1,status)
    call ftclos(u,status)
    call ftfiou(u,status)

  end subroutine fits_keep_scratch

  subroutine fits_open_file_canonical(unit,filename,mode,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: filename
    integer, intent(in) :: mode
    integer, intent(in out) :: status

    integer :: blocksize
    character(len=FLEN_FILENAME) :: scratch

    scratch = ""

    if( .not. (1 <= unit .and. unit <= size(fitsfiles)) ) &
         stop 'FITSIO error: Unit number is out of range.'

    if( status /= 0 ) return

    if( mode == READONLY ) then

       call ftopen(unit,filename,READONLY,blocksize,status)

    else if( mode == READWRITE ) then

       call fits_open_scratch(unit,filename,READWRITE,scratch,status)

    end if

    fitsfiles(unit)%filename = filename
    fitsfiles(unit)%scratch = scratch
    fitsfiles(unit)%output = ''

  end subroutine fits_open_file_canonical

  subroutine fits_open_image(unit,filename,mode,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: filename
    integer, intent(in) :: mode
    integer, intent(in out) :: status

    integer :: nhdu, hdutype, chdu, naxis

    call fits_open_file(unit,filename,mode,status)

    if( status /= 0 ) return

    call ftthdu(unit,nhdu,status)
    do chdu = 1, nhdu
       call ftmahd(unit,chdu,hdutype,status)
       if( status /= 0 ) exit
       if( hdutype == IMAGE_HDU ) then
          call ftgidm(unit,naxis,status)
          if( naxis > 0 .and. status == 0 ) return
       end if
    end do

  end subroutine fits_open_image

  subroutine fits_open_table(unit,filename,mode,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: filename
    integer, intent(in) :: mode
    integer, intent(in out) :: status

    integer :: nhdu, hdutype, chdu, nrows

    call fits_open_file(unit,filename,mode,status)
    if( status /= 0 ) return

    call ftthdu(unit,nhdu,status)
    do chdu = 1, nhdu
       call ftmahd(unit,chdu,hdutype,status)
       if( status /= 0 ) exit
       if( hdutype == ASCII_TBL .or. hdutype == BINARY_TBL ) then
          call ftgnrw(unit,nrows,status)
          if( nrows > 0 .and. status == 0 ) return
       end if
    end do

  end subroutine fits_open_table

  subroutine fits_open_file_output(unit,filename,output,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: filename, output
    integer, intent(in out) :: status

    character(len=FLEN_FILENAME) :: scratch

    scratch = ""

    if( .not. (1 <= unit .and. unit <= size(fitsfiles)) ) &
         stop 'FITSIO error: Unit number is out of range.'

    if( status /= 0 ) return

    if( filename == '' ) then
       status = READ_ERROR
       return
    end if

    call fits_open_scratch(unit,filename,READWRITE,scratch,status)

    fitsfiles(unit)%filename = filename
    fitsfiles(unit)%scratch = scratch
    fitsfiles(unit)%output = output

  end subroutine fits_open_file_output

  subroutine fits_create_file(unit,filename,status,overwrite)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: filename
    integer, intent(in out) :: status
    logical, optional :: overwrite

    integer, parameter :: blocksize = 1
    character(len=FLEN_FILENAME) :: scratch
    logical :: destroy

    scratch = ""

    if( present(overwrite) ) then
       destroy = overwrite
    else
       destroy = .true.
    end if

    if( .not. (1 <= unit .and. unit <= size(fitsfiles)) ) &
         stop 'FITSIO error: Unit number is out of range.'

    if( status /= 0 ) return

    call ftpmrk
    call ftinit(unit,filename,blocksize,status)
    if( status == FILE_NOT_CREATED .and. destroy ) then
       status = 0
       call fits_init_scratch(unit,filename,scratch,status)
       call ftcmrk
    end if

    fitsfiles(unit)%filename = filename
    fitsfiles(unit)%scratch = scratch
    fitsfiles(unit)%output = ""

  end subroutine fits_create_file

  subroutine fits_close_file(unit,status)

    integer, intent(in) :: unit
    integer, intent(in out) :: status

    character(len=FLEN_FILENAME) :: filename, scratch, output

    if( .not. (1 <= unit .and. unit <= size(fitsfiles)) ) &
         stop 'FITSIO error: Unit number is out of range.'

    filename = fitsfiles(unit)%filename
    scratch = fitsfiles(unit)%scratch
    output = fitsfiles(unit)%output

    if( scratch /= "" ) then
       if( output == '' ) then
          call fits_keep_scratch(unit,filename,status)
       else
          call fits_keep_scratch(unit,output,status)
       end if
    end if

    call ftclos(unit,status)
    call fits_unlink(scratch)

  end subroutine fits_close_file

  subroutine fits_read_wcs(unit,ctype,crval,crpix,cd,crder,status)

    integer, intent(in) :: unit
    character(len=*), dimension(:), intent(out) :: ctype
    real(selected_real_kind(15)), dimension(:), intent(out) :: crval,crpix,crder
    real(selected_real_kind(15)), dimension(:,:), intent(out) :: cd
    integer, intent(in out) :: status

    character(len=FLEN_CARD) :: buf
    character(len=FLEN_KEYWORD) :: keyword, key
    integer :: n,m

    buf = ''
    keyword = ''
    key = ''
    ctype = ''

    ! read astrometric calibration
    do n = 1, size(ctype)
       call ftkeyn('CTYPE',n,keyword,status)
       call ftgkys(unit,keyword,ctype(n),buf,status)
    end do
    do n = 1, size(crval)
       call ftkeyn('CRVAL',n,keyword,status)
       call ftgkyd(unit,keyword,crval(n),buf,status)
    end do
    do n = 1, size(crpix)
       call ftkeyn('CRPIX',n,keyword,status)
       call ftgkyd(unit,keyword,crpix(n),buf,status)
    end do
    do n = 1, size(cd,1)
       call ftkeyn('CD',n,keyword,status)
       do m = 1, size(cd,2)
          call ftkeyn(trim(keyword)//"_",m,key,status)
          call ftgkyd(unit,key,cd(n,m),buf,status)
       end do
    end do

    if( status /= 0 ) return

    ! optional keywords
    call fits_write_errmark
    do n = 1, size(crder)
       call ftkeyn('CRDER',n,keyword,status)
       call ftgkyd(unit,keyword,crder(n),buf,status)
    end do
    if( status == KEYWORD_NOT_FOUND ) then
       call fits_clear_errmark
       status = 0
       crder = 0
    end if

  end subroutine fits_read_wcs

  subroutine fits_update_wcs(unit,ctype,crval,crpix,cd,crder,status)

    integer, intent(in) :: unit
    character(len=*), dimension(:), intent(in) :: ctype
    real(selected_real_kind(15)), dimension(:), intent(in) :: crval,crpix
    real(selected_real_kind(15)), dimension(:,:), intent(in) :: cd
    real(selected_real_kind(15)), dimension(:), intent(in) :: crder
    integer, intent(in out) :: status

    integer, parameter :: digits = 15
    character(len=FLEN_KEYWORD) :: keyword, key
    integer :: n,m

    keyword = ''
    key = ''

    ! read astrometric calibration
    do n = 1, size(ctype)
       call ftkeyn('CTYPE',n,keyword,status)
       call ftukys(unit,keyword,ctype(n),'coordinate projection type',status)
    end do
    do n = 1, size(crval)
       call ftkeyn('CRVAL',n,keyword,status)
       call ftukyd(unit,keyword,crval(n),digits, &
            '[deg] spherical coordinates of center of projection',status)
    end do
    do n = 1, size(crpix)
       call ftkeyn('CRPIX',n,keyword,status)
       call ftukyd(unit,keyword,crpix(n),-6, &
            '[pix] reference pixel in focal plane',status)
    end do
    do n = 1, size(cd,1)
       call ftkeyn('CD',n,keyword,status)
       do m = 1, size(cd,2)
          call ftkeyn(trim(keyword)//"_",m,key,status)
          call ftukyd(unit,key,cd(n,m),digits, &
               '[deg/pix] scaled rotation matrix',status)
       end do
    end do

    ! optional
    if( all(crder > 0) ) then
       do n = 1, size(crder)
          call ftkeyn('CRDER',n,keyword,status)
          call ftukyd(unit,keyword,crder(n),1,'[deg] standard errors',status)
       end do
    end if

    do n = 1, size(ctype)
       call ftkeyn('CUNIT',n,keyword,status)
       call ftukys(unit,keyword,'deg','units of the CRVALx axis',status)
    end do

  end subroutine fits_update_wcs

  !
  !  Fortran 2008+ INTERFACES of FITSIO
  !

  subroutine fits_str2date(dateobs,year,month,day,hour,minute,second,status)

    character(len=*), intent(in) :: dateobs
    integer, intent(out) :: year,month,day,hour,minute
    real(selected_real_kind(15)), intent(out) :: second
    integer, intent(in out) :: status

    call fts2tm(dateobs,year,month,day,hour,minute,second,status)

  end subroutine fits_str2date

  subroutine fits_report_error(stream,status)

    character(len=*), intent(in) :: stream
    integer, intent(in out) :: status

    call ftrprt(stream,status)

  end subroutine fits_report_error

  subroutine fits_write_errmark

    call ftpmrk

  end subroutine fits_write_errmark

  subroutine fits_clear_errmark

    call ftcmrk

  end subroutine fits_clear_errmark

  subroutine fits_movnam_hdu(unit,hdutype,extname,extver,status)

    integer, intent(in) :: unit, hdutype, extver
    character(len=*), intent(in) :: extname
    integer, intent(in out) :: status

    call ftmnhd(unit,hdutype,extname,extver,status)

  end subroutine fits_movnam_hdu

  subroutine fits_read_keyword(unit,keyword,keyval,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: keyword
    character(len=*), intent(out) :: keyval, comment
    integer, intent(in out) :: status

    keyval = ''
    comment = ''
    call ftgkey(unit,keyword,keyval,comment,status)

  end subroutine fits_read_keyword

  subroutine fits_read_key_str(unit,keyword,keyval,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: keyword
    character(len=*), intent(out) :: keyval, comment
    integer, intent(in out) :: status

    keyval = ''
    comment = ''
    call ftgkys(unit,keyword,keyval,comment,status)

  end subroutine fits_read_key_str

  subroutine fits_read_key_int(unit,keyword,keyval,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: keyword
    integer, intent(out) :: keyval
    character(len=*), intent(out) :: comment
    integer, intent(in out) :: status

    comment = ''
    keyval = 0
    call ftgkyj(unit,keyword,keyval,comment,status)

  end subroutine fits_read_key_int

  subroutine fits_read_key_real(unit,keyword,keyval,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: keyword
    real, intent(out) :: keyval
    character(len=*), intent(out) :: comment
    integer, intent(in out) :: status

    comment = ''
    keyval = 1.0
    call ftgkye(unit,keyword,keyval,comment,status)

  end subroutine fits_read_key_real

  subroutine fits_read_key_double(unit,keyword,keyval,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: keyword
    real(selected_real_kind(15)), intent(out) :: keyval
    character(len=*), intent(out) :: comment
    integer, intent(in out) :: status

    comment = ''
    keyval = real(1,selected_real_kind(15))
    call ftgkyd(unit,keyword,keyval,comment,status)

  end subroutine fits_read_key_double

  subroutine fits_write_key_str(unit,key,val,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key,val,comment
    integer, intent(in out) :: status

    call ftpkys(unit,key,val,comment,status)

  end subroutine fits_write_key_str

  subroutine fits_write_key_int(unit,key,numval,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key, comment
    integer, intent(in) :: numval
    integer, intent(in out) :: status

    call ftpkyj(unit,key,numval,comment,status)

  end subroutine fits_write_key_int

  subroutine fits_write_key_real(unit,key,numval,decimals,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key, comment
    real, intent(in) :: numval
    integer, intent(in) :: decimals
    integer, intent(in out) :: status

    call ftpkye(unit,key,numval,decimals,comment,status)

  end subroutine fits_write_key_real

  subroutine fits_write_key_double(unit,key,numval,decimals,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key, comment
    real(selected_real_kind(15)), intent(in) :: numval
    integer, intent(in) :: decimals
    integer, intent(in out) :: status

    call ftpkyd(unit,key,numval,decimals,comment,status)

  end subroutine fits_write_key_double

  subroutine fits_update_key_str(unit,key,val,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key,val,comment
    integer, intent(in out) :: status

    call ftukys(unit,key,val,comment,status)

  end subroutine fits_update_key_str

  subroutine fits_update_key_int(unit,key,numval,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key, comment
    integer, intent(in) :: numval
    integer, intent(in out) :: status

    call ftukyj(unit,key,numval,comment,status)

  end subroutine fits_update_key_int

  subroutine fits_update_key_real(unit,key,numval,decimals,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key, comment
    real, intent(in) :: numval
    integer, intent(in) :: decimals
    integer, intent(in out) :: status

    call ftukye(unit,key,numval,decimals,comment,status)

  end subroutine fits_update_key_real

  subroutine fits_update_key_double(unit,key,numval,decimals,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key, comment
    real(selected_real_kind(15)), intent(in) :: numval
    integer, intent(in) :: decimals
    integer, intent(in out) :: status

    call ftukyd(unit,key,numval,decimals,comment,status)

  end subroutine fits_update_key_double

  subroutine fits_delete_key(unit,key,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: key
    integer, intent(in out) :: status

    call ftdkey(unit,key,status)

  end subroutine fits_delete_key

  subroutine fits_write_comment(unit,comment,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: comment
    integer, intent(in out) :: status

    call ftpcom(unit,comment,status)

  end subroutine fits_write_comment

  subroutine fits_write_history(unit,history,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: history
    integer, intent(in out) :: status

    call ftphis(unit,history,status)

  end subroutine fits_write_history

  subroutine fits_get_img_type(unit,bitpix,status)

    integer, intent(in) :: unit
    integer, intent(out) :: bitpix
    integer, intent(in out) :: status

    call ftgidt(unit,bitpix,status)

  end subroutine fits_get_img_type

  subroutine fits_get_img_dim(unit,naxis,status)

    integer, intent(in) :: unit
    integer, intent(out) :: naxis
    integer, intent(in out) :: status

    call ftgidm(unit,naxis,status)

  end subroutine fits_get_img_dim

  subroutine fits_get_img_size(unit,naxis,naxes,status)

    integer, intent(in) :: unit, naxis
    integer, dimension(:), intent(out) :: naxes
    integer, intent(in out) :: status

    call ftgisz(unit,naxis,naxes,status)

  end subroutine fits_get_img_size

  subroutine fits_insert_img(unit,bitpix,naxis,naxes,status)

    integer, intent(in) :: unit, bitpix, naxis
    integer, dimension(:), intent(in) :: naxes
    integer, intent(in out) :: status

    integer, dimension(size(naxes)) :: maxes

    ! A local copy of naxes is required to enable use
    ! of constants or constructors as parameters:
    ! the direct pass causes runtime crash.
    maxes = naxes
    call ftiimg(unit,bitpix,naxis,maxes,status)

    ! the image is scaled to appropriate range by default
    if( bitpix > 0 ) then
       call fits_write_key(unit,'BSCALE',1,'',status)

       if( bitpix == 32 ) then
          call fits_write_key(unit,'BZERO',huge(bitpix),'',status)
       else
          call fits_write_key(unit,'BZERO',2**(bitpix-1),'',status)
       end if
    endif

  end subroutine fits_insert_img


  subroutine fits_read_2d(unit,image,status)

    real, parameter :: nullval = 0.0
    integer, parameter :: group = 0

    integer, intent(in) :: unit
    real, dimension(:,:), intent(out) :: image
    integer, intent(in out) :: status

    integer :: dim1, naxes1, naxes2
    logical :: anyf

    dim1 = size(image,1)
    naxes1 = size(image,1)
    naxes2 = size(image,2)

    call ftg2de(unit,group,nullval,dim1,naxes1,naxes2,image,anyf,status)

  end subroutine fits_read_2d

  subroutine fits_write_2d(unit,image,status)

    integer, parameter :: group = 0

    integer, intent(in) :: unit
    real, dimension(:,:), intent(in) :: image
    integer, intent(in out) :: status

    integer :: dim1, naxes1, naxes2

    dim1 = size(image,1)
    naxes1 = size(image,1)
    naxes2 = size(image,2)

    call ftp2de(unit,group,dim1,naxes1,naxes2,image,status)

  end subroutine fits_write_2d

  subroutine fits_read_3d(unit,image,status)

    real, parameter :: nullval = 0.0
    integer, parameter :: group = 0

    integer, intent(in) :: unit
    real, dimension(:,:,:), intent(out) :: image
    integer, intent(in out) :: status

    integer :: dim1, dim2, naxes1, naxes2, naxes3
    logical :: anyf

    dim1 = size(image,1)
    dim2 = size(image,2)
    naxes1 = size(image,1)
    naxes2 = size(image,2)
    naxes3 = size(image,3)

    call ftg3de(unit,group,nullval,dim1,dim2,naxes1,naxes2,naxes3, &
         image,anyf,status)

  end subroutine fits_read_3d

  subroutine fits_write_3d(unit,image,status)

    integer, parameter :: group = 0

    integer, intent(in) :: unit
    real, dimension(:,:,:), intent(in) :: image
    integer, intent(in out) :: status

    integer :: dim1, dim2, naxes1, naxes2, naxes3

    dim1 = size(image,1)
    dim2 = size(image,2)
    naxes1 = size(image,1)
    naxes2 = size(image,2)
    naxes3 = size(image,3)

    call ftp3de(unit,group,dim1,dim2,naxes1,naxes2,naxes3,image,status)

  end subroutine fits_write_3d

  subroutine fits_copy_file(iunit,ounit,previous,current,following,status)

    integer, intent(in) :: iunit, ounit, previous,current,following
    integer, intent(in out) :: status

    call ftcpfl(iunit,ounit,previous,current,following,status)

  end subroutine fits_copy_file

  subroutine fits_copy_header(iunit,ounit,status)

    integer, intent(in) :: iunit, ounit
    integer, intent(in out) :: status

    call ftcphd(iunit, ounit, status)

  end subroutine fits_copy_header

  subroutine fits_movabs_hdu(unit, nhdu, hdutype, status)

    integer, intent(in) :: unit, nhdu
    integer, intent(out) :: hdutype
    integer, intent(in out) :: status

    call ftmahd(unit, nhdu, hdutype, status)

  end subroutine fits_movabs_hdu

  subroutine fits_delete_hdu(unit,hdutype,status)

    integer, intent(in) :: unit
    integer, intent(out) :: hdutype
    integer, intent(in out) :: status

    call ftdhdu(unit,hdutype,status)

  end subroutine fits_delete_hdu

  subroutine fits_get_num_hdus(unit,hdunum,status)

    integer, intent(in) :: unit
    integer, intent(out) :: hdunum
    integer, intent(in out) :: status

    call ftthdu(unit, hdunum, status)

  end subroutine fits_get_num_hdus

  subroutine fits_get_hdu_num(unit,hdunum)

    integer, intent(in) :: unit
    integer, intent(out) :: hdunum

    call ftghdn(unit, hdunum)

  end subroutine fits_get_hdu_num

  subroutine fits_get_num_rows(unit,nrows, status)

    integer, intent(in) :: unit
    integer, intent(out) :: nrows
    integer, intent(in out) :: status

    call ftgnrw(unit,nrows,status)

  end subroutine fits_get_num_rows

  subroutine fits_get_num_cols(unit,ncols, status)

    integer, intent(in) :: unit
    integer, intent(out) :: ncols
    integer, intent(in out) :: status

    call ftgncl(unit,ncols,status)

  end subroutine fits_get_num_cols

  subroutine fits_get_colnum(unit,casesen,coltemplate,colnum,status)

    integer, intent(in) :: unit
    logical, intent(in) :: casesen
    character(len=*), intent(in) :: coltemplate
    integer, intent(out) :: colnum
    integer, intent(in out) :: status

    call ftgcno(unit,casesen,coltemplate,colnum,status)

  end subroutine fits_get_colnum

  subroutine fits_read_col_str(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    character(len=*), dimension(:), intent(out) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    character, parameter :: nullval = ''
    integer :: nrows
    logical :: anyf

    nrows = size(values)
    values = ''
    call ftgcvs(unit,colnum,frow,felem,nrows,nullval,values,anyf,status)

  end subroutine fits_read_col_str

  subroutine fits_read_col_int(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    integer, dimension(:), intent(out) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    integer, parameter :: nullval = 0
    integer :: nrows
    logical :: anyf

    nrows = size(values)
    call ftgcvj(unit,colnum,frow,felem,nrows,nullval,values,anyf,status)

  end subroutine fits_read_col_int

  subroutine fits_read_col_logical(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    logical, dimension(:), intent(out) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    integer, parameter :: nullval = 0
    integer :: nrows
    logical :: anyf

    nrows = size(values)
    call ftgcvl(unit,colnum,frow,felem,nrows,nullval,values,anyf,status)

  end subroutine fits_read_col_logical

  subroutine fits_read_col_real(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    real, dimension(:), intent(out) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    real, parameter :: nullval = 0
    integer :: nrows
    logical :: anyf

    nrows = size(values)
    call ftgcve(unit,colnum,frow,felem,nrows,nullval,values,anyf,status)

  end subroutine fits_read_col_real

  subroutine fits_read_col_double(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    real(selected_real_kind(15)), dimension(:), intent(out) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    real(selected_real_kind(15)), parameter :: nullval = 0
    integer :: nrows
    logical :: anyf

    nrows = size(values)
    call ftgcvd(unit,colnum,frow,felem,nrows,nullval,values,anyf,status)

  end subroutine fits_read_col_double

  subroutine fits_make_keyn(keyroot,seq_no,keyword,status)

    character(len=*), intent(in) :: keyroot
    integer, intent(in) :: seq_no
    character(len=*), intent(out) :: keyword
    integer, intent(in out) :: status

    keyword = ''
    call ftkeyn(keyroot,seq_no,keyword,status)

  end subroutine fits_make_keyn

  subroutine fits_insert_btbl(unit,nrows,ttype,tform,tunit,extname,status)

    integer, intent(in) :: unit, nrows
    character(len=*), dimension(:), intent(in) :: ttype, tform, tunit
    character(len=*), intent(in) :: extname
    integer, intent(in out) :: status

    integer, parameter :: varidat = 0
    integer :: tfields

    tfields = size(ttype)
    call ftibin(unit,nrows,tfields,ttype,tform,tunit,extname,varidat,status)

  end subroutine fits_insert_btbl

  subroutine fits_write_col_str(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    character(len=*), dimension(:), intent(in) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    integer :: nrows

    nrows = size(values)
    call ftpcls(unit,colnum,frow,felem,nrows,values,status)

  end subroutine fits_write_col_str

  subroutine fits_write_col_logical(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    logical, dimension(:), intent(in) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    integer :: nrows

    nrows = size(values)
    call ftpcll(unit,colnum,frow,felem,nrows,values,status)

  end subroutine fits_write_col_logical

  subroutine fits_write_col_int(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    integer, dimension(:), intent(in) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    integer :: nrows

    nrows = size(values)
    call ftpclj(unit,colnum,frow,felem,nrows,values,status)

  end subroutine fits_write_col_int

  subroutine fits_write_col_real(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    real, dimension(:), intent(in) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    integer :: nrows

    nrows = size(values)
    call ftpcle(unit,colnum,frow,felem,nrows,values,status)

  end subroutine fits_write_col_real

  subroutine fits_write_col_double(unit,colnum,values,status)

    integer, intent(in) :: unit
    integer, intent(in) :: colnum
    real(selected_real_kind(15)), dimension(:), intent(in) :: values
    integer, intent(in out) :: status

    integer, parameter :: frow = 1
    integer, parameter :: felem = 1
    integer :: nrows

    nrows = size(values)
    call ftpcld(unit,colnum,frow,felem,nrows,values,status)

  end subroutine fits_write_col_double

  subroutine fits_get_keytype(value,dtype,status)

    character(len=*), intent(in) :: value
    character(len=*), intent(out) :: dtype
    integer, intent(in out) :: status

    ! provided utility works only with values
    ! obtained by fits_read_keyword / ftgkey
    call ftdtyp(value,dtype,status)

  end subroutine fits_get_keytype

  subroutine fits_read_key_unit(unit,keyword,units,status)

    integer, intent(in) :: unit
    character(len=*), intent(in) :: keyword
    character(len=*), intent(out) :: units
    integer, intent(in out) :: status

    units = ''
    call ftgunt(unit,keyword,units,status)

  end subroutine fits_read_key_unit


end module fitsio
