/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 


/*
 * $Id: screen.h,v 1.8 2003/11/23 08:11:30 alexis Exp $
 */

#ifndef _SCREEN_H		/* include once only */
#define _SCREEN_H

typedef unsigned char text_t;

#if defined(MULTICHAR_SET)
typedef R_u_int32_t	rend_t;
#else
typedef R_u_int16_t	rend_t;
#endif

/*
 * screen accounting:
 * screen_t elements
 *   text:      Contains all text information including the scrollback buffer.
 *              Each line is length TermWin.ncol
 *   tlen:      The length of the line or -1 for wrapped lines.
 *   rend:      Contains rendition information: font, bold, colour, etc.
 * * Note: Each line for both text and rend are only allocated on demand, and
 *         text[x] is allocated <=> rend[x] is allocated  for all x.
 *   row:       Cursor row position                   : 0 <= row < TermWin.nrow
 *   col:       Cursor column position                : 0 <= col < TermWin.ncol
 *   tscroll:   Scrolling region top row inclusive    : 0 <= row < TermWin.nrow
 *   bscroll:   Scrolling region bottom row inclusive : 0 <= row < TermWin.nrow
 * Horizontal scroll region :
 *   lscroll:   Scrolling region left column inclusive: 0 <= col < TermWin.min_bcol
 *   rscroll:   Scrolling region right column inclusive : 0 <= col < TermWin.min_bcol
 *
 * selection_t elements
 *   clicks:    1, 2 or 3 clicks - 4 indicates a special condition of 1 where
 *              nothing is selected
 *   beg:       row/column of beginning of selection  : never past mark
 *   mark:      row/column of initial click           : never past end
 *   end:       row/column of one character past end of selection
 * * Note: -TermWin.nscrolled <= beg.row <= mark.row <= end.row < TermWin.nrow
 * * Note: col == -1 ==> we're left of screen
 *
 * TermWin.saveLines:
 *              Maximum number of lines in the scrollback buffer.
 *              This is fixed for each rxvt instance.
 * TermWin.nscrolled:
 *              Actual number of lines we've used of the scrollback buffer
 *              0 <= TermWin.nscrolled <= TermWin.saveLines
 * TermWin.view_start:  
 *              Offset back into the scrollback buffer for out current view
 *              0 <= TermWin.view_start <= TermWin.nscrolled
 *
 * Layout of text/rend information in the screen_t text/rend structures:
 *   Rows [0] ... [TermWin.saveLines - 1]
 *     scrollback region : we're only here if TermWin.view_start != 0
 *   Rows [TermWin.saveLines] ... [TermWin.saveLines + TermWin.nrow - 1]
 *     normal `unscrolled' screen region
 */
typedef struct {
    text_t        **text;	/* _all_ the text                            */
    R_int16_t	   *tlen;	/* length of each text line                  */
    rend_t        **rend;	/* rendition, uses RS_ flags                 */
    row_col_t       cur;	/* cursor position on the screen             */
    R_int16_t       tscroll,	/* top of settable scroll region             */
                    bscroll,	/* bottom of settable scroll region          */
                    rscroll,	/* right of settable scroll region           */
                    lscroll,	/* left of settable scroll region            */
		    charset;	/* character set number [0..3]               */
    unsigned int    flags;	/* see below                                 */
} screen_t;

typedef struct {
    row_col_t	    cur;	/* cursor position                           */
    R_int16_t	    charset;	/* character set number [0..3]               */
    char            charset_char;
    rend_t          rstyle;	/* rendition style                           */
} save_t;

typedef struct {
    unsigned char  *text;	/* selected text                             */
    int             len;	/* length of selected text                   */
    enum {
	SELECTION_CLEAR = 0,	/* nothing selected                          */
	SELECTION_INIT,		/* marked a point                            */
	SELECTION_BEGIN,	/* started a selection                       */
	SELECTION_CONT,		/* continued selection                       */
	SELECTION_DONE		/* selection put in CUT_BUFFER0              */
    } op;			/* current operation                         */
    short           screen;	/* screen being used                         */
    short           clicks;	/* number of clicks                          */
    row_col_t       beg, mark, end;
    int vt; /* vt of the last selection*/
} selection_t;

/* ------------------------------------------------------------------------- */

/* screen_t flags */
#define Screen_Relative		(1<<0)	/* relative origin mode flag         */
#define Screen_VisibleCursor	(1<<1)	/* cursor visible?                   */
#define Screen_Autowrap		(1<<2)	/* auto-wrap flag                    */
#define Screen_Insert		(1<<3)	/* insert mode (vs. overstrike)      */
#define Screen_WrapNext		(1<<4)	/* need to wrap for next char?       */
#define Screen_DefaultFlags	(Screen_VisibleCursor|Screen_Autowrap)

/* ------------------------------------------------------------------------- *
 *                             MODULE VARIABLES                              *
 * ------------------------------------------------------------------------- */

#ifdef INTERN_SCREEN
# define EXTSCR
#else
# define EXTSCR	extern
#endif

/* This tells what's actually on the screen */
EXTSCR text_t    **drawn_text;
EXTSCR rend_t    **drawn_rend;
EXTSCR text_t    **buf_text;
EXTSCR rend_t    **buf_rend;
EXTSCR R_int16_t  *buf_tlen;
EXTSCR char       *tabs;	/* a 1 for a location with a tab-stop */
EXTSCR screen_t    swap;
EXTSCR save_t      save[MAX_PAGES]; // pour sauver les positions du curseur
EXTSCR selection_t selection;
EXTSCR char        charsets[4];
EXTSCR short       current_screen;
EXTSCR rend_t      rstyle;
EXTSCR short       rvideo;


//#define scr_clear()  f_scr_clear_tint(0,1)

//#define make_screen_mem(tp,rp,row,rstyle) f_make_screen_mem(0,tp,rp,row,rstyle)
#define f_scr_clear(page)  f_scr_clear_tint(page,1)

void print_info_screen();
void scr_init();
void blank_line (text_t * et, rend_t * er, int width, rend_t efs);
void f_make_screen_mem (int page,text_t ** tp, rend_t ** rp, int row, rend_t rstyle);
void f_scr_reset (int page);
void f_scr_release (int page,int release_drawn);
void f_scr_poweron (int page);
void f_scr_cursor (int page,int mode);
int f_scr_change_screen (int page,int scrn);
void scr_color (unsigned int color, unsigned int Intensity);
void scr_rendition (int set, int style);
int f_scroll_text (int page,int row1, int row2, int count, int spec);
void f_scr_add_lines (int page,const unsigned char *str, int nlines, int len);
void f_scr_backspace (int page);
void f_scr_tab (int page,int count);
void f_scr_gotorc (int page,int row, int col, int relative);
void f_scr_index (int page,int direction);
void f_scr_erase_line (int page,int mode);
void f_scr_erase_screen (int page,int mode);
void f_scr_E (int page);
void f_scr_insdel_lines (int page,int count, int insdel);
void f_scr_insdel_chars (int page,int count, int insdel);
void f_scr_scroll_region (int page,int top, int bot);
void f_scr_cursor_visible (int page,int mode);
void f_scr_autowrap (int page,int mode);
void f_scr_relative_origin (int page,int mode);
void f_scr_insert_mode (int page,int mode);
void f_scr_set_tab (int page,int mode);
void f_scr_rvideo_mode (int page,int mode);
void f_scr_report_position (int page);
void f_set_font_style (int page);
void f_scr_charset_choose (int page,int set);
void f_scr_charset_set (int page,int set, unsigned int ch);
void eucj2jis (unsigned char *str, int len);
void sjis2jis (unsigned char *str, int len);
void big5dummy (unsigned char *str, int len);
void set_multichar_encoding (const char *str);
int scr_get_fgcolor (void);
int scr_get_bgcolor (void);
void f_scr_expose (int page,int x, int y, int width, int height);
void f_scr_touch (int page);
int f_scr_move_to (int page,int y, int len);
int f_scr_page (int page,int direction, int nlines);
void scr_bell (void);
void f_scr_printscreen (int page,int fullhist);
void f_scr_refresh (int page,int type);
void f_scr_clear_tint (int page,int bWithTinting);
void f_on_colors_changed (int page,int idx);
void f_scr_reverse_selection (int page);
void f_selection_check (int page,int check_more);
void f_PasteIt (int page,unsigned char *data, unsigned int nitems);
void selection_paste (Window win, unsigned prop, int Delete);
void f_selection_request (int page,Time tm, int x, int y);
void selection_clear (void);
void f_selection_make (int page,Time tm);
void selection_click (int clicks, int x, int y);
void selection_start_colrow (int col, int row);
void f_selection_delimit_word (int page,int dirn, row_col_t * mark, row_col_t * ret);
void selection_extend (int x, int y, int flag);
void f_selection_extend_colrow (int page,int col, int row, int button3, int buttonpress,int clickchange);
void selection_rotate (int x, int y);
void selection_send (XSelectionRequestEvent * rq);
void pixel_position (int *x, int *y);
void mouse_tracking (int report, int x, int y, int firstrow, int lastrow);
void debug_PasteIt (unsigned char *data, int nitems);
void debug_colors (void);
void pat_scr_remove(int page,int all);
#endif				/* repeat inclusion protection */
