#!/bin/bash


help()
{
	echo "Usage:"
	echo ""
	echo "$(basename $0) -h | --help to display this help;"
	echo ""
	echo "This program takes for granted that the version number is in the form"
	echo "<major>.<minor>.<patch>, like for example version 1.3.0."
	echo ""
	echo "$(basename $0) fromCML | [ incMajor [ incMinor [ incPatch ] ] iss tag"
	echo ""
	echo "The arguments need to be provided in this order but not all are required"
	echo ""
	echo "fromCML: read version number from CMakeLists"
	echo "and optionally use it later to perform actions with"
	echo "that version (iss and/or tag). Note that this is exclusive with the"
	echo "other incXyyy options."
	echo ""
	echo "incMajor: increment the major component of the version in CMakeLists.txt"
	echo "incMinor: increment the minor component of the version in CMakeLists.txt"
	echo "incPatch: increment the patch component of the version in CMakeLists.txt"

	echo "iss: update the version in the InnoSetup files in the winInstaller directory." 
	echo "tag: after committing the CMakeListsTxt file, also create a tag by the new version".
	echo ""
}

version=""
major=""
minor=""
patch=""


isTopSourceDir()
{
	if [ -f CMakeLists.txt ]
	then
		# This is not enough, we could be in a subdirectory also managed by CMake.
		# Check if there is a specific bit of info:

		versionLine=$(grep 'set(VERSION' CMakeLists.txt)

		if [ "$?" = "0" ]
		then
			return 0
		fi

		echo "The CMakeLists.txt file does not contain any VERSION set."
		echo "Not in the top source directory."

		return 1
	fi

	# echo "CMakeLists.txt not found. Please, make sure you are in the top source directory. Exiting."
	return 1
}


CMLversion()
{
	# This is the line we want to deal with:
	# set(VERSION "0.2.1")

	versionLine=$(grep 'set(VERSION' CMakeLists.txt)
	if [ "$?" != "0" ]
	then
		echo "The CMakeLists.txt file does not contain any VERSION set. Exiting."
		return 1
	fi

	# echo ${versionLine}

	version=$(echo ${versionLine} | sed 's/set(VERSION \+"\([0-9]\+\.[0-9]\+\.[0-9]\+\)")/\1/')
	if [ "$?" != "0" ]
	then
		return 1
	fi

	# echo "CMakeList.txt version is ${version}"

	major=$(echo ${version} | sed 's/^\([0-9]\)\.\([0-9]\)\.\([0-9]\)/\1/')
	minor=$(echo ${version} | sed 's/^\([0-9]\)\.\([0-9]\)\.\([0-9]\)/\2/')
	patch=$(echo ${version} | sed 's/^\([0-9]\)\.\([0-9]\)\.\([0-9]\)/\3/')

	# echo "major: ${major}"
	# echo "minor: ${minor}"
	# echo "patch: ${patch}"

	# The value version is global.
	# echo ${version}
	return 0
}


isMasterBranch()
{
	currentBranch=$(git branch | grep "\*" | awk '{print $2}')
	echo "currentBranch: ${currentBranch}"

	if [ "${currentBranch}" = "master" ]
	then
		return 0
	fi
	return 1
}

gitStatusClean()
{
	gitStatus=$(git status)
	gitStatusLineCount=$(git status -s | wc -l)

	if [ "${gitStatusLineCount}" = "0" ]
	then
		return 0
	else
		echo "${gitStatus}"
		return 1
	fi
}


tag()
{
	if [ "x${version}" = "x" ]
	then 
		echo "version was not set. Cannot perform the git tag operation"
		echo "without a version number." 

		return 1
	fi

	# We can only accept to commit if the tree is clean.

	gitStatus=$(gitStatusClean)
	if [ "$?" != "0" ]
	then
		echo ""
		echo "The current branch is not clean:"
		echo ""
		echo "8<~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
		echo "${gitStatus}"
		echo "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~>8"
		echo ""
		echo ""
		echo "Should a commit be performed? If so, enter the commit message below."
		echo "Else just hit <Enter> or [Ctrl-C] to break the execution of this"
		echo "program. The commit will include *all* the indexed files."
		echo ""

		read msg

		if [ "x${msg}" = "x" ]
		then 
			echo "Aborted by the user."
			exit 1
		else
			git commit -m"${msg}" -a
			if [ "$?" != "0" ]
			then 
				echo "The commit failed. Exiting."
				exit 1
			fi
		fi
	fi

	# Now, yes, we can tag
	gitTagCmd="git tag -u 41AB484D7694CF42 -m\"v${version}\" ${version} HEAD"
	echo "Tag command: ${gitTagCmd}"

	eval ${gitTagCmd}
	return $?
}
# End of tag()


iss()
{
	if [ "x${version}" = "x" ]
	then 
		echo "version was not set. Cannot perform the iss sed operation"
		echo "without a version number." 

		return 1
	fi

	if [ ! -d winInstaller ]
	then
		echo "The winInstaller directory does not exist. Exiting."

		return 1
	fi

	if [ ! -f winInstaller/msxpertsuite-x86-win7+.iss ] || [ ! -f winInstaller/msxpertsuite-amd64-win7+.iss ]
	then
		echo "The expected ISS files were not found. Exiting."

		return 1
	fi

	file="winInstaller/msxpertsuite-x86-win7+.iss"
	# Verify that we can find the proper version defining statement

	grep '#define public version\s\+"[0-9]\.[0-9]\.[0-9]"' ${file} > /dev/null 2>&1

	if [ "$?" != "0" ]
	then
		echo "Failed to find the version defining statement in file ${file}.  Exiting."
		return 1
	fi

	command="sed -i 's/\(#define public version\s\+\"\)\([0-9]\.[0-9]\.[0-9]\)/\1${version}/'"
	command="${command} ${file}"
	echo "ISS sed command: ${command}"
	eval ${command}

	file="winInstaller/msxpertsuite-amd64-win7+.iss"
	# Verify that we can find the proper version defining statement

	grep '#define public version\s\+"[0-9]\.[0-9]\.[0-9]"' ${file} > /dev/null 2>&1

	if [ "$?" != "0" ]
	then
		echo "Failed to find the version defining statement in file ${file}.  Exiting."
		return 1
	fi

	command="sed -i 's/\(#define public version\s\+\"\)\([0-9]\.[0-9]\.[0-9]\)/\1${version}/'"
	command="${command} ${file}"
	echo "ISS sed command: ${command}"

	eval ${command}

	return $?
}
# End of iss()


if [ "$1" = "help" ] || [ "$1" = "--help" ] || [ "$1" = "-h" ] || [ "$#" = "0" ]
then
	help
	exit 0
fi

#####################################################################################
#														START OF THE SCRIPT WORK

# If we are not located in the top source directory, the CMakeLists.txt file
# will not contain the version number. Without that, we do nothing here.

isTopSourceDir || exit 1

# Now call CMLversion that we'll need each time.
CMLversion || exit 1
echo "CML version: ${version}"

if [ "$1" = "fromCML" ]
then
	shift

	if [ "$1" = "iss" ]
	then
		shift
		echo "fromCML and iss"

		iss || echo "Failed to perform the ISS files update."
	fi

	if [ "$1" = "tag" ]
	then
		shift

		echo "fromCML and tag"

		tag
	fi

	exit 0
fi

# If we are here, that means that the incXyyy route was chosen, thus we can
# start by getting the version from the CMakeLists.txt file, of which we'll
# modified the major/minor/patch components according to the parameters passed
# to this script.

if [ "$?" != "0" ]
then
	echo "Failed to properly get the version of the software. Exiting."
	exit 1
fi


if [ "$1" = "incMajor" ]
then
	shift 
	major=$((major+1))
	echo "New major: ${major}"
	version=${major}.${minor}.${patch}
fi

if [ "$1" = "incMinor" ]
then
	shift
	minor=$((minor+1))
	echo "New minor: ${minor}"
	version=${major}.${minor}.${patch}
fi

if [ "$1" = "incPatch" ]
then
	shift
	patch=$((patch+1))
	echo "New patch: ${patch}"
	version=${major}.${minor}.${patch}
fi

command="sed -i 's/${versionLine}/set(VERSION \"${major}.${minor}.${patch}\")/' CMakeLists.txt"

echo ""
echo "New version: \"${major}.${minor}.${patch}\""
echo "Run the following command ? (y|Y|n)"
echo ""
echo "${command}"
echo ""
read answer

if [ "${answer}" != "y" ] && [ "${answer}" != "Y" ]
then
	echo "Aborted by the user. Exiting."
	exit 0
fi

eval ${command}

# Now update the ISS files
if [ "$1" = "iss" ]
then
	shift
	iss
fi

# Now update the tag
if [ "$1" = "tag" ]
then
	shift
	tag
fi

exit 0



