
#include <iostream>
#include <ctime>
#include <random>

#include "catch.hpp"

#include "MassSpectrumList.hpp"
#include "globals/globals.hpp"

#include <QDebug>
#include <QFile>

using std::cout;

#define TEST_DATA_DIR "/home/rusconi/devel/msxpertsuite/development/tests/data"

//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
//<< "Should compile libmass tests.";

namespace msXpSlibmass
{

	TEST_CASE("Construct a mass spectrum list with a name.", "[MassSpectrumList]")
	{
		QString title = "This is the title";
		MassSpectrumList massSpectra(title);
		REQUIRE(massSpectra.title() == title);
	}


	TEST_CASE("Construct a mass spectrum list as a copy of another list "
			"and test its features.", "[MassSpectrumList]")
	{
		// Craft the initial spectrum list.

		QStringList xyFormatStringList;
		xyFormatStringList
			<< "694.5166264003 133.0000000000"
			<< "694.5333824250 195.0000000000"
			<< "694.5501386518 225.0000000000"
			<< "694.5668950807 220.0000000000"
			<< "694.5836517118 286.0000000000"
			<< "694.6004085450 332.0000000000"
			<< "694.6171655803 282.0000000000"
			<< "694.6339228177 164.0000000000"
			<< "694.6506802573 140.0000000000"
			<< "694.6674378989 156.0000000000"
			<< "694.6841957428 153.0000000000"
			<< "694.7009537887 89.0000000000"
			<< "694.7177120368 104.0000000000"
			<< "694.7344704869 143.0000000000"
			<< "694.7512291393 196.0000000000"
			<< "694.7679879937 295.0000000000"
			<< "694.7847470503 396.0000000000"
			<< "694.8015063090 399.0000000000"
			<< "694.8182657698 369.0000000000"
			<< "694.8350254327 345.0000000000" ;

		int xyFormatStringListSize = xyFormatStringList.size();
		QString xyFormatString = xyFormatStringList.join("\n");
		MassSpectrum *massSpecIdx0 = new MassSpectrum;
		massSpecIdx0->initialize(xyFormatString);
		REQUIRE(xyFormatStringListSize == massSpecIdx0->size());

		xyFormatStringList.clear();
		xyFormatStringList
			<< "694.5166130065 186.0000000000"
			<< "694.5333690308 264.0000000000"
			<< "694.5501252573 288.0000000000"
			<< "694.5668816859 264.0000000000"
			<< "694.5836383166 276.0000000000"
			<< "694.6003951495 303.0000000000"
			<< "694.6171521845 267.0000000000"
			<< "694.6339094216 189.0000000000"
			<< "694.6506668608 170.0000000000"
			<< "694.6674245022 165.0000000000"
			<< "694.6841823457 173.0000000000"
			<< "694.7009403913 159.0000000000"
			<< "694.7176986390 180.0000000000"
			<< "694.7344570889 212.0000000000"
			<< "694.7512157409 245.0000000000"
			<< "694.7679745950 299.0000000000"
			<< "694.7847336512 357.0000000000"
			<< "694.8014929096 381.0000000000"
			<< "694.8182523701 343.0000000000"
			<< "694.8350120327 285.0000000000" ;

		xyFormatStringListSize = xyFormatStringList.size();
		xyFormatString = xyFormatStringList.join("\n");
		MassSpectrum *massSpecIdx1 = new MassSpectrum;
		massSpecIdx1->initialize(xyFormatString);
		REQUIRE(xyFormatStringListSize == massSpecIdx1->size());

		xyFormatStringList.clear();
		xyFormatStringList 
			<< "694.5166130065 204.0000000000"
			<< "694.5333690308 277.0000000000"
			<< "694.5501252573 300.0000000000"
			<< "694.5668816859 286.0000000000"
			<< "694.5836383166 273.0000000000"
			<< "694.6003951495 276.0000000000"
			<< "694.6171521845 268.0000000000"
			<< "694.6339094216 235.0000000000"
			<< "694.6506668608 223.0000000000"
			<< "694.6674245022 195.0000000000"
			<< "694.6841823457 220.0000000000"
			<< "694.7009403913 233.0000000000"
			<< "694.7176986390 249.0000000000"
			<< "694.7344570889 247.0000000000"
			<< "694.7512157409 263.0000000000"
			<< "694.7679745950 299.0000000000"
			<< "694.7847336512 335.0000000000"
			<< "694.8014929096 354.0000000000"
			<< "694.8182523701 317.0000000000"
			<< "694.8350120327 249.0000000000" ;

		xyFormatStringListSize = xyFormatStringList.size();
		xyFormatString = xyFormatStringList.join("\n");
		MassSpectrum *massSpecIdx2 = new MassSpectrum;
		massSpecIdx2->initialize(xyFormatString);
		REQUIRE(xyFormatStringListSize == massSpecIdx2->size());

		xyFormatStringList.clear();
		xyFormatStringList
			<< "694.5165997465 215.0000000000"
			<< "694.5333557706 264.0000000000"
			<< "694.5501119967 292.0000000000"
			<< "694.5668684250 285.0000000000"
			<< "694.5836250554 296.0000000000"
			<< "694.6003818880 310.0000000000"
			<< "694.6171389226 304.0000000000"
			<< "694.6338961594 256.0000000000"
			<< "694.6506535983 232.0000000000"
			<< "694.6674112394 212.0000000000"
			<< "694.6841690826 235.0000000000"
			<< "694.7009271278 255.0000000000"
			<< "694.7176853753 284.0000000000"
			<< "694.7344438248 268.0000000000"
			<< "694.7512024765 272.0000000000"
			<< "694.7679613303 304.0000000000"
			<< "694.7847203862 316.0000000000"
			<< "694.8014796442 304.0000000000"
			<< "694.8182391044 288.0000000000"
			<< "694.8349987667 250.0000000000" ;

		xyFormatStringListSize = xyFormatStringList.size();
		xyFormatString = xyFormatStringList.join("\n");
		MassSpectrum *massSpecIdx3 = new MassSpectrum;
		massSpecIdx3->initialize(xyFormatString);
		REQUIRE(xyFormatStringListSize == massSpecIdx3->size());

		xyFormatStringList.clear();
		xyFormatStringList
			<< "694.5165997465 237.0000000000"
			<< "694.5333557706 261.0000000000"
			<< "694.5501119967 304.0000000000"
			<< "694.5668684250 312.0000000000"
			<< "694.5836250554 297.0000000000"
			<< "694.6003818880 279.0000000000"
			<< "694.6171389226 288.0000000000"
			<< "694.6338961594 268.0000000000"
			<< "694.6506535983 269.0000000000"
			<< "694.6674112394 238.0000000000"
			<< "694.6841690826 248.0000000000"
			<< "694.7009271278 242.0000000000"
			<< "694.7176853753 250.0000000000"
			<< "694.7344438248 256.0000000000"
			<< "694.7512024765 266.0000000000"
			<< "694.7679613303 276.0000000000"
			<< "694.7847203862 270.0000000000"
			<< "694.8014796442 269.0000000000"
			<< "694.8182391044 281.0000000000"
			<< "694.8349987667 262.0000000000" ;

		xyFormatStringListSize = xyFormatStringList.size();
		xyFormatString = xyFormatStringList.join("\n");
		MassSpectrum *massSpecIdx4 = new MassSpectrum;
		massSpecIdx4->initialize(xyFormatString);
		REQUIRE(xyFormatStringListSize == massSpecIdx4->size());

		xyFormatStringList.clear();

		QList<MassSpectrum *> massSpectra;
		massSpectra << massSpecIdx0 << massSpecIdx1 << massSpecIdx2 << massSpecIdx3 << massSpecIdx4;

		SECTION("Test construction of the mass spectrum list.", "[MassSpectrumList]")
		{
			MassSpectrumList copyList(massSpectra);

			REQUIRE(copyList.size() == massSpectra.size());

			int failedComparisons = 0;

			for(int iter = 0; iter < copyList.size(); ++iter)
			{
				if(copyList.at(iter) != massSpectra.at(iter))
					++failedComparisons;
			}

			REQUIRE(failedComparisons == 0);
		}

		SECTION("Test destruction of the mass spectra.", "[MassSpectrumList]")
		{
			MassSpectrumList copyList(massSpectra);

			copyList.deleteSpectra();
			REQUIRE(copyList.size() == 0);
		}

		SECTION("Test the mz min max feature.", "[MassSpectrumList]")
		{
			MassSpectrumList copyList(massSpectra);

			// Take advantage to check the min max feature.
			double min = 0;
			double max = 0;

			copyList.mzMinMax(&min, &max);

			REQUIRE(min == 694.5165997465);
			REQUIRE(max == 694.8350254327);
		}

		SECTION("Test if the spectra of the list are of the same length.", "[MassSpectrumList]")
		{
			MassSpectrumList copyList(massSpectra);
			REQUIRE(copyList.areSpectraSameLength() == true);

			// Now start removing peaks from some of the spectra.

			// 1 peak less in first spectrum.

			massSpecIdx0->takeFirst();

			REQUIRE(copyList.areSpectraSameLength() == false);
		}

		SECTION("Test the total ion current intensity calculation "
				"without mz range.", "[MassSpectrumList]")
		{
			MassSpectrumList copyList(massSpectra);
			REQUIRE(copyList.areSpectraSameLength() == true);

			// Calculate it manually here.
			
			double tic = 0;

			for(int iter = 0; iter < copyList.size(); ++iter)
			{
				tic += copyList.at(iter)->tic();
			}

			REQUIRE(copyList.tic() == tic);
			REQUIRE(MassSpectrumList::tic(copyList) == tic);
		}
				
		SECTION("Test the total ion current intensity calculation "
				"without mz range.", "[MassSpectrumList]")
		{
			MassSpectrumList copyList(massSpectra);
			REQUIRE(copyList.areSpectraSameLength() == true);

			// Calculate it manually here.
			
			double mzStart = 694.5836250554;
			double mzEnd = 694.7847203862;

			double tic = 0;

			for(int iter = 0; iter < copyList.size(); ++iter)
			{
				tic += copyList.at(iter)->tic(mzStart, mzEnd);
			}

			REQUIRE(copyList.tic(mzStart, mzEnd) == tic);
		}
				





	}
	// End of
	// TEST_CASE("Construct a mass spectrum list as a copy of another list "
	//		"and test its features.", "[MassSpectrumList]")







	//TEST_CASE("", "[MassPeak]")
	//{

	//}

} // namespace msXpSlibmass
