/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



/////////////////////// Qt includes
#include <QPushButton>


/////////////////////// Local includes
#include "globals.hpp"
#include "TicChromWnd.hpp"
#include "Application.hpp"
#include "MainWindow.hpp"
#include "libmass/MsMultiHash.hpp"
#include "MassSpecDataFileLoaderPwiz.hpp"
#include "MassSpecDataFileLoaderDx.hpp"
#include "MassSpecDataFileLoaderMs1.hpp"
#include "MassDataIntegrator.hpp"


#include <omp.h>

namespace msXpSmineXpert
{


	//! Initialize the widget count to 0.
	int TicChromWnd::widgetCounter = 0;



	//! Construct an initialized TicChromWnd instance.
	/*!

		This function calls initialize() to perform the initialization of the
		widgets.

		\param parent parent widget.

*/
	TicChromWnd::TicChromWnd(QWidget *parent)
		: AbstractMultiPlotWnd{parent, "mineXpert" /*m_name*/,
			"TIC-chromatogram" /*m_desc*/}
	{
		initialize();
	}


	//! Destruct \c this TicChromWnd instance.
	TicChromWnd::~TicChromWnd()
	{
	}


	//! Initialize the various widgets. Calls initializePlotRegion().
	void
		TicChromWnd::initialize()
		{
			initializePlotRegion();

			show();
		}


	//! Perform the initialization of the plot region of the window.
	void
		TicChromWnd::initializePlotRegion()
		{
			// There are two regions in the window. The upper part of the mp_splitter
			// is gonna receive the multi-graph plotwidget. The lower part of the
			// splitter will receive all the individual plot widgets.

			// We will need later to know that this plot widget is actually for
			// displaying many different graphs, contrary to the normal situation. See
			// the true parameter in the function call below.

			mp_multiGraphPlotWidget =
				new TicChromPlotWidget(this, this, m_name, m_desc, Q_NULLPTR /* MassSpecDataSet * */,
						QString() /* fileName */, true /* isMultiGraph */);
			mp_multiGraphPlotWidget->setObjectName("multiGraphPlot");


			// We need to remove the single graph that was created upon creation of
			// the QCustomPlot widget because we'll later add graphs, and we want to
			// have a proper correlation between the index of the graphs and their
			// pointer value. If we did not remove that graph "by default", then there
			// would be a gap of 1 between the index of the first added mass graph and
			// its expected value.
			mp_multiGraphPlotWidget->clearGraphs();

			// Ensure that there is a connection between this plot widget and this
			// containing window, such that if there is a requirement to replot, the
			// widget gets the signal.
			connect(this, &TicChromWnd::replotWithAxisRange, mp_multiGraphPlotWidget,
					&TicChromPlotWidget::replotWithAxisRange);

			// This window is responsible for the writing of the analysis stanzas to
			// the file. We need to catch the signals.
			connect(mp_multiGraphPlotWidget, &TicChromPlotWidget::recordAnalysisStanza,
					this, &TicChromWnd::recordAnalysisStanza);

			// This window is responsible for displaying the key/value pairs of the
			// graphs onto which the mouse is moved. We need to catch the signal.
			connect(mp_multiGraphPlotWidget, &AbstractPlotWidget::updateStatusBarSignal, this,
					&AbstractMultiPlotWnd::updateStatusBar);

			// We want to create a scroll area in which the multigraph plot widget
			// will be sitting.
			QScrollArea *scrollArea = new QScrollArea();
			scrollArea->setObjectName(QStringLiteral("scrollArea"));
			scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
			scrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
			// We accept that the plots are resized, but we set a minimum size when we
			// create them.
			scrollArea->setWidgetResizable(true);
			scrollArea->setAlignment(Qt::AlignCenter);
			scrollArea->setWidget(mp_multiGraphPlotWidget);
			// And now prepend to the mp_splitter that new scroll area. Remember that
			// the mp_splitter had already a widget set to it, that is, the scroll
			// area that is now located below the mp_multiGraphPlotWidget, that is
			// responsible for the hosting of all the single-graph plot widgets.
			mp_splitter->insertWidget(0, scrollArea);
		}


	//! Add a new plot widget.
	/*!

		\param keyVector vector of double values to be used as the x-axis data.

		\param valVector vector of double values to be used as the y-axis data.

		\param desc string containg the description of the plot.

		\param massSpecDataSet the MassSpecDataSet instance in which the mass data
		have been stored in the first place.

		\param isMultiGraph tell if the plot widget will be plot widget into which
		multiple graph will be plotted.

		\return the allocated plot widget.

*/	
	TicChromPlotWidget *
		TicChromWnd::addPlotWidget(const QVector<double> &keyVector,
				const QVector<double> &valVector, const QString &desc,
				MassSpecDataSet *massSpecDataSet, bool isMultiGraph)
		{
			MainWindow *mainWindow = static_cast<MainWindow *>(parent());

			// Beware that massSpecDataSet might be Q_NULLPTR.
			QString fileName;
			if(massSpecDataSet != Q_NULLPTR)
				fileName = massSpecDataSet->fileName();

			TicChromPlotWidget *widget = 
				new TicChromPlotWidget(this, this, m_name, desc, 
						massSpecDataSet, fileName, isMultiGraph);

			widget->setObjectName("monoGraphPlot");


			///////////////// BEGIN Script availability of the new widget //////////////////

			// Craft a name that differentiates this widget from the previous and next
			// ones. We use the static counter number to affix to the string
			// "ticChrom".

			QString propName =
				QString("ticChromPlot%1").arg(TicChromWnd::widgetCounter);
			QString alias = "lastTicChromPlot";

			QString explanation = 
				QString("Make available a TIC chromatogram plot under name \"%1\" "
						"with alias \"%2\".\n").arg(propName).arg(alias);

			// Make the new plot widget globally available under the crafted property
			// name and alias. Also, put out the explanation text. Note that by
			// providing 'this', we make the new object appear in the scripting objet
			// tree widget as a child item of this.

			QScriptValue scriptValue;
			mainWindow->mp_scriptingWnd->publishQObject(widget, 
					this /* parent object */, 
					propName, alias, explanation, QString() /* help string */,
					scriptValue,
					QScriptEngine::QtOwnership, 
					QScriptEngine::QObjectWrapOptions());

			++TicChromWnd::widgetCounter;

			///////////////// END Script availability of the new widget //////////////////


			widget->setMinimumSize(400, 150);
			mp_scrollAreaVBoxLayout->addWidget(widget);

			widget->graph()->setData(keyVector, valVector);

			// Let's see if one color was set as the desired color, otherwise use
			// black.
			QPen currentPen = widget->graph()->pen();

			if(!m_wishedColor.isValid())
				currentPen.setColor(Qt::black);
			else
				currentPen.setColor(m_wishedColor);

			widget->graph()->setPen(currentPen);

			widget->rescaleAxes(true /*only enlarge*/);
			widget->replot();

			// Store in the history the new ranges of both x and y axes.
			widget->updateAxisRangeHistory();

			// At this point, make sure that the same data are represented on the
			// multi-graph plot widget.

			QCPGraph *newGraph = mp_multiGraphPlotWidget->addGraph();
			newGraph->setData(keyVector, valVector);
			newGraph->setPen(currentPen);
			mp_multiGraphPlotWidget->rescaleAxes(true /*only enlarge*/);
			mp_multiGraphPlotWidget->replot();

			// Maintain a hash connection between any given plot widget and the
			// corresponding graph in the multigraph widget.
			m_plotWidgetGraphMap.insert(widget, newGraph);

			// Finally append the new widget to the list of widgets that belong to
			// this window.
			m_plotWidgetList.append(widget);

			// Signal/slot connections.

			connect(widget, &TicChromPlotWidget::recordAnalysisStanza, this,
					&TicChromWnd::recordAnalysisStanza);

			connect(widget, &AbstractPlotWidget::updateStatusBarSignal, this,
					&AbstractMultiPlotWnd::updateStatusBar);

			return widget;
		}


	AbstractPlotWidget *
		TicChromWnd::initialTic(MassSpecDataSet *massSpecDataSet)
		{
			MassDataIntegrator integrator(massSpecDataSet);

			integrator.initialTic();

			History localHistory = integrator.history();
			
			// At this point what we should do is transfer the data to the plotting
			// widget creation routine. Note that we are creating the very first plot
			// widget with data just loaded from a mass spec file. That plot widget
			// will be later connected to the destruction of the mass spec data set
			// when the plot widget itself is deleted (see true param in newRelation
			// call below).

			AbstractPlotWidget *widget = addPlotWidget(
					integrator.keyVector(), integrator.valVector(), 
					massSpecDataSet->fileName(), massSpecDataSet);

			// For this specific TIC chrom widget, that is created for storing the
			// initial mass spec data set as read from file, we set it as parent of
			// the mass spec data set, so that when the widget is destroyed, the mass
			// spec data set also.
			massSpecDataSet->setParent(widget);

			// At this point, immediately craft a new HistoryItem instance to append
			// to the widget's History.

			widget->rhistory().copyHistory(localHistory, true /*remove duplicates*/);

			qDebug() << __FILE__ << __LINE__
				<< "After initialTic was computed, history:"
				<< widget->history().asText();

			// Log the history to the console.
			MainWindow *mainWindow = static_cast<MainWindow *>(parent());
			mainWindow->logConsoleMessage(widget->history().innermostRangesAsText());

			// The plot widget reacts to the Qt::Key_Delete key by calling a function
			// that emits a set of signals, of which the destroyPlotWidget signal
			// actually indirectly triggers the destruction of the plot widget. This
			// signal is now connected to the main window where a function will call
			// for the destruction of all the target plot widgets, that is all the
			// plot widgets that were created as the result of an integration
			// computation of the plot widget currently being destroyed.

			connect(widget, &AbstractPlotWidget::destroyPlotWidget, mainWindow,
					&MainWindow::destroyAlsoAllTargetPlotWidgets);

			connect(widget, &AbstractPlotWidget::toggleMultiGraphSignal,
					mainWindow, &MainWindow::toggleMultiGraph);

			// Ensure that there is a connexion between this new widget and the
			// MassSpecWnd instance that will receive the signal requiring to make a
			// mass spectrum on the basis of an integration selection in the TIC
			// chromatogram plot.

			connect(static_cast<TicChromPlotWidget *>(widget),
					&TicChromPlotWidget::newMassSpectrum,
					mainWindow->mp_massSpecWnd, &MassSpecWnd::newMassSpectrum);

			connect(static_cast<TicChromPlotWidget *>(widget),
					&TicChromPlotWidget::newDriftSpectrum,
					mainWindow->mp_driftSpecWnd, &DriftSpecWnd::newDriftSpectrum);

			// Ensure that there is a connection between this plot widget and this
			// containing window, such that if there is a requirement to replot, the
			// widget gets the signal.
			connect(this, &TicChromWnd::replotWithAxisRange, widget,
					&TicChromPlotWidget::replotWithAxisRange);

			// Make sure we get the signal requiring to redraw the background of the
			// widget, depending on it being focused or not.
			connect(this, &AbstractMultiPlotWnd::redrawPlotBackground, widget,
					&AbstractPlotWidget::redrawPlotBackground);

			return widget;
		}


	//! Integrate data to TIC chromatogram starting from a mass spectrum.
	/*!

		\param massSpecDataSet pointer to the MassSpecDataSet where the mass data
		are stored.

		\param integrationType IntegrationType of the integration to be performed.

		\param history History to account for the integration.

		\param color to use to plot the graph of the new mass spectrum.

*/
	void
		TicChromWnd::newTicChromatogram(MassSpecDataSet *massSpecDataSet,
				const QString &msg, History history, QColor color)
		{
			if(massSpecDataSet == Q_NULLPTR)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			m_wishedColor = color;

			// Make sure the cancel button is visible.
			mp_statusBarCancelPushButton->setVisible(true);

			MassDataIntegrator integrator(massSpecDataSet, history);

			connect(&integrator,
					&MassDataIntegrator::setupFeedbackSignal,
					this,
					&AbstractMultiPlotWnd::setupFeedback);

			connect(&integrator,
					&MassDataIntegrator::updateFeedbackSignal,
					this,
					&AbstractMultiPlotWnd::updateFeedback);

			connect(this,
					&AbstractMultiPlotWnd::cancelOperationSignal,
					&integrator,
					&MassDataIntegrator::cancelOperation);

			integrator.integrate();

			qInfo() << "At this point the integration is finished. Desetup the feedback.";

			//De-setup the feedback.
			QTimer::singleShot(2000, this, [=]() 
					{ 
					setupFeedback("", -1, -1, -1, LogType::LOG_TO_STATUS_BAR); 
					}
					);

			// Make sure the cancel button is hidden.
			mp_statusBarCancelPushButton->setVisible(false);

			// Create a text item to indicate the last history item.

			QString historyItemText = history.newestHistoryItem()->asText(true /* brief */);

			QString plotText = QString("%1\n%2 - %3 ")
				.arg(massSpecDataSet->fileName())
				.arg(m_desc)
				.arg(historyItemText);

			// Allocate the new plot widget in which the data are to be plot.

			TicChromPlotWidget *widget = static_cast<TicChromPlotWidget *>(
					addPlotWidget(integrator.keyVector(), integrator.valVector(),
						plotText, massSpecDataSet));

			// The history that we get as a parameter needs to be appended to the new
			// widget's history so that we have a full account of the history.

			widget->rhistory().copyHistory(history, true /*remove duplicates*/);

			MainWindow *mainWindow = static_cast<MainWindow *>(parent());
			mainWindow->logConsoleMessage(widget->history().asText(
						"Plot widget history: \n", "End plot widget history:\n"));
			// And now only the innermost ranges of the whole history.
			mainWindow->logConsoleMessage(widget->history().innermostRangesAsText());

			// The plot widget reacts to the Qt::Key_Delete key by calling a function
			// that emits a set of signals, of which the destroyPlotWidget signal
			// actually indirectly triggers the destruction of the plot widget. This
			// signal is now connected to the main window where a function will call for
			// the destruction of all the target plot widgets, that is all the plot
			// widgets that were created as the result of an integration computation of
			// the plot widget currently being destroyed.

			connect(widget, &AbstractPlotWidget::destroyPlotWidget, mainWindow,
					&MainWindow::destroyAlsoAllTargetPlotWidgets);

			connect(widget, &AbstractPlotWidget::toggleMultiGraphSignal,
					mainWindow, &MainWindow::toggleMultiGraph);

			// Ensure that there is a connection between this plot widget and this
			// containing window, such that if there is a requirement to replot, the
			// widget gets the signal.

			connect(this, &AbstractMultiPlotWnd::replotWithAxisRange, widget,
					&AbstractPlotWidget::replotWithAxisRange);

			// Make sure we get the signal requiring to redraw the background of the
			// widget, depending on it being focused or not.

			connect(this, &AbstractMultiPlotWnd::redrawPlotBackground, widget,
					&AbstractPlotWidget::redrawPlotBackground);

			connect(widget, &TicChromPlotWidget::newMassSpectrum,
					mainWindow->mp_massSpecWnd, &MassSpecWnd::newMassSpectrum);

			connect(widget, &TicChromPlotWidget::newDriftSpectrum,
					mainWindow->mp_driftSpecWnd, &DriftSpecWnd::newDriftSpectrum);

			AbstractPlotWidget *senderPlotWidget =
				dynamic_cast<AbstractPlotWidget *>(QObject::sender());

			if(senderPlotWidget == Q_NULLPTR)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			mainWindow->m_dataPlotWidgetRelationer.newRelation(
					massSpecDataSet, senderPlotWidget, widget,
					Q_NULLPTR /* open spectra dialog list widget item */);
		}


	//! Toggle the visibility of the multigraph graph and the target widgets.
	/*!

		The base class simply toggles the visibility of the multigraph graph
		corresponding to the signal emitter plot widget. This overridden function
		does more, not only does it toggle the visibility of the multigraph graph
		corresponding to the sender plot widget, but does the same for all the
		target plot widgets. 

		This is useful in the case where there are many opened spectra in the
		program, thus many initial (top) TIC chromatogram plot widget that may have
		given rise to many mass spectra/drift spectra. It is sometimes useful to
		clear up the multigraph plot widgets in the mass and drift spectrum
		multigraph plot widgets by hiding/showing all the multigraph graphs relating
		to the sender plot widget.

*/
	void
		TicChromWnd::toggleMultiGraph()
		{
			AbstractPlotWidget *ticChromPlotWidget =
				dynamic_cast<AbstractPlotWidget *>(QObject::sender());
			QCPGraph *graph = m_plotWidgetGraphMap.value(ticChromPlotWidget);

			if(graph == Q_NULLPTR)
				return;

			// First toggle visibility of this plot widget graph.
			if(graph->visible())
				graph->setVisible(false);
			else
				graph->setVisible(true);

			// And now ask for a replot of the multi-graph plot widget.
			mp_multiGraphPlotWidget->replot();

			// But we know that each tic chrom plot widget might have a number of mass
			// spec window plot widget that derive from it by way of mass spectrum
			// combination or also any number of drift spectra by the same integration
			// mechanism. All these plot widgets must be have
			// their visibility toggled also. That is, we need to get a recursive
			// search list of all the plot widget targets that were directly or
			// indirectly generated from this ticChromPlotWidget.

			MainWindow *mainWindow = static_cast<MainWindow *>(parent());

			QList<AbstractPlotWidget *> plotWidgetList;
			mainWindow->m_dataPlotWidgetRelationer.targetsRecursive(ticChromPlotWidget,
					&plotWidgetList);

			for(int iter = 0; iter < plotWidgetList.size(); ++iter)
			{
				plotWidgetList.at(iter)->toggleMultiGraph();
			}
		}


	//! Script-invocable override of the base class function.
	void
		TicChromWnd::clearPlots()
		{
			AbstractMultiPlotWnd::clearPlots();
		}



	//! Record an analysis stanza.
	/*!

		\param stanza string to print out in the record.

		\param color color to use for the printout.

*/
	void
		TicChromWnd::recordAnalysisStanza(const QString &stanza, const QColor &color)
		{
			// The record might go to a file or to the console or both.

			if((mp_analysisPreferences->m_recordTarget &
						RecordTarget::RECORD_TO_FILE) == RecordTarget::RECORD_TO_FILE)
			{
				if(mp_analysisFile != Q_NULLPTR)
				{

					// Write the stanza, that was crafted by the calling plot widget to
					// the file.

					if(!mp_analysisFile->open(QIODevice::Append))
					{
						statusBar()->showMessage(
								QString("Could not record the step because "
									"the file could not be opened."),
								4000);
					}
					else
					{
						mp_analysisFile->write(stanza.toLatin1());
						// Force writing because we may want to have tail -f work fine
						// on the
						// file, and see modifications live to change fiels in a text
						// editor.
						mp_analysisFile->flush();
						mp_analysisFile->close();
					}
				}
				else
				{
					qDebug() << __FILE__ << __LINE__
						<< "The mp_analysisFile pointer is Q_NULLPTR.";

					statusBar()->showMessage(
							QString("Could not record the analysis step to file. "
								"Please define a file to write the data to."),
							4000);
				}
			}

			// Also, if recording to the console is asked for, then do that also.
			if((mp_analysisPreferences->m_recordTarget &
						RecordTarget::RECORD_TO_CONSOLE) == RecordTarget::RECORD_TO_CONSOLE)
			{
				MainWindow *mainWindow = static_cast<MainWindow *>(parent());
				mainWindow->logConsoleMessage(stanza, color);
			}

			// Also, if recording to the clipboard is asked for, then do that also.
			if((mp_analysisPreferences->m_recordTarget &
						RecordTarget::RECORD_TO_CLIPBOARD) == RecordTarget::RECORD_TO_CLIPBOARD)
			{
				QClipboard *clipboard = QApplication::clipboard();
				clipboard->setText(clipboard->text() + stanza, QClipboard::Clipboard);
			}

			return;
		}


} // namespace msXpSmineXpert
