/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



///////////////////////////// Qt include


///////////////////////////// Local includes
#include "TicChromPlotWidget.hpp"
#include "TicChromWnd.hpp"
#include "AbstractMultiPlotWnd.hpp"
#include "MassSpecSqlite3Handler.hpp"
#include "AnalysisPreferences.hpp"
#include "MainWindow.hpp"
#include "History.hpp"
#include "MassDataIntegrator.hpp"


namespace msXpSmineXpert
{


	//! Construct an initialized TicChromPlotWidget instance.
	/*!

		This function registers the key codes that, in conjunction to mouse button
		clicks trigger specific actions, like data integrations.

		\param parent parent widget.

		\param parentWnd parent window.

		\param name name of the module.

		\param desc description of the wiget type.

		\param massSpecDataSet MassSpecDataSet holding the mass data.

		\param fileName name of the file from which the data were loaded.

		\param isMultiGraph tells if the widget to be instanciated is for displaying
		multiple graphs.

*/ 
	TicChromPlotWidget::TicChromPlotWidget(QWidget *parent, QWidget *parentWnd,
			const QString &name, const QString &desc,
			MassSpecDataSet *massSpecDataSet, const QString &fileName,
			bool isMultiGraph)
		: AbstractPlotWidget{parent, parentWnd, 
			name, desc, 
			massSpecDataSet, fileName, isMultiGraph}
	{
		registerQtKey(Qt::Key_S); // to mass spectrum : MZ
		registerQtKey(Qt::Key_D); // drift spectrum : DT
		registerQtKey(Qt::Key_I); // to intensity: I

		// Give the axes some labels:
		xAxis->setLabel("time (min)");
		yAxis->setLabel("tic");
	}


	//! Destruct \c this TicChromPlotWidget instance.
	TicChromPlotWidget::~TicChromPlotWidget()
	{
		// No need to delete the mp_massSpecDataSet
	}


	//! Craft a string representing the data of the peak that was analysed.
	/*!

		To do so, this functions get the data format string from the parent window's
		AnalysisPreferences instance. The format string is interpreted and the
		relevant values are substituted to craft a stanza (a paragraph) that matches
		the peak data at hand.

		\return a string containing the stanza.
		*/
	QString
		TicChromPlotWidget::craftAnalysisStanza(QCPGraph *theGraph,
				const QString &fileName)
		{

			// This function might be called for a graph that sits in a single-graph
			// plot widget or for a graph that sits in a multi-graph plot widget.
			// Depending on the parameters, we'll know what is the case. If the
			// parameters are not nullptr or empty, then the plot widget is a
			// multi-graph plot widget and we need to set the data in the analysis
			// record only for the graph
			// passed as parameter.

			QCPGraph *origGraph = nullptr;

			if(theGraph == nullptr)
				origGraph = graph();
			else
				origGraph = theGraph;

			QString destFileName;

			if(fileName.isEmpty())
				destFileName = m_fileName;
			else
				destFileName = fileName;


			// When the mouse moves over the plot widget, its position is recorded
			// real time. That position is both a m/z value and an i (intensity)
			// value. We cannot rely on the i value, because it is valid only the
			// for the last added graph. But the m/z value is the same whatever the
			// graph we are interested in.

			// Craft the analysis stanza:
			QString stanza;

			TicChromWnd *wnd = dynamic_cast<TicChromWnd *>(mp_parentWnd);
			if(wnd == nullptr)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			const AnalysisPreferences *analPrefs = wnd->analysisPreferences();
			if(analPrefs == nullptr)
				return stanza;

			// The way we work here is that we get a format string that specifies how
			// the user wants to have the data formatted in the stanza. That format
			// string is located in a DataFormatStringSpecif object as the m_format
			// member. There is also a m_formatType member that indicates what is the
			// format that we request (mass spec, tic chrom or drift spec). That
			// format type member is an int that also is the key of the hash that is
			// located in the analPrefs: QHash<int, DataFormatStringSpecif *>
			// m_dataFormatStringSpecifHash.

			DataFormatStringSpecif *specif =
				analPrefs->m_dataFormatStringSpecifHash.value(FormatType::DRIFT_SPEC);

			if(specif == nullptr)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			// Handy local copy.
			QString formatString = specif->m_format;

			QChar prevChar = ' ';

			for(int iter = 0; iter < formatString.size(); ++iter)
			{
				QChar curChar = formatString.at(iter);

				// qDebug() << __FILE__ << __LINE__
				// << "Current char:" << curChar;

				if(curChar == '\\')
				{
					if(prevChar == '\\')
					{
						stanza += '\\';
						prevChar = ' ';
						continue;
					}
					else
					{
						prevChar = '\\';
						continue;
					}
				}

				if(curChar == '%')
				{
					if(prevChar == '\\')
					{
						stanza += '%';
						prevChar = ' ';
						continue;
					}
					else
					{
						prevChar = '%';
						continue;
					}
				}

				if(curChar == 'n')
				{
					if(prevChar == '\\')
					{
						stanza += QString("\n");

						// Because a newline only works if it is followed by something,
						// and
						// if the user wants a termination newline, then we need to
						// duplicate that new line char if we are at the end of the
						// string.

						if(iter == formatString.size() - 1)
						{
							// This is the last character of the line, then, duplicate
							// the
							// newline so that it actually creates a new line in the
							// text.

							stanza += QString("\n");
						}

						prevChar = ' ';
						continue;
					}
				}

				if(prevChar == '%')
				{
					// The current character might have a specific signification.
					if(curChar == 'f')
					{
						QFileInfo fileInfo(destFileName);
						if(fileInfo.exists())
							stanza += fileInfo.fileName();
						else
							stanza += "Untitled";
						prevChar = ' ';
						continue;
					}
					if(curChar == 'X')
					{
						double startDt = m_startDragPoint.x();

						stanza += QString("%1").arg(startDt, 0, 'g', 6);
						prevChar = ' ';
						continue;
					}
					if(curChar == 'Y')
					{
						// We need to get the value of the intensity for the graph.
						double startDt = m_startDragPoint.x();
						double intensity = getYatX(startDt, origGraph);

						if(!intensity)
							qDebug() << __FILE__ << __LINE__
								<< "Warning, intensity for dt " << startDt
								<< "is zero.";
						else
							stanza += QString("%1").arg(intensity, 0, 'g', 6);

						prevChar = ' ';
						continue;
					}
					if(curChar == 'x')
					{
						stanza += QString("%1").arg(m_xDelta, 0, 'g', 6);
						prevChar = ' ';
						continue;
					}
					if(curChar == 'y')
					{
						stanza += "this is the drift count delta y";

						prevChar = ' ';
						continue;
					}

					// At this point the '%' is not followed by any special character
					// above, so we skip them both from the text. If the '%' is to be
					// printed, then it needs to be escaped.

					continue;
				}
				// End of
				// if(prevChar == '%')

				// The character prior this current one was not '%' so we just append
				// the current character.
				stanza += curChar;
			}
			// End of
			// for (int iter = 0; iter < pattern.size(); ++iter)

			//qDebug() << __FILE__ << __LINE__ << "Returning stanza: " << stanza;

			return stanza;
		}


	//! Handler for the mouse release event.
	/*!

		This function checks for the pressed keyboard key code and, depending on
		that key code, triggers specific actions, like integrating data to drift
		spectrum or to retention time (XIC) or to single TIC intensity value.

*/
	void
		TicChromPlotWidget::mouseReleaseHandledEvent(QMouseEvent *event)
		{
			if(m_pressedKeyCode == Qt::Key_S && event->button() == Qt::RightButton)
			{
				integrateToMz(xRangeMin(), xRangeMax());
			}

			if(m_pressedKeyCode == Qt::Key_D && event->button() == Qt::RightButton)
			{
				integrateToDt(xRangeMin(), xRangeMax());
			}

			if(m_pressedKeyCode == Qt::Key_I && event->button() == Qt::RightButton)
			{
				integrateToTicIntensity(xRangeMin(), xRangeMax());
			}
		}


	//! Perform a ranged [\p lower -- \p upper] data integration to a mass spectrum.
	void
		TicChromPlotWidget::integrateToMz(double lower, double upper)
		{
			// Get the pointer to the x-axis (key axis).

			QCPAxis *xAxis = graph()->keyAxis();

			double rangeStart = lower; 
			if(qIsNaN(rangeStart))
			{
				rangeStart = xAxis->range().lower;
			}

			double rangeEnd = upper;

			if(qIsNaN(rangeEnd))
			{
				rangeEnd = xAxis->range().upper;
			}

			History localHistory = m_history;
			HistoryItem *histItem = new HistoryItem;
			histItem->newIntegrationRange(IntegrationType::RT_TO_MZ, rangeStart, rangeEnd);

			// Aggreate the new item to the history.
			localHistory.appendHistoryItem(histItem);

			emit newMassSpectrum(mp_massSpecDataSet, "Calculating mass spectrum.",
					localHistory, graph()->pen().color());
		}


	//! Perform a ranged [\p lower -- \p upper] data integration to a drift spectrum.
	void
		TicChromPlotWidget::integrateToDt(double lower, double upper)
		{	
			// Get the pointer to the x-axis (key axis).

			QCPAxis *xAxis = graph()->keyAxis();

			double rangeStart = lower; 
			if(qIsNaN(rangeStart))
			{
				rangeStart = xAxis->range().lower;
			}

			double rangeEnd = upper;
			if(qIsNaN(rangeEnd))
			{
				rangeEnd = xAxis->range().upper;
			}

			History localHistory = m_history;
			HistoryItem *histItem = new HistoryItem;
			histItem->newIntegrationRange(IntegrationType::RT_TO_DT, rangeStart, rangeEnd);

			// Aggreate the new item to the history.
			localHistory.appendHistoryItem(histItem);

			emit newDriftSpectrum(mp_massSpecDataSet, "Calculating drift spectrum.",
					localHistory, graph()->pen().color());
		}


	//! Perform a ranged [\p lower -- \p upper] data integration to a single TIC intensity value.
	double
		TicChromPlotWidget::integrateToTicIntensity(double lower, double upper)
		{
			// Get the pointer to the x-axis (key axis).

			QCPAxis *xAxis = graph()->keyAxis();

			double rangeStart = lower; 
			if(qIsNaN(rangeStart))
			{
				rangeStart = xAxis->range().lower;
			}

			double rangeEnd = upper;
			if(qIsNaN(rangeEnd))
			{
				rangeEnd = xAxis->range().upper;
			}

			// Reset the axis range data because we may return prematurely and we want
			// to craft a stanza with proper values (even if they are qINan()).
			m_keyRangeStart = qSNaN();
			m_keyRangeEnd = qSNaN();

			// Also reset the other datum that might hold invalid values.
			m_lastI = qSNaN();

			if(rangeStart == rangeEnd)
			{
				return false;
			}

			// Slot for sorting stuff.
			double tempVal;

			// Set the values in sorted order for later stanza crafting.
			if(rangeStart > rangeEnd)
			{
				tempVal = rangeStart;
				rangeStart = rangeEnd;
				rangeStart = tempVal;
			}

			History localHistory = m_history;
			HistoryItem *histItem = new HistoryItem;
			histItem->newIntegrationRange(IntegrationType::RT_TO_TIC_INT, rangeStart, rangeEnd);

			// At this point store the range data so that the user may use it in
			// analysis reporting.
			m_keyRangeStart = rangeStart;
			m_keyRangeEnd = rangeEnd;

			// Aggreate the new item to the history.
			localHistory.appendHistoryItem(histItem);

			TicChromWnd *parentWnd = static_cast<TicChromWnd *>(mp_parentWnd);

			// Capture immediately the text that is located in the status bar
			// because if the computation is long that text will have disappeared
			// and we won't be able to show it in the TIC int text.

			QString currentMessage = parentWnd->statusBar()->currentMessage();

			QApplication::setOverrideCursor(QCursor(Qt::BusyCursor));

			MassDataIntegrator integrator(mp_massSpecDataSet, localHistory);

			integrator.integrateToTicIntensity();

			m_lastI = integrator.ticIntensity();

			QApplication::restoreOverrideCursor();

			if(currentMessage.isEmpty())
				parentWnd->statusBar()->showMessage(QString("range [%1-%2]: TIC I = %3")
						.arg(m_keyRangeStart, 0, 'f', 3)
						.arg(m_keyRangeEnd, 0, 'f', 3)
						.arg(m_lastI, 0, 'g', 3));
			else
				parentWnd->statusBar()->showMessage(currentMessage +
						QString(" ; range [%1-%2]: TIC I = %3")
						.arg(m_keyRangeStart, 0, 'f', 3)
						.arg(m_keyRangeEnd, 0, 'f', 3)
						.arg(m_lastI, 0, 'g', 3));

			return m_lastI;
		}


	//! Key release event handler.
	/*! 

		Depending on the type of plot widget (multigraph or not), the actions
		triggered in this function will differ.

		For example, the space key will trigger the crafting of an analysis stanza
		(see craftAnalysisStanza()). The 'L' key triggers the printout of the
		currently pointed graph point data to the console window. The TAB key will
		trigger the cycling of the selected item in the open spectra dialog window
		(see OpenSpectraDlg).

*/
	void
		TicChromPlotWidget::keyReleaseEvent(QKeyEvent *event)
		{
			TicChromWnd *parentWnd = static_cast<TicChromWnd *>(mp_parentWnd);
			MainWindow * mainWindow = static_cast<MainWindow *>(parentWnd->parent());

			// First, check if the space bar was pressed.
			if(event->key() == Qt::Key_Space)
			{
				// The handling of the stanza creation is different in these two
				// situations:
				// 1. the plot widget where this event occurred is a conventional
				// mono-graph plot widget that sits in the lower part of the window. In
				// this case, the filename of the spectrum data file is available.
				// 2. the plot widget is a multi-graph plot widget that sits in the
				// upper part of the multi plot window. In that case, it is more
				// difficult to know what is the file that initially contained the data
				// plotted. We want to know what graph corresponds to what list widget
				// item that is selected (or not) in the open mass spectra dialog window
				// (see MainWindow).

				QString stanza;
				if(m_isMultiGraph)
				{
					// The event occurred in the multi-graph plot widget where many
					// graphs
					// are displayed. We need to craft a stanza for each graph that
					// replicates the data of a plot that is proxied in the list widget
					// of
					// the OpenSpectraDlg. Only handle the spectra of which the list
					// widget item is currently selected and for which the corresponding
					// mass multi-graph plot is visible.

					for(int iter = 0; iter < graphCount(); ++iter)
					{
						QCPGraph *iterGraph = graph(iter);

						// Only work on this graph if it is visible:
						if(!iterGraph->visible())
							continue;

						// Get the color of the graph, so that we can print the stanza
						// in
						// the console window with the proper color.
						QPen curPen = iterGraph->pen();
						QColor curColor = curPen.color();

						// Get a handle to the conventional plot widget of which this
						// multi-graph plot widget's graph is a replica.

						AbstractPlotWidget *ticChromPlotWidget =
							parentWnd->monoGraphPlot(iterGraph);

						if(ticChromPlotWidget == nullptr)
						{
							qFatal("Fatal error at %s@%d -- %s. "
									"Cannot be that a multiGraph plot widget has not corresponding monoGraphPlot."
									"Program aborted.",
									__FILE__, __LINE__, __FUNCTION__);
						}

						// At this point, we know what plot widget in the Tic chrom
						// window
						// is at the origin of the graph currently iterated into. We
						// need to
						// know if that plot widget has a corresponding item in the open
						// spectrum dialog list widget that is currently selected.

						if(mainWindow->mp_openSpectraDlg->isPlotWidgetSelected(ticChromPlotWidget))
						{
							// This means that we have to get the filename out of this
							// tic
							// chrom plot widget's mass spec data set:
							QString fileName =
								static_cast<TicChromPlotWidget *>(ticChromPlotWidget)
								->mp_massSpecDataSet->fileName();

							stanza = craftAnalysisStanza(iterGraph, fileName);

							if(stanza.isEmpty())
							{
								parentWnd->statusBar()->showMessage(
										"Failed to craft an analysis stanza. "
										"Please set the analysis preferences.");

								event->accept();

								return;
							}
							else
							{
								emit recordAnalysisStanza(stanza, curColor);
							}
						}
					}
					// End of
					// for(int iter = 0; iter < graphCount(); ++iter)
				}
				// End of
				// if(m_isMultiGraph)
				else
				{
					// We are in a plot widget that is a single-graph plot widget, the
					// situation is easier:

					stanza = craftAnalysisStanza();

					if(stanza.isEmpty())
					{
						parentWnd->statusBar()->showMessage(
								"Failed to craft an analysis stanza. "
								"Please set the analysis preferences.");

						event->accept();

						return;
					}
					else
					{
						// Send the stanza to the console window, so that we can make a
						// copy
						// paste.

						// Get the color of the graph, so that we can print the stanza
						// in
						// the console window with the proper color.
						QPen curPen = graph()->pen();
						QColor curColor = curPen.color();

						emit recordAnalysisStanza(stanza, curColor);
					}
				}

				event->accept();

				return;
			}
			else if(event->key() == Qt::Key_L)
			{
				// The user wants to copy the current cursor location to the console
				// window, such that it remains available after having moved the cursor.
				// The handling of the text creation is different in these two
				// situations:
				//
				// 1. the plot widget where this event occurred is a conventional
				// mono-graph plot widget that sits in the lower part of the window. In
				// this case, the color to be used to display the data label in the
				// console window is easily gotten from the graph's pen.
				//
				// 2. the plot widget is a multi-graph plot widget that sits in the
				// upper part of the multi plot window. In that case, it is more
				// difficult to know what is the proper color to use, because we need to
				// go to the open spectra dialog and get the list of selected files that
				// initially contained the data plotted. We want to know what graph
				// corresponds to what list widget item that is selected (or not) in the
				// open mass spectra dialog window (see MainWindow).

				QString label;
				if(m_isMultiGraph)
				{

					// The event occurred in the multi-graph plot widget where many
					// graphs are displayed. We need to craft a label for each graph
					// that replicates the data of a plot that is proxied in the list
					// widget of the OpenSpectraDlg. Only handle the spectra of which
					// the list widget item is currently selected and for which the
					// corresponding mass multi-graph plot is visible.

					for(int iter = 0; iter < graphCount(); ++iter)
					{
						QCPGraph *iterGraph = graph(iter);

						// Only work on this graph if it is visible:
						if(!iterGraph->visible())
							continue;

						// Get the color of the graph, so that we can print the label in
						// the console window with the proper color.
						QPen curPen = iterGraph->pen();
						QColor curColor = curPen.color();

						// Get a handle to the conventional plot widget of which this
						// multi-graph plot widget's graph is a replica.

						AbstractPlotWidget *ticChromPlotWidget =
							parentWnd->monoGraphPlot(iterGraph);

						if(ticChromPlotWidget == nullptr)
						{
							qFatal("Fatal error at %s@%d -- %s. "
									"Cannot be that a multiGraph plot widget has not corresponding monoGraphPlot."
									"Program aborted.",
									__FILE__, __LINE__, __FUNCTION__);

							return;
						}

						// At this point, we know what plot widget in the Tic chrom
						// window is at the origin of the graph currently iterated into.
						// We need to know if that plot widget has a corresponding item
						// in the open spectrum dialog list widget that is currently
						// selected.

						if(mainWindow->mp_openSpectraDlg->isPlotWidgetSelected(
									ticChromPlotWidget))
						{
							// We can craft the label text to send to the console
							// window, and
							// that, with the proper color.

							label = QString("(%1,%2)\n")
								.arg(m_lastMousedPlotPoint.x())
								.arg(m_lastMousedPlotPoint.y());

							if(label.isEmpty())
							{
								parentWnd->statusBar()->showMessage(
										"Failed to craft a text label.");
								event->accept();
								return;
							}
							else
							{
								// Send the label to the console window, so that we can
								// make a copy
								// paste.
								mainWindow->logConsoleMessage(label, curColor);
							}
						}
					}
					// End of
					// for(int iter = 0; iter < graphCount(); ++iter)
				}
				// End of
				// if(m_isMultiGraph)
				else
				{
					// We are in a plot widget that is a single-graph plot widget, the
					// situation is easier:

					label = QString("(%1,%2)\n")
						.arg(m_lastMousedPlotPoint.x())
						.arg(m_lastMousedPlotPoint.y());

					if(label.isEmpty())
					{
						parentWnd->statusBar()->showMessage(
								"Failed to craft a text label. ");
						event->accept();
						return;
					}
					else
					{
						// Send the label to the console window, so that we can make a
						// copy
						// paste.

						// Get the color of the graph, so that we can print the stanza
						// in
						// the console window with the proper color.
						QPen curPen = graph()->pen();
						QColor curColor = curPen.color();

						mainWindow->logConsoleMessage(label, curColor);
					}
				}

				event->accept();

				return;
			}
			// End of
			// else if(event->key() == Qt::Key_C)

			// Now let the base class do its work.
			AbstractPlotWidget::keyReleaseEvent(event);
		}


	//! Return the arbitrary integration type that this plot widget can handle (IntegrationType::ANY_TO_RT).
	int 
		TicChromPlotWidget::arbitraryIntegrationType()
		{
			return IntegrationType::ANY_TO_RT;
		}

} 

// namespace msXpSmineXpert
