/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



/////////////////////// Qt includes
#include <QApplication> // for the clipboard
#include <QClipboard>
#include <QFile>
#include <QDebug>
#include <qmath.h>


/////////////////////// Local includes
#include "MassSpectrumList.hpp"


namespace msXpSlibmass
{



	//! Construct a totally empty MassSpectrumList
	/*!
	*/
	MassSpectrumList::MassSpectrumList()
		: QList{}
	{
	}


	//! Construct a MassSpectrumList initialized with a title
	/*!

		\param title title of the list
		*/
	MassSpectrumList::MassSpectrumList(const QString &title)
		: QList{}, m_title{title}
	{
	}


	//! Construct a MassSpectrumList initialized with MassSpectrum instances
	/*!

		\c this MassSpectrumList instance is initialized by appending all the
		MassSpectrum instances in the \p massSpectra parameter. Note that \c this
		MassSpecrumList does not take ownership of the MassSpectrum instances.

		\param massSpectra list of MassSpectrum instance pointers

*/
	MassSpectrumList::MassSpectrumList(const QList<MassSpectrum *> &massSpectra)
	{
		for(int iter = 0; iter < massSpectra.size(); iter++)
			append(massSpectra.at(iter));
	}


	//! Destruct the MassSpectrumList instance.
	/*!

		This function does not do anything because the allocated MassSpectrum
		instances does not belong to it.

*/
	MassSpectrumList::~MassSpectrumList()
	{
		// We cannot delete the MassSpectrum instances allocated on the heap here:
		// this class is used to get pointers to MassSpectrum instances that do not
		// belong to this object. The user is responsible for calling
		// deleteSpectra() at the might moment (typically from inside a
		// MassSpecDataSet object).  qDebug() << __FILE__ << __LINE__ << "Entering
		// MassSpectrumList::~MassSpectrumList";
	}


	//! Delete all the MassSpectrum instances stored in \c this MassSpectrumList.
	/*!

		Use this function with caution as MassSpectrum instances of which the
		pointers are stored in \c this MassSpectrumList instance do usually not
		belong to it.

*/
	void
		MassSpectrumList::deleteSpectra()
		{

			//qDebug() << __FILE__ << __LINE__
				//<< "Entering MassSpectrumList::deleteSpectra";

			// Iterate in the list of spectra and delete each one.
			for(int iter = 0; iter < size(); ++iter)
				delete at(iter);

			clear();
		}


	//! Get the name of the MassSpectrumList instance.
	QString 
		MassSpectrumList::title() const
		{
			return m_title;
		}


	//! Fill in the characteristics of \c this MassSpectrumList
	/*!

		The set of MassSpectrum instances stored in \c this MassSpectrumList is
		analyzed to extract characteristic data and to set them into the variables
		pointed to by the parameters:

		\param minMz the minimum m/z value of all the spectra. If nullptr, that bit of information is not set

		\param maxMz the maximum m/z value of all the spectra. If nullptr, that bit of information is not set

*/
	void 
		MassSpectrumList::mzMinMax(double *minMz, double *maxMz) const
		{

			MassSpectrum *massSpectrum = Q_NULLPTR;

			double finalMinMz = std::numeric_limits<double>::max();
			double finalMaxMz = 0;

			for(int iter = 0; iter < size(); ++iter)
			{

				massSpectrum = at(iter);

				if(maxMz != Q_NULLPTR)
				{
					double tempMaxMz = massSpectrum->last()->mz();
					if(tempMaxMz > finalMaxMz)
						finalMaxMz = tempMaxMz;
				}

				if(minMz != Q_NULLPTR)
				{
					double tempMinMz = massSpectrum->first()->mz();
					if(tempMinMz < finalMinMz)
						finalMinMz = tempMinMz;
				}
			}
			// End of 
			// for(int iter = 0; iter < size(); ++iter)

			if(minMz != Q_NULLPTR)
				*minMz = finalMinMz;
			if(maxMz != Q_NULLPTR)
				*maxMz = finalMaxMz;
		}


	//! Check if all the MassSpectrum instances have the same length.
	/*!

		The computation here is statistic: the set of MassSpectrum instances is
		iterated in and for each spectrum, its size (number of MassPeak instances)
		is recorded. Then, the standard deviation is computed.  \c .

		\return true if the standard deviation is less or equal to 0.1, false otherwise.

*/
	bool 
		MassSpectrumList::areSpectraSameLength() const
		{
			int specCount = size();

			double specSizeSum = 0;
			double specSizeAvg = 0;

			for(int iter = 0; iter < specCount; ++iter)
			{
				specSizeSum += at(iter)->size();
			}

			specSizeAvg = specSizeSum / specCount;

			double specSizeVarN = 0;

			for(int iter = 0; iter < specCount; ++iter)
			{
				specSizeVarN += (at(iter)->size() - specSizeAvg) * (at(iter)->size() - specSizeAvg);
			}

			double specSizeVar = specSizeVarN / specCount;

			double specSizeStdDev = sqrt(specSizeVar);

			//QString msg = QString("specCount = %1 - specSizeSum: %2 - specSizeAvg = %3 - specSizeVarN = %4"
					//" - specSizeVar = %5 - specSizeStdDev = %6\n")
				//.arg(specCount)
				//.arg(specSizeSum, 0, 'f', 7)
				//.arg(specSizeAvg, 0, 'f', 7)
				//.arg(specSizeVarN, 0, 'f', 7)
				//.arg(specSizeVar, 0, 'f', 7)
				//.arg(specSizeStdDev, 0, 'f', 7);
			//qWarning() << __FILE__ << __LINE__
				//<< msg;

			if(specSizeStdDev > specSizeAvg / 100)
				return false;

			return true;
		}


	//! Compute the total ion current of \c this MassSpectrumList
	/*!

		Each MassSpectrum instance in \c this list computes its TIC that is added to a
		local variable that is returned.

		\return a double value containing the sum of all the total ion current of
		the whole set of MassSpectrum instances.

		\sa tic(double mzStart, double mzEnd)

*/
	double
		MassSpectrumList::tic()
		{
			// Iterate in each mass spectrum of this mass spectrum list and get the tic.
			// Then add it recursively.

			double totalTic = 0;

			for(int iter = 0; iter < size(); ++iter)
			{
				totalTic += at(iter)->tic();
			}

			return totalTic;
		}


	//! Compute the total ion current of \c this MassSpectrumList
	/*!

		Each MassSpectrum instance in \c this list computes its TIC by accounting only
		MassPeak instances having their m/z value contained in the [\p mzStart --
		\p mzEnd] range. That TIC value is add to a local variable that is returned.

		\param mzStart start of the authorized m/z range

		\param mzEnd end of the authorized m/z range

		\return a double value containing the sum of all the total ion current
		values of the whole set of MassSpectrum instances.

		\sa tic()

*/	double
	MassSpectrumList::tic(double mzStart, double mzEnd)
	{
		// Iterate in each mass spectrum of this mass spectrum list and get the
		// tic. Then add it recursively. Do that only if the mz value is in the
		// ranged passed as parameters.

		double totalTic = 0;

		for(int iter = 0; iter < size(); ++iter)
		{
			totalTic += at(iter)->tic(mzStart, mzEnd);
		}

		return totalTic;
	}


	//! Compute the total ion current of a MassSpectrumList instance
	/*!

		The \p massSpectrumList list of MassSpectrum instances is iterated in and
		for each MassSpectrum instance the TIC is computed and added to a local
		variable that is returned.

		\param massSpectrumList list of MassSpectrum instances for which the total
		TIC value is to be computed.

		\return a double value containing the sum of all the total ion current
		values of the whole set of MassSpectrum instances in \p massSpectrumList.

		\sa tic()
		\sa tic(double mzStart, double mzEnd)

*/
	double
		MassSpectrumList::tic(const MassSpectrumList &massSpectrumList)
		{
			double totalTic = 0;

			//#pragma omp parallel for reduction (+:totalTic)
			for(int jter = 0; jter < massSpectrumList.size(); ++jter)
			{
				totalTic += massSpectrumList.at(jter)->tic();
			}

			return totalTic;
		}


} // namespace msXpSlibmass
