/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */



#include <omp.h>
#include <limits>


/////////////////////// Qt includes
#include <QDebug>
#include <QFile>
#include <qmath.h>


/////////////////////// Local includes
#include "MassSpectrum.hpp"
#include "MassSpectrumList.hpp"
#include "globals/globals.hpp"

#include "pwiz/data/msdata/MSDataFile.hpp"
#include "pwiz/data/msdata/DefaultReaderList.hpp"

using namespace pwiz::msdata;


namespace msXpSlibmass
{

	double MassSpectrum::m_maxAvgMzBinSize = 0.05;
	double MassSpectrum::m_maxBinSizeStdDev = MassSpectrum::m_maxAvgMzBinSize / 5;

	//! Construct a totally empty MassSpectrum
	/*!
	*/
	MassSpectrum::MassSpectrum()
	{
	}


	//! Construct a MassSpectrum with \p title
	/*!
	*/
	MassSpectrum::MassSpectrum(const QString &title)
	{
		m_title = title;
	}


	//! Destruct the MassSpectrum instance.
	/*!
		This function actually calls deleteMassPeaks().
		*/
	MassSpectrum::~MassSpectrum()
	{
		deleteMassPeaks();
	}


	//! Reset the MassSpectrum
	/*!
		This function calls deleteMassPeaks() to delete all the MassPeak instances it
		contains and clears the title. The other member data are reset to 0.
		*/

	void
		MassSpectrum::reset()
		{
			deleteMassPeaks();

			m_title.clear();

			m_lastModifIdx = 0;
			m_lastBinIdx = 0;
			m_binCount = 0;

			m_dt = 0;
			m_rt = 0;

			m_decimalPlaces = -1;

			m_greatestStep = qSNaN();

			m_isBinnable = false;

			m_binMinMz = 0;
			m_binMaxMz = 0;

			m_mzShift = 0;
		}


	//! Iterate in the MassPeak list and for each item deletes it
	/*!

		The various MassPeak items of the list are deletes and the list is
		clear()'ed.
		*/

	void
		MassSpectrum::deleteMassPeaks()
		{
#pragma omp parallel for
			for(int iter = 0; iter < size(); ++iter)
			{
				//qDebug() << __FILE__ << __LINE__
				//<< "Deleting MassPeak at " << iter;

				delete at(iter);
			}

			// Call the parent class clear() to actually empty the QList<MassPeak>
			clear();
		}


	//! Get the title.
	QString
		MassSpectrum::title() const
		{
			return m_title;
		}


	//! Get the retention time member datum value
	/*!
		\return the retention time value at which \c this spectrum was acquired.
		*/
	double
		MassSpectrum::rt() const
		{
			return m_rt;
		}


	//! Get a reference to the retention time member datum value
	/*!
		\return a reference to the retention time value at which \c this spectrum
		was acquired. The reference handle allows for modification of the retention
		time value.
		*/
	double & MassSpectrum::rrt()
	{
		return m_rt;
	}


	//! Get the drift time member datum value
	double
		MassSpectrum::dt() const
		{
			return m_dt;
		}


	//! Get a reference to the drift time member datum value
	/*!
		\return a reference to the drift time value at which \c this spectrum was
		acquired. The reference handle allows for modification of the drift time
		value.
		*/
	double & MassSpectrum::rdt()
	{
		return m_dt;
	}


	//! Set the decimal places member datum value
	/*! Cannot be negative. If negative, the member value is set to 0.
	*/
	void
		MassSpectrum::setDecimalPlaces(int decimalPlaces)
		{
			if(decimalPlaces < 0)
				m_decimalPlaces = 0;

			m_decimalPlaces = decimalPlaces;
		}

	//! Get the decimal places member datum value
	int
		MassSpectrum::decimalPlaces()
		{
			return m_decimalPlaces;
		}


	//! Get a reference to the decimal places member datum value
	/*!

		\return a reference to the member decimal places value.
		*/
	int &
		MassSpectrum::rdecimalPlaces()
		{
			return m_decimalPlaces;
		}


	//! Initialize the mass spectrum using a mass spectrum XY-formatted textual representation.
	/*!

		The \p xyFormatString contains a number of lines of text having the
		following generic format: <number1><non-number><number2> with number1 a
		string representation of m/z, non-number any string element that is not a
		number and number2 a string representation of i. The decomposition of each
		line into m/z and i values and crafting of a new MassPeak instance is
		performed using the MassPeak specialied constructor.

		\param xyFormatString string containing a series of lines describing m/z,i pairs

		\return 0 if \p xyFormatString is empty or the number of added MassPeak instances.

*/
	int
		MassSpectrum::initialize(const QString &xyFormatString)
		{
			int massPeakCount = 0;
			if(xyFormatString.size() == 0)
			{
				return massPeakCount;
			}

			QStringList lineList = xyFormatString.split("\n");

			for(int iter = 0; iter < lineList.size(); ++iter)
			{
				QString line = lineList.at(iter);

				if(line.isEmpty())
					continue;

				// If the line has only a newline character, go on.
				if(msXpS::gEndOfLineRegExp.match(line).hasMatch())
				{
					// qDebug() << __FILE__ << __LINE__
					// << "Was only end of line";
					continue;
				}

				if(line.startsWith("#"))
					continue;

				// At this point, if the line does not match the real data regexp, then
				// that would be an error.

				msXpSlibmass::MassPeak *massPeak = new MassPeak(line);
				if(massPeak && massPeak->validate())
				{
					append(massPeak);
					++massPeakCount;
				}
			}

			return massPeakCount;
		}


	//! Initialize the MassSpectrum with data
	/*!
		The spectrum is first cleared by calling deleteMassPeaks(). Then, new
		MassPeak instances are allocated and initialized using the values in \p
		mzList and \p iList.

		Note that mzList and iList cannot have different sizes
		otherwise the program crashes.

		\param mzList list of m/z values
		\param iList list of intensity values
		*/
	void
		MassSpectrum::initialize(const QList<double> &mzList,
				const QList<double> &iList)
		{
			if(mzList.size() != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			deleteMassPeaks();

			for(int iter = 0; iter < mzList.size(); ++iter)
			{
				MassPeak *massPeak = new MassPeak(mzList.at(iter), iList.at(iter));
				append(massPeak);
			}
		}


	//! Initialize the MassSpectrum with data
	/*!
		The spectrum is first cleared by calling deleteMassPeaks(). Then, new
		MassPeak instances are allocated and initialized using the values in \p
		mzList and \p iList. The \p mzList and \p iList data are only transposed
		into new MassPeak instances if the m/z value in \p mzList is contained in
		the [\p mzStart--\p mzEnd] range.

		Note that mzList and iList cannot have different sizes otherwise the program
		crashes.

		\param mzList list of m/z values
		\param iList list of intensity values
		\param mzStart start value of the authorized m/z range
		\param mzEnd end value of the authorized m/z range
		*/
	void
		MassSpectrum::initialize(const QList<double> &mzList,
				const QList<double> &iList, double mzStart,
				double mzEnd)
		{
			if(mzList.size() != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			deleteMassPeaks();

			for(int iter = 0; iter < mzList.size(); ++iter)
			{
				double mz = mzList.at(iter);

				if(mz >= mzStart && mz <= mzEnd)
				{
					MassPeak *massPeak = new MassPeak(mz, iList.at(iter));
					append(massPeak);
				}
			}
		}


	//! Initialize the MassSpectrum with data
	/*!
		The spectrum is first cleared by calling deleteMassPeaks(). Then, new
		MassPeak instances are allocated and initialized using the values in \p
		mzList and \p iList.

		Finally, the \c m_dt is set to \p driftTime.

		Note that mzList and iList cannot have different sizes otherwise the program
		crashes.

		\param mzList list of m/z values
		\param iList list of intensity values
		\param driftTime the drift time at which the mass spectrum was acquired
		\param mzStart start value of the authorized m/z range
		\param mzEnd end value of the authorized m/z range
		*/
	void
		MassSpectrum::initialize(const QList<double> &mzList,
				const QList<double> &iList, double driftTime)
		{
			if(mzList.size() != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			deleteMassPeaks();

			for(int iter = 0; iter < mzList.size(); ++iter)
			{
				MassPeak *massPeak = new MassPeak(mzList.at(iter), iList.at(iter));
				append(massPeak);
			}

			m_dt = driftTime;
		}


	//! Initialize the MassSpectrum with data
	/*!
		The spectrum is first cleared by calling deleteMassPeaks(). Then, new
		MassPeak instances are allocated and initialized using the values in \p
		mzList and \p iList. The \p mzList and \p iList data are only transposed
		into new MassPeak instances if the m/z value in \p mzList is contained in
		the [\p mzStart--\p mzEnd] range.

		Finally, the \c m_dt is set to \p driftTime.

		Note that mzList and iList cannot have different sizes otherwise the program
		crashes.

		\param mzList list of m/z values
		\param iList list of intensity values
		\param driftTime the drift time at which the mass spectrum was acquired
		\param mzStart start value of the authorized m/z range
		\param mzEnd end value of the authorized m/z range
		*/
	void
		MassSpectrum::initialize(const QList<double> &mzList,
				const QList<double> &iList, double driftTime,
				double mzStart, double mzEnd)
		{
			if(mzList.size() != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			deleteMassPeaks();

			for(int iter = 0; iter < mzList.size(); ++iter)
			{
				double mz = mzList.at(iter);

				if(mz >= mzStart && mz <= mzEnd)
				{
					MassPeak *massPeak = new MassPeak(mz, iList.at(iter));
					append(massPeak);
				}
			}

			m_dt = driftTime;
		}


	//! Initialize the MassSpectrum with data
	/*!
		The spectrum is first cleared by calling deleteMassPeaks(). Then, new
		MassPeak instances are allocated and initialized using the values in \p
		mzVector and \p iVector. The \p mzVector and \p iVector data are only
		transposed into new MassPeak instances if the m/z value in \p mzVector is
		contained in the [\p mzStart--\p mzEnd] range.

		Finally, the \c m_dt is set to \p driftTime.

		Note that mzVector and iVector cannot have different sizes otherwise the program
		crashes.

		\param mzVector vector of m/z values
		\param iVector vector of intensity values
		\param mzStart start value of the authorized m/z range
		\param mzEnd end value of the authorized m/z range
		*/
	void
		MassSpectrum::initialize(const QVector<double> &mzVector, const QVector<double> &iVector,
				double mzStart, double mzEnd)
		{
			if(mzVector.size() != iVector.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			deleteMassPeaks();

			// There are two possibilities: either we are feeding a new mass spectrum
			// with mz,i data with a precondition of mz value or not.

			if(qIsNaN(mzStart) || qIsNaN(mzEnd))
			{
				// No mz value precondition.

				for(int iter = 0; iter < mzVector.size(); ++iter)
					append(new MassPeak(mzVector.at(iter), iVector.at(iter)));
			}
			else
			{
				// Only feed the mass spectrum if mz value is in the mz range
				// precondition.

				for(int iter = 0; iter < mzVector.size(); ++iter)
				{
					double mz = mzVector.at(iter);

					if(mz >= mzStart && mz <= mzEnd)
						append(new MassPeak(mzVector.at(iter), iVector.at(iter)));
				}

			}
		}


	//! Setup the binning infrastructure
	/*!

		Binning is an operation by which a set of mass spectra (at least two
		spectra) are analyzed and checked for a number of properties. If these
		properties conform to specific criteria, the binning is setup in such a
		manner that any subsequent operation (mostly combine operations) are taking
		advantage of the binning that was setup. The binning infrastructure is only
		created if:

		- the spectra in the \p massSpectra MassSpectrumList have the same length.
		This is logical: if we want to make a binning operation on mass spectra, we
		need to have spectra that have the same number of points because binning
		involves creating bins that faithfully represent the initial mass data. If
		the spectra do not have the same size and the binning is done on a shorter
		spectrum than others, then, when we will perform combinations there will be
		missing bins!

		Ensuring that the spectra have the same size helps ensuring that the m/z
		intervals between points keeps being the same throughout the spectra in \p
		massSpectra, although it might be a non-constant interval throughout the m/z
		range, as in the case of TOF data.

		- \c this spectrum is empty (\c this spectrum is only a holder spectrum for
		later combinations to be performed according to the binning setup).

		Overall working of this function:

		If \p massSpectra is empty, return false.

		The mass spectra in \p massSpectra are checked to establish if they all have
		approximately the same length. If not, then m_isBinnable is set to false and
		the function returns true. The computation otherwise go on if the following
		condition is true: \c this mass spectrum is empty.

		Then the characteristics of the \p massSpectra list are gotten (minimum and
		maximum m/z values of the whole set of mass spectra.). Use the first
		spectrum of the \p massSpectra list to fill-in the bins, that is, fill-in a
		list of double values with all the steps in the spectrum's m/z data (an
		interval between a given m/z value and the following one is called a step).
		Get from fillInBins() the greatest m/z step in that spectrum. Check that the
		bin list contains binnable data. Binnable data are data that have uniform
		steps (relatively similar steps) and steps that are much less than unity. If
		the data are binnable, then setupBins() is called to terminate the binning
		setup. \c m_binCount is set to the number of bins in the bin list.

		\param massSpectra list of MassSpectrum instances

		\return true in all cases unless \p massSpectra is empty, in which case it
		returns false.

*/
	bool
		MassSpectrum::setupBinnability(const MassSpectrumList &massSpectra)
		{

			// Since setting up the bins must be performed only once, and that
			// operation needs to be performed prior to the first combination, we only
			// perform binning if this spectrum is empty, that is, it is the pristine
			// recipient of a combination operation yet to be started.

			if(size())
				qFatal("Fatal error at %s@%d -- %s. "
						"Setting up bins means that *this spectrum is empty. Which it is not."
						" Programming error. Program aborted.",
						__FILE__, __LINE__, __FUNCTION__);

			// If there are no spectra in the list, then we have nothing to do.

			if(massSpectra.size() == 0)
			{
				return false;
			}

			bool areSameLengthSpectra = massSpectra.areSpectraSameLength();

			if(!areSameLengthSpectra)
			{
				m_isBinnable = false;
				return true;
			}

			//qDebug() << __FILE__ << __LINE__
			//<< "Spectrum list has same-length spectra:" << areSameLengthSpectra
			//<< "and might be binnable:" << m_isBinnable;

			if(areSameLengthSpectra)
			{
				double minMz = 0;
				double maxMz = 0;

				QList<double> binList;

				massSpectra.mzMinMax(&minMz, &maxMz);

				m_binMinMz = minMz;
				m_binMaxMz = maxMz;

				// Use the first spectrum of the list to fill in the binList

				if(qIsNaN(massSpectra.first()->fillInBins(&binList)))
					return false;

				//qWarning() << __FILE__ << __LINE__
				//<< QString("First spectrum to combine has %1 peaks and produced %2 bins:")
				//.arg(massSpectra.first()->size())
				//.arg(binList.size())
				//<< binList;

				m_isBinnable = isBinnable(binList);

				//qDebug() << __FILE__ << __LINE__
				//<< "Spectrum is binnable:" << m_isBinnable;

				if(m_isBinnable)
				{

					// Now that we know the characteristics of the mass spec list, we can
					// configure *this spectrum if not done already, that is, if size() is 0.

					m_binCount = binList.size();

					//qDebug() << __FILE__ << __LINE__
					//<< "minMz:" << minMz
					//<< "maxMz:" << maxMz
					//<< "spectrum size:" << size()
					//<< "m_binCount:" << m_binCount;

					setupBins(binList);

					//qWarning() << __FILE__ << __LINE__
					//<< "After having setup the bins, the number of mass peaks:"
					//<< size();
				}
			}

			return true;
		}


	//! Fill-in \p binList with all the m/z intervals in \c this spectrum
	/*!
		A bin is a m/z interval separating two consecutive m/z values in a mass
		spectrum. By definition, a mass spectrum that has n MassPeak instances has
		(n-1) intervals. A mass spectrum with less than two MassPeak instances is by
		definition not binnable.

		This function postulates that the data in the mass spectrum are sorted in
		m/z increasing order. If not, the program crashes.

		\c this list of MassPeak instances is iterated into and for each previous \e
		vs current m/z value of two contiguous MassPeak instances, the (current -
		previous) m/z difference is computed and appended in \p binList.

		\param binList pointer to a list of double values corresponding to all the
		m/z intervals in the spectrum. Cannot be nullptr.

		\return a double value with the greatest step (interval) found in \c this
		mass spectrum.

*/
	double
		MassSpectrum::fillInBins(QList<double> *binList)
		{
			if(binList == Q_NULLPTR)
				qFatal("Fatal error at %s@%d. Program aborted.",
						__FILE__, __LINE__);

			// We need at least two mass peaks in the spectrum to compute the step
			// between the second and the first.

			if(size() < 2)
			{
				return qSNaN();
			}

			double greatestStep = 0;
			double tempStep = 0;

			double prevMz = at(0)->mz();

			for(int iter = 1; iter < size(); ++iter)
			{
				double curMz = at(iter)->mz();

				tempStep = curMz - prevMz;

				// Spectra are ascending-ordered for the mz value. If not, that's fatal.

				if(tempStep < 0)
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);

				binList->append(tempStep);

				//QString msg = QString("Append step: %1 from computation %2 - %3 with result: %4")
				//.arg(tempStep, 0, 'f', 10)
				//.arg(curMz, 0, 'f', 10)
				//.arg(prevMz, 0, 'f', 10)
				//.arg(curMz - prevMz, 0, 'f', 10);

				//qDebug() << __FILE__ << __LINE__ << msg;

				if(tempStep > greatestStep)
					greatestStep = tempStep;

				prevMz = curMz;
			}

			//qDebug() << __FILE__ << __LINE__
			//<< "Number of elements in binList:" << binList->size();

#if 0

			qDebug() << __FILE__ << __LINE__
				<< "Writing the list of bins in file /tmp/initial-bins.txt";

			QFile file("/tmp/initial-bins.txt");
			file.open(QIODevice::WriteOnly);

			QTextStream fileStream(&file);

			for(int iter = 0; iter < binList->size(); ++iter)
			{

				fileStream << QString("%1 %2\n")
					.arg(binList->at(iter), 0, 'f', 10)
					.arg(0);
			}

			fileStream.flush();
			file.close();

#endif

			// Only set to the member datum if that was not computed already. To set
			// it forcefully, use the other function.

			if(qIsNaN(m_greatestStep))
				m_greatestStep = greatestStep;

			return greatestStep;
		}


	//! Compute the average of the bin size for the bins in \p binList.
	/*!

		\p binList contains all the bin sizes of the sprectrum. A bin is the
		distance between two consecutive m/z values in a spectrum. If a spectrum
		contains n MassPeak instances, then the binList will contain (n-1) bins.

		Knowing the average size of the bins is important to establish the
		binnability of the spectrum. The bins need to be roughly of the same size
		throughout all the length of the m/z axis values. 

		That value should then be compared to MassSpectrum::m_maxAvgMzBinSize.

		\param binList list of bins for which the computation is to be performed.

		\return the average of all the values in \p binList.

*/
	double 
		MassSpectrum::binSizeAvg(const QList<double> &binList)
		{
			if(binList.isEmpty())
				return qSNaN();

			int binCount = binList.size();

			double binSizeSum = 0;
			double binSizeAvg = 0;

			for(int iter = 0; iter < binCount; ++iter)
			{
				binSizeSum += binList.at(iter);
			}

			binSizeAvg = binSizeSum / binCount;

			return binSizeAvg;
		}


	//! Compute the standard deviation of the bin size for the bins in \p binList.
	/*!

		\p binList contains all the bin sizes of the sprectrum. A bin is the
		distance between two consecutive m/z values in a spectrum. If a spectrum
		contains n MassPeak instances, then the binList will contain (n-1) bins.

		Knowing the standard deviation of the size of the bins is important to
		establish the binnability of the spectrum. The bins need to be roughly of
		the same size throughout all the length of the m/z axis values. 

		\param binList list of bins for which the computation is to be performed.

		\return the standard deviation of all the values in \p binList.

*/
	double 
		MassSpectrum::binSizeStdDev(const QList<double> &binList)
		{
			if(binList.isEmpty())
				return qSNaN();

			double binCount = binList.size();

			double sizeAvg = binSizeAvg(binList);

			// If a spectrum has not a proper m/z value ladder it will not be
			// binnable. This is typically the case with the spectra obtained from
			// Water's Synapt MS in mobility mode. Below is a rough test.

			//if(sizeAvg > MassSpectrum::m_maxAvgMzBinSize)
			//{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
			//<< "sizeAvg:" << sizeAvg << "greater than" << MassSpectrum::m_maxAvgMzBinSize;
			//}
			//else
			//{
			//qDebug() << "Fine, the spectrum is binnable.";
			//}

			double binSizeVarN = 0;

			for(int iter = 0; iter < binCount; ++iter)
			{
				binSizeVarN += (binList.at(iter) - sizeAvg) * (binList.at(iter) - sizeAvg);
			}

			double binSizeVar = binSizeVarN / binCount;

			return sqrt(binSizeVar);
		}


	//! Check that \p binList contains binnable data.
	/*!

		A mass spectrum is considered binnable if all the m/z intervals are much
		less than unity (that is, << MassSpectrum::m_maxAvgMzBinSize) and if they
		are ismilar (peaks are uniformly distributed along the m/z axis). The
		uniformity of the m/z values is evaluated by means of very simple
		statistics\: the variance must be less than an arbitrary value (0.001, at
		the moment). Also, the m/z interval average must not be greater than 0.1.

		\param binList list of double values to be analyzed

		\return true if the m/z intervals (bins) have an average less than
		MassSpectrum::m_maxAvgMzBinSize and a variance less than
		(MassSpectrum::m_maxAvgMzBinSize / 5).

*/
	bool
		MassSpectrum::isBinnable(const QList<double> &binList)
		{

			// A spectrum is binnable if when looking at all the m/z intervals in it,
			// they are all relatively similar and much less than unity (<< 1).

			if(binList.isEmpty())
				return false;

			double sizeAvg = binSizeAvg(binList);
			double sizeStdDev = binSizeStdDev(binList);

			//QString msg = QString("binCount = %1 - binSizeAvg = %2 - binSizeStdDev = %3\n")
			//.arg(binList.size())
			//.arg(sizeAvg, 0, 'f', 10)
			//.arg(sizeStdDev, 0, 'f', 10);

			//qDebug() << __FILE__ << __LINE__ << msg;

			return (sizeAvg <= MassSpectrum::m_maxAvgMzBinSize &&
					sizeStdDev <= MassSpectrum::m_maxBinSizeStdDev);
		}


	//! Populate \c this MassSpectrum using the m/z values in \p binList
	/*!

		Interate in \p binList and for each m/z value contained in it, create a new
		MassPeak instance with a 0-intensity value and append it to \c this
		MassSpectrum. At the end of the process, \c this MassSpectrum will have a
		list of MassPeak instances, all with 0-intensity values but with the m/z
		values representing the bins.

		\param binList list of m/z values representing the bins

*/
	void
		MassSpectrum::setupBins(const QList<double> &binList)
		{
			if(size())
				qFatal("Fatal error at %s@%d. Program aborted.",
						__FILE__, __LINE__);

			// Sanity check to ensure that the binning setup process was performed in
			// a proper way. Compare the present size of the list of bins with the
			// size that should have been computed earlier.

			if(binList.size() != m_binCount)
			{
				QString msg = QString("binList size: %1 -- m_binCount: %2\n")
					.arg(binList.size()).arg(m_binCount);

				qFatal("Fatal error at %s@%d with msg: %s. Program aborted.",
						__FILE__, __LINE__, msg.toLatin1().data());
			}

			// Seed the mass spectrum with the very first mz value that was found in
			// all the spectra of the MassSpectrumList used by setupBinnability().
			// That is going to be the very first append. Then we'll append as many
			// mass peaks as there are bins. This way, this spectrum will have as many
			// mass peak items as there were in the first mass spectrum of the
			// MassSpectraList that was used in setupBinnability().  Thus, the for
			// loop then starts at index 1.

			append(new MassPeak(m_binMinMz, 0));
			double previousMzBin = m_binMinMz;

			for(int iter = 0; iter < m_binCount; ++iter)
			{
				double tempMzBin = previousMzBin + binList.at(iter);

				//QString msg = QString("Append bin mz: %1 from computation %2 + %3 with result: %4")
				//.arg(tempMzBin, 0, 'f', 10)
				//.arg(previousMzBin, 0, 'f', 10)
				//.arg(binList.at(iter), 0, 'f', 10)
				//.arg(previousMzBin + binList.at(iter), 0, 'f', 10);

				//qDebug() << __FILE__ << __LINE__ << msg;

				previousMzBin = tempMzBin;

				append(new MassPeak(previousMzBin, 0));
			}

#if 0

			qDebug() << __FILE__ << __LINE__
				<< "Writing the list of bins setup in the mass spectrum in file /tmp/massSpecBins.txt";

			QFile file("/tmp/massSpecBins.txt");
			file.open(QIODevice::WriteOnly);

			QTextStream fileStream(&file);

			// We use the size of *this spectrum as the right border of the bin range.
			// We cannot use m_binCount because its size is one less then the actual
			// *this.size() because we added one extra smallest mz value before
			// looping in binList above. So *this mass spectrum ends having the same
			// number of mass peaks a the intial spectra in the mass spectrum list
			// that was to be combined in this initially empty mass spectrum.
			for(int iter = 0; iter < size(); ++iter)
			{

				fileStream << QString("%1 %2\n")
					.arg(at(iter)->mz(), 0, 'f', 10)
					.arg(0);
			}

			fileStream.flush();
			file.close();

#endif

			//qDebug() << __FILE__ << __LINE__
			//<< "Prepared bins with " << size() << "elements."
			//<< "starting with mz" << first()->mz()
			//<< "ending with mz" << last()->mz();
		}


	//! Find the index of the bin containing \p mz
	/*!

		Iterates in \c this MassSpectrum and searches the MassPeak instance having
		the proper m/z value.

		\param mz m/z value whose corresponding bin is searched.

		\return an int containing the index of the MassPeak instance having the
		right m/z value.

*/
	int
		MassSpectrum::binIndex(double mz)
		{
			//QString msg = QString("Entering binIndex with mz: %1 while m_lastBinIdx = %2 (%3,%4) and m_mzShift = %5")
			//.arg(mz, 0, 'f', 10)
			//.arg(m_lastBinIdx)
			//.arg(at(m_lastBinIdx)->mz(), 0, 'f', 10)
			//.arg(at(m_lastBinIdx)->i(), 0, 'f', 10)
			//.arg(m_mzShift, 0, 'f', 10);
			//qDebug() << __FILE__ << __LINE__
			//<< msg;

			if(!size())
				return 0;

			if(mz > at(size() - 1)->m_mz)
			{
				m_lastBinIdx = size() -1;
				return m_lastBinIdx;
			}

			// FIXME: it is a problem that we get an mz value that is less than the
			// first bin's mz value. Because, during binnability, we set the very
			// first bin's mz value to the smallest of all the mz values of all the
			// mass spectra in the mass spectrum list (see setupBinnability()). So
			// this situation should never occur.
			if(mz <= at(0)->m_mz)
			{
				m_lastBinIdx = 0;

				//qFatal("Fatal error at %s@%d -- %s. "
				//"The binning framework was not setup properly. Exiting."
				//"Program aborted.",
				//__FILE__, __LINE__, __FUNCTION__);	
			}

			if(mz < at(m_lastBinIdx)->mz())
				m_lastBinIdx = 0;

			int lastTestIndex = 0;
			int returnIndex = -1;

			for(int iter = m_lastBinIdx; iter < size(); ++iter)
			{
				double iterBinMz = at(iter)->mz() ;

				//QString msg = QString("Current bin index: %1 (%2,%3")
				//.arg(iter)
				//.arg(iterBinMz, 0, 'f', 10)
				//.arg(at(iter)->i(), 0, 'f', 10);
				//qDebug() << __FILE__ << __LINE__
				//<< msg;

				if(mz >= iterBinMz)
				{
					if(mz == iterBinMz)
					{
						returnIndex = iter;
						break;
					}
					else
					{
						// If this is the last bin, then we need to return iter.

						if(iter == size())
						{
							returnIndex = iter;
							break;
						}
						else
							lastTestIndex = iter;
					}
				}
				else if(mz < iterBinMz)
				{
					returnIndex = lastTestIndex;
					break;
				}
				else
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);
			}

			// Due to binning, we might be here if mz was greater than the last mz of
			// the spectrum. In this case we just return the last spectrum index.

			if(returnIndex == -1)
			{
				returnIndex = size() - 1;

				qWarning() << __FILE__ << __LINE__
					<< "needed to set returnIndex to size() - 1;"
					<< "mz: " << QString("%1").arg(mz, 0, 'f', 10)
					<< "bin index:" << returnIndex
					<< "with bin's mz:" << QString("%1").arg(at(returnIndex)->mz(), 0, 'f', 10)
					<< "bin count: size():" << size()
					<< "last bin's mz:" << QString("%1").arg(at(size() - 1)->mz(), 0, 'f', 10);
			}

			m_lastBinIdx = returnIndex;

			//qWarning() << __FILE__ << __LINE__
			//<< "mz: " << QString("%1").arg(mz, 0, 'f', 10)
			//<< "bin index:" << returnIndex
			//<< "with bin's mz:" << QString("%1").arg(at(returnIndex)->mz(), 0, 'f', 10)
			//<< "bin count: size():" << size()
			//<< "last bin's mz:" << QString("%1").arg(at(size() - 1)->mz(), 0, 'f', 10);

			//qWarning() << __FILE__ << __LINE__
			//<< "returning" << returnIndex;

			return returnIndex;
		}


	//! Computes the mz shift between \p massSpectrum and \c this MassSpectrum
	/*!

		The shift is computed by making the difference between the m/z values of the
		first MassPeak instance in \p massSpectrum and \this MassSpectrum instance.

		As a side effect, \c m_mzShift is set to this difference value.

		\param massSpectrum the MassSpectrum to use for the m/z shift calculation
		against \c this MassSpectrum.

		\return a double containing the m/z shift.

*/
	double
		MassSpectrum::determineMzShift(const MassSpectrum &massSpectrum)
		{
			if(!size() || !massSpectrum.size())
				return qSNaN();

			m_mzShift = massSpectrum.first()->mz() - first()->mz();

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__
			//<< "m_mzShift:" << m_mzShift;

			return m_mzShift;
		}


	//! Computes the mz shift between \p otherMzValue and \c this MassSpectrum
	/*!

		The shift is computed by making the difference between the m/z values \p
		otherMzValue and \this MassSpectrum instance' first MassPeak instance.

		As a side effect, \c m_mzShift is set to this difference value.

		\param otherMzValue mz value to use for the m/z shift calculation against \c
		this MassSpectrum's first MassPeak instance.

		\return a double containing the m/z shift.

*/	double
	MassSpectrum::determineMzShift(double otherMzValue)
	{

		m_mzShift = otherMzValue - first()->mz();

		//qDebug() << __FILE__ << __LINE__ << __FUNCTION__
		//<< "m_mzShift:" << m_mzShift;

		return m_mzShift;
	}




	//! Compute the total ion current value of \c this MassSpectrum
	/*!

		The TIC value is nothing but the sum of all the intensity values of all the
		MassPeak instances in \c MassSpectrum.

		\return the TIC value of \c this MassSpectrum.

*/
	double
		MassSpectrum::tic() const
		{
			// Iterate in each mass peak of this spectrum and sum up its intensity with
			// all the other peaks' intensity. Return that value.

			double tic = 0;

#pragma omp parallel
			{
				// int maxNumbThreads = omp_get_max_threads();
				// omp_set_num_threads(10);
				// int numbThreads = omp_get_num_threads();
				// qDebug() << __FILE__ << __LINE__
				// << "Number of threads: " << numbThreads << "over" << maxNumbThreads;

#pragma omp for reduction(+ : tic)
				// 20160921 - This code was tested to provide identical results to the
				// serial version
				// and to provide an increase in the execution speed of roughly 2x with
				// 4 threads.

				for(int iter = 0; iter < size(); ++iter)
				{
					// int numbThreads = omp_get_num_threads();
					// qDebug() << __FILE__ << __LINE__
					// << "Number of threads: " << numbThreads;

					tic += at(iter)->i();
				}
			}

			return tic;
		}


	//! Compute the total ion current value of \c this MassSpectrum
	/*!

		The TIC value is nothing but the sum of all the intensity values of all the
		MassPeak instances in \c MassSpectrum. Only the MassPeak instances of which
		the m/z value is contained in the [\p mzStart--\p mzEnd] m/z range are
		accounted for in this calculation.

		\sa tic()

		\return the TIC value of \c this MassSpectrum.

*/
	double
		MassSpectrum::tic(double mzStart, double mzEnd) const
		{
			// Iterate in each mass peak of this spectrum and sum up its intensity with
			// all the other peaks' intensity. Return that value. Do that only if the
			// mz value is in the ranged passed as parameters.

			double tic = 0;

			for(int iter = 0; iter < size(); ++iter)
			{
				double mz = at(iter)->mz();

				if(mz >= mzStart && mz <= mzEnd)
					tic += at(iter)->i();
			}

			return tic;
		}


	//! Combine \p massPeak into \c this MassSpectrum
	/*!

		A combine operation is a merge of a MassPeak into a preexisting set of
		MassPeak instances. If, in the current spectrum, MassPeak instance exists
		that has the same m/z value of \p massPeak, then the intensity value of \p
		massPeak is added to the preexisting MassPeak instance intensity value. If
		there is no MassPeak instance having the same m/z value of \p massPeak, then
		\p massPeak is copied and inserted at the right place in the list of
		MassPeak instances of \c this MassSpectrum.

		The \p massPeak m/z value is copied and modified according to this:

		- if m_decimalPlaces is not -1, then the decimal places are taken into
		account for the computation.

		To speed up the combine process, each time a MassPeak instance is modified
		(created or updated), its index in the list of MassPeak instances is
		recorded in m_lastModifIdx. This variable is tested for the value it
		contains and if that value is meaningful, then it is used to access the list
		of MassPeak instances right from that index. This is powerful to speed up
		combine operations because a mass spectrum is inherently sorted in ascending
		order of the m/z value and the next MassPeak combination into \c this
		MassSpectrum will occur most certainly below the m_lastModifIdx index value
		and that speeds up the operation.

		\param massPeak the MassPeak instance to be combined to \c this MassSpectrum

		\return 1 if a new MassPeak instance was inserted in \c this MassSpectrum;
		0 if a preexisting MassPeak instance was found having the same m/z value as
		that in \p massPeak (the intensity value was thus incremented).

*/
	int
		MassSpectrum::combine(const MassPeak &massPeak)
		{
			//qDebug() << __FILE__ << __LINE__
			//<< "Entering combine";

			// This function only provide very fast additions, insertion, because a
			// mass spectrum is inherently sorted in increasing order. Otherwise this
			// function would not work.

			// Returns 0 if the peak was not added but only the intensity was
			// increment, and 1 if the peak was actually added (either prepended,
			// inserted or appended).

			// We get a MassPeak instance and we must integrate it into *this list. If
			// a mass peak by the same mz is found, then the intensity is increased.
			// Otherwise the mass peak is integrated in such a manner that the
			// spectrum sorts correctly in ascending order with respect to mz values.

			// The index of the modified element of the list, be that modification an
			// insertion or an intensity increment, the m_lastModifIdx is set to that
			// index such that the next call can make use of it to accelerate the
			// search process.

			double mz = massPeak.mz();

			// Check if decimals are specified.
			if(m_decimalPlaces != -1)
			{
				mz = ceil((mz * pow(10, m_decimalPlaces)) - 0.49) / pow(10, m_decimalPlaces);

				// if no decimal at all is required: int mz = (double) (int) mz;

				// qDebug("Rounding from %.5f to %.5f\n",
				// massPeak.mz(), mz);
			}

			if(!size())
			{
				append(new MassPeak(mz, massPeak.i()));

				m_lastModifIdx = 0;
				return 1;
			}

			if(size() == 1)
			{
				if(mz == last()->mz())
				{
					last()->ri() += massPeak.i();

					m_lastModifIdx = 0;
					return 0;
				}
				else if(mz > last()->mz())
				{
					append(new MassPeak(mz, massPeak.i()));

					m_lastModifIdx = 1;
					return 1;
				}
				else
				{
					prepend(new MassPeak(mz, massPeak.i()));

					m_lastModifIdx = 1;
					return 1;
				}
			}

			// At this point we know that there are at least two MassPeak in this
			// spectrum.

			if(m_lastModifIdx >= size())
			{
				m_lastModifIdx = size() / 2;
			}

			if(mz < at(m_lastModifIdx)->mz())
			{
				m_lastModifIdx = 0;
			}

			int oneLessThanSpecSize = size() - 1;

			for(int iter = m_lastModifIdx; iter < oneLessThanSpecSize; ++iter)
			{
				if(mz < at(iter)->mz())
				{
					insert(iter, new MassPeak(mz, massPeak.i()));
					m_lastModifIdx = iter;
					return 1;
				}

				if(mz == at(iter)->mz())
				{

					// There is a MassPeak that has that same mz, only simply increment
					// the intensity.

					at(iter)->ri() += massPeak.i();
					m_lastModifIdx = iter;
					return 0;
				}

				if(mz > at(iter)->mz() && mz < at(iter + 1)->mz())
				{
					insert(iter + 1, new MassPeak(mz, massPeak.i()));
					m_lastModifIdx = iter + 1;
					return 1;
				}
			}

			// At this point, we did not insert the mass peak, nor did we increment a
			// preexisting mass peak. There are two possibilities:

			if(mz == last()->mz())
			{
				last()->ri() += massPeak.i();
				m_lastModifIdx = size() - 1;
				return 0;
			}
			else
			{
				// The mz value is greater than the last value of the
				// spectrum. Simply perform an append.
				append(new MassPeak(mz, massPeak.i()));
				m_lastModifIdx = size() - 1;

				//qWarning() << __FILE__ << __LINE__
				//<< "modified index:" << m_lastModifIdx
				//<< "mz:" << at(m_lastModifIdx)->m_mz
				//<< "i:" <<  at(m_lastModifIdx)->m_i;

				return 1;
			}
		}


	//! Combine \p massPeak into \c this MassSpectrum using bins.
	/*!

		The combine operation is performed using the bin that corresponds to the m/z
		value of \p massPeak. The intensity value of the found bin is incremented by
		the intensity of \p massPeak. If no bin is found, the program crashes.

		Note that if \c this spectrum is empty, the program crashes, because, by
		definition a binned combine operation requires that the bins be already
		available in \c this MassSpectrum.

		The \p massPeak m/z value is first copied and the copy is modified according
		to this:

		- m_mzShift is systematically added to the \m massPeak mz value. By default,
		m_mzShift is 0, so that does not hurt. However, if a shift was computed, then
		it is systematically accounted for.

		- if m_decimalPlaces is not -1, then the decimal places are taken into account
		for the computation.

		Once the steps above have been performed, the bin corresponding to the m/z
		value of \p massPeak is determined and its intensity value is incremented by
		the intensity value of \p massPeak.

		\param massPeak MassPeak instance to be combined to \c this MassSpectrum

		\sa combine(const MassPeak &massPeak)

		\return an integer containing the index of the updated bin

*/
	int
		MassSpectrum::combineBinned(const MassPeak &massPeak)
		{
			//qDebug() << __FILE__ << __LINE__
			//<< "Entering combineBinned";

			// Since this combine process is binned, it is absolutely necessary that
			// this spectrum contains data, that is, at least the bins !

			if(!size())
				qFatal("Fatal error at %s@%d -- %s. "
						"In a binned combine operation, this spectrum cannot be empty\n. "
						"Program aborted.", __FILE__, __LINE__, __FUNCTION__);

			// Account for m_mzShift, that was computed (might be 0 or not) in
			// combine(MassSpectrum).

			double mz = massPeak.mz() + m_mzShift;

			if(m_decimalPlaces != -1)
			{
				mz = ceil((mz * pow(10, m_decimalPlaces)) - 0.49) / pow(10, m_decimalPlaces);

				// if no decimal at all is required: int mz = (double) (int) mz;

				qDebug("%s %d Rounding from %.5f to %.5f\n",
						__FILE__, __LINE__,
						massPeak.mz(), mz);
			}

			//qDebug() << __FILE__ << __LINE__
			//<< "Calling binIndex with mz:" << QString("%1").arg(massPeak.mz(), 0, 'f', 10)
			//<< "and mzShift:" << QString("%1").arg(m_mzShift, 0, 'f', 10);

			int binIdx = binIndex(mz);

			if(binIdx == -1)
				qFatal("Fatal error at %s@%d -- %s. "
						"Failed to find the bin for the m/z value of MassPeak. Program aborted.",
						__FILE__, __LINE__, __FUNCTION__);

			//QString msg = QString("must combine (%1,%2) with bin[%3] (%4, %5)")
			//.arg(mz, 0, 'f', 10)
			//.arg(massPeak.i())
			//.arg(binIdx)
			//.arg(at(binIdx)->mz(), 0, 'f', 10)
			//.arg(at(binIdx)->i(), 0, 'f', 10);
			//qDebug() << __FILE__ << __LINE__ << msg;

			MassPeak *targetMassPeak = at(binIdx);

			targetMassPeak->ri() += massPeak.i();

			//msg = QString("after combination: (mz,i): (%1,%2)")
			//.arg(at(binIdx)->mz(), 0, 'f', 10)
			//.arg(at(binIdx)->i(), 0, 'f', 10);
			//qDebug() << __FILE__ << __LINE__ << msg;

			return binIdx;
		}

#if 0
	int
		MassSpectrum::combineSlow(const MassPeak &massPeak)
		{
			// Returns 0 if the peak was not added but only the intensity was increment,
			// and
			// 1 if the peak was actually added (either prepended inserted or appended).

			// We get a MassPeak instance and we must integrate it into *this list. If
			// a mass peak by the same mz is found, then the intensity is increased.
			// Otherwise the mass peak is integrated in such a manner that the
			// spectrum sorts correctly in ascending order with respect to mz values.

			double mz = massPeak.mz() + m_mzShift;

			if(m_decimalPlaces != -1)
			{
				mz = ceil((mz * pow(10, m_decimalPlaces)) - 0.49) / pow(10, m_decimalPlaces);

				// if no decimal at all is required: int mz = (double) (int) mz;

				// qDebug("Rounding from %.5f to %.5f\n",
				// massPeak.mz(), mz);
			}

			if(!size())
			{
				append(new MassPeak(mz, massPeak.i()));
				return 1;
			}

			if(size() == 1)
			{
				if(mz == last()->mz())
				{
					last()->ri() += massPeak.i();
					return 0;
				}
				else if(mz > last()->mz())
				{
					append(new MassPeak(mz, massPeak.i()));
					return 1;
				}
				else
				{
					prepend(new MassPeak(mz, massPeak.i()));
					return 1;
				}
			}

			// Now that we have handled two simple cases, go on with the "sorting".

			int oneLessThanSpecSize = size() - 1;

			for(int iter = 0; iter < oneLessThanSpecSize; ++iter)
			{
				if(mz < at(iter)->mz())
				{
					insert(iter, new MassPeak(mz, massPeak.i()));
					return 1;
				}

				if(mz == at(iter)->mz())
				{
					// There is a MassPeak that has that same mz, only simply increment
					// the
					// intensity.
					at(iter)->ri() += massPeak.i();
					return 0;
				}

				if(mz > at(iter)->mz() && mz < at(iter + 1)->mz())
				{
					insert(iter + 1, new MassPeak(mz, massPeak.i()));
					return 1;
				}
			}

			// At this point, we did not insert the mass peak, nor did we increment a
			// preexisting mass peak. There are two possibilities:

			if(mz == last()->mz())
			{
				last()->ri() += massPeak.i();
				return 0;
			}
			else
			{
				// The mz value is greater than the last value of the
				// spectrum. Simply perform an append.
				append(new MassPeak(mz, massPeak.i()));
				return 1;
			}
		}
#endif


	//! Combine \p massSpectrum into \c this MassSpectrum.
	/*!

		The massSpectrum is combined into \c this MassSpectrum by iterating in all
		the MassPeak instances it contains and calling either combine(const MassPeak
		&) or combineBinned(const MassPeak &) depending on the \c m_isBinnable
		member.

		Note that if m_isBinnable is true, then determineMzShift() is called.

		\param massSpectrum MassSpectrum instance to be combined to \c this MassSpectrum

		\return an integer containing the number of combined MassPeak instances

*/
	int
		MassSpectrum::combine(const MassSpectrum &massSpectrum)
		{

			// We receive a spectrum and we need to combine it in this mass spectrum.

			//qDebug() << __FILE__ << __LINE__
			//<< "Starting combination of the spectrum with "
			//<< massSpectrum.size() << "peaks in it.";

			// Only apply the mz shift if this spectrum has undergone successfully the
			// binning setup.

			if(m_isBinnable)
				determineMzShift(massSpectrum);

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int iter = 0;

			for(iter = 0; iter < massSpectrum.size(); ++iter)
			{
				if(m_isBinnable)
					combineBinned(*massSpectrum.at(iter));
				else
					combine(*massSpectrum.at(iter));
			}

			// qDebug() << __FILE__ << __LINE__
			// << "Done combining a new spectrum in this spectrum, that has now "
			// << size() << "mass peaks";

			// Return the number of combined mass peaks.
			return iter;
		}


	//! Combine \p massSpectrum into \c this MassSpectrum.
	/*!

		Same as combine(const MassSpectrum &massSpectrum) unless that only the
		MassPeak instances having a m/z value comtained in the [\p mzStart -- \p
		mzEnd] range are taken into account.

		\param massSpectrum MassSpectrum instance to be combined to \c this MassSpectrum

		\param mzStart left value of the m/z range

		\param mzEnd right value of the m/z range

		\return an integer containing the number of combined MassPeak instances

*/
	int
		MassSpectrum::combine(const MassSpectrum &massSpectrum,
				double mzStart, double mzEnd)
		{
			// We receive a spectrum and we need to combine this spectrum in this mass
			// spectrum. However, we need to check each individual MassPeak of the
			// MassSpectrum instance for their m/z value, because it can only be
			// combined if its value is in between both mzStart and mzEnd arguments to
			// this function.

			// qDebug() << __FILE__ << __LINE__
			// << "Starting combination of the spectrum with " <<
			// << massSpectrum.size()
			// << "peaks in it.";

			// Only apply the mz shift if this spectrum has undergone successfully the
			// binning setup.

			if(m_isBinnable)
				determineMzShift(massSpectrum);

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int combinedPeaksCount  = 0;

			for(int iter = 0; iter < massSpectrum.size(); ++iter)
			{
				MassPeak *iterMassPeak = massSpectrum.at(iter);

				if(iterMassPeak->mz() >= mzStart && iterMassPeak->mz() <= mzEnd)
				{
					//QString msg = QString("Range: [%1-%2] - this mass peak is in the range: mz %3\n")
					//.arg(mzStart, 0, 'f', 10)
					//.arg(mzEnd, 0, 'f', 10)
					//.arg(iterMassPeak->mz(), 0, 'f', 10);
					//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
					//<< msg; 

					if(m_isBinnable)
						combineBinned(*iterMassPeak);
					else
						combine(*iterMassPeak);

					++combinedPeaksCount;
				}
				// Because the spectra are ordered, if the currently iterated mass peak
				// has a mass greater than the accepted mzEnd, then break the loop as
				// this cannot be any better later.
				else if(iterMassPeak->mz() > mzEnd)
					break;
			}
			// qDebug() << __FILE__ << __LINE__
			// << "Done combining a new spectrum in this spectrum, current with "
			// << size() << "mass peaks";

			// Return the number of combined mass peaks.
			return combinedPeaksCount;
		}


#if 0

	// For the record, the function used before when we did not encounter
	// the Bruker microQTof spectra that were misaligned and which required
	// binning for proper combination.

	int
		MassSpectrum::combine(const MassSpectrumList &massSpectra)
		{
			// We receive a list of mass spectra and wee need to combine all the
			// spectra thereof in this mass spectrum.

			// qDebug() << __FILE__ << __LINE__
			// << "Starting combination of " << massSpectra.size()
			// << "mass spectra into this mass spectrum";

			int specCount = massSpectra.size();

			int combinedSpectra = 0;

			for(int iter = 0; iter < specCount; ++iter)
			{
				MassSpectrum *massSpectrum = massSpectra.at(iter);

				combine(*massSpectrum);

				combinedSpectra++;
			}

			// qDebug() << __FILE__ << __LINE__
			// << "Effectively combined " << combinedSpectra
			// << "in this single spectrum for a total of "
			// << size() << "mass peaks";

			return combinedSpectra;

		}
#endif


	//! Combine the MassSpectrum instances of \p massSpectra into \c this MassSpectrum
	/*!

		This function might be called recursively with multiple different
		MassSpectrumList instances.

		The workings of this function are the following:

		- if the \p massSpectra list is empty, return 0

		- if \c this MassSpectrum is empty.

		- iterate in the \p massSpectra list of MassSpectrum instances and for each
		instance call combine(const MassSpectrum &). Note that the combine() call
		will check if the combine operation should take advantage of binning or not.

		\param massSpectra list of MassSpectrum instances to be combined to \c this
		MassSpectrum

		\return an int containing the number of combined MassSpectrum instances.

		\sa combine(const MassSpectrum &)
		*/
	int
		MassSpectrum::combine(const MassSpectrumList &massSpectra)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
			//<< "";

			// We receive a list of mass spectra and we need to combine all the
			// spectra thereof in this mass spectrum.

			if(massSpectra.size() == 0)
				return 0;

			//qDebug() << __FILE__ << __LINE__
			//<< "Starting combination of " << massSpectra.size()
			//<< "mass spectra into this mass spectrum";

			// This function might be called recursively with various mass spectrum
			// lists. We only need to seed the bin combination data if that has not
			// been done already, that is this mass spectrum is empty (binning
			// actually fills-in the very first set of (m/z,i) pairs.

			if(!size())
				if(!setupBinnability(massSpectra))
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);

			MassSpectrum *massSpectrum = nullptr;
			int specCount = massSpectra.size();

			int combinedSpectra = 0;

			for(int iter = 0; iter < specCount; ++iter)
			{
				massSpectrum = massSpectra.at(iter);

				combine(*massSpectrum);

#if 0
				QString fileName = QString("/tmp/unranged-combined-spectrum-after-%1-combinations.txt")
					.arg(iter + 1);

				QFile file(fileName);

				file.open(QIODevice::WriteOnly);

				QTextStream fileStream(&file);

				for(int iter = 0; iter < size(); ++iter)
				{
					fileStream << QString("%1 %2\n")
						.arg(at(iter)->mz(), 0, 'f', 10)
						.arg(at(iter)->i(), 0, 'f', 10);
				}

				fileStream.flush();
				file.close();
#endif

				combinedSpectra++;
			}
			// qDebug() << __FILE__ << __LINE__
			// << "Effectively combined " << combinedSpectra
			// << "in this single spectrum for a total of "
			// << size() << "mass peaks";

#if 0
			QString fileName = "/tmp/unranged-combined-spectrum-after-finishing-combinations.txt";

			QFile file(fileName);

			file.open(QIODevice::WriteOnly);

			QTextStream fileStream(&file);

			for(int iter = 0; iter < size(); ++iter)
			{
				fileStream << QString("%1 %2\n")
					.arg(at(iter)->mz(), 0, 'f', 10)
					.arg(at(iter)->i(), 0, 'f', 10);
			}

			fileStream.flush();
			file.close();
#endif

			return combinedSpectra;
		}


	//! Combine the MassSpectrum instances of \c massSpectra into \c this MassSpectrum
	/*!

		This function acts like

		combine(const MassSpectrumList &massSpectra) with the difference that it calls

		combine(const MassSpectrum &massSpectrum, double mzStart, double mzEnd) to
		only account for MassPeak instances in the spectra that have their m/z value
		contained in the [\p mzStart -- \p mzEnd] range.

		\param massSpectra list of MassSpectrum instances to be combined to \c this
		MassSpectrum

		\param mzStart start value of the acceptable m/z range

		\param mzEnd end value of the acceptable m/z range

		\return an int containing the number of combined MassSpectrum instances.

		\sa combine(const MassSpectrumList &massSpectra)

*/
	int
		MassSpectrum::combine(const MassSpectrumList &massSpectra,
				double mzStart, double mzEnd)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
			//<< "";

			// We receive a list of mass spectra, that is a list of (list of
			// MassPeak*) and wee need to combine all the spectra thereof in this mass
			// spectrum. However, we need to check each individual MassPeak of the
			// MassSpectrum instances for their m/z value, because it can only be
			// combined if its value is in between both mzStart and mzEnd arguments to
			// this function.

			if(massSpectra.size() == 0)
				return 0;

			//qDebug() << __FILE__ << __LINE__
			//<< "Starting combination of " << massSpectra.size()
			//<< "mass spectra into this mass spectrum";

			if(!size())
				if(!setupBinnability(massSpectra))
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);

			int combinedSpectra = 0;

			for(int iter = 0; iter < massSpectra.size(); ++iter)
			{
				MassSpectrum *iterMassSpectrum = massSpectra.at(iter);

				combine(*iterMassSpectrum, mzStart, mzEnd);

				// qDebug() << __FILE__ << __LINE__
				// << "Currently processed mass spectrum has"
				// << iterMassSpectrum->size() << "peaks";

				combinedSpectra++;
			}

			// qDebug() << __FILE__ << __LINE__
			// << "Effectively combined " << combinedSpectra
			// << "in this single spectrum for a total of "
			// << size() << "mass peaks";

#if 0
			QString fileName = "/tmp/mzranged-combined-spectrum-after-finishing-combinations.txt";

			QFile file(fileName);

			file.open(QIODevice::WriteOnly | QIODevice::Truncate);

			QTextStream fileStream(&file);

			for(int iter = 0; iter < size(); ++iter)
			{
				fileStream << QString("%1 %2\n")
					.arg(at(iter)->mz(), 0, 'f', 10)
					.arg(at(iter)->i(), 0, 'f', 10);
			}

			fileStream.flush();
			file.close();
#endif

			return combinedSpectra;
		}


	//! Combine the mass spectrum represented by \p mzList and \iList in \c this MassSpectrum.
	/*!

		The \p mzList and \p iList represent a mass spectrum. Both the lists must
		contain at least one value and have the same number of items. If \p mzList
		is empty, the function returns immediately. If both lists do not have the
		same size, the program crashes.

		Note that if \c this MassSpectrum is binnable, then the m/z shift is
		determined by calling determineMzShift().

		\param mzList list of double values representing the m/z values of the mass
		spectrum.

		\param iList list of double values representing the intensity values of the
		mass spectrum.

		\return -1 if \p mzList is empty, otherwise the number of combined (m/z,i) pairs.

*/
	int
		MassSpectrum::combine(const QList<double> &mzList, const QList<double> &iList)
		{

			// We get two lists of mz and i, typically out of the data file, and we
			// need to construct (or continue constructing) a mass spectrum. For each
			// mz in the mzList, we iterate in the spectrum and we check if that mz
			// exists. If so the intensity of the found MassPeak is increased. If not
			// a new MassPeak is inserted at the proper place with the corresonding
			// intensity from iList.

			// Sanity check

			int listSize = mzList.size();
			if(listSize < 1)
				return -1;

			if(listSize != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			if(m_isBinnable)
				determineMzShift(mzList.first());

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int count = 0;

			for(int iter = 0; iter < listSize; ++iter)
			{
				if(m_isBinnable)
					count += combineBinned(MassPeak(mzList.at(iter), iList.at(iter)));
				else
					count += combine(MassPeak(mzList.at(iter), iList.at(iter)));
			}

			return count;
		}


	//! Combine the mass spectrum represented by \p mzList and \iList in \c this MassSpectrum.
	/*!

		Works like combine(const QList<double> &mzList, const QList<double> &iList)
		unless the (m/z,i) pairs out of the lists are accounted for only if the m/z
		value is contained in the [\p mzStart -- \p mzEnd] range.

		\param mzList list of double values representing the m/z values of the mass
		spectrum

		\param iList list of double values representing the intensity values of the
		mass spectrum

		\param mzStart start value of the acceptable m/z range

		\param mzEnd end value of the acceptable m/z range

		\return -1 if \p mzList is empty, otherwise the number of combined (m/z,i) pairs.

		\sa combine(const QList<double> &mzList, const QList<double> &iList).

*/
	int
		MassSpectrum::combine(const QList<double> &mzList, const QList<double> &iList,
				double mzStart, double mzEnd)
		{

			// Sanity check

			int listSize = mzList.size();
			if(listSize < 1)
				return -1;

			if(listSize != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			if(m_isBinnable)
				determineMzShift(mzList.first());

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int count = 0;

			for(int iter = 0; iter < listSize; ++iter)
			{
				double mz = mzList.at(iter);

				if(mz >= mzStart && mz <= mzEnd)
				{
					if(m_isBinnable)
						count += combineBinned(MassPeak(mzList.at(iter), iList.at(iter)));
					else
						count += combine(MassPeak(mzList.at(iter), iList.at(iter)));
				}
			}

			return count;
		}


	//! Searches for \p mz in \c this MassSpectrum
	/*!

		Each MassPeak instance in \c this MassSpectrum is tested for \p mz in its \c
		m_mz member datum.

		\param mz double value that is to be searched as a m/z value of MassPeak
		instances in \c this MassSpectrum

		\return an int containing the index of the matching MassPeak instance; -1
		if the \p mz value was not found.

*/
	int
		MassSpectrum::contains(double mz) const
		{
			int specSize = size();

			for(int iter = 0; iter < specSize; ++iter)
				if(at(iter)->mz() == mz)
					return iter;

			return -1;
		}


	//! Fills in a list of double values containing all the mz values in \c this MassSpectrum.
	/*!

		\return a QList of all the m/z values in \c this MassSpectrum.

*/
	QList<double>
		MassSpectrum::mzList() const
		{
			QList<double> list;

			for(int iter = 0; iter < size(); ++iter)
			{
				list.append(at(iter)->mz());
			}

			return list;
		}



	//! Fills in a list of double values containing all the i values in \c this MassSpectrum.
	/*!

		\return a QList of all the intensity values in \c this MassSpectrum.

*/
	QList<double>
		MassSpectrum::iList() const
		{
			QList<double> list;

			for(int iter = 0; iter < size(); ++iter)
			{
				list.append(at(iter)->i());
			}

			return list;
		}


	//! Fills in lists of double values containing the data of \c this MassSpectrum.
	/*!

		The program crashes if any of the pointers is nullptr.

		\param mzList pointer to a double list where to store all the m/z values of
		the MassPeak instances of \c this MassSpectrum

		\param iList pointer to a double list where to store all the intensity
		values of the MassPeak instances of \c this MassSpectrum

		The lists are filled in the same order as the MassPeak instances are listed
		in \c this MassSpectrum, thus reflecting perfectly the mass spectral data.

*/
	void
		MassSpectrum::toLists(QList<double> *mzList, QList<double> *iList) const
		{
			if(!mzList)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			if(!iList)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			for(int iter = 0; iter < size(); ++iter)
			{
				mzList->append(at(iter)->mz());
				iList->append(at(iter)->i());
			}

			return;
		}


	//! Fills in vectors of double values containing the data of \c this MassSpectrum.
	/*!

		The program crashes if any of the pointers is nullptr.

		\param mzVector pointer to a double vector where to store all the m/z values of
		the MassPeak instances of \c this MassSpectrum

		\param iVector pointer to a double vector where to store all the intensity
		values of the MassPeak instances of \c this MassSpectrum

		The vectors are filled in the same order as the MassPeak instances are listed
		in \c this MassSpectrum, thus reflecting perfectly the mass spectral data.

*/	void
	MassSpectrum::toVectors(QVector<double> *mzVector,
			QVector<double> *iVector) const
	{
		if(!mzVector || !iVector)
			qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

		for(int iter = 0; iter < size(); ++iter)
		{
			mzVector->append(at(iter)->mz());
			iVector->append(at(iter)->i());
		}

		return;
	}


	//! Fills in a map with the data of \c this MassSpectrum.
	/*!

		The program crashes if the pointer is nullptr.

		\param map pointer to a <double,double> map in which to store all the
		(m/z,i) pairs of this \c MassSpectrum.

		The map is ordered by definition in ascending order with respect to the key
		(the m/z value), irrespective of the value (the intensity value).

*/	void
	MassSpectrum::toMap(QMap<double, double> *map) const
	{
		if(!map)
			qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

		for(int iter = 0; iter < size(); ++iter)
		{
			map->insert(at(iter)->mz(), at(iter)->i());
		}

		return;
	}




	//! Allocates a text string filled with \c this MassSpectrum data.
	/*!

		For each MassPeak instance of \c this MassSpectrum, append a textual
		representation of the (m/z,i) pair to a heap-allocated string.

		\return a heap-allocated string containing a textual representation of \c this
		MassSpectrum. The object must be delete when no more in use.

*/
	QString *
		MassSpectrum::asText(int startIndex, int endIndex)
		{
			QList<double> mzLista = mzList();
			QList<double> iLista = iList();

			int massPeakCount = mzLista.size();

			if(massPeakCount == 0)
				return Q_NULLPTR;

			int indexStart = (startIndex <= 0 ? 0 : startIndex);
			if(indexStart >= massPeakCount - 1)
				indexStart = 0;

			int indexEnd = (endIndex <= 0 ? massPeakCount -1 : endIndex);
			if(indexEnd >= massPeakCount - 1)
				indexEnd = massPeakCount -1;

			QString *text = new QString();

			for(int iter = indexStart; iter <= indexEnd; ++iter)
			{
				text->append(QString("%1 %2\n")
						.arg(mzLista.at(iter), 0, 'f', 10)
						.arg(iLista.at(iter), 0, 'f', 10));
			}

			return text;
		}


	//! Allocates a new MassSpectrum filled with data.
	/*!

		The new MassSpectrum is filled with data obtained by decoding (and possibly
		decompressing) the data contained in the byte arrays passed as parameters.
		The data in the arrays are first decoded/uncompressed in vectors.

		Note that, if after decoding/uncompressing of the byte arrays, the obtained
		data vectors do not have the same size, the program crashes.

		Initialization of \c this MassSpectrum with the data contained in the
		vectors is performed by calling a specialized initialize function.

		\param mzByteArray byte array containing the m/z values

		\param iByteArray byte array containing the intensity values

		\return a newly allocated MassSpectrum filled with MassPeak instances
		created by using the (m/z,i) pairs contained in the byte arrays.

		\sa initialize(const QVector<double> &mzVector, const QVector<double> &iVector,
		double mzStart = qSNaN(), double mzEnd = qSNaN()).
		*/
	MassSpectrum *
		MassSpectrum::createFromBase64EncByteArrays(const QByteArray *mzByteArray,
				const QByteArray *iByteArray, int compressionType,
				double mzStart, double mzEnd)
		{
			if(mzByteArray == nullptr || iByteArray == nullptr)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			// Because pwiz provides the mass spectral data in the form of
			// std::vector<double> data, we need to encode these data before we
			// can inject them to the database.

			pwiz::msdata::BinaryDataEncoder::Config config;

			config.precision =
				pwiz::msdata::BinaryDataEncoder::Precision::Precision_64;

			// mzML standard stipulates that data are always
			// ByteOrder::ByteOrder_LittleEndian.

			config.byteOrder = pwiz::msdata::BinaryDataEncoder::ByteOrder::
				ByteOrder_LittleEndian;

			if(compressionType == msXpS::DataCompression::DATA_COMPRESSION_ZLIB)
				config.compression =
					pwiz::msdata::BinaryDataEncoder::Compression::Compression_Zlib;

			BinaryDataEncoder decoder(config);

			// Start by crunching the mz data.

			std::vector<double> mzStdVector;

			std::string mzString = mzByteArray->toStdString();

			decoder.decode(mzString, mzStdVector);

			QVector<double> mzVector = QVector<double>::fromStdVector(mzStdVector);

			// Clear the standard vector data.
			mzStdVector.clear();

			//qDebug() << __FILE__ << __LINE__
				//<< "Done creating mzVector of items: " << mzVector.size();

			// And now do the same for the i data.

			std::vector<double> iStdVector;

			std::string iString = iByteArray->toStdString();

			decoder.decode(iString, iStdVector);

			QVector<double> iVector = QVector<double>::fromStdVector(iStdVector);

			// Clear the standard vector data.
			iStdVector.clear();

			//qDebug() << __FILE__ << __LINE__
				//<< "Done creating iVector of items: " << iVector.size();

			// At this point we have all the data required to craft and fill-in a
			// new mass spectrum:

			// Sanity check.
			if(mzVector.size() != iVector.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			MassSpectrum *spectrum = new MassSpectrum;
			spectrum->initialize(mzVector, iVector, mzStart, mzEnd);

			// qDebug() << __FILE__ << __LINE__
			// << "Allocated new spectrum of " << mzVector.size() << "peaks";

			return spectrum;
		}

} // namespace msXpSlibmass
