/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */


#pragma once

#include <QVector>

#include "MassSpecDataSet.hpp"
#include "History.hpp"


namespace msXpSmineXpert
{


	class QCPColorMapData;

	//! The MassDataIntegrator class provides the integration calculations.
	/*!

		Any mass data mining session starts with the opening of a mass spectrometric
		data file. Immediately, the program computes a TIC chromatogram and a
		mz=f(dt) color map, if the data are for an ion mobility mass spectrometry
		experiment. From these two plots, the user starts making integrations to
		mine the data in the preferred direction. For example, one might want to
		look at the mass spectrum corresponding to the combination of all the
		spectra acquired between retention time1 and retention time2. The
		combination of the spectra actually correspond to an integration over the
		time lapse (time2 - time1). This class performs this kind of calculation.

*/
	class MassDataIntegrator : public QObject
	{
		Q_OBJECT;

		private:

		//! Mass spectral data set.
		MassSpecDataSet *mp_massSpecDataSet = Q_NULLPTR;

		// History of the data set to take into account for the integration.
		History m_history;

		// Tells if the integration was cancelled by the user.
		bool m_isOperationCancelled = false;

		// Vector of double values to hold the x-axis data.
		QVector<double> m_keyVector;

		// Vector of double values to hold the y-axis data.
		QVector<double> m_valVector;

		double m_ticIntensity;

		// Map relating x-axis data with y-axix data.
		QMap<double, double> m_map;

		// Map relating double data (dt, typically) with mass spectra.
		QHash<double, msXpSlibmass::MassSpectrum *> m_doubleMassSpecHash;


		public:

		MassDataIntegrator(MassSpecDataSet *massSpecDataSet);

		MassDataIntegrator(MassSpecDataSet *massSpecDataSet,
				History history);
		virtual ~MassDataIntegrator();

		const QVector<double> &keyVector() const;
		const QVector<double> &valVector() const;

		const QMap<double, double> &map();
		const QHash<double, msXpSlibmass::MassSpectrum *> &doubleMassSpecHash();

		History history() const;

		double ticIntensity() const;

		bool integrate();

		bool initialTic();
		bool streamInitialTic();

		int fillInDtMzHash(int* dtValues, int*mzCells, 
				double *firstKey, double *lastKey, 
				double *firstVal, double *lastVal);
		int streamFillInDtMzHash(int* dtValues, int*mzCells, 
				double *firstKey, double *lastKey, 
				double *firstVal, double *lastVal);

		bool integrateToRt();
		bool streamIntegrateToRt();

		bool integrateToMz();
		bool streamIntegrateToMz();

		bool integrateToDt();
		bool streamIntegrateToDt();

		bool integrateToTicIntensity();
		bool streamIntegrateToTicIntensity();

		public slots:

			//! React to the signal that tells the operation was cancelled by the
			//user.
			void cancelOperation();

		//! Relay a signal to another object.
		void setupFeedbackRelay(QString msg, int startValue, int endValue,
				int currentValue, int logType = LogType::LOG_TO_CONSOLE);

		//! Relay a signal to another object.
		void updateFeedbackRelay(QString msg, int currentValue,
				int logType = LogType::LOG_TO_CONSOLE);

signals:
		void setupFeedbackSignal(QString msg, int startValue, int endValue,
				int currentValue, int logType = LogType::LOG_TO_CONSOLE);

		void updateFeedbackSignal(QString msg, int currentValue,
				int logType = LogType::LOG_TO_CONSOLE);

		void cancelOperationSignal();
	};


} // namespace msXpSmineXpert
