/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */


/////////////////////// Qt includes
#include <QDebug>

/////////////////////// Local includes
#include "History.hpp"


namespace msXpSmineXpert
{

	//! Construct a HistoryItem instance
	/*!

		The instance has its date-time member datum set to current date and time and
		is further initialized by running initializeIntegrationTypes(), that
		initializes the \c m_intTypeDescMap member map.

		\sa initializeIntegrationTypes()

*/
	HistoryItem::HistoryItem()
	{
		m_dateTime = QDateTime::currentDateTimeUtc();

		initializeIntegrationTypes();
	}


	//! Construct a HistoryItem instance as a copy of \p other
	/*!

		\c this instance is initialized by running initializeIntegrationTypes(), that
		initializes the \c m_intTypeDescMap member map.

		Further initialization is performed as a deep copy of the member data in \p
		other into \c this instance.

		\param other HistoryItem instance to use for the initialization of \c this
		instance

		\sa HistoryItem()

*/
	HistoryItem::HistoryItem(const HistoryItem &other)
	{

		// Fill in all the Integration types that we know in the "static" QMap
		// member.

		initializeIntegrationTypes();

		m_dateTime = other.m_dateTime;

		// And now duplicate the other map.
		QList<IntegrationRange *> list = other.m_integrationMap.values();

		for(int iter = 0; iter < list.size(); ++iter)
		{
			IntegrationRange *intRange = list.at(iter);
			int intType = other.m_integrationMap.key(intRange);

			IntegrationRange *newIntRange = new IntegrationRange;
			newIntRange->start = intRange->start;
			newIntRange->end = intRange->end;

			m_integrationMap.insert(intType, newIntRange);
		}
	}


	//! Assignment operator.
	/*!

		The assignment of \p other to \c this instance runs a deep copy of the
		member data in \p other into \c this instance.

		Note that the m_intTypeDescMap is initialized by calling
		initializeIntegrationTypes().

		\param other HistoryItem instance to be used for the initialization of \c
		this instance.

		\return a reference to \c this HistoryItem instance.

*/
	HistoryItem &
		HistoryItem::operator=(const HistoryItem &other)
		{
			// Fill in all the Integration types that we know in the "static" QMap
			// member.

			initializeIntegrationTypes();

			m_dateTime = other.m_dateTime;

			// And now duplicate the map.
			QList<IntegrationRange *> list = other.m_integrationMap.values();

			for(int iter = 0; iter < list.size(); ++iter)
			{
				IntegrationRange *intRange = list.at(iter);
				int intType = other.m_integrationMap.key(intRange);

				IntegrationRange *newIntRange = new IntegrationRange;
				newIntRange->start = intRange->start;
				newIntRange->end = intRange->end;

				m_integrationMap.insert(intType, newIntRange);
			}

			return *this;
		}


	//! Destruct \c this HistoryItem instance
	/*!

		All the heap-allocated IntegrationRange and IntegrationTypeDesc instances
		are deleted.

*/
	HistoryItem::~HistoryItem()
	{
		QList<IntegrationRange *> irList = m_integrationMap.values();

		while(irList.size())
			delete irList.takeFirst();

		// Finally, clear the map.
		m_integrationMap.clear();

		QList<IntegrationTypeDesc *> itdList = m_intTypeDescMap.values();

		while(itdList.size())
			delete itdList.takeFirst();

		// Finally, clear the map.
		m_intTypeDescMap.clear();
	}


	//! Initialize the IntegrationType vs IntegrationTypeDesc map
	/*!

		The various IntegrationTypeDesc structures required to document the various
		IntegrationType elements of the enumeration are allocated and filled-in in
		this function. The \c m_intTypeDescMap map is then filled-in. See code for
		the details.

		Note that this function is called by the constructors and the assignement
		operator.

		\sa HistoryItem()

*/
	void
		HistoryItem::initializeIntegrationTypes()
		{

			// The m_intTypeDescMap member is "static". Fill-in the data once and for
			// all.

			int intType = IntegrationType::NOT_SET;
			IntegrationTypeDesc *intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "NOT_SET";
			intTypeDesc->detailed = "Not set";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::DATA_TO_RT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[FILE->RT]";
			intTypeDesc->detailed = "TIC from data";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::DATA_TO_MZ;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[FILE->MZ]";
			intTypeDesc->detailed = "Mass spectrum from data";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::DATA_TO_DT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[FILE->DT]";
			intTypeDesc->detailed = "Drift spectrum from data";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::RT_TO_MZ;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[RT->MZ]";
			intTypeDesc->detailed = "Mass spectrum from Tic chromatogram";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::RT_TO_DT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[RT->DT]";
			intTypeDesc->detailed = "Drift time spectrum from Tic chromatogram";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::RT_TO_TIC_INT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[RT->TICint]";
			intTypeDesc->detailed = "TIC intensity (single value) from Tic chromatogram";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::MZ_TO_RT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[MZ->RT]";
			intTypeDesc->detailed = "XIC chromatogram from mass spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::MZ_TO_MZ;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[MZ->MZ]";
			intTypeDesc->detailed = "Mass spectrum from mass spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::MZ_TO_DT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[MZ->DT]";
			intTypeDesc->detailed = "Drift time spectrum from mass spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::MZ_TO_TIC_INT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[MZ->TICint]";
			intTypeDesc->detailed = "TIC intensity (single value) from mass spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::DT_TO_RT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[DT->RT]";
			intTypeDesc->detailed = "XIC chromatogram from drift time spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::DT_TO_MZ;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[DT->MZ]";
			intTypeDesc->detailed = "Mass spectrum from drift time spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::DT_TO_DT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[DT->DT]";
			intTypeDesc->detailed = "Drift spectrum from drift time spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::DT_TO_TIC_INT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[DT->TICint]";
			intTypeDesc->detailed = "TIC intensity (single value) from drift spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::RT_TO_ANY;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[RT-to-any]";
			intTypeDesc->detailed = "Tic to any";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::MZ_TO_ANY;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[MZ-to-any]";
			intTypeDesc->detailed = "Mass spectrum to any";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::DT_TO_ANY;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[DT-to-any]";
			intTypeDesc->detailed = "Drift spectrum to any";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::ANY_TO_RT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[any-to-RT]";
			intTypeDesc->detailed = "Any to Tic";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::ANY_TO_MZ;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[any-to-MZ]";
			intTypeDesc->detailed = "Any to Mass spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::ANY_TO_DT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[any-to-DT]";
			intTypeDesc->detailed = "Any to Drift spectrum";
			m_intTypeDescMap.insert(intType, intTypeDesc);

			intType = IntegrationType::ANY_TO_TIC_INT;
			intTypeDesc = new IntegrationTypeDesc;
			intTypeDesc->brief = "[any-to-DT]";
			intTypeDesc->detailed = "Any to TIC intensity (single value)";
			m_intTypeDescMap.insert(intType, intTypeDesc);
		}


	//! Equality operator.
	/*!

		Two HistoryItem instances are equal if all their members are identical. Note
		that the order of the items in the \c m_integrationMap is not relevant and
		that the \c m_intTypeDescMap is not checked since it is considered
		immutable.

		Note that the IntegrationType instances pointed to by items in the \c
		m_integrationMap map are tested for \c start and \c end values, not by
		pointer value.

		\param other reference to a HistoryItem instance to compare to \c this
		instance.

		\return true if both HistoryItem instances are identical.

		\sa contains()

*/
	bool
		HistoryItem::operator==(const HistoryItem &other) const
		{
			if(this == &other)
				return true;

			if(m_dateTime != other.m_dateTime)
				return false;

			// Now we need to compare all the pairs in the map.

			if(m_integrationMap.size() != other.m_integrationMap.size())
				return false;

			QList<int> intTypeList = m_integrationMap.keys();

			for(int iter = 0; iter < intTypeList.size(); ++iter)
			{
				int intType = intTypeList.at(iter);

				IntegrationRange *intRange = m_integrationMap.value(intType);

				if(other.contains(intType, *intRange))
					continue;
				else
					return false;
			}

			return true;
		}


	//! Verifies if the \c m_integrationMap contains an item
	/*!

		The \c m_integrationMap map is checked for having a value mapped to the \p
		intType parameter that has the same IntegrationRange values start and end as
		the \p &intRange.

		\param intType IntegrationType key to search in the \c m_integrationMap.

		\param intRange reference to an IntegrationRange instance to be used for the
		comparison.

		\return True if an IntegrationRange instance is found as value in \c
		m_integrationMap that has the same \c start and \c end member values as \p
		&intRange.

		\sa operator==()

*/
	bool
		HistoryItem::contains(int intType, const IntegrationRange &intRange) const
		{
			IntegrationRange *localRange = m_integrationMap.value(intType, Q_NULLPTR);

			if(localRange == Q_NULLPTR)
				return false;

			if(localRange->start == intRange.start && localRange->end == intRange.end)
				return true;

			return false;
		}


	//! Set the date and time to the current date and time
	/*!

		The date and time format is UTC (universal time coordinated).

*/
	void
		HistoryItem::setDateTime()
		{
			m_dateTime = QDateTime::currentDateTimeUtc();
		}


	//! Get the date and time of \c this HistoryItem instance.
	/*!

		\return The date and time in QDateTime format.

		\sa setDateTime()

*/
	QDateTime
		HistoryItem::dateTime() const
		{
			return m_dateTime;
		}


	//! Checks if \c this HistoryItem instance has an IntegrationType
	/*!

		The \c m_integrationMap map is queried for a key having the same value as \p integrationType.

		\param integrationType IntegrationType value to search as a key of the \c
		m_integrationMap map.

		\return true if \c m_integrationMap has at an integration type identical to
		\p integrationType; false otherwise.

		\sa

*/
	bool
		HistoryItem::hasIntegrationType(int integrationType)
		{
			QList<int> list = m_integrationMap.keys();

			if(list.contains(integrationType))
				return true;

			return false;
		}


	//! Creates a new IntegrationRange instance and inserts it in the member map
	/*!

		\param intType IntegrationType value used for the map insertion

		\param start start of range used for the creation of the IntegrationRange instance

		\param end end of range used for the creation of the IntegrationRange instance


*/
	void
		HistoryItem::newIntegrationRange(int intType, double start,
				double end)
		{
			IntegrationRange *intRange = new IntegrationRange;
			intRange->start = start;
			intRange->end = end;

			m_integrationMap.insert(intType, intRange);
		}


	IntegrationRange *
		HistoryItem::integrationRange(int intType) const
		{
			IntegrationRange *intRange = m_integrationMap.value(intType, Q_NULLPTR);
			return intRange;
		}


	int  
		HistoryItem::integrationType(IntegrationRange *integrationRange)
		{
			if(integrationRange == Q_NULLPTR)
				qFatal("Fatal error at %s@%d -- %s. "
						"Q_NULLPTR."
						"Program aborted.",
						__FILE__, __LINE__, __FUNCTION__);

			return m_integrationMap.key(integrationRange);
		}


	//! Get a list of IntegrationType values matching the integration ranges.
	/*!

		A given HistoryItem instance might have more than one integration range, and
		for each one of these integration ranges, it might have different
		IntegrationType values. This function returns the list of keys of the \c
		m_integrationMap QMap<IntegrationType, IntegrationRange *> map.

		One typical situation is when performing an integration from the color map
		widget, that is, integrating using two ranges: the mz range and the dt range
		(ion mobility mass spectrometry only).

*/
	QList<int> 
		HistoryItem::integrationTypes() const
		{
			return m_integrationMap.keys();
		}


	QList<IntegrationRange *> 
		HistoryItem::integrationRanges() const
		{
			return m_integrationMap.values();
		}


	bool
		HistoryItem::isValid() const
		{
			if(!m_dateTime.isValid())
				return false;

			if(m_integrationMap.isEmpty())
				return false;

			return true;
		}


	QString
		HistoryItem::asText(bool brief) const
		{
			QString text;

			// For each integration range in the map, craft a text element.

			QList<IntegrationRange *> list = m_integrationMap.values();

			//qDebug() << __FILE__ << __LINE__ << "There are " << list.size()
			//<< " integration ranges in the map";

			for(int iter = 0; iter < list.size(); ++iter)
			{
				IntegrationRange *intRange = list.at(iter);
				int intType = m_integrationMap.key(intRange);

				qDebug() << __FILE__ << __LINE__
					<< "Currently iterated intType:" << intType;

				IntegrationTypeDesc *intTypeDesc = m_intTypeDescMap.value(intType);

				if(intTypeDesc == nullptr)
					qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

				QString label;

				if(brief)
					label = intTypeDesc->brief + " ";
				else
					label = intTypeDesc->detailed + ":\n";

				if(brief)
				{
					text += label +
						QString("range: [%1-%2]")
						.arg(intRange->start, 0, 'f', 3)
						.arg(intRange->end, 0, 'f', 3);

					if(iter > 0 && iter < list.size() - 1)
						text += " | ";
					if(iter == list.size() - 1)
						text += "\n";
				}
				else
					text += label +
						QString("\tRange: [%1-%2]\n")
						.arg(intRange->start, 0, 'f', 3)
						.arg(intRange->end, 0, 'f', 3);
			}

			// Finally, only print the date time if we are verbose.
			if(!brief)
			{
				text += "\t" + m_dateTime.toString(Qt::ISODate);
				text += "\n";
			}


			// qDebug() << __FILE__ << __LINE__ << "text: " << text;

			return text;
		}


	//! Construct an emtpy History instance.
	/*!

*/
	History::History()
	{
	}


	//! Construct a History instance as the copy of another.
	/*!

		The copy is deep, with all the HistoryItem instances referenced in \c
		m_historyItemList being copied to newly allocated ones.

		\param other reference to another History instance to be used for the copy.

		\sa History()

*/
	History::History(const History &other)
	{
		for(int iter = 0; iter < other.m_historyItemList.size(); ++iter)
			m_historyItemList.append(new HistoryItem(*(other.m_historyItemList.at(iter))));
	}


	//! Assignment operator.
	/*!

		Copies all the member data of the other History item into \c this instance.
		The copying is deep, with all the items referenced in \c m_historyItemList being
		coied to newly allocated ones.

		\param other reference to the other History instance to be used for the copy.

		\return a reference to \c this History instance.

*/
	History &
		History::operator=(const History &other)
		{
			if(&other == this)
				return *this;

			for(int iter = 0; iter < other.m_historyItemList.size(); ++iter)
				m_historyItemList.append(new HistoryItem(*(other.m_historyItemList.at(iter))));

			return *this;
		}


	//! Destruct this History instance.
	/*!

		All the items in the member \c m_historyItemList are deleted.

		\sa freeList()

*/
	History::~History()
	{
		freeList();
	}


	//! Deletes all the items in the History item list.
	/*!

*/
	void
		History::freeList()
		{
			while(m_historyItemList.size())
				delete m_historyItemList.takeFirst();

			m_historyItemList.clear();
		}


	//! Tell if History contains a HistoryItem.
	/*!

		Iterates in \c m_historyItemList and for each HistoryItem checks if it is identical
		to \p other. Note that the comparison is deep, that is, the values inside the
		member data are tested.

		\param other HistoryItem instance to be searched for in this History item.

		\return true if a HistoryItem instance was found identical to \p other.

		\sa HistoryItem::operator==()

*/
	int
		History::containsHistoryItem(const HistoryItem &item)
		{
			for(int iter = 0; iter < m_historyItemList.size(); ++iter)
			{
				HistoryItem *curItem = m_historyItemList.at(iter);

				if(*curItem == item)
					return iter;
			}

			return -1;
		}


	//! Get the first HistoryItem that has a specific IntegrationType.
	/*!

		Iterate in m_historyItemList in search for a HistoryItem having the same
		IntegrationType as \p intType. As soon as one is found its IntegrationRange
		pointer is set to \p intRange.

		\param intType IntegrationType value to search for.

		\param intRange slot to set the pointer to the IntegrationRange instance
		pointer of the HistoryItem having the right IntegrationType.

		\return a pointer to the HistoryItem instance containing an IntegrationType
		\p intType.

		\sa lastHistoryItem()

*/
	HistoryItem *
		History::firstHistoryItem(int  intType, IntegrationRange **intRange)
		{
			// We want to return the first item that has integrationType.
			// We iterate in normal order because we want to return the first added
			// item.

			for(int iter = 0; iter < m_historyItemList.size(); ++iter)
			{
				HistoryItem *item = m_historyItemList.at(iter);

				if(item->hasIntegrationType(intType))
				{
					if(intRange != nullptr)
						*intRange = item->integrationRange(intType);

					return item;
				}
			}

			return nullptr;
		}


	//! Get the last HistoryItem that has a specific IntegrationType.
	/*!

		Reverse-iterate in m_historyItemList in search for a HistoryItem having the same
		IntegrationType as \p intType. As soon as one is found its IntegrationRange
		pointer is set to \p intRange.

		\param intType IntegrationType value to search for.

		\param intRange slot to set the pointer to the IntegrationRange instance
		pointer of the HistoryItem having the right IntegrationType.

		\return a pointer to the HistoryItem instance containing an IntegrationType
		\p intType.

		\sa firstHistoryItem().

*/
	HistoryItem *
		History::lastHistoryItem(int intType, IntegrationRange **intRange)
		{
			// We want to return the last item that has integration type intType.
			// We iterate in reverse order because we want to return the last added
			// item.

			for(int iter = m_historyItemList.size() - 1; iter >= 0; --iter)
			{
				HistoryItem *item = m_historyItemList.at(iter);

				if(item->hasIntegrationType(intType))
				{
					if(intRange != nullptr)
						*intRange = item->integrationRange(intType);

					return item;
				}
			}

			return nullptr;
		}


	//! Return the most recent HistoryItem instance in \c this History.
	/*!

		Iterates in \c m_historyItemList and checks for the most recent HistoryItem
		element of the list.

		\return The most recent HistoryItem instance in \c this History.

*/
	const HistoryItem *
		History::newestHistoryItem() const
		{
			if(m_historyItemList.isEmpty())
			{
				return Q_NULLPTR;
			}

			// Seed the holder time from the first history item and set the index to
			// that first history item. Most probably, that will be the oldest-dated
			// item because history items are appended to the list as the time goes
			// by.

			QDateTime newestDateTime = m_historyItemList.first()->m_dateTime;
			int newestDateTimeIndex = 0;

			//qInfo() << __FILE__ << __LINE__ << __FUNCTION__ 
				//<< "First history item:" << m_historyItemList.first()->asText();

			for(int iter = 1; iter < m_historyItemList.size(); ++iter)
			{
				//qInfo() << __FILE__ << __LINE__ << __FUNCTION__ 
					//<< "iterating in second history item.";

				HistoryItem *historyItem = m_historyItemList.at(iter);

				QDateTime iterDateTime = historyItem->m_dateTime;

				if(iterDateTime > newestDateTime)
				{
					//qInfo() << __FILE__ << __LINE__ << __FUNCTION__ 
						//<< "This history item is more recent:"
						//<< historyItem->asText();

					newestDateTime = iterDateTime;
					newestDateTimeIndex = iter;
				}
			}

			return m_historyItemList.at(newestDateTimeIndex);
		}


	//! Copy a History instance to \c this History.
	/*!

		The \p history History instance is copied deeply to \c this History.
		HistoryItem elements are appended to \c this History. Upon copying, if
		HistoryItem elements of \p history are found in \c this History, they are
		not duplicated if \p noDuplicates is true.

		\param history reference to History to be copied into \c this History.

		\param noDuplicates boolean value telling if HistoryItem element in \p
		history found in \c this History may be duplicated or not.

		\sa appendHistoryItem()

*/
	void
		History::copyHistory(const History &history, bool noDuplicates)
		{

			// qDebug() << __FILE__ << __LINE__
			//<< "the history to be appended:" << history.asText();

			for(int iter = 0; iter < history.m_historyItemList.size(); ++iter)
			{
				HistoryItem *item = history.m_historyItemList.at(iter);

				if(containsHistoryItem(*item) >= 0 && noDuplicates)
				{
					qDebug() << __FILE__ << __LINE__
						<< "Not appending history item as it would be duplicated.";
				}
				else
				{
					HistoryItem *newItem = new HistoryItem(*item);
					m_historyItemList.append(newItem);
				}
			}
		}


	//! Append a HistoryItem instance to \c this History.
	/*!

		The \p item HistoryItem instance pointer is appended to \c m_historyItemList.
		Ownership of the HistoryItem instance is transferred to \c this History.

		\param item pointer to the HistoryItem instance to take ownership of.

		\return The index of the appended item.

		\sa duplicateHistoryItem().

*/
	int
		History::appendHistoryItem(HistoryItem *item)
		{
			if(item == nullptr)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			m_historyItemList.append(item);

			return m_historyItemList.size() - 1;
		}


	//! Copy a HistoryItem into \c this History.
	/*!

		The \p item HistoryItem is copied to a newly allocated item that is then
		appended to \c this History. Note that there is no check to verify if \c
		this History contains a HistoryItem instance identical to \p item.

		\param item reference to the HistoryItem to be copied into \c this History.

		\return the index of the newly appended item.

		\sa appendHistoryItem()

*/
	int
		History::copyHistoryItem(const HistoryItem &item)
		{
			HistoryItem *newItem = new HistoryItem(item);
			m_historyItemList.append(newItem);

			return m_historyItemList.size() - 1;
		}


	//! Search a RT range in the HistoryItem elements of \c this History.
	/*!

		Iterate in m_historyItemList and for each HistoryItem check if they have an
		IntegrationRange of type RT_*.  While iterating, there might be more than
		one HistoryItem having IntegrationRange of type RT_*. Each time one is
		found, its IntegrationRange members start and end are checked and the
		HistoryItem having the IntegrationRange greatest \c start value and the
		smallest \c end value is retained. These "innermost" range \c start and \c
		end values are copied into \p start and \p end double value pointers passed
		as parameters.

		\param start pointer to double to receive the range \c start value.

		\param end pointer to double to receive the range \c end value.

		\return true if a HistoryItem of IntegrationType RT* was found, false
		otherwise.

		\sa innermostMzRange()

		\sa innermostDtRange()

*/
	bool
		History::innermostRtRange(double *start, double *end) const
		{
			if(start == nullptr || end == nullptr)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			double tempStart = -1;
			double tempEnd = -1;

			bool itemFound = false;
			bool firstItem = true;

			QList<IntegrationType> intTypeList{IntegrationType::RT_TO_MZ,
				IntegrationType::RT_TO_DT,
				IntegrationType::RT_TO_TIC_INT,
				IntegrationType::ANY_TO_RT};

			// Iterate in the list of HistoryItem instances and for each check if there
			// is a (IntegrationType, IntegrationRange) pair in the QMap that matches
			// one of the RT_XXXX integration types. Thas is we search for integrations
			// that are defined *starting* from an RT range. If so store the start and
			// end values in temporary variables and make sure we finally get the
			// greatest start value and the smallest end value (we are seeking innermost
			// range values.

			for(int iter = 0; iter < m_historyItemList.size(); ++iter)
			{
				HistoryItem *item = m_historyItemList.at(iter);

				// Each history item my have one of the IntegrationType values related
				// to RT integration. Let's test them all in sequence.

				for(int iter = 0; iter < intTypeList.size(); ++iter)
				{
					IntegrationType intType = intTypeList.at(iter);

					IntegrationRange *intRange = item->integrationRange(intType);

					if(intRange != nullptr)
					{
						if(firstItem)
						{
							tempStart = intRange->start;
							tempEnd = intRange->end;

							firstItem = false;
						}
						else
						{
							// We seek the *innermost* range, so be aware of the
							// relational < >
							// operators.
							if(intRange->start > tempStart)
								tempStart = intRange->start;
							if(intRange->end < tempEnd)
								tempEnd = intRange->end;
						}

						// Let the caller know that we found at least one item and thus
						// that the start and end params now have the encountered
						// values.

						itemFound = true;
					}
					else
						// The returned intRange is nullptr.
						continue;
				}
			}
			// We have finally iterated in all the HistoryItem elements and we can
			// return the range values.
			*start = tempStart;
			*end = tempEnd;

			return itemFound;
		}


	//! Search a MZ range in the HistoryItem elements of \c this History.
	/*!

		Iterate in m_historyItemList and for each HistoryItem check if they have an
		IntegrationRange of type MZ_*.  While iterating, there might be more than
		one HistoryItem having IntegrationRange of type MZ_*. Each time one is
		found, its IntegrationRange members start and end are checked and the
		HistoryItem having the IntegrationRange greatest \c start value and the
		smallest \c end value is retained. These "innermost" range \c start and \c
		end values are copied into \p start and \p end double value pointers passed
		as parameters.

		\param start pointer to double to receive the range \c start value.

		\param end pointer to double to receive the range \c end value.

		\return true if a HistoryItem of IntegrationType MZ* was found, false
		otherwise.

		\sa innermostRtRange()

		\sa innermostDtRange()

*/
	bool
		History::innermostMzRange(double *start, double *end) const
		{
			if(start == nullptr || end == nullptr)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			double tempStart = -1;
			double tempEnd = -1;

			bool itemFound = false;
			bool firstItem = true;

			for(int iter = 0; iter < m_historyItemList.size(); ++iter)
			{
				HistoryItem *item = m_historyItemList.at(iter);

				// A history item might contain more than one integration range.
				// Ask a list of these.

				QList<IntegrationRange *> intRangeList = item->integrationRanges();

				for(int iter = 0; iter < intRangeList.size(); ++iter)
				{
					IntegrationRange *intRange = intRangeList.at(iter);

					if(item->integrationType(intRange) & MZ_TO_ANY)
					{
						if(firstItem)
						{
							tempStart = intRange->start;
							tempEnd = intRange->end;

							firstItem = false;
						}
						else
						{

							// We seek the *innermost* range, so be aware of the relational <
							// > operators that may look used at reverse but are not.

							if(intRange->start > tempStart)
								tempStart = intRange->start;

							if(intRange->end < tempEnd)
								tempEnd = intRange->end;
						}

						itemFound = true;
					}
				}
				// End of
				// for(int iter = 0; iter < intRangeList.size(); ++iter)

			}
			// End of
			// for(int iter = 0; iter < m_historyItemList.size(); ++iter)

			// We have finally iterated in all the HistoryItem elements and we can
			// return the range values.
			*start = tempStart;
			*end = tempEnd;

			return itemFound;
		}


	//! Search a DT range in the HistoryItem elements of \c this History.
	/*!

		Iterate in m_historyItemList and for each HistoryItem check if they have an
		IntegrationRange of type DT_*.  While iterating, there might be more than
		one HistoryItem having IntegrationRange of type DT_*. Each time one is
		found, its IntegrationRange members start and end are checked and the
		HistoryItem having the IntegrationRange greatest \c start value and the
		smallest \c end value is retained. These "innermost" range \c start and \c
		end values are copied into \p start and \p end double value pointers passed
		as parameters.

		\param start pointer to double to receive the range \c start value.

		\param end pointer to double to receive the range \c end value.

		\return true if a HistoryItem of IntegrationType DT* was found, false
		otherwise.

		\sa innermostMzRange()

		\sa innermostRtRange()

*/
	bool
		History::innermostDtRange(double *start, double *end) const
		{
			if(start == nullptr || end == nullptr)
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			double tempStart = -1;
			double tempEnd = -1;

			bool itemFound = false;
			bool firstItem = true;

			for(int iter = 0; iter < m_historyItemList.size(); ++iter)
			{
				HistoryItem *item = m_historyItemList.at(iter);

				//qDebug() << __FILE__ << __LINE__ << "Iterating in new history item.";

				// A history item might contain more than one integration range.
				// Ask a list of these.

				QList<IntegrationRange *> intRangeList = item->integrationRanges();

				for(int iter = 0; iter < intRangeList.size(); ++iter)
				{
					IntegrationRange *intRange = intRangeList.at(iter);

					if(item->integrationType(intRange) & DT_TO_ANY)
					{
						if(firstItem)
						{
							tempStart = intRange->start;
							tempEnd = intRange->end;

							firstItem = false;
						}
						else
						{

							// We seek the *innermost* range, so be aware of the relational <
							// > operators that may look used at reverse but are not.

							if(intRange->start > tempStart)
								tempStart = intRange->start;

							if(intRange->end < tempEnd)
								tempEnd = intRange->end;
						}

						itemFound = true;
					}
				}
				// End of
				// for(int iter = 0; iter < intRangeList.size(); ++iter)

			}
			// End of
			// for(int iter = 0; iter < m_historyItemList.size(); ++iter)

			// We have finally iterated in all the HistoryItem elements and we can
			// return the range values.
			*start = tempStart;
			*end = tempEnd;

			return itemFound;
		}


	//! Search an integration range in the HistoryItem elements of \c this History.
	/*!

		The IntegrationRange must be for the last history item added to the history,
		as determined by looking at the date time member datum.  The found
		IntegrationRange \c start and \c end values are then copied to \p start and
		\p end parameters.

		This function is actually a dispatcher function calling specialized
		functions (see the See also section below).

		\param start pointer to double to receive the range \c start value.

		\param end pointer to double to receive the range \c end value.

		\return true if a HistoryItem was found, false otherwise.

		\sa innermostRtRange()

		\sa innermostMzRange()

		\sa innermostDtRange()

*/
	bool
		History::innermostRange(int integrationType, double *start,
				double *end) const
		{
			if(integrationType & IntegrationType::RT_TO_ANY)
				return innermostRtRange(start, end);
			else if(integrationType & IntegrationType::MZ_TO_ANY)
				return innermostMzRange(start, end);
			else if(integrationType & IntegrationType::DT_TO_ANY)
				return innermostDtRange(start, end);

			return false;
		}



	//! Craft a string with a textual representation of integration ranges.
	/*!

		For each kind of IntegrationRange (RT, MZ, DT), gets the innermost range out
		of \c this History and appends a textual representation of that integration
		range. The whole crafted string is returned.

		\return As string with all the textual representations of the found
		integration innermost ranges.

		\sa innermostRange()

*/
	QString
		History::innermostRangesAsText() const
		{
			QString text = "History - innermost ranges:\n";

			double startValue;
			double endValue;

			if(innermostRtRange(&startValue, &endValue))
				text += QString("RT range: [%1-%2]\n")
					.arg(startValue, 0, 'f', 3)
					.arg(endValue, 0, 'f', 3);
			if(innermostMzRange(&startValue, &endValue))
				text += QString("MZ range: [%1-%2]\n")
					.arg(startValue, 0, 'f', 5)
					.arg(endValue, 0, 'f', 5);
			if(innermostDtRange(&startValue, &endValue))
				text += QString("DT range: [%1-%2]\n")
					.arg(startValue, 0, 'f', 10)
					.arg(endValue, 0, 'f', 10);

			return text;
		}


	//! Tells if \c this History is valid.
	/*!

		A History is valid if all of its HistoryItem elements are valid.

		\return true if \c this History is valid; false otherwise.

		\sa

*/
	bool
		History::isValid() const
		{
			for(int iter = 0; iter < m_historyItemList.size(); ++iter)
			{
				if(!m_historyItemList.at(iter)->isValid())
					return false;
			}

			return true;
		}


	//! Craft a string with a textual representation of \c this History
	/*!

		Iterate in all the HistoryItem elements of \c this History and get, for each
		element, a string with a textual representation of its contents. Append that
		string to a local string.

		\param header string to be used as a header in the final returned string

		\param footer string to be used as a footer in the final returned string

		\param brief boolean value telling if the brief description of the
		IntegrationType is to be used. The detailed description is used if that
		boolean value is false.

		\return the string with a textual representation of \c this History.

		\sa

*/
	QString
		History::asText(const QString &header, const QString &footer,
				bool brief) const
		{
			QString text = header;

			for(int iter = 0; iter < m_historyItemList.size(); ++iter)
				text += m_historyItemList.at(iter)->asText(brief);

			text += footer;

			return text;
		}


} // namespace msXpSmineXpert
