/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



///////////////////////////// Qt include


///////////////////////////// Local includes
#include "ColorMapPlotWidget.hpp"
#include "ColorMapWnd.hpp"
#include "AnalysisPreferences.hpp"
#include "MainWindow.hpp"
#include "History.hpp"
#include "MassDataIntegrator.hpp"


namespace msXpSmineXpert
{


	//! Construct a ColorMapPlotWidget instance.
	/*!

		The instance will be initialized with the parameters.

		\param parent parent widget.

		\param parentWnd parent window.

		\param name name of this window to be used for the decoration (title)

		\param desc description of this window to be used for the plot decoration.

		\param massSpecDataSet MassSpecDataSet instance in which to get the data.
		Defaults to nullptr.

		\param fileName name of the file from which the data were initially read
		(also for decoration of the plots). Defaults to the empty string.

		\param isMultiGraph tells if the graph should be considered to be multi.
		Defaults to false.

*/
	ColorMapPlotWidget::ColorMapPlotWidget(QWidget *parent,
			AbstractMultiPlotWnd *parentWnd,
			const QString &name,
			const QString &desc,
			MassSpecDataSet *massSpecDataSet,
			const QString &fileName,
			bool isMultiGraph)
		: AbstractPlotWidget{parent, parentWnd, name,
			desc, massSpecDataSet, fileName, isMultiGraph}
	{
		registerQtKey(Qt::Key_R); // Retention time (TIC chromatogram) calculation
		registerQtKey(Qt::Key_S); // Mass spectrum calculation
		registerQtKey(Qt::Key_D); // Drift spectrum calculation
		registerQtKey(Qt::Key_I); // TIC intensity calculation

		// Give the axes some labels:
		xAxis->setLabel("drift time (ms)");
		yAxis->setLabel("m/z");

		clearPlottables();

		setupWidget();

		mp_colorMap = new QCPColorMap(xAxis, yAxis);
		addPlottable(mp_colorMap);
	}


	//! Destruct \c this ColorMapPlotWidget instance.
	ColorMapPlotWidget::~ColorMapPlotWidget()
	{
	}


	//! Return the integration type that yielded the data in this plot.
	int	
		ColorMapPlotWidget::arbitraryIntegrationType()
		{
			return 
				IntegrationType::DATA_TO_DT | IntegrationType::DATA_TO_MZ;
		}


	//! Setup the widgetry.
	bool
		ColorMapPlotWidget::setupWidget()
		{
			QPen pen(m_pen);

			pen.setColor("white");
			m_zoomRect->setPen(pen);
			m_descText->setColor("white");
			m_hStartTracer->setPen(pen);
			m_selectLine->setPen(m_pen);
			m_xDeltaText->setColor("white");

			return true;
		}


	//! Manually set the data for one of the color map data points.
	/*!

		Since this widget is a color map plot widget, the z component of the (x,y,z)
		triad will be converted into a color that matches the z value in a color
		scale.

*/
	void 
		ColorMapPlotWidget::setColorMapPointData(double key, double val, double z)
		{
			mp_colorMap->data()->setData(key, val, z);
		}


	//! Initialize the color map.
	/*!

		\param keySize the number of cells on the x-axis (keys)

		\param valSize the number of cells on the y-axix (values)

		\param keyRange the borders of the x-axis

		\param valRange the borders on the y-axis

		\param fillingValue value to use to fill-in all the (\p keySize * \p
		valSize) cells of the map.

*/
	void 
		ColorMapPlotWidget::setColorMapSizeAndRanges(int keySize, int valSize, 
				QCPRange keyRange, QCPRange valRange, double fillingValue)
		{
			mp_colorMap->data()->setSize(keySize, valSize);
			mp_colorMap->data()->setRange(keyRange, valRange);
			mp_colorMap->data()->fill(fillingValue);
		}


	//! Set the color to be used for decorating the color map.
	/*!

		When created, a new color map must have an identifying color that matches
		the color of the initial TIC chromatogram plot so that it can be related to
		the same initial data set.

*/
	void ColorMapPlotWidget::setDecoratingColor(const QColor &color)
	{
		QCPAxis *xAxis = mp_colorMap->keyAxis();
		QCPAxis *yAxis = mp_colorMap->valueAxis();

		QPen basePen;
		basePen = xAxis->basePen();
		basePen.setColor(color);

		xAxis->setBasePen(basePen);
		xAxis->setLabelColor(color);
		xAxis->setTickLabelColor(color);

		yAxis->setBasePen(basePen);
		yAxis->setLabelColor(color);
		yAxis->setTickLabelColor(color);
	}


	//! Choose the color scale (color gradient) to be used for the color map plot.
	void ColorMapPlotWidget::setColorMapGradient(QCPColorGradient colorGradient)
	{
		mp_colorMap->setGradient(colorGradient);
	}


	//! Change the scales according to new ranges for x, y and z axes.
	/*!

		This function is typically called after having manually set the cell values
		in the color map, to ensure that the map is redraw with proper scales.

*/
	void 
		ColorMapPlotWidget::rescaleColorMapDataRange(bool recalculateDataBounds)
		{
			mp_colorMap->rescaleDataRange(recalculateDataBounds);
		}


	//! Handle the keyboard key press along with the mouse button release event.
	void
		ColorMapPlotWidget::mouseReleaseHandledEvent(QMouseEvent *event)
		{
			if(m_pressedKeyCode == Qt::Key_R && event->button() == Qt::RightButton)
			{
				integrateToRt(xRangeMin(), xRangeMax(),
						yRangeMin(), yRangeMax());
			}

			if(m_pressedKeyCode == Qt::Key_S && event->button() == Qt::RightButton)
			{
				integrateToMz(xRangeMin(), xRangeMax(),
						yRangeMin(), yRangeMax());
			}

			if(m_pressedKeyCode == Qt::Key_D && event->button() == Qt::RightButton)
			{
				integrateToDt(xRangeMin(), xRangeMax(),
						yRangeMin(), yRangeMax());
			}

			if(m_pressedKeyCode == Qt::Key_I && event->button() == Qt::RightButton)
			{
				integrateToTicIntensity(xRangeMin(), xRangeMax(),
						yRangeMin(), yRangeMax());
			}
		}


	//! Integrate data to a retention time (TIC/XIC) chromatogram
	/*!

		The data to be integrated are bordered by the parameters that describe data
		ranges on both the x and y axes.

*/
	void 
		ColorMapPlotWidget::integrateToRt(double xAxisLower, double xAxisUpper,
				double yAxisLower, double yAxisUpper)
		{
			QCPAxis *xAxis = mp_colorMap->keyAxis();
			QPen basePen = xAxis->basePen();
			QColor color = basePen.color();

			History localHistory = m_history;
			HistoryItem *histItem = new HistoryItem;

			histItem->newIntegrationRange(IntegrationType::DT_TO_RT, 
					xAxisLower, xAxisUpper);

			histItem->newIntegrationRange(IntegrationType::MZ_TO_RT, 
					yAxisLower, yAxisUpper);

			// Aggreate the new item to the local history copy we made.
			localHistory.appendHistoryItem(histItem);

			emit newTicChromatogram(mp_massSpecDataSet, 
					"Calculating extracted ion chromatogram.", localHistory, color);
		}


	//! Integrate data to a mass spectrum
	/*!

		The data to be integrated are bordered by the parameters that describe data
		ranges on both the x and y axes.

*/
	void 
		ColorMapPlotWidget::integrateToMz(double xAxisLower, double xAxisUpper,
				double yAxisLower, double yAxisUpper)
		{
			QCPAxis *xAxis = mp_colorMap->keyAxis();
			QPen basePen = xAxis->basePen();
			QColor color = basePen.color();

			History localHistory = m_history;
			HistoryItem *histItem = new HistoryItem;

			histItem->newIntegrationRange(IntegrationType::DT_TO_MZ, 
					xAxisLower, xAxisUpper);

			histItem->newIntegrationRange(IntegrationType::MZ_TO_MZ, 
					yAxisLower, yAxisUpper);

			// Aggreate the new item to the local history copy we made.
			localHistory.appendHistoryItem(histItem);

			emit newMassSpectrum(mp_massSpecDataSet,
					"Calculating mass spectrum.", localHistory, color);
		}


	//! Integrate data to a drift spectrum
	/*!

		The data to be integrated are bordered by the parameters that describe data
		ranges on both the x and y axes.

*/
	void 
		ColorMapPlotWidget::integrateToDt(double xAxisLower, double xAxisUpper,
				double yAxisLower, double yAxisUpper)
		{
			QCPAxis *xAxis = mp_colorMap->keyAxis();
			QPen basePen = xAxis->basePen();
			QColor color = basePen.color();

			History localHistory = m_history;
			HistoryItem *histItem = new HistoryItem;

			histItem->newIntegrationRange(IntegrationType::DT_TO_DT, 
					xAxisLower, xAxisUpper);

			histItem->newIntegrationRange(IntegrationType::MZ_TO_DT, 
					yAxisLower, yAxisUpper);

			//qInfo() << __FILE__ << __LINE__ << __FUNCTION__
				//<< "histItem:" << histItem->asText();

			// Aggreate the new item to the local history copy we made.
			localHistory.appendHistoryItem(histItem);

			emit newDriftSpectrum(mp_massSpecDataSet,
					"Calculating drift spectrum.", localHistory, color);
		}


	//! Integrate data to a single TIC intensity value 
	/*!

		The data to be integrated are bordered by the parameters that describe data
		ranges on both the x and y axes.

		This is not a conventional integration, because it does not produce any
		graph, it only produces a single TIC intensity value. 

*/
	double 
		ColorMapPlotWidget::integrateToTicIntensity(double xAxisLower, double xAxisUpper,
				double yAxisLower, double yAxisUpper)
		{
			History localHistory = m_history;
			HistoryItem *histItem = new HistoryItem;

			histItem->newIntegrationRange(IntegrationType::DT_TO_TIC_INT, 
					xAxisLower, xAxisUpper);

			histItem->newIntegrationRange(IntegrationType::MZ_TO_TIC_INT, 
					yAxisLower, yAxisUpper);

			// Aggreate the new item to the local history copy we made.
			localHistory.appendHistoryItem(histItem);

			ColorMapWnd *parentWnd = static_cast<ColorMapWnd *>(mp_parentWnd);

			// Capture immediately the text that is located in the status bar
			// because if the computation is long that text will have disappeared
			// and we won't be able to show it in the TIC int text.

			QString currentMessage = parentWnd->statusBar()->currentMessage();

			QApplication::setOverrideCursor(QCursor(Qt::BusyCursor));

			MassDataIntegrator integrator(mp_massSpecDataSet, localHistory);

			integrator.integrateToTicIntensity();

			m_lastI = integrator.ticIntensity();

			QApplication::restoreOverrideCursor();

			if(currentMessage.isEmpty())
				parentWnd->statusBar()->showMessage(QString("dt range [%1-%2] ; m/z range [%3-%4]: TIC I = %5")
						.arg(xAxisLower, 0, 'f', 3)
						.arg(xAxisUpper, 0, 'f', 3)
						.arg(yAxisLower, 0, 'f', 3)
						.arg(yAxisUpper, 0, 'f', 3)
						.arg(m_lastI, 0, 'g', 3));
			else
				parentWnd->statusBar()->showMessage(currentMessage +
						QString(" ; (Tic I = %1)").arg(m_lastI, 0, 'g', 3));

			return m_lastI;
		}


	//! Handle the key code of the keyboard key that was released.
	void
		ColorMapPlotWidget::keyReleaseEvent(QKeyEvent *event)
		{
			ColorMapWnd *parentWnd = static_cast<ColorMapWnd *>(mp_parentWnd);
			MainWindow * mainWindow = static_cast<MainWindow *>(parentWnd->parent());

			if(event->key() == Qt::Key_L)
			{
				// The user wants to copy the current cursor location to the console
				// window, such that it remains available after having moved the cursor.

				QCPAxis *xAxis = mp_colorMap->keyAxis();
				QPen basePen = xAxis->basePen();
				QColor color = basePen.color();

				QString label;

				label = QString("(%1,%2)\n")
					.arg(m_lastMousedPlotPoint.x())
					.arg(m_lastMousedPlotPoint.y());

				if(label.isEmpty())
				{
					parentWnd->statusBar()->showMessage(
							"Failed to craft a text label.");

					event->accept();
					return;
				}
				else
				{

					// Send the label to the console window, so that we can make a
					// copy paste.

					mainWindow->logConsoleMessage(label, color);
				}

				event->accept();

				return;
			}

			// Now let the base class do its work.
			AbstractPlotWidget::keyReleaseEvent(event);
		}


} // namespace msXpSmineXpert
