/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



#pragma once


/////////////////////// Qt includes
#include <QObject>
#include <QList>


/////////////////////// Local includes
#include "MassPeak.hpp"


namespace msXpSlibmass
{

	class MassSpectrumList;


	//! The MassSpectrum class provides a list of MassPeak instances.
	/*!

		The list implementation is based on QList.

		Since a mass spectrum is a collection of mass peaks, that is acquired at a
		specific point in time, it needs to be characterized by a retention time
		member datum (\c m_rt). In experiments of ion mobility mass spectrometry,
		another value is needed to fully characterize the mass spectrum, the drift
		time value (\c m_dt).

		The MassSpectrum class has all the methods required to combine mass spectra
		into a single mass spectrum.

*/
	class MassSpectrum 
		: 
			public QObject,
			public QList<MassPeak *> 
	{
		Q_OBJECT

		private:
			//! Title of the spectrum.
			QString m_title;

			//! Index of the MassPeak instance that was last modified.
			/*!

				When performing lengthy operations, like mass spectrum combination, this
				value allows for a considerable speedup of the operations. Indeed,
				storing the last modified MassPeak instance allows for almost direct
				MassPeak addressing when performing next combination (this is possible
				because by nature a mass spectrum is sorted in increasing mz value
				order).
				*/
			int m_lastModifIdx = 0;

			//! Index of the bin that was last modified.
			/*!
				This value has the same function as \c m_lastModifIdx but in cases where
				binning is required.
				*/
			int m_lastBinIdx = 0;

			//! Retention time at which this mass spectrum was acquired.
			double m_rt = 0.0;

			//! Drift time at which this mass spectrum was acquired.
			double m_dt = 0.0;

			//! Decimal places to use when working with m/z values
			/*!
				Limiting the number of decimal places is useful in some occasions as a
				poor man binning system. In particular, it is useful when creating the
				mz=f(dt) color map.
				*/
			int m_decimalPlaces = -1;

			//! Greatest difference between two consecutive m/z values in the spectrum
			/*!
				This value is useful when performing alignment of the spectra of an
				acquisition.
				*/
			double m_greatestStep = qSNaN();

			//! Number of bins in the spectrum.
			/*! 
				The number of bins is useful for spectral alignment.
				*/
			int m_binCount = 0;

			//! Tells if the spectrum can undergo a binning process
			/*!
				This value is useful to store the ability of the spectral data to undergo a binning process.
				*/
			bool m_isBinnable = false;	

			//! The smallest m/z value contained in the spectrum list being processed
			/*!
				This value is useful when checking for the ability of a mass spectrum
				list to undergo a binning process. It stores the smallest m/z value of a
				set of mass spectra.
				*/
			double m_binMinMz = 0;

			//! The greatest m/z value contained in the spectrum list being processed
			/*!

				This value is useful when checking for the ability of a mass spectrum
				list to undergo a binning process. It stores the greatest m/z value of a
				set of mass spectra.
				*/
			double m_binMaxMz = 0;

			//! Value by which the mass spectra need to be shifted for an alignment
			/*!

				This value holds the m/z delta value by which this mass spectrum needs
				to be shifted in order to align it properly.
				*/
			double m_mzShift = 0;

		public:

			static double m_maxAvgMzBinSize;
			static double m_maxBinSizeStdDev;

			MassSpectrum();
			MassSpectrum(const QString &title);
			virtual ~MassSpectrum();

			void reset();
			void deleteMassPeaks();

			QString title() const;

			double rt() const;
			double &rrt();

			double dt() const;
			double &rdt();

			void setDecimalPlaces(int decimalPlaces);
			int decimalPlaces();
			int &rdecimalPlaces();

			int initialize(const QString &xyFormatString);
			void initialize(const QList<double> &mzList, const QList<double> &iList);
			void initialize(const QList<double> &mzList, const QList<double> &iList,
					double mzStart, double mzEnd);

			void initialize(const QList<double> &mzList, const QList<double> &iList,
					double driftTime);

			void initialize(const QList<double> &mzList, const QList<double> &iList,
					double driftTime, double mzStart, double mzEnd);

			void initialize(const QVector<double> &mzVector, 
					const QVector<double> &iVector,
					double mzStart = qSNaN(),
					double mzEnd = qSNaN());

			bool setupBinnability(const MassSpectrumList &massSpectra);
			double fillInBins(QList<double> *binList);
			double binSizeAvg(const QList<double> &binList);
			double binSizeStdDev(const QList<double> &binList);
			bool isBinnable(const QList<double> &binList);
			void setupBins(const QList<double> &binList);
			int binIndex(double mz);

			double determineMzShift(const MassSpectrum &massSpectrum);
			double determineMzShift(double otherMzValue);

			double tic() const;
			double tic(double mzStart, double mzEnd) const;

			int combine(const MassPeak &massPeak);
			int combineBinned(const MassPeak &massPeak);

			// int combineSlow(const MassPeak &massPeak);

			int combine(const MassSpectrum &massSpectrum);
			int combine(const MassSpectrum &massSpectrum, double mzStart, double mzEnd);

			int combine(const MassSpectrumList &massSpectra);
			int combine(const MassSpectrumList &massSpectra,
					double mzStart, double mzEnd);

			int combine(const QList<double> &mzList, const QList<double> &iList);
			int combine(const QList<double> &mzList, const QList<double> &iList,
					double mzStart, double mzEnd);

			int contains(double mz) const;

			QList<double> mzList() const;
			QList<double> iList() const;
			void toLists(QList<double> *mzList, QList<double> *iList) const;
			void toVectors(QVector<double> *mzVector, QVector<double> *iVector) const;
			void toMap(QMap<double, double> *map) const;

			QString *asText(int startIndex = -1, int endIndex = -1);

			static MassSpectrum *createFromBase64EncByteArrays(
					const QByteArray *mzByteArray, const QByteArray *iByteArray,
					int compressionType, double mzStart = qSNaN(), double mzEnd = qSNaN());

	};


} // namespace msXpSlibmass

