/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



/////////////////////// Qt includes
#include <QDebug>
#include <QRegularExpressionMatch>


/////////////////////// Local includes
#include "globals/globals.hpp"
#include "MassPeak.hpp"


namespace msXpSlibmass
{

	//! Constructor.
	/*! 

		Initialize \c m_mz and \c m_i to qSNaN(), which makes it easy to validate.

*/
	MassPeak::MassPeak()
	{
	}


	//! Constructor.
	/*!

		Initialize the member data with the parameters.

		\param mz copied to \c m_mz
		\param i copied to \c m_i

*/
	MassPeak::MassPeak(double mz, double i)
	{
		m_mz = mz;
		m_i = i;
	}


	//! Constructor.
	/*!

		Constructs \c this to be identical to \p massPeak.

*/
	MassPeak::MassPeak(const MassPeak &massPeak)
	{
		m_mz = massPeak.m_mz;
		m_i = massPeak.m_i;
	}



	//! Constructor.
	/*!

		Construct \c this by initializing its member values from their string representation in \p xyLine.

		The xyLine is a line of text having the following generic format:
		<number1><non-number><number2> with number1 a string representation of m/z,
		non-number any string element that is not a number and number2 a string
		representation of i. The decomposition of xyLine into m/z and i values is
		performed using initialize();

		\sa bool initialize(const QString &xyLine).

*/	
	MassPeak::MassPeak(const QString &xyLine)
	{
		initialize(xyLine);
	}


	//! Destructor.
	MassPeak::~MassPeak()
	{
	}



	//! Initialize \c this MassPeak instance using the string representation of m/z and i in \p xyLine.
	/*!

		The xyLine is a line of text having the following generic format:
		<number1><non-number><number2> with number1 a string representation of m/z,
		non-number any string element that is not a number and number2 a string
		representation of i. The decomposition of xyLine into m/z and i values is
		performed using a globally available regular expression (msXpS::gXyFormatMassDataRegExp).

*/
	bool 
		MassPeak::initialize(const QString &xyLine)
		{

			QRegularExpressionMatch regExpMatch;

			regExpMatch = msXpS::gXyFormatMassDataRegExp.match(xyLine);

			if(!regExpMatch.hasMatch())
				return false;

			bool ok = false;

			double	mz = regExpMatch.captured(1).toDouble(&ok);

			if(!ok)
				return false;

			// Note that group 2 is the separator group.

			double i = regExpMatch.captured(3).toDouble(&ok);

			if(!ok)
				return false;

			m_mz = mz;
			m_i = i;

			return true;
		}



	//! Return the \c m_mz value.
	/*!

		\return \c m_mz.

*/
	double
		MassPeak::mz() const
		{
			return m_mz;
		}


	//! Return the \c m_i value.
	/*!

		\return \c m_i.

*/
	double 
		MassPeak::i() const
		{
			return m_i;
		}


	//! Return a <em> non-const reference</em> to \c m_i.
	/*!

		\return a <em> non-const reference</em> to \c m_i.

*/
	double &
		MassPeak::ri()
		{
			return m_i;
		}

	//! Validate \c this MassPeak instance.
	/*!

		A MassPeak is validated if both \c m_mz and \c m_i members are not NaN and
		if the \c m_mz is not negative.

		\return true if the MassPeak validates, false otherwise.
		*/
	bool 
		MassPeak::validate() const
		{
			if(!qIsNaN(m_mz) && !qIsNaN(m_i) && m_mz >= 0)
				return true;

			return false;
		}



} // namespace msXpSlibmass
