# Copyright (c) 2011, Peter A. Bigot, licensed under New BSD (see COPYING)
# This file is part of msp430mcu (http://sourceforge.net/projects/mspgcc/)
#
# Generate linker scripts to define the memory regions supported on
# particular MSP430 chips.

import sys
import csv
import os

msp430_root = os.environ.get('MSP430_ROOT', '/msp430')
analysis_dir = os.path.join(msp430_root, 'msp430mcu', 'analysis')
upstream_dir = os.path.join(msp430_root, 'msp430mcu', 'upstream')

def analysis_path (filename):
    global analysis_dir
    return os.path.join(analysis_dir, filename)

def upstream_path (filename):
    global upstream_dir
    return os.path.join(upstream_dir, filename)

regMcu_map = { 'msp430bt5190' : 'msp430f5438' }

# Device Name,CPU_TYPE,CPU_Bugs,MPY_TYPE,STACKSIZE,RAMStart,RAMEnd,BSLStart,BSLSize,BSLEnd,INFOStart,INFOSize,INFOEnd,INFOA,INFOB,INFOC,INFOD,FStart,FEnd,FStart2,FEnd2,INTStart,INTEnd
reader = csv.reader(file(upstream_path('devices.csv')))

class Region (object):
    _Regions = set()
    name = None
    attributes = None
    address_width = 4
    __is_fixed = False
    origin = None
    length = None

    def __init__ (self, name, attributes=None, address_width=4):
        self.name = name
        self.attributes = attributes
        self.address_width = address_width
        self.reset()
        self._Regions.add(self)

    def set (self, origin, length, fixed=False):
        if fixed or ((0 != origin) and (0 != length)):
            self.origin = origin
            self.length = length
        self.__is_fixed = fixed
        return self

    def reset (self):
        if not self.__is_fixed:
            self.origin = self.length = 0

    def __cmp__ (self, other):
        # Sort infomem before other info sections
        if self.name.startswith('info') and other.name.startswith('info') and (len(self.name) != len(other.name)):
            return - cmp(len(self.name), len(other.name))
        # If both lengths zero, sort by name within near/far
        if (0 == self.length) and (0 == other.length):
            if self.address_width != other.address_width:
                return cmp(self.address_width, other.address_width)
            return cmp(self.name, other.name)
        # If one length zero, sort it after
        if (0 == self.length) or (0 == other.length):
            return - cmp(self.length, other.length)
        return cmp(self.origin, other.origin) or cmp(self.length, other.length)

    __M = 1024 * 1024
    __K = 1024

    def _integerToLinkConstants (self, b):
        m = b / self.__M
        if (1 <= m) and (0 == (b % self.__M)):
            return '%uM' % (m,)
        k = b / self.__K
        if (1 <= k) and (0 == (b % self.__K)):
            return '%uK' % (k,)
        return '%u' % (b,)

    def _formatAsString (self, name_width):
        attr_str = ''
        if self.attributes:
            attr_str = ' (%s)' % (self.attributes,)
        rv = []
        rv.append('%-*s : ' % (name_width, self.name + attr_str))
        rv.append('ORIGIN = 0x%0*x, ' % (self.address_width, self.origin))
        rv.append('LENGTH = 0x%0*x' % (self.address_width, self.length))
        if 0 < self.length:
            rv.append(' /* END=0x%0*x, size %s */' % (self.address_width, self.origin + self.length, self._integerToLinkConstants(self.length)))
        return ''.join(rv)

    @classmethod
    def Memory (cls):
        regions = sorted(cls._Regions)
        name_width = 0
        valid_bank = True
        for r in regions:
            n = len(r.name)
            if r.attributes:
                n += 3 + len(r.attributes)
            name_width = max(name_width, n)
        rv = [ 'MEMORY {']
        for r in regions:
            if valid_bank and (0 == r.length):
                rv.append('  /* Remaining banks are absent */')
                valid_bank = False
            rv.append('  ' + r._formatAsString(name_width))
        rv.append('}')
        return "\n".join(rv);
    
    @classmethod
    def Reset (cls):
        [ _r.reset() for _r in cls._Regions ]

sfr = Region('sfr').set(0x0000, 0x0010, True)
peripheral_8bit = Region('peripheral_8bit').set(0x0010, 0x00F0, True)
peripheral_16bit = Region('peripheral_16bit').set(0x0100, 0x0100, True)
bsl = Region('bsl')
infoa = Region('infoa')
infob = Region('infob')
infoc = Region('infoc')
infod = Region('infod')
infomem = Region('infomem')
ram = Region('ram', attributes='wx')
rom = Region('rom', attributes='rx')
far_rom = Region('far_rom', address_width=8)
vectors = Region('vectors')

for row in reader:
    if (0 == len(row)) or row[0].startswith('#'):
        continue

    Region.Reset()
    mcu = row.pop(0)            # Device Name
    cpu = row.pop(0)            # CPU_TYPE
    bugs = row.pop(0)           # CPU_Bugs
    mpy = row.pop(0)            # MPY_TYPE
    row.pop(0)                  # STACKSIZE

    (rams, rame, bsls, bslsz, bsle, infos, infosz, infoe, infoas, infobs, infocs, infods,
     flashs, flashe, flash2s, flash2e, ints, inte) = [ int(_x, 16) for _x in row ]

    bsl.set(bsls, bslsz)
    infoa.set(infoas, infosz)
    infob.set(infobs, infosz)
    infoc.set(infocs, infosz)
    infod.set(infods, infosz)
    infomem.set(infos, infoe - infos + 1)
    ram.set(rams, 1 + rame - rams)
    rom.set(flashs, 1 + flashe - flashs)
    far_rom.set(flash2s, 1 + flash2e - flash2s)
    vectors.set(ints, 1 + inte - ints)

    mdir = os.path.join(analysis_dir, 'ldscripts', mcu)
    try:
        os.makedirs(mdir)
    except OSError, e:
        pass

    mpath = os.path.join(mdir, 'memory.x')
    outf = file(mpath, 'w')
    outf.write(Region.Memory())
    outf.write('''
REGION_ALIAS("REGION_TEXT", rom);
REGION_ALIAS("REGION_DATA", ram);
REGION_ALIAS("REGION_FAR_ROM", far_rom);
''')
