# Copyright (c) 2011, Peter A. Bigot, licensed under New BSD (see COPYING)
# This file is part of msp430mcu (http://sourceforge.net/projects/mspgcc/)
#
# Check for consistency between devices.csv and individual headers.
# Generate cpu/mpy/iv data file: chipflags.csv

import re
import csv
import glob
import os.path

msp430_root = os.environ.get('MSP430_ROOT', '/msp430')
analysis_dir = os.path.join(msp430_root, 'msp430mcu', 'analysis')
upstream_dir = os.path.join(msp430_root, 'msp430mcu', 'upstream')

def analysis_path (filename):
    global analysis_dir
    return os.path.join(analysis_dir, filename)

def upstream_path (filename):
    global upstream_dir
    return os.path.join(upstream_dir, filename)

# Part-number regex.
pn_re = re.compile('''
^(?P<processorFamily>[a-z]*)
(?P<platform>430)
(?P<memoryType>[cefgpl])
(?P<endEquipmentOptimized>[gew]?)
(?P<generation>[0-9])
(?P<series>[0-9])
(?P<family>[0-9]+)
(?P<variant>.*)
$''', re.VERBOSE + re.IGNORECASE)

# Alternative regex.  This exists solely because TI decided to denote
# a bluetooth-optimized device (msp430bt5190) without any regard for
# consistency with their existing part number scheme.

alt_pn_re = re.compile('''
^(?P<processorFamily>[a-z]*)
(?P<platform>430)
(?P<special>[^0-9]+)
(?P<generation>[0-9])
(?P<series>[0-9])
(?P<family>[0-9]+)
(?P<variant>.*)
$''', re.VERBOSE + re.IGNORECASE)

PN_FIELDS = ( 'processorFamily', 'platform', 'memoryType', 'endEquipmentOptimized',
              'generation', 'series', 'family', 'variant' )

# List of generic MCU identifiers
generics = [ ]
# Map from a generic MCU identifier to list of devices described by the generic
chip_equiv = { }
# Map from a specific MCU device to the generic device that describes it
chip_generic = { }
for line in file(analysis_path('chip-equiv.txt')).readlines():
    # Strip off .h extension
    chips = [ _fn[:-2] for _fn in line.split() ]
    generic = chips.pop(0)
    suffix = generic[generic.find('430x')+4:]
    if 0 > suffix.find('x'):
        # Not really a generic; multiple chips with identical headers
        # e.g. msp430f415 msp430f417.  @TODO@ Generalize this, currently
        # handles only the one case.
        chips = [ generic.replace('x', 'f') ]
    generics.append(generic)
    assert generic not in chip_equiv
    chip_equiv[generic] = chips
    [ chip_generic.setdefault(_c, generic) for _c in chips ]

# Map from MCU identifier to a tuple (cpu, mpy, ivlen, hex_ivaddr)
mcus = { }

# devices.csv represents CPU architecture with these values
CPU_MAP = { '0' : '430',        # MSP430: 16-bit MCU
            '1' : '430x',       # MSP430X: 16-bit MCU with 20-bit addresses
            '2' : '430xv2' }    # MSP430X with altered timing characteristics
# devices.csv represents MPY hardware with these values
MPY_MAP = { '0' : 'none',
            '1' : '16',         # 16-bit
            '2' : '16se',       # 16-bit with sign extension
            '4' : '32',         # 32-bit
            '8' : '32dw' }      # 32-bit with delayed write

# Device Name,CPU_TYPE,CPU_Bugs,MPY_TYPE,STACKSIZE,RAMStart,RAMEnd,BSLStart,BSLSize,BSLEnd,INFOStart,INFOSize,INFOEnd,INFOA,INFOB,INFOC,INFOD,FStart,FEnd,FStart2,FEnd2,INTStart,INTEnd
reader = csv.reader(open(upstream_path('devices.csv')))

for row in reader:
    if (0 == len(row)) or row[0].startswith('#'):
        continue
    mcu = row.pop(0)            # Device Name
    cpu = CPU_MAP[row.pop(0)]   # CPU_TYPE
    bugs = row.pop(0)           # CPU_Bugs
    mpy = MPY_MAP[row.pop(0)]   # MPY_TYPE
    row.pop(0)                  # STACKSIZE
    (rams, rame, bsls, bslsz, bsle, infos, infosz, infoe, infoas, infobs, infocs, infods,
     flashs, flashe, flash2s, flash2e, ints, inte) = [ int(_x, 16) for _x in row ]
    ivlen = (1 + (inte - ints)) / 2
    mcus[mcu] = ( cpu, mpy, ivlen, hex(ints) )

# mcu -- chip to which data pertains
# xmcu -- genericized chip name per TI standards
# cpu -- 430, 430x, 430xv2
# mpy -- none, 16, 16se, 32, 32dw
# ivcnt -- 16, 32, 64
# ivaddr -- start address of vectors section, as hex literal
chipflags = csv.writer(file(analysis_path('chipflags.csv'), 'w'))
chipflags.writerow( ('# mcu', 'xmcu', 'cpu', 'mpy', 'ivcnt', 'ivaddr') )
chipflags.writerows([ (_mcu, chip_generic.get(_mcu))+mcus.get(_mcu) for _mcu in sorted(mcus.keys()) ])


def grepline (pattern, lines):
    """Find first element of lines in which pattern exists and return
    the second token in the line.

    Generally this is finding lines that check for or define
    functional preprocessor symbol definitions."""
    
    for l in lines:
        if 0 <= l.find(pattern):
            return l.split()[1]
    return None

print '''# Sanity-check header contents against MCU characteristics.  Known
# inconsistencies:
#
# - MSP430FE42x devices have a hardware multiplier (noted in header),
#   but it is unavailable when the second watt-hour meter CPU is
#   active (disabled in characteristics)
#
# - MSP430x09x devices have a MSP430X cpu but the extended
#   instructions do not work correctly.  Byte access to word
#   registers is OK.
#
'''
headers = glob.glob(upstream_path('*430*.h'))
for h in headers:
    mcu = os.path.split(h)[1]
    if 'msp430.h' == mcu:
        continue
    mcu = mcu[:mcu.find('.')]
    if not (mcu in mcus):
        if not (mcu in chip_equiv):
            print 'ERROR: Header %s has no device description' % (mcu,)
        continue
    (cpu, mpy, ivlen, ivaddr) = mcus[mcu]
    header_contents = file(h).readlines()
    hcpu = grepline('_CPU__', header_contents)
    if hcpu is None:
        hcpu = CPU_MAP['0']
    else:
        hcpu = hcpu.split('_')[4][3:].lower()

    # Re msp430x092 devices: "The Header has the tag so that the byte
    # access to word registers are enabled but the compiler should not
    # use the extended instructions set for this CPU as it has some
    # issues on this devices.  It also only has the limited memory of
    # <64k."
    if hcpu != cpu:
        print 'WARNING: %s inconsistent cpu %s %s' % (mcu, cpu, hcpu)
    hmpy = grepline('_HAS_MPY', header_contents)
    if hmpy is None:
        hmpy = MPY_MAP['0']
    else:
        hmpy = hmpy.split('_')[4][3:]
        if '' == hmpy:
            hmpy = '16'

    # Re msp430fe43x devices: "This devices have a 2nd CPU in it which
    # is used to do the energy processing for Watt Hour Meter. If this
    # is enabled the MPY is used by the 2nd CPU and therefore not
    # available for the main CPU. To avoid that it is used by the
    # linker for the math functions it is not set for the device
    # features in the devices.csv file."
    if not mpy.startswith(hmpy):
        print 'WARNING: %s inconsistent mpy %s hdr %s' % (mcu, mpy, hmpy)
    #print '%s: %s %s ; %s %s ; %d' % (mcu, cpu, hcpu, mpy, hmpy, ivlen)
    
    
