# Copyright (c) 2011, Peter A. Bigot, licensed under New BSD (see COPYING)
# This file is part of msp430mcu (http://sourceforge.net/projects/mspgcc/)
#
# Read through all the devices, and identify those where eliding the
# memoryType field would result in inconsistencies in chip
# characteristics.  This mostly applies to ValueLine devices; for
# example the msp430g2131 and msp430f2131 differ in RAM size and FLASH
# start and size.

import os
import csv
import re

msp430_root = os.environ.get('MSP430_ROOT', '/msp430')
analysis_dir = os.path.join(msp430_root, 'msp430mcu', 'analysis')
upstream_dir = os.path.join(msp430_root, 'msp430mcu', 'upstream')

def analysis_path (filename):
    global analysis_dir
    return os.path.join(analysis_dir, filename)

def upstream_path (filename):
    global upstream_dir
    return os.path.join(upstream_dir, filename)

# Part-number regex.
pn_re = re.compile('''
^(?P<processorFamily>[a-z]*)
(?P<platform>430)
(?P<memoryType>[cefgpl])
(?P<endEquipmentOptimized>[gew]?)
(?P<generation>[0-9])
(?P<series>[0-9])
(?P<family>[0-9]+)
(?P<variant>.*)
$''', re.VERBOSE + re.IGNORECASE)

# Alternative regex.  This exists solely because TI decided to denote
# a bluetooth-optimized device (msp430bt5190) without any regard for
# consistency with their existing part number scheme.

alt_pn_re = re.compile('''
^(?P<processorFamily>[a-z]*)
(?P<platform>430)
(?P<special>[^0-9]+)
(?P<generation>[0-9])
(?P<series>[0-9])
(?P<family>[0-9]+)
(?P<variant>.*)
$''', re.VERBOSE + re.IGNORECASE)

PN_FIELDS = ( 'processorFamily', 'platform', 'memoryType', 'endEquipmentOptimized',
              'generation', 'series', 'family', 'variant' )

reader = csv.reader(open(upstream_path('devices.csv')))

include_data = { }

# Map from genericized part numbers to specific part numbers that
# correspond to them.
xmap = { }

mcus = set()

# Set of genericized mcu names for which the devices.csv table
# indicates a difference in peripherals or address map among the
# devices that have the same genericized mcu name.
device_mismatch_xmcus = set()

# Set of genericized mcu names for which there are multiple devices
# with the same generic name, and they do not share the same legacy
# header file (hence, have a difference in peripherals or other
# characteristics that was not indicated in the devices table).
header_mismatch_xmcus = set()

# Set of genericized mcu names for which multiple devices have the
# same generic name, even if they appear otherwise identical.
multi_mismatch_xmcus = set()

fields = None
last_row = None
for row in reader:
    if (0 == len(row)) or row[0].startswith('#'):
        last_row = row
        continue
    if fields is None:
        last_row.pop(0)
        fields = last_row

    mcu = row.pop(0)
    mcus.add(mcu)

    # Extract the part number fields
    mcu_pndict = None
    m = pn_re.match(mcu)
    if m is not None:
        mcu_pndict = m.groupdict()
    else:
        m = alt_pn_re.match(mcu)
        if m is not None:
            mcu_pndict = m.groupdict()
            special = mcu_pndict.pop('special')
            if 'bt' == special:
                # Bluetooth-optimized device
                mcu_pndict['memoryType'] = 'f'
                mcu_pndict['endEquipmentOptimized'] = ''
            elif 2 == len(special):
                mcu_pndict['memoryType'] = special[0]
                mcu_pndict['endEquipmentOptimized'] = special[1]
    if mcu_pndict is None:
        print 'No match to "%s"' % (mcu,)
        continue

    # Create a copy that uses a genericized memory type, as with legacy mspgcc
    xmcu_pndict = mcu_pndict.copy()
    xmcu_pndict['memoryType'] = '_'
    xmcu_pndict['endEquipmentOptimized'] = ''
    xmcu_pndict['variant'] = ''
    xmcu = ''.join([ xmcu_pndict.get(_field) for _field in PN_FIELDS ])

    # Store this chip within its generic, first saving a tag
    # identifying the other chips already in the generic.
    matchables = xmap.setdefault(xmcu, set())
    omcu = ' '.join(matchables)
    matchables.add(mcu)

    # See if we already have data on an MCU that uses this genericized
    # part number.  If so, see whether the characteristics of the new
    # chip are consistent with the old one.
    old_row = include_data.get(xmcu)
    if old_row is None:
        include_data[xmcu] = row
    elif old_row != row:
        changes = []
        for i in xrange(len(fields)):
            if old_row[i] != row[i]:
                changes.append('%s %s[%s] != %s[%s]' % (fields[i], omcu, old_row[i], mcu, row[i]))
        print "'MISMATCH: %s differ from %s:\n\t%s" % (mcu, omcu, '\n\t'.join(changes))
        device_mismatch_xmcus.add(xmcu)
        
nof_variants = 0
for xmcu in sorted(xmap.keys()):
    mcu = xmcu.replace('430_', '430f')
    if mcu not in xmap[xmcu]:
        print '%s does not have F variant' % (mcu,)
        nof_variants += 1
print '%d generics have an F variant' % (len(xmap) - nof_variants,)

match_xmcus = set(include_data.keys())
match_xmcus.difference_update(device_mismatch_xmcus)

# Create a map identifying the generic header for each chip which has
# one.
legacy_map = { }
for line in file(analysis_path('chip-equiv.txt')).readlines():
    devices = [ _s.replace('.h', '') for _s in line.split() ]
    legacy = devices.pop(0)
    for d in devices:
        legacy_map[d] = legacy

# Use the legacy map to identify genericized devices for which
# specific devices have different headers.
for xmcu in sorted(match_xmcus):
    if 1 < len(xmap[xmcu]):
        multi_mismatch_xmcus.add(xmcu)
    legacy_headers = set([ legacy_map.get(_m, _m) for _m in xmap[xmcu] ])
    if 1 < len(legacy_headers):
        header_mismatch_xmcus.add(xmcu)

match_xmcus.difference_update(header_mismatch_xmcus)
match_xmcus.difference_update(multi_mismatch_xmcus)

nomatch_mcus = set()

def emitMcuInclude (mcu, tag, xmcu=None):
    global iof
    tmcu = []
    if xmcu is not None:
        tmcu.append(xmcu)
        tmcu.append(xmcu.replace('_', 'X'))
    tmcu.append(mcu)
    iof.write("""
#if %s
#include <%s.h>
#endif /* %s */
""" % (' || '.join([ 'defined(__%s__)' % (_t.upper(),) for _t in tmcu ]), mcu, tag))
    
for xmcu in sorted(device_mismatch_xmcus):
    print 'Device characteristic mismatch: %s' % (' '.join(xmap[xmcu]),)
    nomatch_mcus.update(xmap[xmcu])

for xmcu in sorted(header_mismatch_xmcus):
    print "Header mismatch: %s: %s" % (xmcu, ' '.join([ '%s:%s' % (_m, legacy_map.get(_m, '-')) for _m in xmap[xmcu] ]))
    nomatch_mcus.update(xmap[xmcu])

for xmcu in sorted(multi_mismatch_xmcus):
    print "Multiple instances: %s = %s" % (xmcu, ' '.join(xmap[xmcu]))
    nomatch_mcus.update(xmap[xmcu])

for xmcu in sorted(match_xmcus):
    print 'Genericized: %s = %s' % (xmcu, ' '.join(xmap[xmcu]))

print '%d mcus, %d conflict' % (len(mcus), len(nomatch_mcus))

iof = file(analysis_path('msp430.h'), 'w')
iof.write("""/* Map MCU preprocessor definitions to chip-specific include files.
 *
 * This file is automatically generated from TI-provided data.  Each device
 * is mapped to a genericized name by ignoring differences in memory type,
 * end-equipment optimization, and some other variances.  Preprocessor
 * directives are generated to include the appropriate header for each
 * device.  Generic names, such as msp430x1611, are recognized only if
 * they are sufficient to uniquely identify a device.  When this is not
 * the case, a comment indicates why a generic is excluded: normally
 * because devices have different peripherals or memory maps.
 */
#ifndef __msp430_h_
#define __msp430_h_

/** Bit-markers for type of CPU present.
 * Check equality against __MSP430_CPU__. */
typedef enum msp430_cpu_e
{
  MSP430_CPU_MSP430 = 0x0000,
  MSP430_CPU_MSP430X = 0x0002,
  MSP430_CPU_MSP430XV2 = 0x0003,
  MSP430_CPU = 0x0003
} msp430_cpu_e;

/** Bit-markers for type of hardware multiplier present.
 * Check equality against __MSP430_MPY__ (undefined if not present). */
typedef enum msp430_mpy_e
{
  MSP430_MPY_NONE = 0x0000,
  MSP430_MPY_16 = 0x0010,
  MSP430_MPY_16SE = 0x0011,
  MSP430_MPY_32 = 0x0020,
  MSP430_MPY_32DW = 0x0022,
  MSP430_MPY = 0x0030
} msp430_mpy_e;

""")
for xmcu in sorted(xmap.keys()):
    mcus = xmap[xmcu]
    if xmcu in device_mismatch_xmcus:
        [ emitMcuInclude(_m, 'Device mismatch %s' % (xmcu,)) for _m in mcus]
    elif xmcu in header_mismatch_xmcus:
        [ emitMcuInclude(_m, 'Header mismatch %s' % (xmcu,)) for _m in mcus]
    elif xmcu in multi_mismatch_xmcus:
        [ emitMcuInclude(_m, 'Multi-device %s' % (xmcu,)) for _m in mcus]
    else:
        assert 1 == len(mcus)
        mcu = mcus.copy().pop()
        emitMcuInclude(mcu, 'Genericizable', xmcu)

iof.write("""
#endif /* __msp430_h_ */
""")
