#include <mrpt/core.h>
#include <vector>
#include <algorithm>
#include <iterator>
#include <utility>
#include <numeric>

using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::random;
using namespace mrpt::utils::metaprogramming;
using namespace std;

const size_t matrixSize=10;
const size_t howManySets=4;
const double eps=1e-7;

typedef CMatrixTemplateNumeric<double> MyMat;

class MatrixGenerator	{
private:
	CRandomGenerator gen;
	size_t sz;
public:
	inline MatrixGenerator(size_t s):gen(123),sz(s)	{}
	MyMat operator()()	{
		MyMat res(sz,sz);
		do gen.drawUniformMatrix(res,-10.0,10.0); while (res.isSingular(1e-4));
		return res;
	}
};

template<typename It1,typename It2> class DoubleIterator	{
public:
	typedef pair<typename It1::value_type,typename It2::value_type> value_type;
	typedef value_type *pointer;
	typedef value_type &reference;
	typedef ptrdiff_t difference_type;
	typedef input_iterator_tag iterator_category;
private:
	It1 base1;
	It2 base2;
public:
	inline DoubleIterator(const It1 &b1,const It2 &b2):base1(b1),base2(b2)	{}
	inline DoubleIterator<It1,It2> operator++()	{
		++base1;
		++base2;
		return *this;
	}
	inline DoubleIterator<It1,It2> operator++(int)	{
		DoubleIterator<It1,It2> res=*this;
		++(*this);
		return res;
	}
	inline bool operator==(const DoubleIterator<It1,It2> &it) const	{
		return (base1==it.base1)&&(base2==it.base2);
	}
	inline bool operator!=(const DoubleIterator<It1,It2> &it) const	{
		return (base1!=it.base1)||(base2!=it.base2);
	}
	inline value_type operator*() const	{
		return make_pair(*base1,*base2);
	}
};
template<typename It1,typename It2> inline DoubleIterator<It1,It2> itDouble(const It1 &i1,const It2 &i2)	{
	return DoubleIterator<It1,It2>(i1,i2);
}

inline MyMat prod(const MyMat &l,const MyMat &r)	{
	return l*r;
}

int compareMatrices(const MyMat &m1,const MyMat &m2)	{
	if ((m1.getRowCount()!=m2.getRowCount())||(m1.getColCount()!=m2.getColCount())) return 1;
	for (size_t i=0;i<m1.getRowCount();++i) for (size_t j=0;j<m1.getColCount();++j) if (abs(m1.get_unsafe(i,j)-m2.get_unsafe(i,j))>eps) 	return 1;
	return 0;
}

template<typename T,typename ConstIt,typename It> void for_eachEX(ConstIt begin1,const ConstIt &end1,ConstIt leftSource,ConstIt rightSource,It leftDest,It rightDest,void (T::*leftFun)(const T &,T &) const,void (T::*rightFun)(const T &,T &) const)	{
	while (begin1!=end1)	{
		//cout << "l" << endl << (*begin1) << endl << *leftSource <<endl;
		//cout << "det b: " << begin1->det() << endl;
		//cout << "det s: " << leftSource->det() << endl;
		((*begin1).*leftFun)(*leftSource,*leftDest);
		((*begin1).*rightFun)(*rightSource,*rightDest);
		begin1++;
		leftSource++;
		rightSource++;
		leftDest++;
		rightDest++;
	}
}

inline int compareAndSum(int val,const pair<MyMat,MyMat > &p)	{
	return val+compareMatrices(p.first,p.second);
}

#define CHECK_AND_RET_ERROR(COND_,TXT_) \
	if (COND_) { std::cerr << TXT_ << std::endl; return 1; }


int test_matrix_divide_main(int argc,char **argv)	{
	try{
#if 1   // Extensive test
	vector<MyMat > leftOps(howManySets,MyMat(0,0));
	vector<MyMat > rightOps(howManySets,MyMat(0,0));
	vector<MyMat > products(howManySets,MyMat(0,0));
	vector<MyMat > leftCheck(howManySets,MyMat(0,0));
	vector<MyMat > rightCheck(howManySets,MyMat(0,0));
	MatrixGenerator gen(matrixSize);
	generate(leftOps.begin(),leftOps.end(),gen);
	generate(rightOps.begin(),rightOps.end(),gen);
	transform(leftOps.begin(),leftOps.end(),rightOps.begin(),products.begin(),&prod);
	for_eachEX(products.begin(),products.end(),leftOps.begin(),rightOps.begin(),leftCheck.begin(),rightCheck.begin(),
				&MyMat::leftDivideSquare<MyMat,MyMat>,
				&MyMat::rightDivideSquare<MyMat,MyMat> );
	return accumulate(itDouble(leftOps.begin(),leftCheck.begin()),itDouble(leftOps.end(),leftCheck.end()),accumulate(itDouble(rightOps.begin(),rightCheck.begin()),itDouble(rightOps.end(),rightCheck.end()),0,&compareAndSum),&compareAndSum);
#else
	// Simple left matrix divide test:
	{
		// RES(5,3) = B(5,5)^-1 * A(5,3)
		const double   dat_A[] = { 0.908476030831065,0.879919891489066,0.250067252262605,0.920138784660366,0.339886367553029,0.060459543365432,0.056627907484451,0.654039310230904,0.974776838855892,0.389544131381595,0.744290814070957,0.499115606151359,0.988576311557799,0.259649006583242,0.644135392887618  };
		CMatrixDouble  A(5,3, dat_A);
		const double   dat_B[] = { 0.831126996450328,0.214121074950156,0.636254505783512,0.692235943443756,0.293813504665217,0.116337996288886,0.455021117980695,0.247202963251884,0.419454326304339,0.337186829539914,0.760430416395189,0.920522595172334,0.939779215293909,0.146875660396882,0.645204546293099,0.963145925110214,0.127435698076257,0.338127761299189,0.160831107251112,0.534522692357088,0.956705203771399,0.968125700691649,0.412805097155230,0.556998524131909,0.523605414807628 };
		CMatrixDouble  B(5,5, dat_B);

		CMatrixDouble  RES(5,3);
		A.leftDivideSquare(B, RES);

		CMatrixDouble  RR = B * RES;
		cout << "(RR-A): " << endl << (RR-A) << endl;
		CHECK_AND_RET_ERROR(fabs((RR-A).sumAll())>1e-4,  "Error in left matrix divide text 5x5\\5x3")
	}
#endif

	}
	catch (std::exception &e)
	{
		cerr << e.what() << endl;
		return 1;
	}
}
