/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/slam/CSensoryFrame.h>
#include <mrpt/slam/CSimplePointsMap.h>

using namespace mrpt::slam;
using namespace mrpt::poses;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;

#include <mrpt/utils/metaprogramming.h>
using namespace mrpt::utils::metaprogramming;

IMPLEMENTS_SERIALIZABLE( CSensoryFrame, CSerializable, mrpt::slam )
// -------------------------------------------------------------------------------
//   The CSensoryFrame-HACK:		*** "Temporary" hack!! :-( ***
//  The name of this class was previously "CSensorialFrame", so this 'hack' is
//  needed to allow portability in the old serializations. This hack will
//  be eventually removed when all old data become ported to the new format!
//                                                          JLBC, 14/DEC/2007
// -------------------------------------------------------------------------------
CLASSINIT CSensoryFrame::_init_CSensorialFrame( (static_cast<TRuntimeClassId*>(&CSensoryFrame::classCSensorialFrame)) );
TRuntimeClassId CSensoryFrame::classCSensorialFrame = {
		"CSensorialFrame",
		sizeof(class CSensoryFrame),
		CSensoryFrame::CreateObject,
		&CSensoryFrame::_GetBaseClass };


/*---------------------------------------------------------------
						Default constructor
  ---------------------------------------------------------------*/
CSensoryFrame::CSensoryFrame() :
	m_auxMap(),
	m_observations()
{
}

/*---------------------------------------------------------------
						Copy constructor
  ---------------------------------------------------------------*/
CSensoryFrame::CSensoryFrame( const CSensoryFrame &o) :
	m_auxMap(),
	m_observations()
{
	*this = o;
}

/*---------------------------------------------------------------
							Copy
  ---------------------------------------------------------------*/
CSensoryFrame & CSensoryFrame::operator =( const CSensoryFrame &o)
{
	MRPT_START;

	clear();

	if (this == &o) return *this;		// It may be used sometimes

	// JL: Why make a copy? Just copy the smart pointers:
//	size_t i,n;
//	n = o.m_observations.size();
//	m_observations.resize( n );
//	for (i=0;i<n;i++)
//	{
//		m_observations[i] = o.m_observations[i]; //static_cast<CObservation*>( o.m_observations[i]->duplicate() );
//		m_observations[i].make_unique();
//	}

	m_observations = o.m_observations;


	m_auxMap.clear();

	return *this;

	MRPT_END;

}

/*---------------------------------------------------------------
							Destructor
  ---------------------------------------------------------------*/
CSensoryFrame::~CSensoryFrame()
{
	clear();
}


/*---------------------------------------------------------------
							clear
  ---------------------------------------------------------------*/
void  CSensoryFrame::clear()
{
	m_observations.clear();
}

/*---------------------------------------------------------------
						writeToStream
  ---------------------------------------------------------------*/
void  CSensoryFrame::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 2;
	else
	{
		uint32_t		i,n;

		n = static_cast<uint32_t>(m_observations.size());
		out << n;
		for (i=0;i<n;i++)
			out << *m_observations[i];
	}
}

/*---------------------------------------------------------------
						readFromStream
  ---------------------------------------------------------------*/
void  CSensoryFrame::readFromStream(CStream &in,int version)
{
	MRPT_START;

	switch(version)
	{
	case 0:
	case 1:
	case 2:
		{
			uint32_t	i,n;
			mrpt::system::TTimeStamp	tempTimeStamp= INVALID_TIMESTAMP;

			clear();
			if (version<2)		// ID was removed in version 2
			{
				uint32_t ID;
				in >> ID;
			}

			if (version==0)
				in.ReadBuffer( &tempTimeStamp, sizeof(tempTimeStamp));

			in >> n;
			m_observations.resize(n);
			for_each( m_observations.begin(), m_observations.end(), ObjectReadFromStream(&in) );

			if (version==0)
				for (i=0;i<n;i++)
					m_observations[i]->timestamp = tempTimeStamp;

			m_auxMap.clear();

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

	MRPT_END;
}


/*---------------------------------------------------------------
						insertObservationInto
  ---------------------------------------------------------------*/
bool  CSensoryFrame::insertObservationsInto(
	CMetricMap	*theMap,
	const CPose3D		*robotPose ) const
{
	bool	anyone = false;
	for (const_iterator it = begin();it!=end();++it)
		anyone|= (*it)->insertObservationInto(theMap, robotPose);

	return anyone;
}

/*---------------------------------------------------------------
						operator +=
  ---------------------------------------------------------------*/
void CSensoryFrame::operator += (const CSensoryFrame &sf)
{
	for (const_iterator it = begin();it!=end();++it)
	{
		CObservationPtr newObs = *it;
		newObs.make_unique();
		m_observations.push_back( newObs ); //static_cast<CObservation*>( (*it)->duplicate()) );
	}
}

/*---------------------------------------------------------------
						operator +=
  ---------------------------------------------------------------*/
void CSensoryFrame::operator += (const CObservationPtr &obs)
{
	m_observations.push_back( obs );
}

/*---------------------------------------------------------------
					buildAuxPointsMap
  ---------------------------------------------------------------*/
const CPointsMap	*CSensoryFrame::buildAuxPointsMap( CPointsMap::TInsertionOptions	*options) const
{
	if (!m_auxMap.get())
	{
		// Create:
		CPointsMap *m = new CSimplePointsMap();
		if (options)
			m->insertionOptions = *options;
		insertObservationsInto(m);
		m_auxMap.set(m);
	}

	return m_auxMap.get();
}

/*---------------------------------------------------------------
					likelihoodWith
  ---------------------------------------------------------------*/
float  CSensoryFrame::likelihoodWith(
	const CSensoryFrame *anotherSF,
	const CPosePDF	  *anotherSFPose) const
{
	float SUM = 0;
	int   nCmps = 0;
	for (const_iterator it1=begin();it1!=end();it1++)
	{
		for (const_iterator it2=anotherSF->begin();it2!=anotherSF->end();it2++)
		{
			SUM += (*it2)->likelihoodWith( (CObservation*)it1->pointer(), anotherSFPose );
			nCmps++;
		}
	}

	return nCmps ? (SUM/nCmps) : 0;
}

/*---------------------------------------------------------------
					size
  ---------------------------------------------------------------*/
size_t CSensoryFrame::size() const
{
	return m_observations.size();
}


/*---------------------------------------------------------------
					push_back
  ---------------------------------------------------------------*/
void CSensoryFrame::push_back(const CObservationPtr &obs)
{
	m_observations.push_back( obs );
}

/*---------------------------------------------------------------
				insert
  ---------------------------------------------------------------*/
void CSensoryFrame::insert(const CObservationPtr &obs)
{
	m_observations.push_back( obs );
}

/*---------------------------------------------------------------
				eraseByIndex
  ---------------------------------------------------------------*/
void CSensoryFrame::eraseByIndex(const size_t &idx)
{
	MRPT_START
	if (idx>=size()) THROW_EXCEPTION_CUSTOM_MSG1("Index %u out of range.", static_cast<unsigned>(idx) );

	iterator it = begin()+idx;
	ASSERT_(it->present());
	//delete (*it);
	m_observations.erase( it );
	MRPT_END
}


/*---------------------------------------------------------------
					getObservationByIndex
  ---------------------------------------------------------------*/
CObservationPtr CSensoryFrame::getObservationByIndex( const size_t &idx ) const
{
	MRPT_START
	if (idx>=size()) THROW_EXCEPTION_CUSTOM_MSG1("Index %u out of range.", static_cast<unsigned>(idx) );

	const_iterator it = begin()+idx;
	return *it;

	MRPT_END
}

/*---------------------------------------------------------------
					erase
  ---------------------------------------------------------------*/
CSensoryFrame::iterator CSensoryFrame::erase( const iterator &it)
{
	MRPT_START
	ASSERT_(it!=end());

	//CObservationPtr *obs = *it;
	//delete obs;

	return m_observations.erase(it);
	MRPT_END
}

/*---------------------------------------------------------------
					getObservationBySensorLabel
  ---------------------------------------------------------------*/
CObservationPtr CSensoryFrame::getObservationBySensorLabel(
	const std::string &label,
	const size_t &idx) const
{
	MRPT_START;

	size_t  foundCount = 0;
	for (const_iterator it = begin();it!=end();++it)
		if ( !os::_strcmpi( (*it)->sensorLabel.c_str(), label.c_str() ) )
			if (foundCount++ == idx)
				return *it;

	return CObservationPtr();

	MRPT_END;
}

/*---------------------------------------------------------------
						moveFrom
  ---------------------------------------------------------------*/
void CSensoryFrame::moveFrom( CSensoryFrame &sf )
{
	copy(sf.m_observations.begin(),sf.m_observations.end(), back_inserter(m_observations) );
	sf.m_observations.clear();
}

/*---------------------------------------------------------------
						swap
  ---------------------------------------------------------------*/
void CSensoryFrame::swap( CSensoryFrame &sf )
{
	m_observations.swap(sf.m_observations);
}

/*---------------------------------------------------------------
						eraseByLabel
  ---------------------------------------------------------------*/
void CSensoryFrame::eraseByLabel(const std::string &label)
{
	for (iterator it = begin();it!=end();  )
	{
		if ( !os::_strcmpi( (*it)->sensorLabel.c_str(), label.c_str() ) )
		{
			it = erase(it);
		}
		else it++;
	}
}
