/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/scan_matching/scan_matching.h>
#include <mrpt/poses/CPosePDFGaussian.h>
#include <mrpt/poses/CPosePDFSOG.h>
#include <mrpt/random.h>
#include <mrpt/math/CMatrixD.h>
#include <mrpt/math/utils.h>
#include <mrpt/math/CQuaternion.h>

#include <algorithm>

using namespace mrpt;
using namespace mrpt::scan_matching;
using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace std;

/*---------------------------------------------------------------
			leastSquareErrorRigidTransformation
  ---------------------------------------------------------------*/
bool  scan_matching::leastSquareErrorRigidTransformation(
	CMetricMap::TMatchingPairList	&in_correspondences,
	CPose2D								&out_transformation,
	CMatrixD							*out_estimateCovariance )
{
	MRPT_TRY_START;

	size_t										nCorrespondences = in_correspondences.size();
	double										lx,ly;
	double										gx, gy;
	CMetricMap::TMatchingPairList::iterator		corrIt;
	double										Ax,Ay,ccos,csin,Ei;	// Aux. terms

	if (nCorrespondences<2)
		return false;

	// Compute the estimated pose.
	//  (Notation from paper of J.Gonzalez, Martinez y Morales)
	// ----------------------------------------------------------------------
	double	Sxj, Sxj1, Syj, Syj1, Sxx, Sxy, Syx, Syy;
	Sxj = Syj = Sxj1 = Syj1 = Sxx = Sxy = Syx = Syy = 0;

	for (corrIt=in_correspondences.begin(); corrIt!=in_correspondences.end(); corrIt++)
	{
		// Get the pair of points in the correspondence:
		gx = corrIt->this_x;
		gy = corrIt->this_y;
		lx = corrIt->other_x;
		ly = corrIt->other_y;

		// Compute the terms:
		Sxj+=gx;
		Syj+=gy;

		Sxj1 += lx;
		Syj1 += ly;

		Sxx += gx * lx;
		Sxy += gx * ly;
		Syx += gy * lx;
		Syy += gy * ly;
	}	// End of "for all correspondences"...

	// Estimation:
	Ay = Sxj * Syj1 + nCorrespondences*(Syx-Sxy)- Sxj1 * Syj;
	Ax = nCorrespondences*(Sxx + Syy) - Sxj*Sxj1 - Syj*Syj1;
	if (Ax!=0 || Ay!=0)
	{
		out_transformation.phi = atan2( Ay, Ax);
		if (Ax==0)
		{
			Ei = 0;
		}
		else
		{
			if (fabs(Ay/Ax)<1e-14f)
					Ei = 1e14f;
			else	Ei = Ax / Ay;
		}
	}
	else
	{
		Ei = 0;
		out_transformation.phi = 0;
	}

	ccos = cos( out_transformation.phi );
	csin = sin( out_transformation.phi );

	out_transformation.x = ( Sxj - ccos * Sxj1 + csin * Syj1 ) / nCorrespondences;
	out_transformation.y = ( Syj - csin * Sxj1 - ccos * Syj1 ) / nCorrespondences;

	if ( out_estimateCovariance )
	{
		// Compute the normalized covariance matrix:
		// -------------------------------------------
		out_estimateCovariance->setSize(3,3);
		out_estimateCovariance->zeros();

		double		mean_x_a = Sxj / nCorrespondences;
		double		mean_y_a = Syj / nCorrespondences;
		double		mean_x_b = Sxj1 / nCorrespondences;
		double		mean_y_b = Syj1 / nCorrespondences;
		double		var_x_a = 0,var_y_a = 0,var_x_b = 0,var_y_b = 0;

		// 0) Precompute the unbiased variances estimations:
		// ----------------------------------------------------
		for (corrIt=in_correspondences.begin(); corrIt!=in_correspondences.end(); corrIt++)
		{
			var_x_a += square( corrIt->this_x - mean_x_a );
			var_y_a += square( corrIt->this_y - mean_y_a );
			var_x_b += square( corrIt->other_x - mean_x_b );
			var_y_b += square( corrIt->other_y - mean_y_b );
		}
		var_x_a /= (nCorrespondences-1);
		var_x_a /= (nCorrespondences-1);
		var_x_a /= (nCorrespondences-1);
		var_x_a /= (nCorrespondences-1);

		// 1) Compute  s_Delta^2 / s_p^2
		// --------------------------------
		double		var_Delta = (var_x_a+var_y_a+var_x_b+var_y_b)*pow(static_cast<double>(nCorrespondences),3.0)/pow(static_cast<double>(nCorrespondences),4.0);

		// 2) Compute  the jacobian J_Delta
		// -----------------------------------
		double		K_1 = square(Ax)+square(Ay);
		double		K_32 = pow(K_1,1.5);
		CMatrixD		J(3,2);

		J(0,0) = -( mean_x_b * Ay*Ay + mean_y_b * Ax * Ay )/K_32;
		J(0,1) =  ( mean_x_b * Ax*Ay + mean_y_b * Ax * Ax )/K_32;
		J(1,0) =  ( mean_x_b * Ax*Ay - mean_y_b * Ay * Ay )/K_32;
		J(1,1) = -( mean_x_b * Ax*Ax - mean_y_b * Ax * Ay )/K_32;
		J(2,0) = - Ay / K_1;
		J(2,1) =   Ax / K_1;

		// 3) And the final covariance matrix:
		// -------------------------------------
		//(*out_estimateCovariance) = J * (~J);
		out_estimateCovariance->multiply_ABt(J,J);
		(*out_estimateCovariance) *= var_Delta;
		(*out_estimateCovariance)(0,0) += 2.0f/square(nCorrespondences);
		(*out_estimateCovariance)(1,1) += 2.0f/square(nCorrespondences);
	}


	return true;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------

					robustRigidTransformation

 This works as follows:
	- Repeat "ransac_nSimulations" times:
		- Randomly pick TWO correspondences from the set "in_correspondences".
		- Compute the associated rigid transformation.
		- For "ransac_maxSetSize" randomly selected correspondences, test for "consensus" with the current group:
			- If if is compatible (ransac_maxErrorXY, ransac_maxErrorPHI), grow the "consensus set"
			- If not, do not add it.
  ---------------------------------------------------------------*/
void  scan_matching::robustRigidTransformation(
	mrpt::slam::CMetricMap::TMatchingPairList	&in_correspondences,
	poses::CPosePDFSOG				&out_transformation,
	float							normalizationStd,
	unsigned int					ransac_minSetSize,
	unsigned int					ransac_maxSetSize,
	float						ransac_mahalanobisDistanceThreshold,
	unsigned int					ransac_nSimulations,
	mrpt::slam::CMetricMap::TMatchingPairList	*out_largestSubSet,
	bool						ransac_fuseByCorrsMatch,
	float						ransac_fuseMaxDiffXY,
	float						ransac_fuseMaxDiffPhi,
	bool						ransac_algorithmForLandmarks
	)
{
	size_t								i,nCorrespondences = in_correspondences.size();
	unsigned int						maxThis=0, maxOther=0;
	int									debugCheckPoint = 0;
	CPosePDFGaussian					temptativeEstimation, referenceEstimation;
	CMetricMap::TMatchingPairList::iterator		matchIt;
	std::vector<bool>					alreadySelectedThis;
	std::vector<bool>					alreadySelectedOther;

//#define DEBUG_OUT

	MRPT_TRY_START;

	// Asserts:
	if( nCorrespondences < ransac_minSetSize )
	{
		// Nothing to do!
		out_transformation.m_modes.clear();
		return;
	}

	// Find the max. index of "this" and "other:
	//for (i=0;i<nCorrespondences;i++)
	for (matchIt=in_correspondences.begin();matchIt!=in_correspondences.end(); matchIt++)
	{
		maxThis = max(maxThis , matchIt->this_idx  );
		maxOther= max(maxOther, matchIt->other_idx );
	}

	debugCheckPoint = 1;

	// Fill out 2 arrays indicating whether each element has a correspondence:
	std::vector<bool>	hasCorrThis(maxThis+1,false);
	std::vector<bool>	hasCorrOther(maxOther+1,false);
	unsigned int		howManyDifCorrs = 0;
	//for (i=0;i<nCorrespondences;i++)
	for (matchIt=in_correspondences.begin();matchIt!=in_correspondences.end(); matchIt++)
	{
		if (!hasCorrThis[matchIt->this_idx] &&
			!hasCorrOther[matchIt->other_idx] )
		{
			hasCorrThis[matchIt->this_idx] = true;
			hasCorrOther[matchIt->other_idx] = true;
			howManyDifCorrs++;
		}
	}

	debugCheckPoint = 2;

	// Clear the set of output particles:
	out_transformation.m_modes.clear();

	debugCheckPoint = 22;

	// The max. number of corrs!
	//ransac_maxSetSize = min(ransac_maxSetSize, max(2,(howManyDifCorrs-1)));
	ransac_maxSetSize = min(ransac_maxSetSize, max((unsigned int)2,howManyDifCorrs) );

	//printf("howManyDifCorrs=%u  ransac_maxSetSize=%u\n",howManyDifCorrs,ransac_maxSetSize);

	//ASSERT_( ransac_maxSetSize>=ransac_minSetSize );
	if ( ransac_maxSetSize < ransac_minSetSize )
	{
		// Nothing we can do here!!! :~$
		if (out_largestSubSet!=NULL)
		{
			CMetricMap::TMatchingPairList		emptySet;
			*out_largestSubSet = emptySet;
		}

		out_transformation.m_modes.clear();
		return;
	}

	debugCheckPoint = 21;

//#define AVOID_MULTIPLE_CORRESPONDENCES

#ifdef  AVOID_MULTIPLE_CORRESPONDENCES
	unsigned 					k;
	// Find duplicated landmarks (from SIFT features with different descriptors,etc...)
	//   this is to avoid establishing multiple correspondences for the same physical point!
	// ------------------------------------------------------------------------------------------------
	std::vector<vector_int>		listDuplicatedLandmarksThis(maxThis+1);
	ASSERT_(nCorrespondences>=1);
	for (k=0;k<nCorrespondences-1;k++)
	{
		vector_int		duplis;
		for (unsigned j=k;j<nCorrespondences-1;j++)
		{
			if ( in_correspondences[k].this_x == in_correspondences[j].this_x &&
				 in_correspondences[k].this_y == in_correspondences[j].this_y &&
				 in_correspondences[k].this_z == in_correspondences[j].this_z )
					duplis.push_back(in_correspondences[j].this_idx);
		}
		listDuplicatedLandmarksThis[in_correspondences[k].this_idx] = duplis;
	}

	std::vector<vector_int>		listDuplicatedLandmarksOther(maxOther+1);
	for (k=0;k<nCorrespondences-1;k++)
	{
		vector_int		duplis;
		for (unsigned j=k;j<nCorrespondences-1;j++)
		{
			if ( in_correspondences[k].other_x == in_correspondences[j].other_x &&
				 in_correspondences[k].other_y == in_correspondences[j].other_y &&
				 in_correspondences[k].other_z == in_correspondences[j].other_z )
					duplis.push_back(in_correspondences[j].other_idx);
		}
		listDuplicatedLandmarksOther[in_correspondences[k].other_idx] = duplis;
	}
#endif

	std::deque<CMetricMap::TMatchingPairList>	alreadyAddedSubSets;
	std::vector<size_t> 	corrsIdxs( nCorrespondences), corrsIdxsPermutation;
	for (i=0;i<nCorrespondences;i++) corrsIdxs[i]= i;

	// If we put this out of the loop, each correspondence will be used just ONCE!
	/**/
	alreadySelectedThis.clear();
	alreadySelectedThis.resize(maxThis+1,false);
	alreadySelectedOther.clear();
	alreadySelectedOther.resize(maxOther+1, false);
	/**/

	CPosePDFGaussian	temptativeEstimation;

	// -------------------------
	//		The RANSAC loop
	// -------------------------
	for (i=0;i<ransac_nSimulations;i++)
	{
		debugCheckPoint = 24;

		CMetricMap::TMatchingPairList		subSet,temptativeSubSet;

		// Select a subset of correspondences at random:
		if (ransac_algorithmForLandmarks)
		{
			alreadySelectedThis.clear();
			alreadySelectedThis.resize(maxThis+1,false);
			alreadySelectedOther.clear();
			alreadySelectedOther.resize(maxOther+1, false);
		}
		else
		{
			// For points: Do not repeat the corrs, and take the numer of corrs as weights
		}

		debugCheckPoint = 3;

		// Try to build a subsetof "ransac_maxSetSize" (maximum) elements that achieve consensus:
		// ------------------------------------------------------------------------------------------
		// First: Build a permutation of the correspondences to pick from it sequentially:
		mrpt::random::randomPermutation( corrsIdxs,corrsIdxsPermutation );

		for (unsigned int j=0;j<ransac_maxSetSize;j++)
		{
			ASSERT_(j<corrsIdxsPermutation.size())

			size_t	idx = corrsIdxsPermutation[j];

			matchIt = in_correspondences.begin() + idx;

			ASSERT_( matchIt->this_idx < alreadySelectedThis.size() );
			ASSERT_( matchIt->other_idx < alreadySelectedOther.size() );

			if ( !(alreadySelectedThis [ matchIt->this_idx ] &&
					alreadySelectedOther[ matchIt->other_idx]) )
//			if ( !alreadySelectedThis [ matchIt->this_idx ] &&
//			     !alreadySelectedOther[ matchIt->other_idx]  )
			{
				// mark as "selected" for this pair not to be selected again:
				//  ***NOTE***: That the expresion of the "if" above requires the
				//  same PAIR not to be selected again, but one of the elements
				//  may be selected again with a diferent matching! This improves the
				//  robustness and posibilities of the algorithm! (JLBC - NOV/2006)

#ifndef  AVOID_MULTIPLE_CORRESPONDENCES
				alreadySelectedThis[ matchIt->this_idx ]= true;
				alreadySelectedOther[ matchIt->other_idx ] = true;
#else
				for (vector_int::iterator it1 = listDuplicatedLandmarksThis[matchIt->this_idx].begin();it1!=listDuplicatedLandmarksThis[matchIt->this_idx].end();it1++)
					alreadySelectedThis[ *it1 ] = true;
				for (vector_int::iterator it2 = listDuplicatedLandmarksOther[matchIt->other_idx].begin();it2!=listDuplicatedLandmarksOther[matchIt->other_idx].end();it2++)
					alreadySelectedOther[ *it2 ] = true;
#endif

				if (subSet.size()<2)
				{
					// ------------------------------------------------------------------------------------------------------
					// If we are within the first two correspondences, just add them to the subset:
					// ------------------------------------------------------------------------------------------------------
					subSet.push_back( *matchIt );

					if (subSet.size()==2)
					{
						temptativeSubSet = subSet;
						// JLBC: Modification DEC/2007: If we leave only ONE correspondence in the ref. set
						//  the algorithm will be pretty much sensible to reject bad correspondences:
						temptativeSubSet.erase( temptativeSubSet.begin() + (temptativeSubSet.size() -1) );

						// Perform estimation:
						scan_matching::leastSquareErrorRigidTransformation(
							subSet,
							referenceEstimation.mean,
							&referenceEstimation.cov );
						// Normalized covariance: scale!
						referenceEstimation.cov *= square(normalizationStd);

						// Additional filter:
						//  If the correspondences as such the transformation has a high ambiguity, we discard it!
						if ( referenceEstimation.cov(2,2)>=square(DEG2RAD(5.0f)) )
						{
						 	// Remove this correspondence & try again with a different pair:
						 	subSet.erase( subSet.begin() + (subSet.size() -1) );
						}
						else
						{
						}
					}
				}
				else
				{
					// ------------------------------------------------------------------------------------------------------
					// The normal case:
					//  - test for "consensus" with the current group:
					//		- If it is compatible (ransac_maxErrorXY, ransac_maxErrorPHI), grow the "consensus set"
					//		- If not, do not add it.
					// ------------------------------------------------------------------------------------------------------

					// Compute the temptative new estimation (matchIt will be removed after the test!):
					temptativeSubSet.push_back( *matchIt );

					scan_matching::leastSquareErrorRigidTransformation(
						temptativeSubSet,
						temptativeEstimation.mean,
						&temptativeEstimation.cov );
					// Normalized covariance: scale!
					temptativeEstimation.cov *= square(normalizationStd);

					// Additional filter:
					//  If the correspondences as such the transformation has a high ambiguity, we discard it!
					if ( temptativeEstimation.cov(2,2)<square(DEG2RAD(5.0f)) )
					{
						// ASSERT minimum covariance!!
						/*temptativeEstimation.cov(0,0) = max( temptativeEstimation.cov(0,0), square( 0.03f ) );
						temptativeEstimation.cov(1,1) = max( temptativeEstimation.cov(1,1), square( 0.03f ) );

						referenceEstimation.cov(0,0) = max( referenceEstimation.cov(0,0), square( 0.03f ) );
						referenceEstimation.cov(1,1) = max( referenceEstimation.cov(1,1), square( 0.03f ) ); */

						temptativeEstimation.cov(2,2) = max( temptativeEstimation.cov(2,2), square( DEG2RAD(0.05) ) );
						referenceEstimation.cov(2,2) = max( referenceEstimation.cov(2,2), square( DEG2RAD(0.05) ) );



						// debug:
						//if (mahaDist>1)
						/*if (matchIt->this_y>-0.3 && matchIt->this_y<0.4 && matchIt->this_x>0.9 && matchIt->this_x<1.7)
								cout << "CENTRO\n==========\n";
						else	cout << "NORMAL\n==========\n";
						cout << "|temptativeSubSet|=" << temptativeSubSet.size() << endl;
						cout << "temptativeEstimation:" << temptativeEstimation.mean << " referenceEstimation:" << referenceEstimation.mean << " mahaDist:" << mahaDist << "\n";
						*/

						// Test for compatibility:
						bool passTest;

						if (ransac_algorithmForLandmarks)
						{
							// Compatibility test: Mahalanobis distance between Gaussians:
							double	mahaDist = temptativeEstimation.mahalanobisDistanceTo( referenceEstimation );
							passTest = mahaDist < ransac_mahalanobisDistanceThreshold;
						}
						else
						{
							// Compatibility test: Euclidean distances
							double diffXY = referenceEstimation.mean.distanceTo( temptativeEstimation.mean );
							double diffPhi = fabs( math::wrapToPi( referenceEstimation.mean.phi - temptativeEstimation.mean.phi ) );
							passTest  = diffXY < 0.02f && diffPhi < DEG2RAD(2.0f);

							//FILE *f=os::fopen("hist.txt","at");
							//fprintf(f,"%f %f\n",diffXY, RAD2DEG(diffPhi) );
							//fclose(f);
						}

						if ( passTest )
						{
							// OK, consensus passed!!
							subSet.push_back( *matchIt );
							referenceEstimation = temptativeEstimation;
						}
						else
						{
							// Test failed!
							//printf("Discarded!:\n");
							//std::cout << "temptativeEstimation:" << temptativeEstimation << " referenceEstimation:" << referenceEstimation << " mahaDist:" << mahaDist << "\n";
						}
					}
					else
					{
						// Test failed!
						//printf("Discarded! stdPhi=%f\n",RAD2DEG(sqrt(temptativeEstimation.cov(2,2))));
					}

					// Remove the temporaly added last correspondence:
					temptativeSubSet.erase( temptativeSubSet.begin() + (temptativeSubSet.size() -1) );

				} // end else "normal case"

			} // end "if" the randomly selected item is new

		} // end for j

		debugCheckPoint = 4;

		// Save the estimation result as a "particle", only if the subSet contains
		//  "ransac_minSetSize" elements at least:
		if (subSet.size()>=ransac_minSetSize)
		{
			debugCheckPoint = 5;

			// If this subset was previously added to the SOG, just increment its weight
			//  and do not add a new mode:
			int		indexFound = -1;



			// JLBC Added DEC-2007: An alternative (optional) method to fuse Gaussian modes:
			if (!ransac_fuseByCorrsMatch)
			{
				// Find matching by approximate match in the X,Y,PHI means
				// -------------------------------------------------------------------
				// Recompute referenceEstimation from all the corrs:
				scan_matching::leastSquareErrorRigidTransformation(
					subSet,
					referenceEstimation.mean,
					&referenceEstimation.cov );
				// Normalized covariance: scale!
				referenceEstimation.cov *= square(normalizationStd);
				for (size_t i=0;i<out_transformation.m_modes.size();i++)
				{
					double diffXY = out_transformation.m_modes[i].mean.distanceTo( referenceEstimation.mean );
					double diffPhi = fabs( math::wrapToPi( out_transformation.m_modes[i].mean.phi - referenceEstimation.mean.phi ) );
					if ( diffXY < ransac_fuseMaxDiffXY && diffPhi < ransac_fuseMaxDiffPhi )
					{
						//printf("Match by distance found: distXY:%f distPhi=%f deg\n",diffXY,RAD2DEG(diffPhi));
						indexFound = i;
						break;
					}
				}
			}
			else
			{
				// Find matching mode by exact match in the list of correspondences:
				// -------------------------------------------------------------------
				// Sort "subSet" in order to compare them easily!
				//std::sort( subSet.begin(), subSet.end() );

				// Try to find matching corrs:
				for (size_t i=0;i<alreadyAddedSubSets.size();i++)
				{
					//
					if ( subSet == alreadyAddedSubSets[i] )
					{
						indexFound = i;
						break;
					}
				}
			}

			if (indexFound!=-1)
			{
				// This is an alrady added mode:
				if (ransac_algorithmForLandmarks)
						out_transformation.m_modes[indexFound].log_w = log(1+ exp(out_transformation.m_modes[indexFound].log_w));
				else	out_transformation.m_modes[indexFound].log_w = log(subSet.size()+ exp(out_transformation.m_modes[indexFound].log_w));
			}
			else
			{
				// Add a new mode to the SOG:
				alreadyAddedSubSets.push_back( subSet );

				//printf("New subset: "); for (size_t k=0;k<subSet.size();k++) printf("%i ",subSet[k].this_idx);	printf("\n");
				//printf("Subset has %i corrs -> ",subSet.size());  cout << referenceEstimation.mean << endl;
				//subSet.saveAsMATLABScript("subset.m");
				//system::pause();

				poses::CPosePDFSOG::TGaussianMode	newSOGMode;
				if (ransac_algorithmForLandmarks)
						newSOGMode.log_w = 0; //log(1);
				else	newSOGMode.log_w = log(static_cast<double>(subSet.size()));

				scan_matching::leastSquareErrorRigidTransformation(
					subSet,
					newSOGMode.mean,
					&newSOGMode.cov );

				// Normalized covariance: scale!
				newSOGMode.cov *= square(normalizationStd);

				// Add a new mode to the SOG!
				out_transformation.m_modes.push_back(newSOGMode);
			}
		}

		debugCheckPoint = 6;

		// Save the largest subset:
		if (out_largestSubSet!=NULL)
		{
			if (subSet.size()>out_largestSubSet->size())
			{
				*out_largestSubSet = subSet;
			}
		}

#ifdef DEBUG_OUT
		printf("[RANSAC] Sim #%i/%i \t--> |subSet|=%u \n",
			(int)i,
			(int)ransac_nSimulations,
			(unsigned)subSet.size()
			);
#endif
	} // end for i

	debugCheckPoint = 7;

	// Set the weights of the particles to sum the unity:
	out_transformation.normalizeWeights();

	// Now the estimation is in the particles set!
	// Done!

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("maxThis=%u, maxOther=%u\n",static_cast<unsigned int>(maxThis), static_cast<unsigned int>(maxOther)); \
		printf("nCorrespondences=%u\n",static_cast<unsigned int>(nCorrespondences)); \
		printf("debugCheckPoint=%i\n",debugCheckPoint); \
		printf("Saving '_debug_in_correspondences.txt'..."); \
		in_correspondences.dumpToFile("_debug_in_correspondences.txt"); \
		printf("Ok\n"); \
		printf("Saving '_debug_out_transformation.txt'..."); \
		out_transformation.saveToTextFile("_debug_out_transformation.txt"); \
		printf("Ok\n"); );

}

/*---------------------------------------------------------------
	leastSquareErrorRigidTransformation6D
  ---------------------------------------------------------------*/
bool  scan_matching::leastSquareErrorRigidTransformation6DRANSAC(
	const CMetricMap::TMatchingPairList	&in_correspondences,
	CPose3D								&out_transformation,
	double								&out_scale,
	vector_int							&out_inliers_idx,
	CMatrixD							*in_rotationMatrix,
	CMatrixD							*out_estimateCovariance )
{
	MRPT_UNUSED_PARAM( in_rotationMatrix );
	MRPT_UNUSED_PARAM( out_estimateCovariance );

	MRPT_TRY_START;

	unsigned int N = (unsigned int)in_correspondences.size();

	// -------------------------------------------
	// Thresholds
	// -------------------------------------------
	vector_float	th(7);
	th[0] = 0.05;		// X (meters)
	th[1] = 0.05;		// Y (meters)
	th[2] = 0.05;		// Z (meters)

	th[3] = 1;			// YAW (degrees)
	th[4] = 1;			// PITCH (degrees)
	th[5] = 1;			// ROLL (degrees)

	th[6] = 0.03;		// SCALE

	// -------------------------------------------
	// RANSAC parameters
	// -------------------------------------------
	unsigned int	n, d, max_it, iterations;
	double			min_err = 1e2;						// Minimum error achieved so far
	unsigned int	max_size = 0;						// Maximum size of the consensus set so far
	double			scale;								// Output scale

	n		= 5;										// Minimum number of points to fit the model
	d		= round( N*0.7 );							// Minimum number of points to be considered a good set
	max_it	= 50;										// Maximum number of iterations

	// -------------------------------------------
	// MAIN loop
	// -------------------------------------------
	for( iterations = 0; iterations < max_it; iterations++ )
	{
		//printf("Iteration %2u of %u", iterations+1, max_it );

		// Generate maybe inliers
		vector_int	rub, mbSet, cSet;
		mrpt::math::linspace( (int)0, (int)N-1, (int)N, rub );
		mrpt::random::randomPermutation( rub, mbSet );

		// Compute first inliers output
		CPose3D							mbOut;
		vector_float					mbOut_vec(7);
		CMetricMap::TMatchingPairList	mbInliers;
		mbInliers.resize( n );
		for( unsigned int i = 0; i < n; i++ )
		{
			mbInliers[i] = in_correspondences[ mbSet[i] ];
			cSet.push_back( mbSet[i] );
		}

		bool res = leastSquareErrorRigidTransformation6D( mbInliers, mbOut, scale );
		if( !res )
			return false;

		// Maybe inliers Output
		mbOut_vec[0] = mbOut.x;
		mbOut_vec[1] = mbOut.y;
		mbOut_vec[2] = mbOut.z;

		mbOut_vec[3] = mbOut.yaw;
		mbOut_vec[4] = mbOut.pitch;
		mbOut_vec[5] = mbOut.roll;

		mbOut_vec[6] = scale;

		// Inner loop: for each point NOT in the maybe inliers
		for( unsigned int k = n; k < N; k++ )
		{
			CPose3D		csOut;

			// Consensus set: Maybe inliers + new point
			mbInliers.push_back( in_correspondences[ mbSet[k] ] );							// Insert
			bool res = leastSquareErrorRigidTransformation6D( mbInliers, csOut, scale );	// Compute
			mbInliers.erase( mbInliers.end()-1 );											// Erase
			if( !res )
				return false;

			// Is this point a supporter of the initial inlier group?
			if( fabs( mbOut_vec[0] - csOut.x ) < th[0] && fabs( mbOut_vec[1] - csOut.y ) < th[1] &&
				fabs( mbOut_vec[2] - csOut.z ) < th[2] && fabs( mbOut_vec[3] - csOut.yaw ) < th[3] &&
				fabs( mbOut_vec[4] - csOut.pitch ) < th[4] && fabs( mbOut_vec[5] - csOut.roll ) < th[5] &&
				fabs( mbOut_vec[6] - scale ) < th[6] )
			{
				// Inlier detected -> add to the inlier list
				cSet.push_back( mbSet[k] );
			} // end if INLIERS
			else
			{
				cout << " It " << iterations << " - RANSAC Outlier Detected: " << k << endl;
			}
		} // end 'inner' for

		// Test cSet size
		if( cSet.size() > d )
		{
			// Good set of points found
			CMetricMap::TMatchingPairList	cSetInliers;
			cSetInliers.resize( cSet.size() );
			for( unsigned int m = 0; m < cSet.size(); m++ )
				cSetInliers[m] = in_correspondences[ cSet[m] ];

			// Compute output: Consensus Set + Initial Inliers Guess
			CPose3D		cIOut;
			bool res = leastSquareErrorRigidTransformation6D( cSetInliers, cIOut, scale );	// Compute output
			if( !res )
				return false;

			// Compute error for consensus_set
			double err = sqrt( square( mbOut_vec[0] - cIOut.x ) + square( mbOut_vec[1] - cIOut.y ) + square( mbOut_vec[2] - cIOut.z ) +
							   square( mbOut_vec[3] - cIOut.yaw ) + square( mbOut_vec[4] - cIOut.pitch ) + square( mbOut_vec[5] - cIOut.roll ) +
							   square( mbOut_vec[6] - scale ));

			// Is the best set of points so far?
			if( err < min_err && cSet.size() >= max_size )
			{
				min_err						= err;
				max_size					= cSet.size();

				out_transformation.x		= cIOut.x;
				out_transformation.y		= cIOut.y;
				out_transformation.z		= cIOut.z;

				out_transformation.yaw		= cIOut.yaw;
				out_transformation.pitch	= cIOut.pitch;
				out_transformation.roll		= cIOut.roll;

				out_scale					= scale;

				out_inliers_idx				= cSet;
			} // end if SCALE ERROR
			//printf(" - Consensus set size: %u - Error: %.6f\n", (unsigned int)cSet.size(), err );
		} // end if cSet.size() > d
		else
		{
			//printf(" - Consensus set size: %u - Not big enough!\n", (unsigned int)cSet.size() );
		}
	} // end 'iterations' for

	MRPT_TRY_END;

	return true;
} // end leastSquareErrorRigidTransformation6DRANSAC


/*---------------------------------------------------------------
	leastSquareErrorRigidTransformation6D
  ---------------------------------------------------------------*/
bool  scan_matching::leastSquareErrorRigidTransformation6D(
	const CMetricMap::TMatchingPairList	&in_correspondences,
	CPose3D								&out_transformation,
	double								&out_scale,
	CMatrixDouble						*in_rotationMatrix,
	CMatrixDouble						*out_estimateCovariance,
	bool 								forceScaleToUnity)
{

	MRPT_UNUSED_PARAM(out_estimateCovariance);
	MRPT_UNUSED_PARAM(in_rotationMatrix);

	MRPT_TRY_START;
	if( in_correspondences.size() < 3 )
	{
		std::cout << "Error in ICP6D: At least 3 correspondences must be provided" << std::endl;
		return false;
	}

	CPoint3D					cL, cR;
	CMatrixD					S, N;
	CMatrixD					Z, D;
	CMatrixD					Rot;				// Rotation matrix

	double						roll, pitch ,yaw;

	std::vector<double>			v;
	int							nMatches = in_correspondences.size();

	double						s;					// Scale

	// Compute the centroid
	CMetricMap::TMatchingPairList::const_iterator	itMatch;

	for(itMatch = in_correspondences.begin(); itMatch != in_correspondences.end(); itMatch++)
	{
		cL.x += itMatch->this_x;
		cL.y += itMatch->this_y;
		cL.z += itMatch->this_z;

		cR.x += itMatch->other_x;
		cR.y += itMatch->other_y;
		cR.z += itMatch->other_z;
	}
	cL.x /= nMatches;	cL.y /= nMatches;	cL.z /= nMatches;
	cR.x /= nMatches;	cR.y /= nMatches;	cR.z /= nMatches;

	CMetricMap::TMatchingPairList			auxList( in_correspondences );
	CMetricMap::TMatchingPairList::iterator auxIt;
	// Substract the centroid
	for( auxIt = auxList.begin(); auxIt != auxList.end(); auxIt++ )
	{
		auxIt->this_x -= cL.x;
		auxIt->this_y -= cL.y;
		auxIt->this_z -= cL.z;

		auxIt->other_x -= cR.x;
		auxIt->other_y -= cR.y;
		auxIt->other_z -= cR.z;
	}

	if( in_rotationMatrix != NULL )
	{
		//std::cout << "Rotation Matrix Entered" << std::endl;
		CPose3D aux( *in_rotationMatrix );
		yaw = aux.yaw;
		pitch = aux.pitch;
		roll = aux.roll;
	}
	else
	{
		//std::cout << "Compute all" << std::endl;
		// Compute the S matrix of products
		S.setSize(3,3);
		S.fill(0);
		for( auxIt = auxList.begin(); auxIt != auxList.end(); auxIt++ )
		{
			S(0,0) += auxIt->this_x * auxIt->other_x;
			S(0,1) += auxIt->this_x * auxIt->other_y;
			S(0,2) += auxIt->this_x * auxIt->other_z;

			S(1,0) += auxIt->this_y * auxIt->other_x;
			S(1,1) += auxIt->this_y * auxIt->other_y;
			S(1,2) += auxIt->this_y * auxIt->other_z;

			S(2,0) += auxIt->this_z * auxIt->other_x;
			S(2,1) += auxIt->this_z * auxIt->other_y;
			S(2,2) += auxIt->this_z * auxIt->other_z;
		}

		N.setSize(4,4);
		N.fill(0);

		N(0,0) = S(0,0) + S(1,1) + S(2,2);
		N(0,1) = S(1,2) - S(2,1);
		N(0,2) = S(2,0) - S(0,2);
		N(0,3) = S(0,1) - S(1,0);

		N(1,0) = N(0,1);
		N(1,1) = S(0,0) - S(1,1) - S(2,2);
		N(1,2) = S(0,1) + S(1,0);
		N(1,3) = S(2,0) + S(0,2);

		N(2,0) = N(0,2);
		N(2,1) = N(1,2);
		N(2,2) = -S(0,0) + S(1,1) - S(2,2);
		N(2,3) = S(1,2) + S(2,1);

		N(3,0) = N(0,3);
		N(3,1) = N(1,3);
		N(3,2) = N(2,3);
		N(3,3) = -S(0,0) - S(1,1) + S(2,2);

		// q is the quaternion correspondent to the greatest eigenvector of the N matrix (last column in Z)
		N.eigenVectors( Z, D );
		Z.extractCol( Z.getColCount()-1, v );

		//CMatrix v = N.largestEigenvector();
		math::CQuaternion<double>	q( v[0], v[1], v[2], v[3] );
		//q = new math::CQuaternion( v[0], v[1], v[2], v[3] );

		// Compute Rotation
		q.rpy( roll, pitch, yaw );					// Rotation angles
		//std::cout << "YPR: " << yaw << "," << pitch << ","  << roll << std::endl;
		q.q_rotation_matrix( Rot );				// Rotation matrix
	} // end else

	// Compute scale
	double	num = 0.0;
	double	den = 0.0;
	for( auxIt = auxList.begin(); auxIt != auxList.end(); auxIt++ )
	{
		num += square(auxIt->other_x) + square(auxIt->other_y) + square(auxIt->other_z);
		den += square(auxIt->this_x) + square(auxIt->this_y) + square(auxIt->this_z);
	}
	s = sqrt( num/den );

	// Enforce scale to be 1
	out_scale = s;
	if (forceScaleToUnity)
		s = 1.0;

	// Compute traslation
	double x = cR.x - s*( Rot(0,0)*cL.x + Rot(0,1)*cL.y + Rot(0,2)*cL.z );
	double y = cR.y - s*( Rot(1,0)*cL.x + Rot(1,1)*cL.y + Rot(1,2)*cL.z );
	double z = cR.z - s*( Rot(2,0)*cL.x + Rot(2,1)*cL.y + Rot(2,2)*cL.z );

	//std::cout << "Rotation matrix: " << std::endl << Rot << std::endl;

	// Set out_transformation
	out_transformation.setFromValues( x, y, z, yaw, pitch, roll );
	//std::cout << "ICP:" << x << "," << y << "," << z << "," << yaw << "," << pitch << "," << roll << std::endl;

	MRPT_TRY_END;

	return true;

/*---------------------------------------------------------------
			leastSquareErrorRigidTransformation in 6D
  ---------------------------------------------------------------*/
	// Algorithm:
	// 0. Preliminary
	//		pLi = { pLix, pLiy, pLiz }
	//		pRi = { pRix, pRiy, pRiz }
	// -------------------------------------------------------
	// 1. Find the centroids of the two sets of measurements:
	//		cL = (1/n)*sum{i}( pLi )		cL = { cLx, cLy, cLz }
	//		cR = (1/n)*sum{i}( pRi )		cR = { cRx, cRy, cRz }
	//
	// 2. Substract centroids from the point coordinates:
	//		pLi' = pLi - cL					pLi' = { pLix', pLiy', pLiz' }
	//		pRi' = pRi - cR					pRi' = { pRix', pRiy', pRiz' }
	//
	// 3. For each pair of coordinates (correspondences) compute the nine possible products:
	//		pi1 = pLix'*pRix'		pi2 = pLix'*pRiy'		pi3 = pLix'*pRiz'
	//		pi4 = pLiy'*pRix'		pi5 = pLiy'*pRiy'		pi6 = pLiy'*pRiz'
	//		pi7 = pLiz'*pRix'		pi8 = pLiz'*pRiy'		pi9 = pLiz'*pRiz'
	//
	// 4. Compute S components:
	//		Sxx = sum{i}( pi1 )		Sxy = sum{i}( pi2 )		Sxz = sum{i}( pi3 )
	//		Syx = sum{i}( pi4 )		Syy = sum{i}( pi5 )		Syz = sum{i}( pi6 )
	//		Szx = sum{i}( pi7 )		Szy = sum{i}( pi8 )		Szz = sum{i}( pi9 )
	//
	// 5. Compute N components:
	//			[ Sxx+Syy+Szz	Syz-Szy			Szx-Sxz			Sxy-Syx		 ]
	//			[ Syz-Szy		Sxx-Syy-Szz		Sxy+Syx			Szx+Sxz		 ]
	//		N = [ Szx-Sxz		Sxy+Syx			-Sxx+Syy-Szz	Syz+Szy		 ]
	//			[ Sxy-Syx		Szx+Sxz			Syz+Szy			-Sxx-Syy+Szz ]
	//
	// 6. Rotation represented by the quaternion eigenvector correspondent to the higher eigenvalue of N
	//
	// 7. Scale computation (symmetric expression)
	//		s = sqrt( sum{i}( square(abs(pRi')) / sum{i}( square(abs(pLi')) ) )
	//
	// 8. Translation computation (distance between the Right centroid and the scaled and rotated Left centroid)
	//		t = cR-sR(cL)
}
