/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/opengl/CPointCloud.h>
#include <mrpt/slam/CPointsMap.h>
#include <mrpt/math/utils.h>

#include "opengl_internals.h"

using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CPointCloud, CRenderizable, mrpt::opengl )

/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
CPointCloud::CPointCloud( ) :
	m_colorFromZ(false),
	m_xs(),m_ys(),m_zs(),
	m_pointSize(1),
	m_z_min(0),
	m_z_max(0),
	m_z_minmax_valid(false),
	m_colorFromZ_min(0,0,0),
	m_colorFromZ_max(0,0,1)
{
}


/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void   CPointCloud::render() const
{
#if MRPT_HAS_OPENGL_GLUT

    ASSERT_(m_xs.size() == m_ys.size());
    ASSERT_(m_xs.size() == m_zs.size());

	float Az=0, Az_1=0;

	if ( m_colorFromZ )
	{
		if (!m_z_minmax_valid)
		{
			m_z_minmax_valid = true;
			math::minimum_maximum(m_zs, m_z_min, m_z_max);

			Az = m_z_max - m_z_min;
			if (Az==0)
				Az=-1;
			else	m_z_min = m_z_max - Az * 1.01f;
		}
		else
		{
			Az = m_z_max - m_z_min;
		}

		Az_1  = 1.0/Az;
	}

    vector<float>::const_iterator itX,itY,itZ;


	if ( m_color_A != 1.0 )
	{
		//glDisable(GL_DEPTH_TEST);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}


	// Slopes of color interpolation:
	const float AR = m_colorFromZ_max.R - m_colorFromZ_min.R;
	const float AG = m_colorFromZ_max.G - m_colorFromZ_min.G;
	const float AB = m_colorFromZ_max.B - m_colorFromZ_min.B;
	float AR_1,AG_1,AB_1;
	if (AR) AR_1 = 1.0/AR;
	if (AG) AG_1 = 1.0/AG;
	if (AB) AB_1 = 1.0/AB;

    glPointSize( m_pointSize );

    glBegin( GL_POINTS );

    glColor4f( m_color_R,m_color_G,m_color_B,m_color_A );

    for (itX=m_xs.begin(), itY=m_ys.begin(), itZ=m_zs.begin();
           itX!=m_xs.end();
         itX++,itY++,itZ++)
    {
		if ( m_colorFromZ && Az>0 )
		{
			float	f = (*itZ - m_z_min) * Az_1;
			f=max(0.0f,min(1.0f,f));

			glColor4f(
				m_colorFromZ_min.R + f*AR,
				m_colorFromZ_min.G + f*AG,
				m_colorFromZ_min.B + f*AB,
				m_color_A );
		}
		glVertex3f( *itX,*itY,*itZ );
    }
    glEnd();

	if ( m_color_A != 1.0 )
	{
		//glEnable(GL_DEPTH_TEST);
		glDisable(GL_BLEND);
	}

	checkOpenGLError();
#endif
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CPointCloud::writeToStream(CStream &out,int *version) const
{

	if (version)
		*version = 2;
	else
	{
		writeToStreamRender(out);
		out << m_colorFromZ;
		out << m_xs << m_ys << m_zs;

		// Added in version 1.
		out << m_pointSize;

		// New in version 2:
		out << m_colorFromZ_min.R << m_colorFromZ_min.G << m_colorFromZ_min.B;
		out << m_colorFromZ_max.R << m_colorFromZ_max.G << m_colorFromZ_max.B;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CPointCloud::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
	case 1:
	case 2:
		{
			m_z_minmax_valid = false;

			readFromStreamRender(in);
			in >> m_colorFromZ;
			in >> m_xs >> m_ys >> m_zs;

			if (version>=1)
				 in >> m_pointSize;
			else m_pointSize = 1;

			if (version>=2)
			{
				in >> m_colorFromZ_min.R >> m_colorFromZ_min.G >> m_colorFromZ_min.B;
				in  >> m_colorFromZ_max.R >> m_colorFromZ_max.G >> m_colorFromZ_max.B;
			}
			else
			{
				m_colorFromZ_min = TColorf(0,0,0);
				m_colorFromZ_max.R = m_color_R;
				m_colorFromZ_max.G = m_color_G;
				m_colorFromZ_max.B = m_color_B;
			}
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
						loadFromPointsMap
---------------------------------------------------------------*/
void  CPointCloud::loadFromPointsMap( const slam::CPointsMap *themap)
{
	themap->getAllPoints(m_xs,m_ys,m_zs);
}

/*---------------------------------------------------------------
						clear
---------------------------------------------------------------*/
void CPointCloud::clear()
{
	m_xs.clear();
	m_ys.clear();
	m_zs.clear();
}

/*---------------------------------------------------------------
						insertPoint
---------------------------------------------------------------*/
void CPointCloud::insertPoint( float x,float y, float z )
{
	m_xs.push_back(x);
	m_ys.push_back(y);
	m_zs.push_back(z);
}

/*---------------------------------------------------------------
					setGradientColors
---------------------------------------------------------------*/
void  CPointCloud::setGradientColors( const mrpt::utils::TColorf &colorMin, const mrpt::utils::TColorf &colorMax )
{
	m_colorFromZ_min = colorMin;
	m_colorFromZ_max = colorMax;
}
