/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef opengl_CGeneralizedCylinder_H
#define opengl_CGeneralizedCylinder_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/opengl/CPolyhedron.h>

namespace mrpt	{
namespace opengl	{
	class MRPTDLLIMPEXP CGeneralizedCylinder;
	// This must be added to any CSerializable derived class:
	DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE(CGeneralizedCylinder,CRenderizable)
	//Version 0.1: axis is supposed to be roughly PERPENDICULAR to the Z axis. So, the plane containing each control point's curve will be 
	//PARALLEL to that axis. This should me modified in the future so that any axis is accepted, although that'll be very dense.
	class MRPTDLLIMPEXP CGeneralizedCylinder:public CRenderizable	{
		DEFINE_SERIALIZABLE(CGeneralizedCylinder)
	public:
		struct MRPTDLLIMPEXP TQuadrilateral	{
		private:
			void calculateNormal();
			TQuadrilateral()	{}
		public:
			CPoint3D points[4];
			float normal[3];
			TQuadrilateral(const CPoint3D &p1,const CPoint3D &p2,const CPoint3D &p3,const CPoint3D &p4)	{
				points[0]=p1;
				points[1]=p2;
				points[2]=p3;
				points[3]=p4;
				calculateNormal();
			}
			TQuadrilateral(const CPoint3D (&p)[4])	{
				for (int i=0;i<4;i++) points[i]=p[i];
				calculateNormal();
			}
		};
	protected:
		vector_serializable<CPoint3D> axis;
		vector_serializable<CPoint3D> generatrix;
		mutable std::vector<TQuadrilateral> mesh;
		mutable bool meshUpToDate;
		bool closed;
	public:
		static CGeneralizedCylinderPtr Create(const std::vector<CPoint3D> &a,const std::vector<CPoint3D> &g)	{
			return CGeneralizedCylinderPtr(new CGeneralizedCylinder(a,g));
		}
		void render() const;
		virtual bool traceRay(const mrpt::poses::CPose3D &o,float &dist) const;
		inline void getAxis(std::vector<CPoint3D> &a) const	{
			a=axis;
		}
		inline void setAxis(const std::vector<CPoint3D> &a)	{
			axis=a;
			meshUpToDate=false;
		}
		inline void getGeneratrix(std::vector<CPoint3D> &g) const	{
			g=generatrix;
		}
		inline void setGeneratrix(const std::vector<CPoint3D> g)	{
			generatrix=g;
			meshUpToDate=false;
		}
		inline bool isClosed() const	{
			return closed;
		}
		inline void setClosed(bool c=true)	{
			closed=c;
			meshUpToDate=false;
		}
		void getOrigin(CPolyhedronPtr &poly) const;
		void getEnd(CPolyhedronPtr &poly) const;
	private:
		void updateMesh() const;
		CGeneralizedCylinder():axis(),generatrix(),mesh(),meshUpToDate(false),closed(false)	{}
		CGeneralizedCylinder(const std::vector<CPoint3D> &a,const std::vector<CPoint3D> &g):axis(a),generatrix(g),mesh(),meshUpToDate(false),closed(false)	{}
		virtual ~CGeneralizedCylinder() {};
	};
}
}
#endif
