/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef CMyGLCanvas_H
#define CMyGLCanvas_H

#include <mrpt/opengl.h>

#if MRPT_HAS_WXWIDGETS

#include <wx/string.h>
#include <wx/intl.h>
#include <wx/bitmap.h>
#include <wx/icon.h>
#include <wx/image.h>
#include <wx/artprov.h>

#include <wx/msgdlg.h>
#include <wx/filedlg.h>
#include <wx/progdlg.h>
#include <wx/imaglist.h>
#include <wx/busyinfo.h>
#include <wx/log.h>
#include <wx/textdlg.h>
#include <wx/dirdlg.h>
#include <wx/colordlg.h>
#include <wx/dcmemory.h>

#if wxUSE_GLCANVAS && MRPT_HAS_OPENGL_GLUT

#include <wx/glcanvas.h>
#include <wx/dcclient.h>

namespace mrpt
{
	namespace gui
	{
		/** This class implements a OpenGL canvas, and it's used in gui::CDisplayWindow3D and a number of standalone applications in the MRPT project.
		  *  In MRPT 0.6.2 a filter has been added to control the user capability of moving the camera with the mouse. See OnUserManuallyMovesCamera
		  */
		class MRPTDLLIMPEXP CMyGLCanvasBase: public wxGLCanvas
		{
		public:
			CMyGLCanvasBase( wxWindow *parent, wxWindowID id = wxID_ANY,
						 const wxPoint& pos = wxDefaultPosition,
						 const wxSize& size = wxDefaultSize,
						 long style = 0, const wxString& name = _T("CMyGLCanvasBase") );

			virtual ~CMyGLCanvasBase();

			void OnPaint(wxPaintEvent& event);
			void OnSize(wxSizeEvent& event);
			void OnEraseBackground(wxEraseEvent& event);
			void OnKeyDown(wxKeyEvent& event);
			void OnKeyUp(wxKeyEvent& event);
			void OnEnterWindow(wxMouseEvent& event);

			void OnLeftDown(wxMouseEvent& event);
			void OnMouseMove(wxMouseEvent& event);
			void OnRightDown(wxMouseEvent& event);
			void OnRightUp(wxMouseEvent& event);
			void OnLeftUp(wxMouseEvent& event);
			void OnMouseWheel(wxMouseEvent& event);

			void Render();
			void InitGL();

			// Visualization params:
			float	cameraPointingX,cameraPointingY,cameraPointingZ;
			float	cameraZoomDistance;
			float	cameraElevationDeg,cameraAzimuthDeg;
			bool	cameraIsProjective;

			float	clearColorR,clearColorG,clearColorB;

			static float  SENSIBILITY_DEG_PER_PIXEL;		// Default = 0.1

			/**  Methods that can be implemented in custom derived classes  */
			virtual void OnKeyDownCustom( wxKeyEvent& event ) {  }
			virtual void OnKeyUpCustom( wxKeyEvent& event ) {  }

			virtual void OnPreRender() { }
			virtual void OnPostRender()  { }
			virtual void OnPostRenderSwapBuffers(double At, wxPaintDC &dc) { }
			virtual void OnRenderError( const wxString &str ) { }

			/** Overload this method to limit the capabilities of the user to move the camera using the mouse.
			  *  For all these variables:
			  *  - cameraPointingX
			  *  - cameraPointingY
			  *  - cameraPointingZ
			  *  - cameraZoomDistance
			  *  - cameraElevationDeg
			  *  - cameraAzimuthDeg
			  *
			  *  A "new_NAME" variable will be passed with the temptative new value after the user action.
			  *   The default behavior should be to copy all the new variables to the variables listed above
			  *   but in the middle any find of user-defined filter can be implemented.
			  */
			virtual void OnUserManuallyMovesCamera(
				float	new_cameraPointingX,
				float 	new_cameraPointingY,
				float 	new_cameraPointingZ,
				float	new_cameraZoomDistance,
				float	new_cameraElevationDeg,
				float	new_cameraAzimuthDeg )
			{
				cameraPointingX 	= new_cameraPointingX;
				cameraPointingY 	= new_cameraPointingY;
				cameraPointingZ 	= new_cameraPointingZ;
				cameraZoomDistance 	= new_cameraZoomDistance;
				cameraElevationDeg 	= new_cameraElevationDeg ;
				cameraAzimuthDeg 	= new_cameraAzimuthDeg;
			}

			/**  At constructor an empty scene is created. The object is freed at GL canvas destructor.
			  */
			opengl::COpenGLScenePtr		m_openGLScene;

		protected:
			bool   m_init;

			int 	mouseClickX,mouseClickY;
			bool 	mouseClicked;

			long           m_Key;
			unsigned long  m_StartTime;
			unsigned long  m_LastTime;
			unsigned long  m_LastRedraw;

			/** Draws a text string on the screen.
			  *   - Coordinates (x,y) are 2D pixels, starting at bottom-left of the viewport.
			  *   - The text color is defined by (color_r,color_g,color_b), each float numbers in the range [0,1].
			  *   - The text font is:
			  *        - font=0 -> GLUT_BITMAP_TIMES_ROMAN_10
			  *        - font=1 -> GLUT_BITMAP_TIMES_ROMAN_24
			  *        - font=2 -> GLUT_BITMAP_HELVETICA_10
			  *        - font=3 -> GLUT_BITMAP_HELVETICA_12
			  *        - font=4 -> GLUT_BITMAP_HELVETICA_18
			  */
			void renderTextBitmap(
				int screen_x,
				int screen_y,
				const char *str,
				float  color_r=1,
				float  color_g=1,
				float  color_b=1,
				int    font = 1
				);

			DECLARE_EVENT_TABLE()

		};  // end of class

	}	// end namespace
}	// end namespace

#endif		// wxUSE_GLCANVAS
#endif 		// MRPT_HAS_WXWIDGETS
#endif 		// CMyGLCanvas_H

