/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

/*---------------------------------------------------------------
    APPLICATION: Camera calibration GUI
    AUTHOR: Jose Luis Blanco, based on code from the OpenCV library.
  ---------------------------------------------------------------*/


#include "camera_calib_guiMain.h"

//(*InternalHeaders(camera_calib_guiDialog)
#include <wx/font.h>
#include <wx/intl.h>
#include <wx/string.h>
//*)

#include <wx/filedlg.h>
#include <wx/msgdlg.h>
#include <wx/progdlg.h>


#include <mrpt/core.h>

#include <mrpt/gui/WxUtils.h>

using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::vision;
using namespace mrpt::gui;
using namespace std;

#include "../wx-common/CMyRedirector.h"

#include "CAboutBox.h"

#include "imgs/icono_main.xpm"


#define USE_EXTERNAL_STORAGE_IMGS 1

// VARIABLES  ================================

TCalibrationImageList  lst_images;	// Here are all the images: file_name -> data

CMatrixDouble			intrinsicParams(1,1);
std::vector<double>		distortionParams;

// END VARIABLES  ============================



//(*IdInit(camera_calib_guiDialog)
const long camera_calib_guiDialog::ID_BUTTON1 = wxNewId();
const long camera_calib_guiDialog::ID_BUTTON2 = wxNewId();
const long camera_calib_guiDialog::ID_LISTBOX1 = wxNewId();
const long camera_calib_guiDialog::ID_STATICTEXT1 = wxNewId();
const long camera_calib_guiDialog::ID_SPINCTRL1 = wxNewId();
const long camera_calib_guiDialog::ID_STATICTEXT2 = wxNewId();
const long camera_calib_guiDialog::ID_SPINCTRL2 = wxNewId();
const long camera_calib_guiDialog::ID_STATICTEXT3 = wxNewId();
const long camera_calib_guiDialog::ID_TEXTCTRL1 = wxNewId();
const long camera_calib_guiDialog::ID_STATICTEXT4 = wxNewId();
const long camera_calib_guiDialog::ID_TEXTCTRL3 = wxNewId();
const long camera_calib_guiDialog::ID_CHECKBOX1 = wxNewId();
const long camera_calib_guiDialog::ID_TEXTCTRL2 = wxNewId();
const long camera_calib_guiDialog::ID_BUTTON3 = wxNewId();
const long camera_calib_guiDialog::ID_BUTTON6 = wxNewId();
const long camera_calib_guiDialog::ID_BUTTON7 = wxNewId();
const long camera_calib_guiDialog::ID_BUTTON5 = wxNewId();
const long camera_calib_guiDialog::ID_BUTTON4 = wxNewId();
const long camera_calib_guiDialog::ID_STATICBITMAP2 = wxNewId();
const long camera_calib_guiDialog::ID_SCROLLEDWINDOW2 = wxNewId();
const long camera_calib_guiDialog::ID_STATICBITMAP3 = wxNewId();
const long camera_calib_guiDialog::ID_SCROLLEDWINDOW3 = wxNewId();
const long camera_calib_guiDialog::ID_NOTEBOOK1 = wxNewId();
//*)

BEGIN_EVENT_TABLE(camera_calib_guiDialog,wxDialog)
    //(*EventTable(camera_calib_guiDialog)
    //*)
END_EVENT_TABLE()

camera_calib_guiDialog::camera_calib_guiDialog(wxWindow* parent,wxWindowID id)
{
    //(*Initialize(camera_calib_guiDialog)
    wxStaticBoxSizer* StaticBoxSizer2;
    wxFlexGridSizer* FlexGridSizer4;
    wxFlexGridSizer* FlexGridSizer10;
    wxFlexGridSizer* FlexGridSizer3;
    wxFlexGridSizer* FlexGridSizer5;
    wxFlexGridSizer* FlexGridSizer9;
    wxFlexGridSizer* FlexGridSizer2;
    wxFlexGridSizer* FlexGridSizer7;
    wxStaticBoxSizer* StaticBoxSizer3;
    wxFlexGridSizer* FlexGridSizer8;
    wxFlexGridSizer* FlexGridSizer6;
    wxStaticBoxSizer* StaticBoxSizer1;
    wxFlexGridSizer* FlexGridSizer1;
    wxFlexGridSizer* FlexGridSizer11;
    
    Create(parent, id, _("Camera calibration GUI - Part of the MRPT project"), wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE|wxRESIZE_BORDER|wxMAXIMIZE_BOX, _T("id"));
    FlexGridSizer1 = new wxFlexGridSizer(1, 2, 0, 0);
    FlexGridSizer1->AddGrowableCol(1);
    FlexGridSizer1->AddGrowableRow(0);
    FlexGridSizer2 = new wxFlexGridSizer(2, 1, 0, 0);
    FlexGridSizer2->AddGrowableCol(0);
    FlexGridSizer2->AddGrowableRow(0);
    FlexGridSizer2->AddGrowableRow(2);
    StaticBoxSizer1 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("List of images"));
    FlexGridSizer4 = new wxFlexGridSizer(2, 1, 0, 0);
    FlexGridSizer4->AddGrowableCol(0);
    FlexGridSizer4->AddGrowableRow(1);
    FlexGridSizer5 = new wxFlexGridSizer(0, 3, 0, 0);
    FlexGridSizer5->AddGrowableCol(0);
    Button1 = new wxButton(this, ID_BUTTON1, _("Add image..."), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON1"));
    FlexGridSizer5->Add(Button1, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    Button2 = new wxButton(this, ID_BUTTON2, _("Clear all"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON2"));
    FlexGridSizer5->Add(Button2, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer4->Add(FlexGridSizer5, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    lbFiles = new wxListBox(this, ID_LISTBOX1, wxDefaultPosition, wxSize(259,102), 0, 0, wxLB_ALWAYS_SB|wxVSCROLL|wxHSCROLL|wxALWAYS_SHOW_SB, wxDefaultValidator, _T("ID_LISTBOX1"));
    FlexGridSizer4->Add(lbFiles, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticBoxSizer1->Add(FlexGridSizer4, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer2->Add(StaticBoxSizer1, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticBoxSizer3 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Checkerboard detection parameters"));
    FlexGridSizer6 = new wxFlexGridSizer(0, 2, 0, 0);
    FlexGridSizer6->AddGrowableCol(1);
    StaticText1 = new wxStaticText(this, ID_STATICTEXT1, _("Size in X (squares):"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT1"));
    FlexGridSizer6->Add(StaticText1, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    edSizeX = new wxSpinCtrl(this, ID_SPINCTRL1, _T("5"), wxDefaultPosition, wxDefaultSize, 0, 1, 200, 5, _T("ID_SPINCTRL1"));
    edSizeX->SetValue(_T("5"));
    FlexGridSizer6->Add(edSizeX, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticText2 = new wxStaticText(this, ID_STATICTEXT2, _("Size in Y (squares):"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT2"));
    FlexGridSizer6->Add(StaticText2, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    edSizeY = new wxSpinCtrl(this, ID_SPINCTRL2, _T("8"), wxDefaultPosition, wxDefaultSize, 0, 1, 200, 8, _T("ID_SPINCTRL2"));
    edSizeY->SetValue(_T("8"));
    FlexGridSizer6->Add(edSizeY, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticText3 = new wxStaticText(this, ID_STATICTEXT3, _("Square length in X (mm):"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT3"));
    FlexGridSizer6->Add(StaticText3, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    edLengthX = new wxTextCtrl(this, ID_TEXTCTRL1, _("40.0"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_TEXTCTRL1"));
    FlexGridSizer6->Add(edLengthX, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticText4 = new wxStaticText(this, ID_STATICTEXT4, _("Square length in Y (mm):"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT4"));
    FlexGridSizer6->Add(StaticText4, 1, wxALL|wxEXPAND|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    edLengthY = new wxTextCtrl(this, ID_TEXTCTRL3, _("40.0"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_TEXTCTRL3"));
    FlexGridSizer6->Add(edLengthY, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer6->Add(-1,-1,1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    cbNormalize = new wxCheckBox(this, ID_CHECKBOX1, _("Normalize image"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_CHECKBOX1"));
    cbNormalize->SetValue(true);
    FlexGridSizer6->Add(cbNormalize, 1, wxALL|wxEXPAND|wxALIGN_LEFT|wxALIGN_CENTER_VERTICAL, 5);
    StaticBoxSizer3->Add(FlexGridSizer6, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer2->Add(StaticBoxSizer3, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    StaticBoxSizer2 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Calibration"));
    FlexGridSizer7 = new wxFlexGridSizer(0, 1, 0, 0);
    FlexGridSizer7->AddGrowableCol(0);
    FlexGridSizer7->AddGrowableRow(1);
    FlexGridSizer9 = new wxFlexGridSizer(1, 1, 0, 0);
    FlexGridSizer9->AddGrowableCol(0);
    FlexGridSizer9->AddGrowableRow(0);
    txtLog = new wxTextCtrl(this, ID_TEXTCTRL2, _("(Calibration results)"), wxDefaultPosition, wxSize(259,119), wxTE_MULTILINE|wxTE_READONLY|wxHSCROLL|wxVSCROLL, wxDefaultValidator, _T("ID_TEXTCTRL2"));
    txtLog->SetMinSize(wxSize(300,300));
    FlexGridSizer9->Add(txtLog, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer7->Add(FlexGridSizer9, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer8 = new wxFlexGridSizer(0, 3, 0, 0);
    FlexGridSizer8->AddGrowableCol(0);
    FlexGridSizer8->AddGrowableCol(1);
    btnRunCalib = new wxButton(this, ID_BUTTON3, _("Calibrate"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON3"));
    btnRunCalib->SetDefault();
    wxFont btnRunCalibFont(wxDEFAULT,wxDEFAULT,wxFONTSTYLE_NORMAL,wxBOLD,false,wxEmptyString,wxFONTENCODING_DEFAULT);
    btnRunCalib->SetFont(btnRunCalibFont);
    FlexGridSizer8->Add(btnRunCalib, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnSave = new wxButton(this, ID_BUTTON6, _("Save matrices..."), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON6"));
    FlexGridSizer8->Add(btnSave, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer8->Add(-1,-1,1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnManualRect = new wxButton(this, ID_BUTTON7, _("Manual params..."), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON7"));
    FlexGridSizer8->Add(btnManualRect, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnAbout = new wxButton(this, ID_BUTTON5, _("About"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON5"));
    FlexGridSizer8->Add(btnAbout, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    btnClose = new wxButton(this, ID_BUTTON4, _("Close"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON4"));
    FlexGridSizer8->Add(btnClose, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer7->Add(FlexGridSizer8, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    StaticBoxSizer2->Add(FlexGridSizer7, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer2->Add(StaticBoxSizer2, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer1->Add(FlexGridSizer2, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    FlexGridSizer3 = new wxFlexGridSizer(1, 1, 0, 0);
    FlexGridSizer3->AddGrowableCol(0);
    FlexGridSizer3->AddGrowableRow(0);
    Notebook1 = new wxNotebook(this, ID_NOTEBOOK1, wxDefaultPosition, wxSize(719,570), 0, _T("ID_NOTEBOOK1"));
    ScrolledWindow2 = new wxScrolledWindow(Notebook1, ID_SCROLLEDWINDOW2, wxDefaultPosition, wxDefaultSize, wxHSCROLL|wxVSCROLL|wxALWAYS_SHOW_SB, _T("ID_SCROLLEDWINDOW2"));
    FlexGridSizer11 = new wxFlexGridSizer(0, 3, 0, 0);
    bmpOriginal = new wxStaticBitmap(ScrolledWindow2, ID_STATICBITMAP2, wxNullBitmap, wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICBITMAP2"));
    FlexGridSizer11->Add(bmpOriginal, 1, wxALL|wxALIGN_LEFT|wxALIGN_TOP, 5);
    ScrolledWindow2->SetSizer(FlexGridSizer11);
    FlexGridSizer11->Fit(ScrolledWindow2);
    FlexGridSizer11->SetSizeHints(ScrolledWindow2);
    ScrolledWindow3 = new wxScrolledWindow(Notebook1, ID_SCROLLEDWINDOW3, wxDefaultPosition, wxDefaultSize, wxHSCROLL|wxVSCROLL|wxALWAYS_SHOW_SB, _T("ID_SCROLLEDWINDOW3"));
    FlexGridSizer10 = new wxFlexGridSizer(0, 3, 0, 0);
    bmpRectified = new wxStaticBitmap(ScrolledWindow3, ID_STATICBITMAP3, wxNullBitmap, wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICBITMAP3"));
    FlexGridSizer10->Add(bmpRectified, 1, wxALL|wxALIGN_LEFT|wxALIGN_TOP, 5);
    ScrolledWindow3->SetSizer(FlexGridSizer10);
    FlexGridSizer10->Fit(ScrolledWindow3);
    FlexGridSizer10->SetSizeHints(ScrolledWindow3);
    Notebook1->AddPage(ScrolledWindow2, _("Original image"), false);
    Notebook1->AddPage(ScrolledWindow3, _("Rectified image and reprojected points"), true);
    FlexGridSizer3->Add(Notebook1, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 2);
    FlexGridSizer1->Add(FlexGridSizer3, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
    SetSizer(FlexGridSizer1);
    FlexGridSizer1->Fit(this);
    FlexGridSizer1->SetSizeHints(this);
    
    Connect(ID_BUTTON1,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&camera_calib_guiDialog::OnAddImage);
    Connect(ID_BUTTON2,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&camera_calib_guiDialog::OnListClear);
    Connect(ID_LISTBOX1,wxEVT_COMMAND_LISTBOX_SELECTED,(wxObjectEventFunction)&camera_calib_guiDialog::OnlbFilesSelect);
    Connect(ID_BUTTON3,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&camera_calib_guiDialog::OnbtnRunCalibClick);
    Connect(ID_BUTTON6,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&camera_calib_guiDialog::OnbtnSaveClick);
    Connect(ID_BUTTON7,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&camera_calib_guiDialog::OnbtnManualRectClick);
    Connect(ID_BUTTON5,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&camera_calib_guiDialog::OnbtnAboutClick);
    Connect(ID_BUTTON4,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&camera_calib_guiDialog::OnbtnCloseClick);
    //*)

	
	wxIcon icon;
	icon.CopyFromBitmap( wxBitmap(wxImage( icono_main_xpm )) );
	this->SetIcon( icon );

	Center();
	this->SetTitle( _U( format("Camera calibration GUI %s - Part of the MRPT project",CAMERA_CALIB_GUI_VERSION).c_str() ) );
}

camera_calib_guiDialog::~camera_calib_guiDialog()
{
    //(*Destroy(camera_calib_guiDialog)
    //*)
	this->clearListImages();
}


// Ask the user for new files to add to the list:
void camera_calib_guiDialog::OnAddImage(wxCommandEvent& event)
{
	wxFileDialog 	dlg(
		this,
		_("Select image(s) to open"),
		_("."),
		_(""),
		_("Image files (*.bmp;*.png;*.jpg)|*.bmp;*.png;*.jpg|All files (*.*)|*.*"),
		wxFD_OPEN | wxFD_FILE_MUST_EXIST | wxFD_MULTIPLE | wxFD_PREVIEW );

	if (wxID_OK!=dlg.ShowModal()) return;

	wxBusyCursor waitcur;

	wxArrayString files;
	dlg.GetPaths(files);

	wxProgressDialog    progDia(
		wxT("Adding image files"),
		wxT("Processing..."),
		files.Count() , // range
		this, // parent
		wxPD_CAN_ABORT |
		wxPD_APP_MODAL |
		wxPD_SMOOTH |
		wxPD_AUTO_HIDE |
		wxPD_ELAPSED_TIME |
		wxPD_ESTIMATED_TIME |
		wxPD_REMAINING_TIME);

	wxTheApp->Yield();  // Let the app. process messages


	int counter_loops=0;

	for (unsigned int i=0;i<files.Count();i++)
	{
		if (counter_loops++ % 5 == 0)
		{
			if (!progDia.Update( i ))
				break;
			wxTheApp->Yield();  // Let the app. process messages
		}

		const string fil = string(files[i].mb_str());

		TImageCalibData	dat;

#if USE_EXTERNAL_STORAGE_IMGS
		// Optimization: Use external storage:
		// ---------------------------------------
		//string tmp_check     = mrpt::system::getTempFileName()+".jpg";
		//string tmp_rectified = mrpt::system::getTempFileName()+".jpg";
		//dat.img_original.saveToFile(tmp_check);
		//dat.img_original.saveToFile(tmp_rectified);

		// mark all imgs. as external storage:
		dat.img_original.setExternalStorage(fil);
		//dat.img_checkboard.setExternalStorage(tmp_check);
		//dat.img_rectified.setExternalStorage(tmp_rectified);

		dat.img_original.unload();
		//dat.img_checkboard.unload();
		//dat.img_rectified.unload();
#else
		// All in memory:
		if (!dat.img_original.loadFromFile(fil))
		{
			wxMessageBox(_U(format("Error loading file: %s",fil.c_str()).c_str()), _("Error"));
			this->updateListOfImages();
			return;
		}

#endif

		lst_images[fil] = dat;
	}

	this->updateListOfImages();
}


void camera_calib_guiDialog::clearListImages()
{
/*	string s;
	for (TCalibrationImageList::iterator i=lst_images.begin();i!=lst_images.end();i++)
	{
		if ( i->second.img_checkboard.isExternallyStored() )
		{
			i->second.img_checkboard.getExternalStorageFileAbsolutePath(s);
			mrpt::system::deleteFile(s);
		}
		if ( i->second.img_rectified.isExternallyStored() )
		{
			i->second.img_rectified.getExternalStorageFileAbsolutePath(s);
			mrpt::system::deleteFile(s);
		}
	}
*/

	lst_images.clear();
}

// Clear list of images.
void camera_calib_guiDialog::OnListClear(wxCommandEvent& event)
{
	this->clearListImages();
	this->updateListOfImages();
}

void camera_calib_guiDialog::OnbtnRunCalibClick(wxCommandEvent& event)
{
	txtLog->Clear();

	CMyRedirector	redire(txtLog, true, 10);

	const unsigned int  check_size_x = edSizeX->GetValue();
	const unsigned int  check_size_y = edSizeY->GetValue();
	const double        check_squares_length_X_meters = 0.001 * atof( string(edLengthX->GetValue().mb_str()).c_str() );
	const double        check_squares_length_Y_meters = 0.001 * atof( string(edLengthY->GetValue().mb_str()).c_str() );

	const bool			normalize_image = cbNormalize->GetValue();

	wxBusyCursor waitcur;

	bool res = mrpt::vision::checkerBoardCameraCalibration(
		lst_images,
		check_size_x,
		check_size_y,
		check_squares_length_X_meters,
		check_squares_length_Y_meters,
		intrinsicParams,
		distortionParams,
		normalize_image );


	refreshDisplayedImage();

	if (!res)
		wxMessageBox(_("Calibration finished with error: Please check the text log to see what's wrong"), _("Error"));

//	if (res)
//		show3Dview();
}

void camera_calib_guiDialog::OnbtnCloseClick(wxCommandEvent& event)
{
	Close();
}

void camera_calib_guiDialog::OnbtnAboutClick(wxCommandEvent& event)
{
	CAboutBox	dlg(this);
	dlg.ShowModal();
}

// save matrices:
void camera_calib_guiDialog::OnbtnSaveClick(wxCommandEvent& event)
{
	if (size(intrinsicParams,1)!=3)
	{
		wxMessageBox(_("Run the calibration first"),_("Error"));
		return;
	}

	{
		wxFileDialog 	dlg(
			this,
			_("Save intrinsic parameters matrix"),
			_("."),
			_("intrinsic_matrix.txt"),
			_("Text files (*.txt)|*.txt|All files (*.*)|*.*"),
			wxFD_SAVE | wxFD_OVERWRITE_PROMPT );

		if (wxID_OK!=dlg.ShowModal()) return;

		intrinsicParams.saveToTextFile(	string(dlg.GetPath().mb_str()) );
	}

	{
		wxFileDialog 	dlg(
			this,
			_("Save distortion parameters"),
			_("."),
			_("distortion_matrix.txt"),
			_("Text files (*.txt)|*.txt|All files (*.*)|*.*"),
			wxFD_SAVE | wxFD_OVERWRITE_PROMPT );

		if (wxID_OK!=dlg.ShowModal()) return;

		CMatrixDouble  M(1,4);
		for (unsigned i=0;i<4;i++)
			M(0,i) = distortionParams[i];

		M.saveToTextFile( string(dlg.GetPath().mb_str()) );
	}
}

// Update the listbox from lst_img_files
void camera_calib_guiDialog::updateListOfImages()
{
	lbFiles->Clear();
	for (TCalibrationImageList::iterator s=lst_images.begin();s!=lst_images.end();s++)
		lbFiles->Append(_U(s->first.c_str()));

	refreshDisplayedImage();
}


// Shows the image selected in the listbox:
void camera_calib_guiDialog::refreshDisplayedImage()
{
	if (!lbFiles->GetCount())
	{
		// No images:
		return;
	}

	// Assure there's one selected:
	if (lbFiles->GetSelection()==wxNOT_FOUND)
		lbFiles->SetSelection(0);

	const string selFile = string(lbFiles->GetStringSelection().mb_str());

	TCalibrationImageList::iterator it = lst_images.find(selFile);
	if (it==lst_images.end()) return;

	wxImage *img_chk  = NULL;
	wxImage *img_rect = NULL;

	// Ok, the image exists:
	if (!it->second.img_original.isExternallyStored())
	{
		img_chk = mrpt::gui::MRPTImage2wxImage( it->second.img_checkboard );
		img_rect = mrpt::gui::MRPTImage2wxImage( it->second.img_rectified );
	}
	else
	{
		// External storage: generate the images on-the-fly:
		CMRPTImage  imgCheck = it->second.img_original;

		// Draw the board:
		for (unsigned int k=0;k<it->second.detected_corners.size();k++)
		{
			imgCheck.cross(it->second.detected_corners[k].x, it->second.detected_corners[k].y, 0x0000FF, 3 );
			imgCheck.drawCircle( it->second.projectedPoints_distorted[k].x, it->second.projectedPoints_distorted[k].y, 4, 0x00FF40 );
		}
		imgCheck.drawCircle( 10,10, 4, 0x00FF40 );
		imgCheck.textOut(18,4,"Reprojected corners",0xFFFFFF);

		imgCheck.cross( 10,30, 0x0000FF, 3 );
		imgCheck.textOut(18,24,"Detected corners",0xFFFFFF);

		// Rectify:
		CMRPTImage  imgRect;
		if (size(intrinsicParams,1)!=3)
		{
			// Not calibrated yet:
			imgRect = it->second.img_original;
		}
		else
		{
			it->second.img_original.rectifyImage(imgRect,intrinsicParams, distortionParams);

			// Draw reprojected:
			for (unsigned int k=0;k<it->second.projectedPoints_undistorted.size();k++)
				imgRect.drawCircle( it->second.projectedPoints_undistorted[k].x, it->second.projectedPoints_undistorted[k].y, 4, 0x00FF40 );

			imgRect.drawCircle( 10,10, 4, 0x00FF40 );
			imgRect.textOut(18,4,"Reprojected corners",0xFFFFFF);
		}

		img_chk = mrpt::gui::MRPTImage2wxImage( imgCheck );
		img_rect = mrpt::gui::MRPTImage2wxImage( imgRect );

		it->second.img_original.unload();
	}

	if (img_chk)
	{
		this->bmpOriginal->SetBitmap( wxBitmap(*img_chk) );
		delete img_chk;
	}
	if (img_rect)
	{
		this->bmpRectified->SetBitmap( wxBitmap(*img_rect) );
		delete img_rect;
	}
	//	this->ScrolledWindow2->FitInside();
	//	this->ScrolledWindow3->FitInside();
}

void camera_calib_guiDialog::OnlbFilesSelect(wxCommandEvent& event)
{
	refreshDisplayedImage();
}

void camera_calib_guiDialog::show3Dview()
{
	if (size(intrinsicParams,1)!=3)
	{
		wxMessageBox(_("Run the calibration first"),_("Error"));
		return;
	}

	win3D.clear();
	mrpt::system::sleep(50);

	win3D = CDisplayWindow3DPtr( new CDisplayWindow3D("Calibration 3D view") );


	mrpt::opengl::COpenGLScenePtr	scene = win3D->get3DSceneAndLock();

	const unsigned int  check_size_x = edSizeX->GetValue();
	const unsigned int  check_size_y = edSizeY->GetValue();
	const double        check_squares_length_X_meters = 0.001 * atof( string(edLengthX->GetValue().mb_str()).c_str() );
	const double        check_squares_length_Y_meters = 0.001 * atof( string(edLengthY->GetValue().mb_str()).c_str() );

	if (!check_squares_length_X_meters || !check_squares_length_Y_meters) return;

	opengl::CGridPlaneXYPtr	grid = opengl::CGridPlaneXY::Create(0,check_size_x*check_squares_length_X_meters, 0, check_size_y*check_squares_length_Y_meters, 0, check_squares_length_X_meters );
	scene->insert( grid );

	for (TCalibrationImageList::iterator it=lst_images.begin();it!=lst_images.end();it++)
	{
		mrpt::opengl::CSetOfObjectsPtr	cor = mrpt::opengl::stock_objects::CornerXYZ();
		cor->setName( mrpt::system::extractFileName(it->first) );
		cor->enableShowName(true);
		cor->setPose( it->second.reconstructed_camera_pose );

		scene->insert( cor );
	}

	scene->insert( mrpt::opengl::stock_objects::CornerXYZ() );

	win3D->unlockAccess3DScene();
	win3D->forceRepaint();
}

// Enter calib. params manually:
void camera_calib_guiDialog::OnbtnManualRectClick(wxCommandEvent& event)
{
	wxMessageBox(_("Please, enter calibration parameters manually next to overpass automatically obtained parameters."),_("Manual parameters"));

	wxString s;

	if (size(intrinsicParams,1)!=3)
	{
		wxMessageBox(_("Run the calibration first"),_("Error"));
		return;
	}

	distortionParams.resize(4);

	s = wxGetTextFromUser(_("Focus length in X pixel size (fx):"),_("Manual parameters"),wxString::Format(wxT("%.07f"),intrinsicParams(0,0)), this );
	if (s.IsEmpty()) return;
	if (!s.ToDouble(&intrinsicParams(0,0))) { wxMessageBox(_("Invalid number")); return; }

	s = wxGetTextFromUser(_("Focus length in Y pixel size (fy):"),_("Manual parameters"),wxString::Format(wxT("%.07f"),intrinsicParams(1,1)), this );
	if (s.IsEmpty()) return;
	if (!s.ToDouble(&intrinsicParams(1,1))) { wxMessageBox(_("Invalid number")); return; }

	s = wxGetTextFromUser(_("Image center X (cx):"),_("Manual parameters"),wxString::Format(wxT("%.07f"),intrinsicParams(0,2)), this );
	if (s.IsEmpty()) return;
	if (!s.ToDouble(&intrinsicParams(0,2))) { wxMessageBox(_("Invalid number")); return; }

	s = wxGetTextFromUser(_("Image center Y (cy):"),_("Manual parameters"),wxString::Format(wxT("%.07f"),intrinsicParams(1,2)), this );
	if (s.IsEmpty()) return;
	if (!s.ToDouble(&intrinsicParams(1,2))) { wxMessageBox(_("Invalid number")); return; }



	s = wxGetTextFromUser(_("Distortion param p1:"),_("Manual parameters"),wxString::Format(wxT("%.07f"),distortionParams[0]), this );
	if (s.IsEmpty()) return;
	if (!s.ToDouble(&distortionParams[0])) { wxMessageBox(_("Invalid number")); return; }

	s = wxGetTextFromUser(_("Distortion param p2:"),_("Manual parameters"),wxString::Format(wxT("%.07f"),distortionParams[1]), this );
	if (s.IsEmpty()) return;
	if (!s.ToDouble(&distortionParams[1])) { wxMessageBox(_("Invalid number")); return; }

	s = wxGetTextFromUser(_("Distortion param k1:"),_("Manual parameters"),wxString::Format(wxT("%.07f"),distortionParams[2]), this );
	if (s.IsEmpty()) return;
	if (!s.ToDouble(&distortionParams[2])) { wxMessageBox(_("Invalid number")); return; }

	s = wxGetTextFromUser(_("Distortion param k2:"),_("Manual parameters"),wxString::Format(wxT("%.07f"),distortionParams[3]), this );
	if (s.IsEmpty()) return;
	if (!s.ToDouble(&distortionParams[3])) { wxMessageBox(_("Invalid number")); return; }


	refreshDisplayedImage();
}
