/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */


#include <mrpt/utils/utils_defs.h>
#include <mrpt/system/os.h>

#include <mrpt/hwdrivers/CSickLaserUSB.h>


IMPLEMENTS_GENERIC_SENSOR(CSickLaserUSB,mrpt::hwdrivers)

using namespace std;
using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::hwdrivers;


int CSickLaserUSB::CRC16_GEN_POL = 0x8005;

/*-------------------------------------------------------------
						CSickLaserUSB
-------------------------------------------------------------*/
CSickLaserUSB::CSickLaserUSB() :
	m_usbConnection( NULL ),
	m_serialNumber ( "LASER001" ),
	m_sensorLabel  ( "SICKLMS"  ),
	m_timeStartUI  ( 0 )
{
	MRPT_TRY_START;
	m_usbConnection = new CInterfaceFTDI();
	MRPT_TRY_END;
}

/*-------------------------------------------------------------
						~CSickLaserUSB
-------------------------------------------------------------*/
CSickLaserUSB::~CSickLaserUSB()
{
	MRPT_TRY_START;
	delete m_usbConnection;
	m_usbConnection = NULL;
	MRPT_TRY_END;
}

/*-------------------------------------------------------------
						doProcessSimple
-------------------------------------------------------------*/
void  CSickLaserUSB::doProcessSimple(
	bool							&outThereIsObservation,
	mrpt::slam::CObservation2DRangeScan	&outObservation,
	bool							&hardwareError )
{
	outThereIsObservation	= false;
	hardwareError			= false;

	if ( !checkControllerIsConnected() )
	{
		hardwareError = true;
		return;
	}

	vector_float 	ranges;
	unsigned char	LMS_stat;
	uint32_t		board_timestamp;
	bool			is_mm_mode;

    m_state = ssWorking;

	// Wait for a scan:
	if (!waitContinuousSampleFrame( ranges,LMS_stat, board_timestamp, is_mm_mode ))
		return;

	// Yes, we have a new scan:

	// -----------------------------------------------
	//   Extract the observation:
	// -----------------------------------------------
	uint32_t AtUI = 0;
	if( m_timeStartUI == 0 )
	{
		m_timeStartUI = board_timestamp;
		m_timeStartTT = mrpt::system::now();
	}
	else	AtUI = board_timestamp - m_timeStartUI;

	mrpt::system::TTimeStamp AtDO	=  mrpt::system::secondsToTimestamp( AtUI * 1e-3 /* Board time is ms */ );
	outObservation.timestamp = m_timeStartTT + AtDO;

	outObservation.sensorLabel  = m_sensorLabel;	// Set label

	// Extract the timestamp of the sensor:

	// And the scan ranges:
	outObservation.rightToLeft = true;
	outObservation.aperture = M_PIf;
	outObservation.maxRange	= is_mm_mode ? 32.7 : 81.0;
	outObservation.stdError = 0.003f;
	outObservation.sensorPose = m_sensorPose;

	outObservation.scan = ranges;
	outObservation.validRange.resize(ranges.size());

	for (size_t i=0;i<ranges.size();i++)
		outObservation.validRange[i] = (outObservation.scan[i] <= outObservation.maxRange);

	outThereIsObservation = true;
}

/*-------------------------------------------------------------
						loadConfig
-------------------------------------------------------------*/
void  CSickLaserUSB::loadConfig(
	const mrpt::utils::CConfigFileBase &configSource,
	const std::string	  &iniSection )
{
	m_serialNumber = configSource.read_string(iniSection,"SICKUSB_serialNumber",m_serialNumber);
	m_sensorLabel  = configSource.read_string(iniSection,"sensorLabel",m_sensorLabel);
	m_sensorPose = CPose3D(
		configSource.read_float(iniSection,"pose_x",0),
		configSource.read_float(iniSection,"pose_y",0),
		configSource.read_float(iniSection,"pose_z",0),
		DEG2RAD( configSource.read_float(iniSection,"pose_yaw",0) ),
		DEG2RAD( configSource.read_float(iniSection,"pose_pitch",0) ),
		DEG2RAD( configSource.read_float(iniSection,"pose_roll",0) )
		);
}

/*-------------------------------------------------------------
						turnOn
-------------------------------------------------------------*/
bool  CSickLaserUSB::turnOn()
{
	return true;
}

/*-------------------------------------------------------------
						turnOff
-------------------------------------------------------------*/
bool  CSickLaserUSB::turnOff()
{
	return true;
}

/*-------------------------------------------------------------
					checkControllerIsConnected
-------------------------------------------------------------*/
bool  CSickLaserUSB::checkControllerIsConnected()
{
	// If device is already open, thats ok:
	if (m_usbConnection->isOpen())
		return true;

	// If it isn't, try to open it now:
	try
	{
		m_usbConnection->OpenBySerialNumber( m_serialNumber );
		m_usbConnection->ResetDevice( );
		system::sleep(10);
		m_usbConnection->SetTimeouts( 10, 20 );	// read, write, in milliseconds
		system::sleep(10);
		m_usbConnection->SetLatencyTimer(16);		// 1ms, the minimum
		system::sleep(10);

		printf_debug("[CSickLaserUSB] USB DEVICE S/N:'%s' OPEN SUCCESSFULLY!!!\n",m_serialNumber.c_str() );
		return true;
	}
	catch(std::exception &e)
	{
		printf_debug(e.what());
		printf_debug("[CSickLaserUSB] ERROR TRYING TO OPEN USB DEVICE S/N:'%s'\n",m_serialNumber.c_str() );
		return false;
	}
}

/*-------------------------------------------------------------
					waitContinuousSampleFrame
-------------------------------------------------------------*/
bool  CSickLaserUSB::waitContinuousSampleFrame(
	vector_float 	&out_ranges_meters,
	unsigned char 	&LMS_status,
	uint32_t 		&out_board_timestamp,
    bool 			&is_mm_mode )
{
	size_t 	nRead,nBytesToRead;
	size_t	nFrameBytes = 0;
	size_t	lenghtField;
	unsigned char	buf[2000];
	buf[2]=buf[3]=0;

	while ( nFrameBytes < (lenghtField=( 6 + (buf[2] | (buf[3] << 8))) ) + 5  /* for 32bit timestamp + end-flag */  )
	{
		if (lenghtField>800)
		{
			cout << "#";
			nFrameBytes = 0;	// No es cabecera de trama correcta
			buf[2]=buf[3]=0;
		}

		if (nFrameBytes<4)
			nBytesToRead = 1;
		else
			nBytesToRead = (5 /* for 32bit timestamp + end-flag */ + lenghtField) - nFrameBytes;

		try
		{
			nRead = m_usbConnection->ReadSync( buf+nFrameBytes,nBytesToRead );
		}
		catch (std::exception &e)
		{
			// Disconnected?
			printf_debug("[CSickLaserUSB::waitContinuousSampleFrame] Disconnecting due to comms error: %s\n", e.what());
			m_usbConnection->Close();
			m_timeStartUI = 0;
			return false;
		}

		if ( nRead != nBytesToRead )
		{
			return false;
		}
		else
		if (nRead>0)
		{
			// Lectura OK:
			// Era la primera?
			if (nFrameBytes>1 || (!nFrameBytes && buf[0]==0x02) || (nFrameBytes==1 && buf[1]==0x80))
					nFrameBytes+=nRead;
			else
			{
				nFrameBytes = 0;	// No es cabecera de trama correcta
				buf[2]=buf[3]=0;
			}
		}
	}

	// Frame received
	// --------------------------------------------------------------------------
	// | STX | ADDR | L1 | L2 | COM | INF1 | INF2 |	DATA	| STA | CRC1 | CRC2 |
	// --------------------------------------------------------------------------

	// Trama completa:
	//  Checkear que el byte de comando es 0xB0:
	if ( buf[4]!=0xB0 )	return false;

	// GET FRAME INFO
	int  info	 = buf[5] | (buf[6] << 8);	// Little Endian
	int  n_points = info & 0x01FF;
	is_mm_mode = 0 != ((info & 0xC000) >> 14);	// 0x00: cm 0x01: mm

	out_ranges_meters.resize(n_points);

	// Copiar rangos:
	short mask = is_mm_mode ? 0x7FFF : 0x1FFF;
	float meters_scale = is_mm_mode ? 0.001f : 0.01f;

	for (int i=0;i<n_points;i++)
		out_ranges_meters[i] = ( (buf[7+i*2] | (buf[8+i*2] << 8)) & mask ) * meters_scale;

	// Status
	LMS_status = buf[729];

	// CRC:
	uint16_t CRC = computeCRC(buf,lenghtField-2);
	uint16_t CRC_packet = buf[lenghtField-2] | ( buf[lenghtField-1] << 8);
	if (CRC_packet!=CRC)
	{
		cerr << format("[CSickLaserUSB::waitContinuousSampleFrame] bad CRC len=%u nptns=%u: %i != %i", unsigned(lenghtField),unsigned(n_points), CRC_packet, CRC) << endl;
		return false; // Bad CRC
	}

	// Get USB board timestamp:
	out_board_timestamp =
		(uint32_t(buf[nFrameBytes-5]) << 24) |
		(uint32_t(buf[nFrameBytes-4]) << 16) |
		(uint32_t(buf[nFrameBytes-3]) << 8) |
		 uint32_t(buf[nFrameBytes-2]);

	// All OK?
	return (buf[nFrameBytes-1]==0x55);
}


/*-------------------------------------------------------------
					ComputeCRC
-------------------------------------------------------------*/
uint16_t CSickLaserUSB::computeCRC(unsigned char *data, unsigned long len)
{
  uint16_t uCrc16;
  uint8_t  abData[2];

  uCrc16 = 0;
  abData[0] = 0;

  while(len-- )
  {
    abData[1] = abData[0];
    abData[0] = *data++;

    if( uCrc16 & 0x8000 )
    {
      uCrc16 = (uCrc16 & 0x7fff) << 1;
      uCrc16 ^= CRC16_GEN_POL;
    }
    else
    {
      uCrc16 <<= 1;
    }
    uCrc16 ^= (abData[0] | (abData[1]<<8));
  }
  return (uCrc16);
}
