/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/hwdrivers/CPtuHokuyo.h>
#include <vector>
#include <mrpt/gui.h>

using namespace mrpt;

using namespace mrpt::gui;
using namespace mrpt::opengl;

using namespace mrpt::hwdrivers;
using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;

/*-------------------------------------------------------------
						  Destructor
-------------------------------------------------------------*/

CPtuHokuyo::~CPtuHokuyo(){

	if(!ptu)
		delete ptu;

	vObs.erase( vObs.begin(), vObs.end() );
}


/*-------------------------------------------------------------
						  init
-------------------------------------------------------------*/

bool CPtuHokuyo::init(const string &portPtu,const string &portHokuyo){

	cout << endl << "[INFO] Using serial port for PTU: " << portPtu << endl;
	cout << "[INFO] Using serial port for Hokuyo: " << portHokuyo << endl;

	ptu = new CPtuDPerception();

	// Ptu initialization

	if(ptu->init(portPtu)){
		printf("\n[TEST] Initialization PTU OK!\n");
	}else{
		printf("[TEST] Initialization PTU failed!\n");
		return false;
	}

	// Hokuyo initialization

	laser.m_com_port = portHokuyo;

	printf("\n[TEST] Turning laser ON...\n");
	if (laser.turnOn())
		printf("[TEST] Initialization Hokuyo OK!\n");
	else
	{
		printf("[TEST] Initialization Hokuyo failed!\n");
		return false;
	}

	return true;

}


/*-------------------------------------------------------------
						  scan
-------------------------------------------------------------*/

bool CPtuHokuyo::scan(char &axis, const int &tWait, double &initial, double &final, const double &radPre, const int &n_mean, const bool &interlaced){

	if(initial<final){
		float aux;
		aux=initial;
		initial = final;
		final = aux;
	}

	// Move PTU to initial position
	ptu->moveToAbsPos(axis,initial);
	ptu->aWait();

	// Obtain total number of steps and movements
	long steps = ptu->radToPos(axis,radPre);

	int initialPos = ptu->radToPos(axis,initial);
	int finalPos = ptu->radToPos(axis,final);

	long totalSteps = abs(initialPos-finalPos);

	int movements = totalSteps/steps;

	// Performs first scan
	if(!singleScan(axis,tWait,movements,-radPre,n_mean))
		return false;

	// Check if we have to perform a second scan
	if(interlaced){

		// Adjust position for perform the second scan
		ptu->moveToAbsPos(axis,initial-radPre/2);
		ptu->aWait();
		mrpt::system::sleep(1500);

		movements=(totalSteps/steps)-1;

		// Performs second scan
		if(!singleScan(axis,tWait,movements,-radPre,n_mean))
			return false;
	}

	// Return to initial position
	ptu->moveToAbsPos(axis,0);

	return true;
}


/*-------------------------------------------------------------
						obtainObs
-------------------------------------------------------------*/

bool CPtuHokuyo::obtainObs(CObservation2DRangeScan & obs){

	bool	thereIsObservation,hardError;

	laser.doProcessSimple( thereIsObservation, obs, hardError );

	if (hardError){
		//printf("[TEST] Hardware error=true!!\n");
		return false;
	}

	if (!thereIsObservation){
		return false;
	}

	return true;
}


/*-------------------------------------------------------------
				   	  saveObservation
-------------------------------------------------------------*/

double CPtuHokuyo::saveObservation(const char &axis, const int &n_mean){

	double j=0;
	vector<CObservation2DRangeScan>		vObsAux;
	CObservation2DRangeScan obs;

	// Empty laser buffer for obtain the 10 last observation
	laser.purgeBuffers();

	// Obtain n_mean observations
	int count=0;
	while(count<n_mean){
		while(!obtainObs(obs)){
			mrpt::system::sleep(1);
		}
		vObsAux.push_back(obs);
		count++;
	}

	int length = minLengthVectors(obs,vObsAux);

	for(int i=0; i<length;i++){
		// the final observation value is the mean of the n_mean observations
		obs.scan.at(i)= 0;
		for(int j=0;j<n_mean;j++){
			obs.scan.at(i)+= vObsAux.at(j).scan.at(i);
		}
		obs.scan.at(i)/= n_mean;
		// If any range is invalid appears as invalid in the mean
		for(int j=0;j<n_mean;j++){
			obs.validRange.at(i) = obs.validRange.at(i)||vObsAux.at(j).validRange.at(i);
		}
	}

	// Obtain actual radians
	ptu->absPosQ(axis,j);

	calculateSensorPose(axis,j,obs);

	// Save observation in a vector for later use

	vObs.push_back(obs);

	return -j;

}


/*-------------------------------------------------------------
					thread_saveObs
-------------------------------------------------------------*/

void thread_saveObs(void *param){

	mrpt::hwdrivers::ThreadParams* t_params = (mrpt::hwdrivers::ThreadParams*) param;
	char axis = t_params->axis;
	const double hokuyo_frec = 0.025;

	CObservation2DRangeScan obs;
	//double j=0;

	while(true){
		if(!t_params->start_capture) break;

		// Obtain a valid observation
		while(!t_params->ptu_hokuyo->obtainObs(obs))
		{
			mrpt::system::sleep(1);
		}

		// Variation of delta pitch while ptu is moving and hokuyo is scanning
		if((axis=='T')||(axis=='t'))
			obs.deltaPitch = (t_params->scan_vel * hokuyo_frec);

		// Obtain time stamp
		obs.timestamp = now();

		// Save obs in vObs vector
		t_params->ptu_hokuyo->vObs.push_back(obs);

		mrpt::system::sleep(1);
	}
}

/*-------------------------------------------------------------
						singleScan
-------------------------------------------------------------*/

bool CPtuHokuyo::singleScan(const char &axis, const int &tWait, const int &movements, const double &radPre, const int &n_mean){

	for(int i=0;i<movements;i++)
	{
		// Save observation
		saveObservation(axis,n_mean);

		// Move PTU
		ptu->moveToOffPos(axis,radPre);

		mrpt::system::sleep(tWait);
	}

	// Save the last observation
	saveObservation(axis,n_mean);

	return true;
}




/*-------------------------------------------------------------
					continuousScan
-------------------------------------------------------------*/

bool CPtuHokuyo::continuousScan(char &axis, const double &velocity, double &initial, double &final){

	if(initial<final){
		float aux;
		aux=initial;
		initial = final;
		final = aux;
	}

	const double defSpeed = 0.897598; // Default speed

	// Restart speed
	ptu->speed(axis,defSpeed);
	ptu->aWait();

	// Move PTU to initial position
	ptu->moveToAbsPos(axis,initial);
	ptu->aWait();

	mrpt::system::sleep(2000);

	// Calculate final position in radians
	int initialPos = ptu->radToPos(axis,initial);
	double initialRad = ptu->posToRad(axis,(long) initialPos);
	int finalPos = ptu->radToPos(axis,final);
	double finalRad = ptu->posToRad(axis, (long) finalPos);

	ptu->speed(axis,velocity);
	ptu->aWait();

	// Move PTU to final position
	ptu->moveToAbsPos(axis,final);

	// Struct for save observations thread
	ThreadParams t_params;
	t_params.axis = axis;
	t_params.start_capture = true;
	t_params.scan_vel = velocity;
	t_params.ptu_hokuyo = this;

	// Create save observations thread
	TThreadHandle my_thread = createThread( thread_saveObs, &t_params );

	double last_position = initialRad;
	while(last_position>finalRad){
		// Obtain PTU position
		ptu->absPosQ(axis,last_position);
		// Insert position and time stamp in v_my_pos vector
		my_pos m_pos;
		m_pos.pos = last_position;
		m_pos.timeStamp = now();

		v_my_pos.push_back(m_pos);

		mrpt::system::sleep(1);
	}

	// Stop and destroy thread
	t_params.start_capture = false;
	joinThread(my_thread);

	// Restore default speed
	ptu->speed(axis,defSpeed);
	ptu->aWait();

	// Return to initial position
	ptu->moveToAbsPos(axis,0);

	// Create vObs
	refineVObs(axis);

	return true;

}

/*-------------------------------------------------------------
					     refineVObs
-------------------------------------------------------------*/

void CPtuHokuyo::refineVObs(const char &axis){

	my_pos m_pos = v_my_pos.at(0);
	CObservation2DRangeScan obs = vObs.at(0);
	unsigned int j=0; // v_my_pos and vObsAux index
	vector<CObservation2DRangeScan> vObsAux;

	// Insert the first observation in vObsAux
	calculateSensorPose(axis,m_pos.pos,obs);

	vObsAux.push_back(obs);

	// Obtain time difference between actual position and next
	double time_b = timeDifference(m_pos.timeStamp,v_my_pos.at(j).timeStamp);
	TTimeStamp t_between_movs = secondsToTimestamp (time_b);

	for(unsigned int i=1; i<vObs.size();i++){
		obs = vObs.at(i);
		// If obs time was pre-intermediate between the two positions
		if(obs.timestamp <= (m_pos.timeStamp + t_between_movs/2)){
			// Obtain min length vectors and performs a mean
			int length = minLengthVectors(vObsAux.at(j),obs,1);
			for(int k=0; k < length ;k++){
				vObsAux.at(j).scan.at(k)= (vObsAux.at(j).scan.at(k)+obs.scan.at(k))/2;
			}
			// See if any range is invalid
			length = minLengthVectors(vObsAux.at(j),obs,0);
			for(int k=0;k< length ;k++){
				vObsAux.at(j).validRange.at(k) = vObsAux.at(j).validRange.at(k)|| obs.validRange.at(k);
			}
		}else{
			if(j<v_my_pos.size()-1){
				// Obtain time difference between actual position and next
				time_b = timeDifference(m_pos.timeStamp,v_my_pos.at(j+1).timeStamp);
				t_between_movs = secondsToTimestamp(time_b);

				// Move to next position
				m_pos = v_my_pos.at(j);
				j++;

				// Calculate sensor pose and insert en vObsAux
				calculateSensorPose(axis,m_pos.pos,obs);

				vObsAux.push_back(obs);

			//If is the last position
			}else if(j==v_my_pos.size()-1){

				t_between_movs = secondsToTimestamp(0);
				m_pos = v_my_pos.at(j);
				j++;

				calculateSensorPose(axis,m_pos.pos,obs);

				vObsAux.push_back(obs);
			}
		}
	}

	vObs.erase( vObs.begin(), vObs.end() );
	vObs = vObsAux;

}


/*-------------------------------------------------------------
					saveVObsPoints2File
-------------------------------------------------------------*/

bool CPtuHokuyo::saveVObsPoints2File(char* fname, const bool &colours){

	ofstream file;

	file.open(fname);

	if(!file.is_open()){
		printf("[TEST] Open file failed!\n");
		return false;
	}

	printf("\n[INFO] Saving Map  to file %s\n",fname);

	for(unsigned int i=0;i<vObs.size();i++){

		mrpt::slam::CSimplePointsMap		theMap;
		theMap.insertionOptions.minDistBetweenLaserPoints	= 0;
		theMap.insertObservation( &vObs[i] );

		vector_float	xs,ys,zs;
		theMap.getAllPoints(xs,ys,zs);

		for (unsigned int i=0; i<xs.size(); i++){
			char cadena[300];
			if(colours){
				sprintf(cadena,"%f\t%f\t%f\t1\t1\t1\n",xs[i],ys[i],zs[i]);
			}else{
				sprintf(cadena,"%f\t%f\t%f\n",xs[i],ys[i],zs[i]);
			}
			file << cadena;
		}
	}

	file.close();

	printf("[INFO] Save Map Points to file COMPLETE\n"); //,fname);

	return true;
}


/*-------------------------------------------------------------
						saveMap2File
-------------------------------------------------------------*/

bool CPtuHokuyo::saveMap2File(mrpt::slam::CSimplePointsMap		&theMap, char* fname, const bool &colours){

	ofstream file;

	file.open(fname);

	if(!file.is_open()){
		printf("[TEST] Open file failed!\n");
		return false;
	}

	printf("\n[INFO] Save Map Points to file %s\n",fname);

	vector_float xs,ys,zs;
	theMap.getAllPoints(xs,ys,zs);

	for (unsigned int i=0; i<xs.size(); i++){
		char cadena[300];
		if(colours){
			sprintf(cadena,"%f\t%f\t%f\t1\t1\t1\n",xs[i],ys[i],zs[i]);
		}else{
			sprintf(cadena,"%f\t%f\t%f\n",xs[i],ys[i],zs[i]);
		}
		file << cadena;
	}

	file.close();

	printf("[INFO] Save Map Points to file COMPLETE\n"); //,fname);

	return true;
}


/*-------------------------------------------------------------
						showGraphic
-------------------------------------------------------------*/

bool CPtuHokuyo::showGraphic(mrpt::slam::CSimplePointsMap	*theMap){

	// Use if theMap is null
	mrpt::slam::CSimplePointsMap		theMap2;

	// Scene Objects
	CDisplayWindow3D	win("Ptu&Hokuyo graphic: Points earned",800,600);

	COpenGLScenePtr &theScene = win.get3DSceneAndLock();

	// Modify the scene
	{
		opengl::CGridPlaneXYPtr obj = opengl::CGridPlaneXY::Create(-10,10,-10,10,0,1);
		obj->m_color_R = obj->m_color_G = obj->m_color_B = 0.4f;
		theScene->insert( obj );
	}

	{
		opengl::CAxisPtr obj = opengl::CAxis::Create();
		obj->m_frecuency = 5;
		obj->m_marks     = true;
		obj->m_xmin      = -10;
		obj->m_xmax      =  10;
		obj->m_ymin      = -10;
		obj->m_ymax      =  10;
		obj->m_zmin      = -10;
		obj->m_zmax      =  10;
		theScene->insert( obj );
	}

	// UnLock scene
	win.unlockAccess3DScene();

	win.setCameraElevationDeg( 25.0f );

	if(!theMap){
		theMap2.insertionOptions.minDistBetweenLaserPoints	= 0;
		for(unsigned int j=0;j < vObs.size();j++){
			theMap2.insertObservation( &vObs.at(j) );
		}
	}

	opengl::CPointCloudPtr obj = opengl::CPointCloud::Create();
	obj->m_color_R = 0;
	obj->m_color_G = 0;
	obj->m_color_B = 1;
	if(!theMap){
		obj->loadFromPointsMap(&theMap2);
	}else{
		obj->loadFromPointsMap(theMap);
	}
	obj->m_colorFromZ = true;
	obj->m_name = "Cloud" ;

	// Lock scene to use
	theScene=win.get3DSceneAndLock();

	// Insert points cloud in the scene
	theScene->insert( obj );

	// Unlock scene
	win.unlockAccess3DScene();

	// Update window
	win.forceRepaint();

	cout << endl << "Press any key and enter to end program" << endl;

	int i;
	cin >> i;

	return true;

}

/*-------------------------------------------------------------
						limitScan
-------------------------------------------------------------*/

bool CPtuHokuyo::limitScan(const char &axis, double &low, double &high, mrpt::slam::CSimplePointsMap		&theMap){

	mrpt::slam::CSimplePointsMap		theMapAux;
	theMap.insertionOptions.minDistBetweenLaserPoints	= 0;

	// Check that axis is correct
	if((axis!='x')&&(axis!='y')&&(axis!='z')){
		cout << "Incorrect axis to limit" << endl;
		return false;
	}

	// Check is low and high values
	if(low>high){
		double aux;
		aux = low;
		low = high;
		high = aux;
	}

	// Check if the map alredy exist
	if(theMap.isEmpty()){
		for(unsigned int j=0;j < vObs.size();j++){
			theMapAux.insertObservation( &vObs.at(j) );
		}
	}else{
		theMapAux.copyFrom(theMap);
		theMap.clear();
	}

	// Get points
	vector_float xs, ys, zs;
	theMapAux.getAllPoints(xs,ys,zs);

	for(unsigned int j=0;j<xs.size();j++){
		switch (axis) {
			case 'x':
				if((low<=xs[j])&&(xs[j]<=high)){
					theMap.insertPoint(xs[j],ys[j],zs[j]);
				}
				break;
			case 'y':
				if((low<=ys[j])&&(ys[j]<=high)){
					theMap.insertPoint(xs[j],ys[j],zs[j]);
				}
				break;
			case 'z':
				if((low<=zs[j])&&(zs[j]<=high)){
					theMap.insertPoint(xs[j],ys[j],zs[j]);
				}
				break;
		}
	}

	return true;
}


/*-------------------------------------------------------------
					   saveVObs2File
-------------------------------------------------------------*/

bool CPtuHokuyo::saveVObs2File(char *fname){

	CFileOutputStream file;

	// Open data file
	file.open(fname);

	if(!file.fileOpenCorrectly()){
		printf("[TEST] Open file failed!\n");
		return false;
	}

	for(unsigned int i=0;i<vObs.size();i++){
		file << vObs.at(i);
	}
	return true;
}


/*-------------------------------------------------------------
					  minLengthVectors
-------------------------------------------------------------*/

int CPtuHokuyo::minLengthVectors(CObservation2DRangeScan &obs, vector<CObservation2DRangeScan> &vObsAux){

	unsigned int min= obs.scan.size();

	for(unsigned int i=0;i<vObsAux.size();i++){
		if(vObsAux.at(i).scan.size() < min)
			min = vObsAux.at(i).scan.size();
	}

	return  min;

}


/*-------------------------------------------------------------
					loadObs2PointsMap
-------------------------------------------------------------*/

void CPtuHokuyo::loadObs2PointsMap(mrpt::slam::CSimplePointsMap	&theMap){

	for(unsigned int j=0;j < vObs.size();j++){
		theMap.insertObservation( &vObs.at(j) );
	}
}


/*-------------------------------------------------------------
						   limit
-------------------------------------------------------------*/

void CPtuHokuyo::limit(CSimplePointsMap &theMap){

	char axis = 'a';

	loadObs2PointsMap(theMap);

	// Scene Objects
	CDisplayWindow3D	win("Ptu&Hokuyo graphic: Points earned",800,600);

	COpenGLScenePtr &theScene = win.get3DSceneAndLock();

	// Modify the scene
	{
		opengl::CGridPlaneXYPtr obj = opengl::CGridPlaneXY::Create(-10,10,-10,10,0,1);
		obj->m_color_R = obj->m_color_G = obj->m_color_B = 0.4f;
		theScene->insert( obj );
	}

	{
		opengl::CAxisPtr obj = opengl::CAxis::Create();
		obj->m_frecuency = 5;
		obj->m_marks     = true;
		obj->m_xmin      = -10;
		obj->m_xmax      =  10;
		obj->m_ymin      = -10;
		obj->m_ymax      =  10;
		obj->m_zmin      = -10;
		obj->m_zmax      =  10;
		theScene->insert( obj );
	}

	// Lock scene to use
	win.unlockAccess3DScene();

	win.setCameraElevationDeg( 25.0f );

	opengl::CPointCloudPtr objMap = opengl::CPointCloud::Create();
	objMap->m_color_R = 0;
	objMap->m_color_G = 0;
	objMap->m_color_B = 1;
	objMap->loadFromPointsMap(&theMap);
	objMap->m_colorFromZ = true;
	objMap->m_name = "Cloud" ;

	// Lock scene to use
	theScene=win.get3DSceneAndLock();

	// Insert points cloud in the scene
	theScene->insert( objMap );

	// Unlock scene
	win.unlockAccess3DScene();

	// Update window
	win.forceRepaint();

	printf("\n===============================\n"
			"Limit valids points application\n"
			"===============================\n\n"
			"Commands list:\n"
			"x : limit x axis\n"
			"y : limit y axis\n"
			"z : limit z axis\n"
			"r : reload map points from the observations vector\n"
			"q : finish limit\n\n");

	while (axis!='q'){

		double low,high;
		cout << ">> Command: ";
		cin >> axis;

		if((axis!='q')&&(axis=='r')){
			loadObs2PointsMap(theMap);
			cout << endl;
		}

		if(axis!='q'&&(axis!='r')){

			cout << ">> Enter low limit: ";
			cin >> low;
			cout << ">> Enter high limit: ";
			cin >> high;
			cout << endl;

			limitScan(axis,low,high,theMap);
		}
			// Lock scene to use
			theScene=win.get3DSceneAndLock();

			theScene->clear();

			// Modify the scene
			{
				opengl::CGridPlaneXYPtr obj = opengl::CGridPlaneXY::Create(-10,10,-10,10,0,1);
				obj->m_color_R = obj->m_color_G = obj->m_color_B = 0.4f;
				theScene->insert( obj );
			}

			{
				opengl::CAxisPtr obj = opengl::CAxis::Create();
				obj->m_frecuency = 5;
				obj->m_marks     = true;
				obj->m_xmin      = -10;
				obj->m_xmax      =  10;
				obj->m_ymin      = -10;
				obj->m_ymax      =  10;
				obj->m_zmin      = -10;
				obj->m_zmax      =  10;
				theScene->insert( obj );
			}

			// Unlock scene
			win.unlockAccess3DScene();

			opengl::CPointCloudPtr objMap = opengl::CPointCloud::Create();
			objMap->m_color_R = 0;
			objMap->m_color_G = 0;
			objMap->m_color_B = 1;
			objMap->loadFromPointsMap(&theMap);
			objMap->m_colorFromZ = true;
			objMap->m_name = "Cloud" ;

			// Lock scene to use
			theScene=win.get3DSceneAndLock();

			// Insert points cloud in the scene
			theScene->insert( objMap );

			// Unlock scene
			win.unlockAccess3DScene();

			// Update window
			win.forceRepaint();


	} // End while

}

/*-------------------------------------------------------------
				 minLengthVectors
-------------------------------------------------------------*/

int CPtuHokuyo::minLengthVectors(mrpt::slam::CObservation2DRangeScan &obs1, mrpt::slam::CObservation2DRangeScan &obs2, const int &mode){

	if(mode){
		if (obs1.scan.size()>=obs2.scan.size()){
			return obs1.scan.size();
		}else{
			return obs2.scan.size();
		}
	}else{
		if (obs1.validRange.size()>=obs2.validRange.size()){
			return obs1.validRange.size();
		}else{
			return obs2.validRange.size();
		}
	}

}

void CPtuHokuyo::calculateSensorPose(const char &axis, const double &pos, mrpt::slam::CObservation2DRangeScan &obs){

	if((axis=='T')||(axis=='t')){
		// Calculate sensor position
		double x=high*cos(pos+DEG2RAD(90.0));
		double z=high*sin(pos+DEG2RAD(90.0));


		obs.sensorPose = CPose3D(x,0,z,0,-pos);

	}else{ // Pan movement

		obs.sensorPose = CPose3D(0,0,high,-pos);
	}
}
