/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/config.h>

#include <mrpt/vision/utils.h>
#include <mrpt/poses/CPoint3D.h>
#include <mrpt/slam/CLandmarksMap.h>
#include <mrpt/slam/CObservationVisualLandmarks.h>
#include <mrpt/vision/CFeature.h>

#include <mrpt/gui/CDisplayWindow.h>
#include <mrpt/utils/CTicTac.h>
#include <mrpt/math/utils.h>
#include <mrpt/math/vector_ops.h>


#if MRPT_HAS_OPENCV

	// OPENCV HEADERS
	#include <cv.h>
	#include <cxcore.h>
	#include <cvaux.h>
	#include <highgui.h>

	// Include the OPENCV libraries:
	#if defined(_MSC_VER)
		#pragma comment (lib,"comctl32.lib")
	#endif

#endif // MRPT_HAS_OPENCV


using namespace mrpt;
using namespace mrpt::vision;
using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::math;
using namespace mrpt::system;
using namespace std;


#ifdef MRPT_OS_WINDOWS
    #include <process.h>
    #include <windows.h>		// TODO: This is temporal!!!
#endif

/*-------------------------------------------------------------
					buildIntrinsicParamsMatrix
-------------------------------------------------------------*/
CMatrix  vision::buildIntrinsicParamsMatrix(
				float	focalLengthX,
				float	focalLengthY,
				float	centerX,
				float	centerY)
{
	CMatrix		A(3,3);

	A.zeros();

	A(0,0) = focalLengthX;
	A(1,1) = focalLengthY;
	A(2,2) = 1;

	A(0,2) = centerX;
	A(1,2) = centerY;

	return A;
}

/*-------------------------------------------------------------
					pixelTo3D
-------------------------------------------------------------*/
CPoint3D  vision::pixelTo3D(float x,float y, const CMatrix &A)
{
	CPoint3D	res;

	// Build the vector:
	res.x = x - A(0,2);
	res.y = y - A(1,2);
	res.z = A(0,0);

	// Normalize:
	float	u = res.norm();
	ASSERT_(u!=0);

	res*=1/u;

	return res;
}

/*-------------------------------------------------------------
					defaultIntrinsicParamsMatrix
-------------------------------------------------------------*/
CMatrix  vision::defaultIntrinsicParamsMatrix(
					unsigned int camIndex,
					unsigned int resX,
					unsigned int resY)
{
	float		fx,fy,cx,cy;

	switch(camIndex)
	{
	case 0:
		// Bumblebee:
		fx=0.79345f;	fy=1.05793f;
		cx=0.55662f;	cy=0.52692f;
		break;

	case 1:
		// Sony:
		fx=0.95666094f;	fy=1.3983423f;
		cx=0.54626328f;	cy=0.4939191f;
		break;

	default:
		{
			THROW_EXCEPTION_CUSTOM_MSG1( "Unknown camera index!! for 'camIndex'=%u",camIndex );
		}
	}

	return buildIntrinsicParamsMatrix(	resX * fx,	resY * fy,
										resX * cx,	resY * cy );

}

/*-------------------------------------------------------------
					projectMatchedFeatures
-------------------------------------------------------------*/
void  vision::projectMatchedFeatures(
	CMatchedFeatureList					&mfList,		// The set of matched features
	const vision::TStereoSystemParams	&param,			// Parameters for the stereo system
	mrpt::slam::CLandmarksMap			&landmarks )	// Output map of 3D landmarks
{
	MRPT_TRY_START;

	landmarks.clear();								// Assert that the output CLandmarksMap is clear

	CMatchedFeatureList::iterator		itList;
	float								stdPixel2	= square( param.stdPixel );
	float								stdDisp2	= square( param.stdDisp );

	// Main loop
	for(itList = mfList.begin(); itList != mfList.end();)
	{
		float disp = ( itList->first->x - itList->second->x );	// Disparity
		if ( disp < 1e-9 )										// Filter out too far points
			itList = mfList.remove( itList );					// Erase the match : itList = mfList.erase( itList );

		else													// This
		{
			// Too much distant features are not taken into account
			float x3D = ( itList->first->x - param.K(0,2) ) * ( (param.baseline) ) / disp;
			float y3D = ( itList->first->y - param.K(1,2) ) * ( (param.baseline) ) / disp;
			float z3D = ( param.K(0,0) ) * ( (param.baseline) ) / disp;

			// Filter out bad points
			if ( (z3D < param.minZ) || (z3D > param.maxZ) )
				itList = mfList.remove( itList );					// Erase the match : itList = mfList.erase( itList );
			else
			{
				CPoint3D	p3D(x3D,y3D,z3D);

				// STORE THE OBTAINED LANDMARK
				CLandmark	lm;

				p3D *= -1/p3D.norm();

				lm.type = CLandmark::vlSIFT;
				lm.normal_x = p3D.x;
				lm.normal_y = p3D.y;
				lm.normal_z = p3D.z;

				lm.pose_mean_x = x3D;
				lm.pose_mean_y = y3D;
				lm.pose_mean_z = z3D;

				lm.ID = itList->first->ID;

				// If the matched landmarks has a (SIFT or SURF) descriptor, asign the left one to the landmark.
				// TO DO: Assign the mean value of the descriptor (between the matches)
				if( (*itList).first->hasDescriptorSIFT || (*itList).first->hasDescriptorSURF )
				{
					// By now, SIFT over SURF. TO DO: support for two different descriptors
					lm.descriptor1		= (*itList).first->hasDescriptorSIFT ? (*itList).first->descriptorSIFT : (*itList).first->descriptorSURF;
					lm.descriptor2.resize(2);
					lm.descriptor2[0]	= (*itList).first->orientation;
					lm.descriptor2[1]	= (*itList).first->scale;
				}

				// Compute the covariance matrix for the landmark
				switch( param.uncPropagation )
				{
				case TStereoSystemParams::Prop_Linear:
				{

					float foc2	= square( param.K(0,0) );
					float c0	= param.K(0,2);
					float r0	= param.K(1,2);
					float base2	= square( param.baseline );
					float disp2	= square( itList->first->x - itList->second->x );

					lm.pose_cov_11 = stdPixel2*base2/disp2 + stdDisp2*base2*square( itList->first->x - c0 )/square(disp2);
					lm.pose_cov_12 = stdDisp2*base2*( itList->first->x - c0 )*( itList->first->y - r0 )/square(disp2);
					lm.pose_cov_13 = stdDisp2*base2*sqrt(foc2)*( itList->first->x - c0 )/square(disp2);
					lm.pose_cov_22 = stdPixel2*base2/disp2 + stdDisp2*base2*square( itList->first->y - r0 )/square(disp2);
					lm.pose_cov_23 = stdDisp2*base2*sqrt(foc2)*( itList->first->y - r0 )/square(disp2);
					lm.pose_cov_33 = stdDisp2*foc2*base2/square(disp2);
				} // end case 'Prop_Linear'
				break;

				case TStereoSystemParams::Prop_UT:
				{
					// Parameters
					unsigned int			Na = 3;
					unsigned int			i;

					float					k = param.factor_k;

					float					w0 = k/(Na + k);
					float					w1 = 1/(2*(Na + k));

					CMatrix					Pa(3,3);
					CMatrix					L(3,3);

					Pa.fill(0);
					Pa(0,0) = Pa(1,1) = ( Na + k ) * square( param.stdPixel );
					Pa(2,2) = ( Na + k ) * square( param.stdDisp );

					// Cholesky decomposition
					math::chol(Pa,L);

					vector<CPoint3D>	B;				// B group
					poses::CPoint3D			meanB;			// Mean value of the B group
					CMatrix					Pb;				// Covariance of the B group

					B.resize( 2*Na + 1 );	// Set of output values
					Pb.fill(0);				// Reset the output covariance

					vector_float			vAux, myPoint;	// Auxiliar vectors
					vector_float			meanA;			// Mean value of the A group

					vAux.resize(3);			// Set the variables size
					meanA.resize(3);
					myPoint.resize(3);

					// Mean input value: (c,r,d)
					meanA[0] = itList->first->x;
					meanA[1] = itList->first->y;
					meanA[2] = disp;

					// Output mean
					meanB.x = w0*x3D; meanB.y = w0*y3D; meanB.z = w0*z3D;	// Add to the mean
					B[0].x = x3D; B[0].y = y3D; B[0].z = z3D;				// Insert into B

					for( i = 1; i <= 2*Na; i++ )
					{
						// Form the Ai value
						if( i <= Na )
						{
							L.extractRow( i-1, vAux );						// Extract the proper row
							myPoint[0] = meanA[0] + vAux[0];
							myPoint[1] = meanA[1] + vAux[1];
							myPoint[2] = meanA[2] + vAux[2];
						}
						else
						{
							L.extractRow( (i-Na)-1, vAux );					// Extract the proper row
							myPoint[0] = meanA[0] - vAux[0];
							myPoint[1] = meanA[1] - vAux[1];
							myPoint[2] = meanA[2] - vAux[2];
						}

						// Pass the Ai through the functions:
						x3D = ( myPoint[0] - param.K(0,2) ) * ( (param.baseline) ) / myPoint[2];
						y3D = ( myPoint[1] - param.K(1,2) ) * ( (param.baseline) ) / myPoint[2];
						z3D = ( param.K(0,0) ) * ( (param.baseline) ) / myPoint[2];

						// Add to the B mean computation and the B vector
						meanB.x = meanB.x + w1*x3D;
						meanB.y = meanB.y + w1*y3D;
						meanB.z = meanB.z + w1*z3D;

						B[i].x = x3D;
						B[i].y = y3D;
						B[i].z = z3D;

					} // end for 'i'

					// Output covariance
					for( i = 0; i <= 2*Na; i++ )
					{
						float		weight = w1;
						CMatrix		v(3,1);

						if( i == 0 )				// The weight for the mean value of A is w0
							weight = w0;

						v(0,0) = B[i].x - meanB.x;
						v(1,0) = B[i].y - meanB.y;
						v(2,0) = B[i].z - meanB.z;

						Pb = Pb + weight*(v*~v);
					} // end for 'i'

					// Store it in the landmark
					lm.pose_cov_11 = Pb(0,0);
					lm.pose_cov_12 = Pb(0,1);
					lm.pose_cov_13 = Pb(0,2);
					lm.pose_cov_22 = Pb(1,1);
					lm.pose_cov_23 = Pb(1,2);
					lm.pose_cov_33 = Pb(2,2);
				} // end case 'Prop_UT'
				break;

				case TStereoSystemParams::Prop_SUT:
				{
					// Parameters
					unsigned int			Na = 3;
					unsigned int			i;

					float					a = param.factor_a;
					float					b = param.factor_b;
					float					k = param.factor_k;

					float					lambda = square(a)*(Na + k) - Na;

					float					w0_m = lambda/(Na + lambda);
					float					w0_c = w0_m + (1 - square(a) + b);
					float					w1 = 1/(2*(Na + lambda));

					CMatrix					Pa(3,3);
					CMatrix					L(3,3);

					Pa.fill(0);
					Pa(0,0) = Pa(1,1) = ( Na + lambda ) * square( param.stdPixel );
					Pa(2,2) = ( Na + lambda ) * square( param.stdDisp );

					// Cholesky decomposition
					math::chol(Pa,L);

					vector<CPoint3D>	B;				// B group
					poses::CPoint3D		meanB;			// Mean value of the B group
					CMatrix				Pb;				// Covariance of the B group

					B.resize( 2*Na + 1 );	// Set of output values
					Pb.fill(0);				// Reset the output covariance

					vector_float		vAux, myPoint;	// Auxiliar vectors
					vector_float		meanA;			// Mean value of the A group

					vAux.resize(3);			// Set the variables size
					meanA.resize(3);
					myPoint.resize(3);

					// Mean input value: (c,r,d)
					meanA[0] = itList->first->x;
					meanA[1] = itList->first->y;
					meanA[2] = disp;

					// Output mean
					meanB.x = w0_m*x3D; meanB.y = w0_m*y3D; meanB.z = w0_m*z3D;		// Add to the mean
					B[0].x = x3D; B[0].y = y3D; B[0].z = z3D;						// Insert into B

					for( i = 1; i <= 2*Na; i++ )
					{
						// Form the Ai value
						if( i <= Na )
						{
							L.extractRow( i-1, vAux );						// Extract the proper row
							myPoint = meanA + vAux;
							//myPoint[0] = meanA[0] + vAux[0];
							//myPoint[1] = meanA[1] + vAux[1];
							//myPoint[2] = meanA[2] + vAux[2];
						}
						else
						{
							L.extractRow( (i-Na)-1, vAux );					// Extract the proper row
							myPoint = meanA - vAux;
							//myPoint[0] = meanA[0] - vAux[0];
							//myPoint[1] = meanA[1] - vAux[1];
							//myPoint[2] = meanA[2] - vAux[2];
						}

						// Pass the Ai through the functions:
						x3D = ( myPoint[0] - param.K(0,2) ) * ( (param.baseline) ) / myPoint[2];
						y3D = ( myPoint[1] - param.K(1,2) ) * ( (param.baseline) ) / myPoint[2];
						z3D = ( param.K(0,0) ) * ( (param.baseline) ) / myPoint[2];

						// Add to the B mean computation and the B vector
						meanB.x = meanB.x + w1*x3D;
						meanB.y = meanB.y + w1*y3D;
						meanB.z = meanB.z + w1*z3D;

						B[i].x = x3D;
						B[i].y = y3D;
						B[i].z = z3D;

					} // end for 'i'

					// Output covariance
					for( i = 0; i <= 2*Na; i++ )
					{
						float		weight = w1;
						CMatrix		v(3,1);

						if( i == 0 )				// The weight for the mean value of A is w0
							weight = w0_c;

						v(0,0) = B[i].x - meanB.x;
						v(1,0) = B[i].y - meanB.y;
						v(2,0) = B[i].z - meanB.z;

						Pb = Pb + weight*(v*~v);
					} // end for 'i'

					// Store it in the landmark
					lm.pose_cov_11 = Pb(0,0);
					lm.pose_cov_12 = Pb(0,1);
					lm.pose_cov_13 = Pb(0,2);
					lm.pose_cov_22 = Pb(1,1);
					lm.pose_cov_23 = Pb(1,2);
					lm.pose_cov_33 = Pb(2,2);
				} // end case 'Prop_SUT'
				break;

				} // end switch
				landmarks.landmarks.push_back( &lm );
				itList++;
			} // end else ( (z3D > param.minZ) && (z3D < param.maxZ) )
		} // end else
	} // end for 'i'

	MRPT_TRY_END;
} // end of projectMatchedFeatures(


/*-------------------------------------------------------------
					deleteRepeatedFeats
-------------------------------------------------------------*/
void  vision::deleteRepeatedFeats( CFeatureList & feat_list )
{
	CFeatureList::iterator	itList1,itList2;
	float					lx = 0.0, ly = 0.0;

	// Look for repeated features in the feat_list of features
	for( itList1 = feat_list.begin(); itList1 != feat_list.end(); itList1++)
	{
		lx = (*itList1)->x;
		ly = (*itList1)->y;
		for( itList2 = itList1; itList2 != feat_list.end(); itList2++)
		{
			if( (lx == (*itList2)->x && ly == (*itList2)->y ) && ( (*itList2)->x > 0.0f && (*itList2)->y > 0.0f ))
			{
				(*itList2)->x = -1.0f;
				(*itList2)->y = -1.0f;
			} // end if
		} // end for
	} // end for

	// Delete the repeated features
	for( itList1 = feat_list.begin(); itList1 != feat_list.end();)
	{
		if( (*itList1)->x == -1.0f && (*itList1)->y == -1.0f )
			itList1 = feat_list.remove( itList1 );
		else
			itList1++;
	} // end for
} // end deleteRepeatedFeats


/*------------------------------------------------------------
					getDispersion
-------------------------------------------------------------*/
void vision::getDispersion( const CFeatureList &list, vector_float &std, vector_float &mean )
{
	std.clear();
	mean.clear();

	std.resize(2,0);
	mean.resize(2,0);

	CFeatureList::const_iterator it;
	double varx = 0, vary = 0;

	for( it = list.begin(); it != list.end(); it++ )
	{
		mean[0] += (*it)->x;
		mean[1] += (*it)->y;
	}
	mean[0] /= list.size();
	mean[1] /= list.size();

	for( it = list.begin(); it != list.end(); it++ )
	{
		varx += square( (*it)->x - mean[0] );
		vary += square( (*it)->y - mean[1] );
	}
	varx /= list.size();
	vary /= list.size();

	std[0] = sqrt( varx );
	std[1] = sqrt( vary );
} // end getDispersion

/*------------------------------------------------------------
					checkTrackedFeatures
-------------------------------------------------------------*/
void vision::checkTrackedFeatures( CFeatureList &leftList,
							    CFeatureList &rightList,
								vision::TMatchingOptions options)
{
	ASSERT_( leftList.size() == rightList.size() );

	std::cout << std::endl << "Tracked features checking ..." << std::endl;

	CFeatureList::iterator	itLeft, itRight;
	size_t					u,v;
	double					res;
	for( itLeft = leftList.begin(), itRight = rightList.begin(); itLeft != leftList.end(); )
	{
		// Compute cross correlation:
		openCV_cross_correlation( (*itLeft)->patch, (*itRight)->patch, u, v, res );

		if( (*itLeft)->x < 0 || (*itLeft)->y < 0 ||
			(*itRight)->x < 0 || (*itRight)->y < 0 ||
			fabs( (*itLeft)->y - (*itRight)->y ) > options.rowCheck_TH ||
			res < options.ccCheck_TH )
		{
			// Show reason
			std::cout << "Bad tracked match:";

			if( res < options.ccCheck_TH )
				std::cout << " CC Value: " << res << std::endl;
			if( (*itLeft)->x < 0 || (*itLeft)->y < 0 || (*itRight)->x < 0 || (*itRight)->y < 0 )
				std::cout << " Out of bounds: (" << (*itLeft)->x << "," << (*itLeft)->y << " & (" << (*itRight)->x << "," << (*itRight)->y << ")";
			if( fabs( (*itLeft)->y - (*itRight)->y ) > options.rowCheck_TH )
				std::cout << " Bad row checking: " << fabs( (*itLeft)->y - (*itRight)->y );

			std::cout << std::endl;

			// Erase the pair of features
			itLeft = leftList.remove( itLeft );
			itRight = rightList.remove( itRight );

		} // end if
		else
		{
			itLeft++;
			itRight++;
		}
	} // end for



} // end checkTrackedFeatures
/*------------------------------------------------------------
					rowChecking
-------------------------------------------------------------*/
void  vision::rowChecking( CFeatureList &leftList,
							    CFeatureList &rightList,
								float threshold )
{
	ASSERT_( leftList.size() == rightList.size() );

	// By now: row checking -> Delete bad correspondences
	std::cout << std::endl << "[ROW CHECKING]------------------------------------------" << std::endl;

	CFeatureList::iterator	itLeft, itRight;
	for( itLeft = leftList.begin(), itRight = rightList.begin(); itLeft != leftList.end(); )
	{
		if( (*itLeft)->x < 0 || (*itLeft)->y < 0 ||
			(*itRight)->x < 0 || (*itRight)->y < 0 ||
			fabs( (*itLeft)->y - (*itRight)->y ) > threshold )
		{
			std::cout << "[Erased Feature] Row Dif: " << fabs( (*itLeft)->y - (*itRight)->y ) << std::endl;
			itLeft = leftList.remove( itLeft );
			itRight = rightList.remove( itRight );
		} // end if
		else
		{
			itLeft++;
			itRight++;
		}
	} // end for

	std::cout << "------------------------------------------" << std::endl;

	std::cout << "Tracked features: " << leftList.size() << " and " << rightList.size() << std::endl;
	ASSERT_( leftList.size() == rightList.size() );

} // end rowChecking

/*------------------------------------------------------------
					trackFeatures
-------------------------------------------------------------*/
void  vision::trackFeatures( const CMRPTImage &inImg1,
				 				  const CMRPTImage &inImg2,
								  CFeatureList &featureList,
								  const unsigned int &window_width,
								  const unsigned int &window_height)
{
	MRPT_TRY_START;

#if MRPT_HAS_OPENCV

MRPT_TRY_START;
// Use OpenCV Implementation

// OpenCV Local Variables
CvPoint2D32f*	points[2] = {0,0};
char*			status = 0;
int				flags = 0;
int				nFeatures = (int)featureList.size();					// Number of features

// Input Images
IplImage* pImg = (IplImage*)inImg1.getAsIplImage();
IplImage* cImg = (IplImage*)inImg2.getAsIplImage();

// Grayscale images
IplImage* pGrey = cvCreateImage( cvGetSize( pImg ), 8, 1 );
IplImage* cGrey = cvCreateImage( cvGetSize( cImg ), 8, 1 );

// Conver to grayscale
cvCvtColor( pImg, pGrey, CV_BGR2GRAY );
cvCvtColor( cImg, cGrey, CV_BGR2GRAY );

// Pyramids
IplImage* pPyr = NULL;
IplImage* cPyr = NULL;

// Arrays definition
points[0] = (CvPoint2D32f*)cvAlloc( nFeatures*sizeof( points[0][0] ) );
points[1] = (CvPoint2D32f*)cvAlloc( nFeatures*sizeof( points[0][0] ) );
status = (char*)cvAlloc( nFeatures );

// Array conversion MRPT->OpenCV
CFeatureList::iterator		itFeat;
int							i;				// Counter
for( i = 0, itFeat = featureList.begin(); i < nFeatures && itFeat != featureList.end(); i++, itFeat++  )
{
	points[0][i].x = (*itFeat)->x;
	points[0][i].y = (*itFeat)->y;
} // end for

// Optical Flow Computation
cvCalcOpticalFlowPyrLK( pGrey, cGrey, pPyr, cPyr,
	points[0], points[1], nFeatures, cvSize( window_width, window_height ), 3, status, NULL,
	cvTermCriteria(CV_TERMCRIT_ITER|CV_TERMCRIT_EPS,20,0.03), flags );

// Array conversion OpenCV->MRPT
for( i = 0, itFeat = featureList.begin(); i < nFeatures && itFeat != featureList.end(); i++, itFeat++  )
{
	if( status[i] == 1 &&
		points[1][i].x > 0 && points[1][i].y > 0 &&
		points[1][i].x < pGrey->width && points[1][i].y < pGrey->height )
	{
		// Feature could be tracked
		(*itFeat)->x			= points[1][i].x;
		(*itFeat)->y			= points[1][i].y;
		(*itFeat)->KLT_status	= statusKLT_TRACKED;
	} // end if
	else
	{
		// Feature could not be tracked
		(*itFeat)->x			= -1;
		(*itFeat)->y			= -1;
		(*itFeat)->KLT_status	= statusKLT_IDLE;
	} // end else
} // end for

// Free memory
cvReleaseImage( &pGrey );
cvReleaseImage( &cGrey );
cvReleaseImage( &pPyr );
cvReleaseImage( &cPyr );
cvFree( (void**)&points[0] );
cvFree( (void**)&points[1] );
cvFree( (void**)&status );

MRPT_TRY_END;

#else
	THROW_EXCEPTION("The MRPT has been compiled with MRPT_HAS_OPENCV=0 !");
#endif

	MRPT_TRY_END;
} // end trackFeatures



/*-------------------------------------------------------------
					correctDistortion
-------------------------------------------------------------*/
void  vision::correctDistortion(
			CMRPTImage		&in_img,
			CMRPTImage		&out_img,
			CMatrix			A,
			CMatrix			dist_coeffs )
{
	out_img = CMRPTImage( in_img.getWidth(),
						  in_img.getHeight(),
						  in_img.isColor() ? 3:1,
						  in_img.isOriginTopLeft() );

	THROW_EXCEPTION("Not IMPLEMENTED!!");

/*    static int inittab = 0;
    uchar* buffer = 0;

    float a[9], k[4];
    int coi1 = 0, coi2 = 0;
    CvMat srcstub, *src = (CvMat*)_src;
    CvMat dststub, *dst = (CvMat*)_dst;
    CvMat _a = cvMat( 3, 3, CV_32F, a ), _k;
    int cn, src_step, dst_step;
    CvSize size;

    if( !inittab )
    {
        icvInitLinearCoeffTab();
        icvInitCubicCoeffTab();
        inittab = 1;
    }

    CV_CALL( src = cvGetMat( src, &srcstub, &coi1 ));
    CV_CALL( dst = cvGetMat( dst, &dststub, &coi2 ));

    if( coi1 != 0 || coi2 != 0 )
        CV_ERROR( CV_BadCOI, "The function does not support COI" );

    if( CV_MAT_DEPTH(src->type) != CV_8U )
        CV_ERROR( CV_StsUnsupportedFormat, "Only 8-bit images are supported" );

    if( !CV_ARE_TYPES_EQ( src, dst ))
        CV_ERROR( CV_StsUnmatchedFormats, "" );

    if( !CV_ARE_SIZES_EQ( src, dst ))
        CV_ERROR( CV_StsUnmatchedSizes, "" );

    if( !CV_IS_MAT(A) || A->rows != 3 || A->cols != 3  ||
        CV_MAT_TYPE(A->type) != CV_32FC1 && CV_MAT_TYPE(A->type) != CV_64FC1 )
        CV_ERROR( CV_StsBadArg, "Intrinsic matrix must be a valid 3x3 floating-point matrix" );

    if( !CV_IS_MAT(dist_coeffs) || dist_coeffs->rows != 1 && dist_coeffs->cols != 1 ||
        dist_coeffs->rows*dist_coeffs->cols*CV_MAT_CN(dist_coeffs->type) != 4 ||
        CV_MAT_DEPTH(dist_coeffs->type) != CV_64F &&
        CV_MAT_DEPTH(dist_coeffs->type) != CV_32F )
        CV_ERROR( CV_StsBadArg,
            "Distortion coefficients must be 1x4 or 4x1 floating-point vector" );

    cn = CV_MAT_CN(src->type);
    size = cvGetMatSize(src);
    src_step = src->step ? src->step : CV_STUB_STEP;
    dst_step = dst->step ? dst->step : CV_STUB_STEP;

    if( fabs((double)k[2]) < 1e-5 && fabs((double)k[3]) < 1e-5 && icvUndistortGetSize_p )
    {
        int buf_size = 0;
        CvUndistortRadialIPPFunc func =
            cn == 1 ? (CvUndistortRadialIPPFunc)icvUndistortRadial_8u_C1R_p :
                      (CvUndistortRadialIPPFunc)icvUndistortRadial_8u_C3R_p;

        if( func && icvUndistortGetSize_p( size, &buf_size ) >= 0 && buf_size > 0 )
        {
            CV_CALL( buffer = (uchar*)cvAlloc( buf_size ));
            if( func( src->data.ptr, src_step, dst->data.ptr,
                      dst_step, size, a[0], a[4],
                      a[2], a[5], k[0], k[1], buffer ) >= 0 )
                EXIT;
        }
    }

    icvUnDistort_8u_CnR( src->data.ptr, src_step,
        dst->data.ptr, dst_step, size, a, k, cn );
*/
}

/*-------------------------------------------------------------
					hsv2rgb
-------------------------------------------------------------*/
void  vision::hsv2rgb(
	float	h,
	float	s,
	float	v,
	float	&r,
	float	&g,
	float	&b)
{
	// See: http://en.wikipedia.org/wiki/HSV_color_space
	h = max(0.0f, min(1.0f,h));
	s = max(0.0f, min(1.0f,s));
	v = max(0.0f, min(1.0f,v));

	int		Hi = ((int)floor(h *6)) % 6;
	float	f  = (h*6) - Hi;
	float	p  = v*(1-s);
	float	q  = v*(1-f*s);
	float	t  = v*(1-(1-f)*s);

	switch (Hi)
	{
	case 0:	r=v; g=t;b=p; break;
	case 1:	r=q; g=v;b=p; break;
	case 2:	r=p; g=v;b=t; break;
	case 3:	r=p; g=q;b=v; break;
	case 4:	r=t; g=p;b=v; break;
	case 5:	r=v; g=p;b=q; break;
	}
}

/*-------------------------------------------------------------
					rgb2hsv
-------------------------------------------------------------*/
void  vision::rgb2hsv(
	float	r,
	float	g,
	float	b,
	float	&h,
	float	&s,
	float	&v )
{
	// See: http://en.wikipedia.org/wiki/HSV_color_space
	r = max(0.0f, min(1.0f,r));
	g = max(0.0f, min(1.0f,g));
	b = max(0.0f, min(1.0f,b));

	float	Max = max3(r,g,b);
	float	Min = min3(r,g,b);

	if (Max==Min)
	{
		h = 0;
	}
	else
	{
		if (Max==r)
		{
			if (g>=b)
					h = (g-b)/(6*(Max-Min));
			else	h = 1-(g-b)/(6*(Max-Min));
		}
		else
		if (Max==g)
				h = 1/3.0f + (b-r)/(6*(Max-Min));
		else	h = 2/3.0f + (r-g)/(6*(Max-Min));
	}

	if (Max == 0)
			s = 0;
	else	s = 1 - Min/Max;

	v = Max;
}

/*-------------------------------------------------------------
						computeMsd
-------------------------------------------------------------*/
double  vision::computeMsd( const mrpt::slam::CMetricMap::TMatchingPairList &feat_list,
								 const mrpt::poses::CPose3D &Rt )
{

	CMatrix	mat = Rt.getHomogeneousMatrix();
	double	acum = 0.0;

	mrpt::slam::CMetricMap::TMatchingPairList::const_iterator it;
	for( it = feat_list.begin(); it != feat_list.end(); it++ )
	{
		double e1 = it->other_x - (it->this_x*mat(0,0) + it->this_y*mat(0,1) + it->this_z*mat(0,2) + Rt.x);
		double e2 = it->other_y - (it->this_x*mat(1,0) + it->this_y*mat(1,1) + it->this_z*mat(1,2) + Rt.y);
		double e3 = it->other_z - (it->this_x*mat(2,0) + it->this_y*mat(2,1) + it->this_z*mat(2,2) + Rt.z);

		acum += sqrt( square(e1) + square(e2) + square(e3) );

	} // end for
	return( acum/feat_list.size() );
} // end msd

/*-------------------------------------------------------------
					filterBadCorrsByDistance
-------------------------------------------------------------*/
void  vision::filterBadCorrsByDistance( mrpt::slam::CMetricMap::TMatchingPairList &feat_list, unsigned int numberOfSigmas )
{
	ASSERT_( numberOfSigmas > 0 );
	//	MRPT_UNUSED_PARAM( numberOfSigmas );
	MRPT_TRY_START;

	CMetricMap::TMatchingPairList::iterator	itPair;
	CMatrix									dist;
	float									v_mean, v_std;
	unsigned int							count = 0;

	dist.setSize( feat_list.size(), 1 );
	//v_mean.resize(1);
	//v_std.resize(1);

	// Compute mean and standard deviation of the distance
	for( itPair = feat_list.begin(); itPair != feat_list.end() ; itPair++, count++ ) {
		//cout << "(" << itPair->other_x << "," << itPair->other_y << "," << itPair->this_z << ")" << "- (" << itPair->this_x << "," << itPair->this_y << "," << itPair->other_z << "): ";
		//cout << sqrt( square( itPair->other_x - itPair->this_x ) + square( itPair->other_y - itPair->this_y ) + square( itPair->other_z - itPair->this_z ) ) << endl;
		dist( count, 0 ) = sqrt( square( itPair->other_x - itPair->this_x ) + square( itPair->other_y - itPair->this_y ) + square( itPair->other_z - itPair->this_z ) );
	}

	dist.meanAndStdAll( v_mean, v_std );

	cout << endl << "*****************************************************" << endl;
	cout << "Mean: " << v_mean << " - STD: " << v_std << endl;
	cout << endl << "*****************************************************" << endl;

	// Filter out bad points
	unsigned int idx = 0;
	//for( int idx = (int)feat_list.size()-1; idx >= 0; idx-- )
	for( itPair = feat_list.begin(); itPair != feat_list.end(); idx++)
	{
		//if( dist( idx, 0 ) > 1.2 )
		if( fabs( dist(idx,0) - v_mean ) > v_std*numberOfSigmas )
		{
			cout << "Outlier deleted: " << dist( idx, 0 ) << " vs " << v_std*numberOfSigmas << endl;
			itPair = feat_list.erase( itPair );
		}
		else
			itPair++;
	}

	MRPT_TRY_END;
} // end filterBadCorrsByDistance


/*-------------------------------------------------------------
					TROI Constructors
-------------------------------------------------------------*/
vision::TROI::TROI(): xMin(0), xMax(0), yMin(0), yMax(0), zMin(0), zMax(0)
{}

vision::TROI::TROI(float x1, float x2, float y1, float y2, float z1, float z2) : xMin(x1), xMax(x2), yMin(y1), yMax(y2), zMin(z1), zMax(z2)
{}

/*-------------------------------------------------------------
					TImageROI Constructors
-------------------------------------------------------------*/
vision::TImageROI::TImageROI(): 
	xMin(0), xMax(0), 
	yMin(0), yMax(0)
{}

vision::TImageROI::TImageROI( float x1, float x2, float y1, float y2 ): 
	xMin(x1), xMax(x2), 
	yMin(y1), yMax(y2)
{}

/*-------------------------------------------------------------
					openCV_cross_correlation
-------------------------------------------------------------*/
void  vision::openCV_cross_correlation(
	const CMRPTImage	&img,
	const CMRPTImage	&patch_img,
	size_t				&x_max,
	size_t				&y_max,
	double				&max_val,
	int					x_search_ini,
	int					y_search_ini,
	int					x_search_size,
	int					y_search_size)
{
	MRPT_TRY_START;

#if MRPT_HAS_OPENCV
	double		mini;
	CvPoint		min_point,max_point;

/*	static gui::CDisplayWindow	win1("IMG");
	static gui::CDisplayWindow	win2("PATCH");
	win1.showImage(img);
	win2.showImage(patch_img);
	win1.waitForKey();
*/

	bool entireImg = (x_search_ini<0 || y_search_ini<0 || x_search_size<0 || y_search_size<0);

	IplImage *im_ = (IplImage*)img.getAsIplImage();
	IplImage *patch_im_ = (IplImage*)patch_img.getAsIplImage();

	IplImage *im = cvCreateImage( cvGetSize( im_ ), 8, 1 );
	IplImage *patch_im = cvCreateImage( cvGetSize( patch_im_ ), 8, 1 );

	cvCvtColor( im_, im, CV_BGR2GRAY );
	cvCvtColor( patch_im_, patch_im, CV_BGR2GRAY );

	if (entireImg)
	{
		x_search_size = im->width - patch_im->width;
		y_search_size = im->height - patch_im->height;
	}

	// JLBC: Perhaps is better to raise the exception always??
	if ((x_search_ini + x_search_size  + patch_im->width-1)>im->width)
		x_search_size -= (x_search_ini + x_search_size + patch_im->width-1) - im->width;

	if ((y_search_ini + y_search_size  + patch_im->height-1)>im->height)
		y_search_size -= (y_search_ini + y_search_size  + patch_im->height-1) - im->height;

	ASSERT_( (x_search_ini + x_search_size  + patch_im->width-1)<=im->width )
	ASSERT_( (y_search_ini + y_search_size  + patch_im->height-1)<=im->height )

	IplImage *result = cvCreateImage(cvSize(x_search_size+1,y_search_size+1),IPL_DEPTH_32F, 1);

	IplImage *ipl_ext;

	if (!entireImg)
	{
		ipl_ext = cvCreateImage(cvSize(patch_im->width+x_search_size,patch_im->height+y_search_size),IPL_DEPTH_8U, 1);
		for (unsigned int i = 0 ; i < (unsigned int)y_search_size ; i++)
		{
			memcpy( &ipl_ext->imageData[i * ipl_ext->widthStep ],
					&im->imageData[(i+y_search_ini) * im->widthStep + x_search_ini * im->nChannels],
					ipl_ext->width * ipl_ext->nChannels ); //widthStep);  <-- JLBC: widthstep SHOULD NOT be used as the length of each row (the last one may be shorter!!)
		}
	}
	else
	{
		ipl_ext = im;
	}

	// Compute cross correlation:
	//cvMatchTemplate(ipl_ext,patch_im,result,CV_TM_CCORR_NORMED);
	cvMatchTemplate(ipl_ext,patch_im,result,CV_TM_CCOEFF_NORMED);

	// Find the max point:
	cvMinMaxLoc(result,&mini,&max_val,&min_point,&max_point,NULL);
	x_max = max_point.x+x_search_ini+(round(patch_im->width-1)/2);
	y_max = max_point.y+y_search_ini+(round(patch_im->height-1)/2);

	// Free memory:
	if (!entireImg) cvReleaseImage( &ipl_ext );
	cvReleaseImage( &result );
#else
	THROW_EXCEPTION("The MRPT has been compiled with MRPT_HAS_OPENCV=0 !");
#endif

	MRPT_TRY_END;
}

/********************************************************************************************/
/********************************************FLIP********************************************/
void  vision::flip(CMRPTImage		&img)
{
	MRPT_TRY_START;

#if MRPT_HAS_OPENCV
	cvFlip((IplImage*)img.getAsIplImage());	// More params exists, they could be added in the future?
	img.setOriginTopLeft(true);
#else
	THROW_EXCEPTION("The MRPT has been compiled with MRPT_HAS_OPENCV=0 !");
#endif

	MRPT_TRY_END;
}

/****************************************** FAMD ********************************************/
void  vision::cloudsToMatchedList(
		const mrpt::slam::CObservationVisualLandmarks &cloud1,
		const mrpt::slam::CObservationVisualLandmarks &cloud2,
		mrpt::slam::CMetricMap::TMatchingPairList &outList)
{
	CLandmarksMap::TCustomSequenceLandmarks::const_iterator	itLand1, itLand2;
	CMetricMap::TMatchingPair								pair;

	for(itLand1 = cloud1.landmarks.landmarks.begin(); itLand1 != cloud1.landmarks.landmarks.end(); itLand1++)
		for(itLand2 = cloud2.landmarks.landmarks.begin(); itLand2 != cloud2.landmarks.landmarks.end(); itLand2++)
			if( itLand1->ID == itLand2->ID )
			{
				// Match found!
				pair.this_idx = pair.other_idx = (unsigned int)itLand1->ID;

				pair.other_x = itLand1->pose_mean_x;
				pair.other_y = itLand1->pose_mean_y;
				pair.other_z = itLand1->pose_mean_z;

				pair.this_x = itLand2->pose_mean_x;
				pair.this_y = itLand2->pose_mean_y;
				pair.this_z = itLand2->pose_mean_z;

				outList.push_back( pair );
			} // end if
}

float vision::computeMainOrientation( const CMRPTImage &image,
										   const unsigned int &x,
										   const unsigned int &y )
{
	MRPT_TRY_START;
	float orientation=0;
	if( ( x-1 >= 0 ) && ( y-1 >= 0 ) && ( x+1 < image.getWidth() ) && ( y+1 < image.getHeight() ) )
		orientation = (float) atan2( (double)*image(x,y+1) - (double)*image(x,y-1), (double)*image(x+1,y) - (double)*image(x-1,y) );

	// Convert from [-pi,pi] to [0,2pi]

	return orientation;


	MRPT_TRY_END;
} // end vision::computeMainOrientation
/*************************************** END FAMD ********************************************/
size_t vision::matchFeatures( const CFeatureList &list1,
									const CFeatureList &list2,
									CMatchedFeatureList &matches,
									const TMatchingOptions &options )
{
	//unsigned int		nMatches = 0;		// Number of matches found
	CTicTac				tictac;

	MRPT_TRY_START;

	// Preliminary comprobations
	ASSERT_( list1.get_type() == list2.get_type() );		// Both lists must be of the same type

	switch( options.matching_method )
	{
	case TMatchingOptions::mmDescriptorSIFT:
	{
		/***********************************************************************
		   SIFT Features -> Matching by Euclidean distance between descriptors
		************************************************************************/
		CFeatureList::const_iterator	itList1, itList2;	// Iterators for the lists

		float							distDesc;			// EDD
		float							minDist1 = 1e5;		// Minimum EDD
		float							minDist2 = 1e5;		// Second minimum EDD
		TFeatureID						minIdx = 0;			// Index of the closest feature

		map<TFeatureID,unsigned int>	featAssigned;		// Right feature status: 0 (idle) 1 (assigned) 2 (ambiguous)
		unsigned int					featIdx;			// Counter

		// For each feature in list1 ...
		for( itList1 = list1.begin(); itList1 != list1.end(); itList1++ )
		{
			minDist1 = 1e5;
			minDist2 = 1e5;
			minIdx	 = 0;
			for( itList2 = list2.begin(), featIdx = 0;
				 itList2 != list2.end();
				 itList2++, featIdx++ )					// ... compare with all the features in list2.
			{
				// Filter out by epipolar constraint (By now: row checking)
				if( fabs( (*itList1)->y - (*itList2)->y ) < options.epipolar_TH && fabs( (*itList1)->x - (*itList2)->x ) > 0 )
				{
					// Ensure that both features have SIFT descriptors
					ASSERT_((*itList1)->hasDescriptorSIFT && (*itList2)->hasDescriptorSIFT );

					// Compute the Euclidean distance between descriptors
					distDesc = (*itList1)->descriptorSIFTDistanceTo( *(*itList2) );

					// Search for the two minimum values
					if( distDesc < minDist1 )
					{
						minDist2 = minDist1;
						minDist1 = distDesc;
						minIdx	 = (*itList2)->ID;
					}
					else if ( distDesc < minDist2 )
						minDist2 = distDesc;
				}
			} // end for 'list2'

			// PROCESS THE RESULTS
			if( minDist1 < options.maxEDD_TH && 					// The minimum distance must be below a threshold
			  ( minDist1/minDist2) < options.EDD_RATIO )			// The difference between it and the second minimum distance must be over a threshold
			{
				switch( featAssigned[ minIdx ] )
				{
				case 0: // Right feature is free
				{
					// Create the match and insert it into the matched list
					std::pair<CFeaturePtr,CFeaturePtr> mPair( *itList1, list2.getByID( minIdx ) );
					matches.insert( mPair );
					featAssigned[ minIdx ] = 1;						// Set the right feature status to ASSIGNED (1)
					break;
				} // end FREE

				case 1: // Right feature is already assigned
				{
					// Not create the match and remove the old one
					CMatchedFeatureList::iterator			itVec;				// We set an iterator for the vector
					for( itVec = matches.begin(); itVec != matches.end(); )		// ... we search it into the vector
					{
						if( itVec->second->ID == minIdx )
						{
							itVec = matches.remove( itVec );
							break;
						}
						else
							itVec++;
					} // end for
					featAssigned[ minIdx ] = 2;						// Set the right feature status to AMBIGUOUS (2)
					break;
				} // end ASSIGNED

				default:{break;}
				} // end switch
			} // end if

		//	if( featAssigned[minIdx] != 2 &&						// If the selected feature is not ambiguous and ...
		//		minDist1 < options.maxEDD_TH && 					// The minimum distance must be below a threshold
		//		(minDist1/minDist2) < options.EDD_RATIO )			// The difference between it and the second minimum distance must be over a threshold
		//	{
		//		// Create the match and insert into the matched list
		//		std::pair<CFeature*,CFeature*> mPair( *itList1, list2.at( minIdx ) );
		//		matches.insert( mPair );
		//		featAssigned[minIdx] = 1;											// Set the right feature status to ASSIGNED (1)

		//		if( featAssigned[minIdx] == 1 )
		//		{
		//			// We have an ambiguous match (the feature in list2 is already assigned to another feature in list1
		//			// Delete the pair
		//			CMatchedFeatureList::iterator		itVec;							// We set an iterator for the vector
		//			for( itVec = matches.begin(); itVec != matches.end(); itVec++)		// ... we search it into the vector
		//			{
		//				if( itVec->second->ID == mPair.second->ID )
		//				{
		//					itVec->first->x = -1; itVec->first->y = -1;
		//					itVec->second->x = -1; itVec->second->y = -1;

		//					mPair.first->x = -1; mPair.first->y = -1;
		//					mPair.second->x = -1; mPair.second->y = -1;
		//					break;
		//				} // end if
		//				else
		//					THROW_EXCEPTION("Previous Match not found!");
		//			} // end for
		//			featAssigned[minIdx] = 2;											// Flag this feature as AMBIGUOUS (2)
		//		} // end if
		//		else
		//		{
		//			// Correct Correspondence so set it
		//			matches.insert( mPair );
		//			featAssigned[minIdx] = 1;											// Set the right feature status to ASSIGNED (1)
		//		} // end else
		//	} // end if
		//} // end for 'list1'
		//CMatchedFeatureList::iterator itMatch;
		//for(itMatch = matches.begin(); itMatch != matches.end();)
		//{
		//	if( (itMatch->first->x == -1) && (itMatch->first->y) == -1 &&
		//		(itMatch->second->x == -1) && (itMatch->second->y) == -1)
		//	{
		//		itMatch = matches.remove( itMatch );
		//	}
		//	else
		//		itMatch++;
		} // end for
		break;
	} // end case mmDescriptorSIFT

	case TMatchingOptions::mmCorrelation:
	{
		///***********************************************************************
		//   Matching by Cross Correlation
		//************************************************************************/
		CFeatureList::const_iterator	itList1, itList2;	// Iterators for the lists

		size_t							u,v;				// Coordinates of the peak
		double							res;				// Value of the peak

		double							maxCC1 = 0;			// Maximum CC
		double							maxCC2 = 0;			// Second maximum CC
		TFeatureID						minIdx = 0;			// Index of the closest feature

		map<TFeatureID,unsigned int>	featAssigned;		// Right feature status: 0 (idle) 1 (assigned) 2 (ambiguous)
		unsigned int					featIdx;			// Counter

		// For each feature in list1 ...
		unsigned int k = 0;
		for( itList1 = list1.begin(); itList1 != list1.end(); itList1++, k++ )
		{
			maxCC1	= 0;
			maxCC2	= 0;
			minIdx	= 0;
			for( itList2 = list2.begin(), featIdx = 0;
				 itList2 != list2.end();
				 itList2++, featIdx++ )			// ... compare with all the features in list2.
			{
				if( fabs( (*itList1)->y - (*itList2)->y ) < options.epipolar_TH && ( (*itList1)->x - (*itList2)->x ) > 0 ) // Quick filtering of bad correspondences
				{
					// Ensure that both features have patches
					ASSERT_(	(*itList1)->patch.getHeight() > 0 && (*itList1)->patch.getWidth() > 0 &&
								(*itList2)->patch.getHeight() > 0 && (*itList2)->patch.getWidth() > 0 );

					vision::openCV_cross_correlation( (*itList1)->patch, (*itList2)->patch, u, v, res );

					// Search for the two maximum values
					if( res > maxCC1 )
					{
						maxCC2 = maxCC1;
						maxCC1 = res;
						minIdx = (*itList2)->ID;
					}
					else if( res > maxCC2 )
						maxCC2 = res;

				} // end if EPIPOLAR CONSTRAINT
			} // end for "2"

			// PROCESS THE RESULTS
			if( maxCC1 > options.minCC_TH && 				// The minimum distance must be below a threshold
			   (maxCC2/maxCC1) < options.rCC_TH )			// The difference between it and the second minimum distance must be over a threshold
			{
				switch( featAssigned[ minIdx ] )
				{
				case 0: // Right feature is free
				{
					// Create the match and insert it into the matched list
					std::pair<CFeaturePtr,CFeaturePtr> mPair( *itList1, list2.getByID( minIdx ) );
					//mPair.first->patch.saveToFile( format( "./LOG/m%u_pLeft_%d.bmp", k, (int)mPair.first->ID ) );
					//mPair.second->patch.saveToFile( format( "./LOG/m%u_pRight_%d.bmp", k, (int)mPair.second->ID ) );

					//FILE *f = mrpt::system::os::fopen( format( "./LOG/m%u_NCC.txt", k ), "wt" );
					//fprintf( f, "%.4f %.4f", maxCC1, maxCC2 );
					//mrpt::system::os::fclose( f );

					matches.insert( mPair );
					featAssigned[ minIdx ] = 1;						// Set the right feature status to ASSIGNED (1)
					break;
				} // end FREE

				case 1: // Right feature is already assigned
				{
					// Not create the match and remove the old one
					CMatchedFeatureList::iterator			itVec;				// We set an iterator for the vector
					for( itVec = matches.begin(); itVec != matches.end(); )		// ... we search it into the vector
					{
						if( itVec->second->ID == minIdx )
						{
							//mrpt::system::deleteFile( format( "./LOG/m%u_pLeft_%d.bmp", k, (int)itVec->first->ID ) );
							//mrpt::system::deleteFile( format( "./LOG/m%u_pRight_%d.bmp", k, (int)itVec->second->ID ) );
							//mrpt::system::deleteFile( format( "./LOG/m%u_NCC.txt", k ) );

							itVec = matches.remove( itVec );
							break;
						}
						else
							itVec++;
					} // end for
					featAssigned[ minIdx ] = 2;						// Set the right feature status to AMBIGUOUS (2)
					break;
				} // end ASSIGNED

				default:{break;}
				} // end switch
			} // end if

		//	if( featAssigned[minIdx] != 2 &&						// If the selected feature is not ambiguous and ...
		//		maxCC1 > options.minCC_TH && 						// The maximum CC must be over a threshold
		//		(maxCC2/maxCC1) < options.rCC_TH )					// Maximum ratio between the two highest CC values
		//	{
		//		// Form the corresponding pair
		//		std::pair<CFeature*,CFeature*> mPair( *itList1, list2.at( minIdx ) );// We form the corresponding pair

		//		if( featAssigned[minIdx] == 1 )
		//		{
		//			// We have an ambiguous match (the feature in list2 is already assigned to another feature in list1
		//			// Delete the pair
		//			CMatchedFeatureList::iterator				itVec;						// We set an iterator for the vector
		//			for( itVec = matches.begin(); itVec != matches.end(); itVec++ )			// ... we search it into the vector
		//				if( itVec->second->ID == mPair.second->ID )
		//				{
		//					itVec->first->x = -1; itVec->first->y = -1;
		//					itVec->second->x = -1; itVec->second->y = -1;

		//					mPair.first->x = -1; mPair.first->y = -1;
		//					mPair.second->x = -1; mPair.second->y = -1;
		//					break;
		//				} // end if

		//			featAssigned[minIdx] = 2;							// Flag this feature as ambiguous
		//		} // end if
		//		else
		//		{
		//			matches.insert( mPair );
		//			featAssigned[minIdx] = 1;
		//		} // end else
		//	} // end if
		//	maxCC1 = 0;
		//	maxCC2 = 0;
		//} // end for "1"
		//CMatchedFeatureList::iterator itMatch;
		//for(itMatch = matches.begin(); itMatch != matches.end();)
		//{
		//	if( (itMatch->first->x == -1) && (itMatch->first->y) == -1 &&
		//		(itMatch->second->x == -1) && (itMatch->second->y) == -1)
		//	{
		//		itMatch = matches.remove( itMatch );
		//	}
		//	else
		//		itMatch++;
		} // end for
		break;
		} // end case mmCorrelation

		case TMatchingOptions::mmDescriptorSURF:
		{
			cout << "To do ... [MATCH SURF FEATURES]" << endl;
			break; // end case featSURF
		}
		default:
		{
			THROW_EXCEPTION( "Invalid type of feature when matching" );
		}
	} // end switch

	return matches.size();

	MRPT_TRY_END;
}
/*-------------------------------------------------------------
			TStereoSystemParams: constructor
-------------------------------------------------------------*/
TStereoSystemParams::TStereoSystemParams() :
	uncPropagation(Prop_Linear),
	K(3,3),
	baseline ( 0.119f ),	// Bumblebee
	stdPixel ( 1 ),
	stdDisp  ( 1 ),
	maxZ	 ( 20.0f ),	// Indoor
	minZ	 ( 0.5f ),	// Indoor
	maxY	 ( 3.0f ),	// Indoor
	factor_k ( 1.5f ),
	factor_a ( 1e-3f ),
	factor_b ( 2.0f )
{
}

/*-------------------------------------------------------------
			TStereoSystemParams: loadFromConfigFile
-------------------------------------------------------------*/
void  TStereoSystemParams::loadFromConfigFile(
			const mrpt::utils::CConfigFileBase	&iniFile,
			const std::string		&section)
{
	int unc;
	unc				= iniFile.read_int(section.c_str(),"uncPropagation",uncPropagation);
	switch (unc)
	{
		case 0:
			uncPropagation = Prop_Linear;
			break;
		case 1:
			uncPropagation = Prop_UT;
			break;
		case 2:
			uncPropagation = Prop_SUT;
			break;
	} // end switch

	K(0,0)			= iniFile.read_float(section.c_str(),"k00",K(0,0));
	K(0,1)			= iniFile.read_float(section.c_str(),"k01",K(0,1));
	K(0,2)			= iniFile.read_float(section.c_str(),"k02",K(0,2));
	K(1,0)			= iniFile.read_float(section.c_str(),"k10",K(1,0));
	K(1,1)			= iniFile.read_float(section.c_str(),"k11",K(1,1));
	K(1,2)			= iniFile.read_float(section.c_str(),"k12",K(1,2));
	K(2,0)			= iniFile.read_float(section.c_str(),"k20",K(2,0));
	K(2,1)			= iniFile.read_float(section.c_str(),"k21",K(2,1));
	K(2,2)			= iniFile.read_float(section.c_str(),"k22",K(2,2));

	baseline		= iniFile.read_float(section.c_str(),"baseline",baseline);
	stdPixel		= iniFile.read_float(section.c_str(),"stdPixel",stdPixel);
	stdDisp			= iniFile.read_float(section.c_str(),"stdDisp",stdDisp);
	maxZ			= iniFile.read_float(section.c_str(),"maxZ",maxZ);
	minZ			= iniFile.read_float(section.c_str(),"minZ",minZ);
	maxY			= iniFile.read_float(section.c_str(),"maxY",maxY);
	factor_k		= iniFile.read_float(section.c_str(),"factor_k",factor_k);
	factor_a		= iniFile.read_float(section.c_str(),"factor_a",factor_a);
	factor_b		= iniFile.read_float(section.c_str(),"factor_b",factor_b);
} // end of loadFromConfigFile

/*---------------------------------------------------------------
					TStereoSystemParams: dumpToTextStream
  ---------------------------------------------------------------*/
void  TStereoSystemParams::dumpToTextStream(CStream	&out)
{
	out.printf("\n----------- [vision::TStereoSystemParams] ------------ \n\n");

	out.printf("Method for 3D Uncert. \t= ");
	switch (uncPropagation)
	{
	case Prop_Linear:
		out.printf("Linear propagation\n");
		break;
	case Prop_UT:
		out.printf("Unscented Transform\n");
		break;
	case Prop_SUT:
		out.printf("Scaled Unscented Transform\n");
		break;
	} // end switch

	out.printf("K\t\t\t= [%f\t%f\t%f]\n", K(0,0), K(0,1), K(0,2));
	out.printf(" \t\t\t  [%f\t%f\t%f]\n", K(1,0), K(1,1), K(1,2));
	out.printf(" \t\t\t  [%f\t%f\t%f]\n", K(2,0), K(2,1), K(2,2));

	out.printf("Baseline \t\t= %f\n", baseline);
	out.printf("Pixel std \t\t= %f\n", stdPixel);
	out.printf("Disparity std\t\t= %f\n", stdDisp);
	out.printf("Z maximum\t\t= %f\n", maxZ);
	out.printf("Z minimum\t\t= %f\n", minZ);
	out.printf("Y maximum\t\t= %f\n", maxY);

	out.printf("k Factor [UT]\t\t= %f\n", factor_k);
	out.printf("a Factor [UT]\t\t= %f\n", factor_a);
	out.printf("b Factor [UT]\t\t= %f\n", factor_b);
}
/*-------------------------------------------------------------
			TMatchingOptions: constructor
-------------------------------------------------------------*/
TMatchingOptions::TMatchingOptions() :
	// General
	matching_method ( mmCorrelation ),	// Matching method
	epipolar_TH	( 1.5f ),				// Epipolar constraint (rows of pixels)
	rowCheck_TH ( 1.0f ),				// Rows of pixels for row checking when using vOdometry
	ccCheck_TH ( 0.88 ),				// Minimum cross correlation between

	// SIFT
	maxEDD_TH	( 90 ),					// Maximum Euclidean Distance Between Descriptors
	EDD_RATIO   ( 0.6 ),				// Ratio between the two smallest EDD

	// KLT
	minCC_TH	( 0.95f ),				// Minimum Value of the Cross Correlation
	minDCC_TH	( 0.025f ),				// Minimum Difference Between the Maximum Cross Correlation Values
	rCC_TH		( 0.92f )				// Maximum Ratio Between the two highest CC values
{
} // end constructor TMatchingOptions

/*-------------------------------------------------------------
			TMatchingOptions: loadFromConfigFile
-------------------------------------------------------------*/
void  TMatchingOptions::loadFromConfigFile(
			const mrpt::utils::CConfigFileBase	&iniFile,
			const std::string		&section)
{
	int mm			= iniFile.read_int(section.c_str(),"matching_method",matching_method);
	switch( mm )
	{
	case 0:
		matching_method = mmCorrelation;
		break;
	case 1:
		matching_method = mmDescriptorSIFT;
		break;
	case 2:
		matching_method = mmDescriptorSURF;
		break;
	} // end switch

	epipolar_TH		= iniFile.read_float(section.c_str(),"epipolar_TH",epipolar_TH);
	maxEDD_TH		= iniFile.read_float(section.c_str(),"maxEDD_TH",maxEDD_TH);
	EDD_RATIO		= iniFile.read_float(section.c_str(),"minDIF_TH",EDD_RATIO);
	minCC_TH		= iniFile.read_float(section.c_str(),"minCC_TH",minCC_TH);
	minDCC_TH		= iniFile.read_float(section.c_str(),"minDCC_TH",minDCC_TH);
	rCC_TH			= iniFile.read_float(section.c_str(),"rCC_TH",rCC_TH);
	rowCheck_TH		= iniFile.read_float(section.c_str(),"rowCheck_TH",rowCheck_TH);
	ccCheck_TH		= iniFile.read_float(section.c_str(),"ccCheck_TH",ccCheck_TH);
} // end TMatchingOptions::loadFromConfigFile

/*---------------------------------------------------------------
					TMatchingOptions: dumpToTextStream
  ---------------------------------------------------------------*/
void  TMatchingOptions::dumpToTextStream(CStream	&out)
{
	out.printf("\n----------- [vision::TMatchingOptions] ------------ \n\n");

	out.printf("Matching method	\t= ");
	switch( matching_method )
	{
	case mmCorrelation:
		out.printf("Cross Correlation\n");
		break;
	case mmDescriptorSIFT:
		out.printf("SIFT descriptor\n");
		break;
	case mmDescriptorSURF:
		out.printf("SURF descriptor\n");
		break;
	} // end switch
	out.printf("Epipolar Thres.\t\t= %f\n", epipolar_TH);
	out.printf("Row Checking Thres.\t= %f\n", rowCheck_TH);
	out.printf("CC Checking Thres.\t= %f\n", ccCheck_TH);
	out.printf("Max. EDD Thres.\t\t= %f\n", maxEDD_TH);
	out.printf("EDD Ratio\t\t= %f\n", EDD_RATIO);
	out.printf("Min. CC Thres.\t\t= %f\n", minCC_TH);
	out.printf("Min. Dif. CC Thres.\t= %f\n", minDCC_TH);
	out.printf("Max. Ratio CC Thres.\t= %f\n", rCC_TH);
} // end TMatchingOptions::dumpToTextStream

/*-------------------------------------------------------------
					colormap
-------------------------------------------------------------*/
void MRPTDLLIMPEXP vision::colormap(
	const TColormap &color_map,
	const float	&color_index,
	float	&r,
	float	&g,
	float	&b)
{
	MRPT_TRY_START
	switch (color_map)
	{
		case cmJET:
			jet2rgb(color_index,r,g,b);
			break;
		case cmGRAYSCALE:
			r = g = b = color_index;
			break;
		default:
			THROW_EXCEPTION("Invalid color_map");
	};
	MRPT_TRY_END
}

/*-------------------------------------------------------------
					jet2rgb
-------------------------------------------------------------*/
void  vision::jet2rgb(
	const float	&color_index,
	float	&r,
	float	&g,
	float	&b)
{
	static bool	jet_table_done = false;
	static vector_float	jet_r,jet_g,jet_b;


	// Initialize tables
	if (!jet_table_done)
	{
		jet_table_done = true;

		// Refer to source code of "jet" in MATLAB:
		float JET_R[] = { 0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.062500,0.125000,0.187500,0.250000,0.312500,0.375000,0.437500,0.500000,0.562500,0.625000,0.687500,0.750000,0.812500,0.875000,0.937500,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,0.937500,0.875000,0.812500,0.750000,0.687500,0.625000,0.562500,0.500000 };
		float JET_G[] = { 0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.062500,0.125000,0.187500,0.250000,0.312500,0.375000,0.437500,0.500000,0.562500,0.625000,0.687500,0.750000,0.812500,0.875000,0.937500,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,0.937500,0.875000,0.812500,0.750000,0.687500,0.625000,0.562500,0.500000,0.437500,0.375000,0.312500,0.250000,0.187500,0.125000,0.062500,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000 };
		float JET_B[] = { 0.562500,0.625000,0.687500,0.750000,0.812500,0.875000,0.937500,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,1.000000,0.937500,0.875000,0.812500,0.750000,0.687500,0.625000,0.562500,0.500000,0.437500,0.375000,0.312500,0.250000,0.187500,0.125000,0.062500,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000,0.000000 };
		const size_t N = sizeof(JET_B)/sizeof(JET_B[0]);

		jet_r.resize(N);
		jet_g.resize(N);
		jet_b.resize(N);
		for (size_t i=0;i<N;i++)
		{
			jet_r[i] = JET_R[i];
			jet_g[i] = JET_G[i];
			jet_b[i] = JET_B[i];
		}
	}

	// Return interpolate value:
	r = math::interpolate(color_index, jet_r, 0.0f,1.0f);
	g = math::interpolate(color_index, jet_g, 0.0f,1.0f);
	b = math::interpolate(color_index, jet_b, 0.0f,1.0f);
}

