/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 



#include <mrpt/slam/CBeacon.h>
#include <mrpt/slam/CBeaconMap.h>
#include <mrpt/slam/CObservation.h>

#include <mrpt/system/os.h>
#include <mrpt/math/geometry.h>
#include <mrpt/opengl.h>

using namespace mrpt;
using namespace mrpt::slam; using namespace mrpt::utils; using namespace mrpt::poses;
using namespace mrpt::utils;
using namespace std;

IMPLEMENTS_SERIALIZABLE(CBeacon, CSerializable,mrpt::slam)


/*---------------------------------------------------------------
						Default constructor
  ---------------------------------------------------------------*/
CBeacon::CBeacon( ) :
	m_typePDF(pdfGauss),
	m_locationMC(1),
	m_locationGauss(),
	m_locationSOG(1),
	m_ID(INVALID_BEACON_ID)
{
}

/*---------------------------------------------------------------
						Destructor
  ---------------------------------------------------------------*/
CBeacon::~CBeacon()
{
}

/*---------------------------------------------------------------
					writeToStream
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CBeacon::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 0;
	else
	{
		uint32_t	i = m_ID;
		uint32_t	j = m_typePDF;
		out << i << j << m_locationMC << m_locationGauss << m_locationSOG;
	}
}

/*---------------------------------------------------------------
					readFromStream
   Implements the reading from a CStream capability of
      CSerializable objects
  ---------------------------------------------------------------*/
void  CBeacon::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
		{
			uint32_t	i,j;
			in >> i >> j >> m_locationMC >> m_locationGauss >> m_locationSOG;
			m_ID = i;
			m_typePDF = static_cast<TTypePDF>(j);
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
					getEstimatedPoint
  ---------------------------------------------------------------*/
CPoint3D  CBeacon::getEstimatedPoint() const
{
	MRPT_TRY_START;
	switch (m_typePDF)
	{
	case pdfMonteCarlo:	return m_locationMC.getEstimatedPoint();		break;
	case pdfGauss:		return m_locationGauss.getEstimatedPoint();		break;
	case pdfSOG:		return m_locationSOG.getEstimatedPoint();		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};
	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					getEstimatedCovariance
  ---------------------------------------------------------------*/
CMatrixD  CBeacon::getEstimatedCovariance() const
{
	MRPT_TRY_START;
	switch (m_typePDF)
	{
	case pdfMonteCarlo:	return m_locationMC.getEstimatedCovariance();		break;
	case pdfGauss:		return m_locationGauss.getEstimatedCovariance();	break;
	case pdfSOG:		return m_locationSOG.getEstimatedCovariance();		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};
	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					bayesianFusion
  ---------------------------------------------------------------*/
void CBeacon::bayesianFusion( CPointPDF &p1, CPointPDF &p2, const double &minMahalanobisDistToDrop)
{
	MRPT_TRY_START;
	switch (m_typePDF)
	{
	case pdfMonteCarlo:	m_locationMC.bayesianFusion(p1,p2,minMahalanobisDistToDrop);	break;
	case pdfGauss:		m_locationGauss.bayesianFusion(p1,p2,minMahalanobisDistToDrop);	break;
	case pdfSOG:		m_locationSOG.bayesianFusion(p1,p2,minMahalanobisDistToDrop);		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};
	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					drawSingleSample
  ---------------------------------------------------------------*/
void CBeacon::drawSingleSample(CPoint3D &outSample) const
{
	MRPT_TRY_START;
	switch (m_typePDF)
	{
	case pdfMonteCarlo:	m_locationMC.drawSingleSample(outSample);		break;
	case pdfGauss:		m_locationGauss.drawSingleSample(outSample);	break;
	case pdfSOG:		m_locationSOG.drawSingleSample(outSample);		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};
	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					copyFrom
  ---------------------------------------------------------------*/
void  CBeacon::copyFrom(const CPointPDF &o)
{
	MRPT_TRY_START;
	switch (m_typePDF)
	{
	case pdfMonteCarlo:	m_locationMC.copyFrom(o);		break;
	case pdfGauss:		m_locationGauss.copyFrom(o);	break;
	case pdfSOG:		m_locationSOG.copyFrom(o);		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};
	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					saveToTextFile
  ---------------------------------------------------------------*/
void  CBeacon::saveToTextFile(const std::string &file) const
{
	MRPT_TRY_START;
	switch (m_typePDF)
	{
	case pdfMonteCarlo:	m_locationMC.saveToTextFile(file);		break;
	case pdfGauss:		m_locationGauss.saveToTextFile(file);	break;
	case pdfSOG:		m_locationSOG.saveToTextFile(file);		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};
	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					changeCoordinatesReference
  ---------------------------------------------------------------*/
void  CBeacon::changeCoordinatesReference( const CPose3D &newReferenceBase )
{
	MRPT_TRY_START;
	switch (m_typePDF)
	{
	case pdfMonteCarlo:	m_locationMC.changeCoordinatesReference(newReferenceBase);		break;
	case pdfGauss:		m_locationGauss.changeCoordinatesReference(newReferenceBase);	break;
	case pdfSOG:		m_locationSOG.changeCoordinatesReference(newReferenceBase);		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};
	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					getAs3DObject
  ---------------------------------------------------------------*/
void  CBeacon::getAs3DObject( mrpt::opengl::CSetOfObjectsPtr	&outObj ) const
{
	MRPT_TRY_START;

	switch (m_typePDF)
	{
	case pdfMonteCarlo:
		{
			opengl::CPointCloudPtr obj = opengl::CPointCloud::Create();
			obj->m_color_R = 1;
			obj->m_color_G = 0;
			obj->m_color_B = 0;

			obj->m_pointSize = 2.5f;

			size_t	i,N = m_locationMC.m_particles.size();
			obj->m_xs.resize( N );
			obj->m_ys.resize( N );
			obj->m_zs.resize( N );

			for (i=0;i<N;i++)
			{
				obj->m_xs[i] = m_locationMC.m_particles[i].d->x;
				obj->m_ys[i] = m_locationMC.m_particles[i].d->y;
				obj->m_zs[i] = m_locationMC.m_particles[i].d->z;
			}

			outObj->insert( obj );
		}
		break;
	case pdfGauss:
		{
			opengl::CEllipsoidPtr obj = opengl::CEllipsoid::Create();

			obj->m_x = m_locationGauss.mean.x;
			obj->m_y = m_locationGauss.mean.y;
			obj->m_z = m_locationGauss.mean.z;

			obj->m_lineWidth = 3;

			obj->m_cov = m_locationGauss.cov;
			if (obj->m_cov(2,2)==0)
				obj->m_cov.setSize(2,2);

			obj->m_quantiles = 3;
			obj->m_drawSolid3D=false;

			obj->m_color_R = 1;
			obj->m_color_G = 0;
			obj->m_color_B = 0;
			obj->m_color_A = 0.85f;

			outObj->insert( obj );
		}
		break;
	case pdfSOG:
		{
			m_locationSOG.getAs3DObject( outObj );
		}
		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};

	opengl::CTextPtr obj2 = opengl::CText::Create();
	char str[100];
	os::sprintf(str,sizeof(str),"#%d",static_cast<int>(m_ID));
	obj2->m_str =std::string(str);

	CPoint3D	meanP( getEstimatedPoint() );

	obj2->m_x = meanP.x+0.10f;
	obj2->m_y = meanP.y+0.10f;
	obj2->m_z = meanP.z;
	outObj->insert( obj2 );

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					getAsMatlabDrawCommands
  ---------------------------------------------------------------*/
void CBeacon::getAsMatlabDrawCommands( utils::CStringList &out_Str  ) const
{
	MRPT_TRY_START;

	out_Str.clear();
	char		auxStr[1000];

	switch (m_typePDF)
	{
	case pdfMonteCarlo:
		{
			// xs=[...];
			// ys=[...];
			// plot(xs,ys,'.','MarkerSize',4);
			size_t			i,N = m_locationMC.m_particles.size();
			std::string		sx,sy;

			sx= "xs=[";
			sy= "ys=[";
			for (i=0;i<N;i++)
			{
				os::sprintf(auxStr,sizeof(auxStr),"%.3f%c",m_locationMC.m_particles[i].d->x, (i==N-1) ? ' ':',' );
				sx=sx+std::string(auxStr);
				os::sprintf(auxStr,sizeof(auxStr),"%.3f%c",m_locationMC.m_particles[i].d->y, (i==N-1) ? ' ':',' );
				sy=sy+std::string(auxStr);
			}
			sx=sx+"];";
			sy=sy+"];";
			out_Str.add(sx);
			out_Str.add(sy);
			out_Str.add(std::string("plot(xs,ys,'k.','MarkerSize',4);"));
		}
		break;
	case pdfGauss:
		{
			// m=[x y];
			// C=[2x2]
			// error_ellipse(C,m,'conf',0.997,'style','k');

			os::sprintf(auxStr,sizeof(auxStr),"m=[%.3f %.3f];",m_locationGauss.mean.x,m_locationGauss.mean.y);
			out_Str.add(std::string(auxStr));
			os::sprintf(auxStr,sizeof(auxStr),"C=[%e %e;%e %e];",
				m_locationGauss.cov(0,0),m_locationGauss.cov(0,1),
				m_locationGauss.cov(1,0),m_locationGauss.cov(1,1) );
			out_Str.add(std::string(auxStr));

			out_Str.add(std::string("error_ellipse(C,m,'conf',0.997,'style','k');"));
		}
		break;
	case pdfSOG:
		{
			CPointPDFSOG::CListGaussianModes::const_iterator it;
		    for ( it = m_locationSOG.m_modes.begin(); it!= m_locationSOG.m_modes.end();it++)
			{
				os::sprintf(auxStr,sizeof(auxStr),"m=[%.3f %.3f];",it->val.mean.x,it->val.mean.y);
				out_Str.add(std::string(auxStr));
				os::sprintf(auxStr,sizeof(auxStr),"C=[%e %e;%e %e];",
					it->val.cov(0,0),it->val.cov(0,1),
					it->val.cov(1,0),it->val.cov(1,1) );
				out_Str.add(std::string(auxStr));
				out_Str.add(std::string("error_ellipse(C,m,'conf',0.997,'style','k');"));
			}
		}
		break;
	default:			THROW_EXCEPTION("ERROR: Invalid 'm_typePDF' value");
	};

	// The text:
	CPoint3D	meanP( getEstimatedPoint() );

	os::sprintf(auxStr,sizeof(auxStr),"text(%f,%f,'#%i');",meanP.x,meanP.y, static_cast<int>(m_ID) );
	out_Str.add(std::string(auxStr));

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					generateObservationModelDistribution
 Compute the observation model p(z_t|x_t) for a given observation (range value), and return it as an approximate SOG.
*  Note that if the beacon is a SOG itself, the number of gaussian modes will be square.
*  As a speed-up, if a "center point"+"maxDistanceFromCenter" is supplied (maxDistanceFromCenter!=0), those modes farther than this sphere will be discarded.
*  Parameters such as the stdSigma of the sensor are gathered from "myBeaconMap"
*  The result is one "ring" for each Gaussian mode that represent the beacon position in this object.
*  The position of the sensor on the robot is used to shift the resulting densities such as they represent the position of the robot, not the sensor.
*  \sa CBeaconMap::insertionOptions, generateRingSOG

  ---------------------------------------------------------------*/
void CBeacon::generateObservationModelDistribution(
	const float &sensedRange,
	CPointPDFSOG	&outPDF,
	const CBeaconMap *myBeaconMap,
	const CPoint3D	&sensorPntOnRobot,
	const CPoint3D &centerPoint,
	const float &maxDistanceFromCenter) const
{
	MRPT_TRY_START


	const CPointPDFSOG *beaconPos=NULL;

	if ( m_typePDF==pdfGauss )
	{
		// Copy the gaussian to the SOG:
		CPointPDFSOG *new_beaconPos= new CPointPDFSOG(1);
		new_beaconPos->m_modes[0].log_w=0;
		new_beaconPos->m_modes[0].val = m_locationGauss;
		beaconPos = new_beaconPos;
	}
	else
	{
		ASSERT_( m_typePDF==pdfSOG )
		beaconPos = static_cast<const CPointPDFSOG*> (&m_locationSOG );
	}

	outPDF.m_modes.clear();

	CPointPDFSOG::CListGaussianModes::const_iterator it;
	for ( it = beaconPos->m_modes.begin(); it!= beaconPos->m_modes.end();it++)
	{
		// The center of the ring to be generated
		CPoint3D ringCenter(
			it->val.mean.x - sensorPntOnRobot.x,
			it->val.mean.y - sensorPntOnRobot.y,
			it->val.mean.z - sensorPntOnRobot.z  ) ;

		size_t startIdx = outPDF.m_modes.size();

		CBeacon::generateRingSOG(
			sensedRange,                    // Sensed range
			outPDF,                         // The ouput (Append all !!)
			myBeaconMap,                    // For params
			ringCenter,                     // The center of the ring to be generated
			&it->val.cov,					// The covariance to ADD to each mode, due to the composition of uncertainty
			false,                          // clearPreviousContentsOutPDF
			centerPoint,
			maxDistanceFromCenter           // Directly, do not create too far modes
			);

		// Adjust the weights to the one of "this" mode:
		for (size_t k=startIdx;k<outPDF.m_modes.size();k++)
			outPDF.m_modes[k].log_w = it->log_w;
	}

	if ( m_typePDF==pdfGauss )
		delete beaconPos;

	MRPT_TRY_END
}


/*---------------------------------------------------------------
					generateRingSOG
  ---------------------------------------------------------------*/
void CBeacon::generateRingSOG(
	const float 	&R,
	CPointPDFSOG	&outPDF,
	const CBeaconMap *myBeaconMap,
	const CPoint3D	&sensorPnt,
	const CMatrixD   *covarianceCompositionToAdd,
	bool  clearPreviousContentsOutPDF,
	const CPoint3D &centerPoint,
	const float &maxDistanceFromCenter
	)
{
	MRPT_TRY_START

	ASSERT_(myBeaconMap)

	if (covarianceCompositionToAdd)
		ASSERT_(covarianceCompositionToAdd->getColCount() ==3 && covarianceCompositionToAdd->getRowCount()==3 )

	// Compute the number of Gaussians:
	float	minEl = DEG2RAD(myBeaconMap->insertionOptions.minElevation_deg);
	float	maxEl = DEG2RAD(myBeaconMap->insertionOptions.maxElevation_deg);
	ASSERT_(myBeaconMap->insertionOptions.minElevation_deg<=myBeaconMap->insertionOptions.maxElevation_deg)
	size_t	B;  // B: Number of gaussians in each cut to the sphere (XY,XZ,...)
	double      el,th,A_ang;
	float	maxDistBetweenGaussians = myBeaconMap->insertionOptions.SOG_maxDistBetweenGaussians ;  // Meters



	B = (size_t)(M_2PIf * R / maxDistBetweenGaussians ) + 1;

	// Assure minimum B (maximum angular increment):
	B = max(B, (size_t)30);

	// B must be even:
	if (B%2) B++;

	A_ang = M_2PI/B;  // Angular increments between modes:



	// The diagonal basic covariance matrix.
	//  (0,0) is the variance in the direction "in->out" of the sphere
	//  (1,1),(2,2) is the var. in the two directions tangent to the sphere
	CMatrixD		S(3,3);
	S.zeros();
	S(0,0) = square( myBeaconMap->likelihoodOptions.rangeStd );
	S(1,1) = S(2,2) = square( A_ang*R / myBeaconMap->insertionOptions.SOG_separationConstant  );  //4.0f * sensedRange * S(0,0);

	CPoint3D	dir;

	// Create the SOG:
	size_t modeIdx;
	if (clearPreviousContentsOutPDF)
	{
		// Overwrite modes:
		modeIdx = 0;
		outPDF.m_modes.resize(B*B);
	}
	else
	{
		// Append modes:
		modeIdx = outPDF.m_modes.size(); // Start here
		outPDF.m_modes.resize(outPDF.m_modes.size()+B*B);
	}

	size_t idxEl, idxTh;  // idxEl:[0,B/2+1]

	for (idxEl=0;idxEl<=(1+B/2);idxEl++)
	{
		el = minEl + idxEl*A_ang;
		if (el>(maxEl+0.5*A_ang)) continue;

		size_t nThSteps = B;
		// Except if we are in the top/bottom of the sphere:
		if (fabs(cos(el))<1e-4)
		{
			nThSteps=1;
		}

		for (idxTh=0;idxTh<nThSteps;idxTh++)
		{
			th = idxTh*A_ang;

			// Compute the mean of the new Gaussian:
			dir.x = (sensorPnt.x + R*cos(th)*cos(el));
			dir.y = (sensorPnt.y + R*sin(th)*cos(el));
			dir.z = (sensorPnt.z + R*sin(el));

			// If we are provided a radius for not creating modes out of it, check it:
			bool reallyCreateIt = true;
			if (maxDistanceFromCenter>0)
				reallyCreateIt = dir.distanceTo( centerPoint ) < maxDistanceFromCenter;

			if (reallyCreateIt)
			{
				// All have equal log-weights:
				outPDF.m_modes[modeIdx].log_w = 0;

				// The mean:
				outPDF.m_modes[modeIdx].val.mean = dir;

				// Compute the covariance:
				dir = dir - sensorPnt;
				CMatrixD		H( math::generateAxisBaseFromDirection( dir.x,dir.y,dir.z ) );	// 3 perpendicular & normalized vectors.
				H.multiplyByMatrixAndByTranspose(
					S,
					outPDF.m_modes[modeIdx].val.cov ); // out = H * S * ~H;
				if (minEl==maxEl)
				{   // We are in 2D:
					// 3rd column/row = 0
					outPDF.m_modes[modeIdx].val.cov.setSize(2,2);
					outPDF.m_modes[modeIdx].val.cov.setSize(3,3);
				}

				// Add covariance for uncertainty composition?
				if (covarianceCompositionToAdd)
					outPDF.m_modes[modeIdx].val.cov += *covarianceCompositionToAdd;

				// One more mode is used:
				modeIdx++;

			} // end if reallyCreateIt

		} // end for idxTh
	} // end for idxEl

	// resize to the number of really used modes:
	outPDF.m_modes.resize(modeIdx);

	MRPT_TRY_END
}
