/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/opengl/CSetOfTriangles.h>
#include "opengl_internals.h"


using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CSetOfTriangles, CRenderizable, mrpt::opengl )


/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void   CSetOfTriangles::render()
{
#if MRPT_HAS_OPENGL_GLUT

	if (m_enableTransparency)
	{
		//glDisable(GL_DEPTH_TEST);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}
    else
    {
		glEnable(GL_DEPTH_TEST);
		glDisable(GL_BLEND);
    }

	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
    glEnable(GL_COLOR_MATERIAL);
    glShadeModel(GL_SMOOTH);

	vector<TTriangle>::iterator	it;

	glBegin(GL_TRIANGLES);

	for (it=m_triangles.begin();it!=m_triangles.end();it++)
	{
        // Compute the normal vector:
        // ---------------------------------
        float	ax= it->x[1] - it->x[0];
        float	ay= it->y[1] - it->y[0];
        float	az= it->z[1] - it->z[0];

        float	bx= it->x[2] - it->x[0];
        float	by= it->y[2] - it->y[0];
        float	bz= it->z[2] - it->z[0];

        glNormal3f(ay*bz-az*by,-ax*bz+az*bx,ax*by-ay*bx);

		glColor4f( it->r[0],it->g[0],it->b[0],it->a[0] );
		glVertex3f(it->x[0],it->y[0],it->z[0]);

		glColor4f( it->r[1],it->g[1],it->b[1],it->a[1] );
		glVertex3f(it->x[1],it->y[1],it->z[1]);

		glColor4f( it->r[2],it->g[2],it->b[2],it->a[2] );
		glVertex3f(it->x[2],it->y[2],it->z[2]);
	}

	glEnd();

	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);
#endif
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CSetOfTriangles::writeToStream(CStream &out,int *version) const
{

	if (version)
		*version = 1;
	else
	{
		writeToStreamRender(out);
		uint32_t	n = (uint32_t)m_triangles.size();
		out << n;
		if (n)
			out.WriteBuffer( &m_triangles[0], n*sizeof(TTriangle) );

		// Version 1:
		out << m_enableTransparency;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CSetOfTriangles::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
	case 1:
		{
			readFromStreamRender(in);
			uint32_t	n;
			in >> n;
			m_triangles.resize(n);
			if (n)
				in.ReadBuffer( &m_triangles[0], n*sizeof(TTriangle) );

			if (version>=1)
					in >> m_enableTransparency;
			else	m_enableTransparency = true;
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

