/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/math/CVectorTemplate.h>

#include <mrpt/opengl/CMesh.h>

#include <mrpt/vision/utils.h>
#include "opengl_internals.h"

using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CMesh, CRenderizable, mrpt::opengl )

/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void   CMesh::render()
{
#if MRPT_HAS_OPENGL_GLUT

	if (m_enableTransparency)
	{
		//glDisable(GL_DEPTH_TEST);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}
    else
    {
		glEnable(GL_DEPTH_TEST);
		glDisable(GL_BLEND);
    }

	//glEnable(GL_LIGHTING);
	//glEnable(GL_LIGHT0);
	glEnable(GL_COLOR_MATERIAL);
	glShadeModel(GL_SMOOTH);

	// Variable declaration
	float	x0,y0,z0,x1,y1,z1,x2,y2,z2;
	float	cR0=0,cG0=0,cB0=0;
	float	cR1=0,cG1=0,cB1=0;
	float	cR2=0,cG2=0,cB2=0;

	const size_t cols = Z.getColCount();
	const size_t rows = Z.getRowCount();
	bool	useMask = false;

	// Look for the maximum height
	if(m_colorFromZ)
	{
		// Make sure C matrix is updated:
		updateColorsMatrix();
	}
	else
	{
		//z_max = 1.0f;
		cR0 = cR1 = cR2 = m_color_R;
		cG0 = cG1 = cG2 = m_color_G;
		cB0 = cB1 = cB2 = m_color_B;
	}

	// Preliminary comprobations
	ASSERT_(cols > 0 && rows > 0);
	ASSERT_(xMax > xMin && yMax > yMin);

	if( mask.getColCount() != 0 && mask.getRowCount() != 0 )
	{
		ASSERT_( mask.getColCount() == cols && mask.getRowCount() == rows );
		useMask = true;
	}

	// Cell size (must be equal for squared grids)
	const float	sCellX = (xMax-xMin)/rows;
	const float	sCellY = (yMax-yMin)/cols;

	// Variables for computing the normal vector of each triangle
	float	ax, ay, az, bx, by, bz;

	const mrpt::vision::TColormap	the_color_map = m_colorMap;

	if (!m_isWireFrame)
		glBegin(GL_TRIANGLES);

	for(size_t i=0; i<rows-1; i++)
	{
		for(size_t j=0; j<cols-1; j++)
		{
			if( (useMask && mask(i,j) &&  mask(i+1,j) && mask(i,j+1)) || !useMask )
			{
				x0 = xMin + sCellX*( i + 0.5f );
				y0 = yMin + sCellY*( j + 0.5f );
				z0 = Z(i,j);
				if(m_colorFromZ)
					mrpt::vision::colormap(the_color_map,C(i,j),cR0,cG0,cB0);

				x1 = xMin + sCellX*( i+1 + 0.5f );
				y1 = yMin + sCellY*( j + 0.5f );
				z1 = Z(i+1,j);
				if(m_colorFromZ)
					mrpt::vision::colormap(the_color_map,C(i+1,j),cR1,cG1,cB1);

				x2 = xMin + sCellX*( i + 0.5f );
				y2 = yMin + sCellY*( j+1 + 0.5f );
				z2 = Z(i,j+1);
				if(m_colorFromZ)
					mrpt::vision::colormap(the_color_map,C(i,j+1),cR2,cG2,cB2);

				ax= x1 - x0;
				ay= y1 - y0;
				az= z1 - z0;

				bx= x2 - x0;
				by= y2 - y0;
				bz= z2 - z0;

				if (m_isWireFrame)
					glBegin(GL_LINE_LOOP);

				glNormal3f(ay*bz-az*by,-ax*bz+az*bx,ax*by-ay*bx);

				glColor4f( cR0, cG0, cB0, m_color_A );
				glVertex3f(x0, y0, z0);
				glColor4f( cR1, cG1, cB1, m_color_A );
				glVertex3f(x1, y1, z1);
				glColor4f( cR2, cG2, cB2, m_color_A );
				glVertex3f(x2, y2, z2);

				if (m_isWireFrame)
					glEnd();
			} //end if

			// Other orientation
			if( (useMask && mask(i+1,j) &&  mask(i+1,j+1) && mask(i,j+1)) || !useMask )
			{
				x0 = xMin + sCellX*( i+1 + 0.5f );
				y0 = yMin + sCellY*( j + 0.5f );
				z0 = Z(i+1,j);
				if(m_colorFromZ)
					mrpt::vision::colormap(the_color_map,C(i+1,j),cR0,cG0,cB0);

				x1 = xMin + sCellX*( i+1 + 0.5f );
				y1 = yMin + sCellY*( j+1 + 0.5f );
				z1 = Z(i+1,j+1);
				if(m_colorFromZ)
					mrpt::vision::colormap(the_color_map,C(i+1,j+1),cR1,cG1,cB1);

				x2 = xMin + sCellX*( i + 0.5f );
				y2 = yMin + sCellY*( j+1 + 0.5f );
				z2 = Z(i,j+1);
				if(m_colorFromZ)
					mrpt::vision::colormap(the_color_map,C(i,j+1),cR2,cG2,cB2);

				ax= x1 - x0;
				ay= y1 - y0;
				az= z1 - z0;

				bx= x2 - x0;
				by= y2 - y0;
				bz= z2 - z0;

				if (m_isWireFrame)
					glBegin(GL_LINE_LOOP);

				glNormal3f(ay*bz-az*by,-ax*bz+az*bx,ax*by-ay*bx);

				glColor4f( cR0, cG0, cB0, m_color_A );
				glVertex3f(x0, y0, z0);
				glColor4f( cR1, cG1, cB1, m_color_A );
				glVertex3f(x1, y1, z1);
				glColor4f( cR2, cG2, cB2, m_color_A );
				glVertex3f(x2, y2, z2);

				if (m_isWireFrame)
					glEnd();
			} // end if
		} // end for
	} // end for

	if (!m_isWireFrame)
		glEnd();

	/** /
	glBegin(GL_TRIANGLE_FAN);

	while( cont != rows-1)
	{
		cont++;
		if( cont%2 != 0 )
		{
			for(i = 0; i < cols; i++)
			{
				for(j = ini; j <= ini+1; j++)
				{
					if( (i == 0) && (j == ini) )
					{
						if(first)
						{
							x = xMin + sCellX*( i + 0.5f );
							y = yMin + sCellY*( j + 0.5f );

							glVertex3f( x, y, Z(j,i) );
							glColor4f( m_color_R, m_color_G, m_color_B, m_color_A );
							//cout << "(" << j << "," << i << ")" << endl;

							first = false;
						} // end if
					} // end if
					else
					{
						x = xMin + sCellX*( i + 0.5f );
						y = yMin + sCellY*( j + 0.5f );

						glVertex3f( x, y, Z(j,i) );
						glColor4f( m_color_R, m_color_G, m_color_B, m_color_A );
						//cout << "(" << j << "," << i << ")" << endl;
					} // end else
				} // end for 'j'
			} // end for 'i'
		} // end if
		else
		{
			for(i = cols-1; i >= 0; i--)
			{
				for(j = ini; j <= ini+1; j++)
				{
					if( (i != cols-1) || (j != ini) )
					{
						x = xMin + sCellX*( i + 0.5f );
						y = yMin + sCellY*( j + 0.5f );

						glVertex3f( x, y, Z(j,i) );
						glColor4f( m_color_R, m_color_G, m_color_B, m_color_A );
						//cout << "(" << j << "," << i << ")" << endl;
					} // end if
				} // end for 'j'
			} // end for 'i'
		} // end else
		ini++;
	} // end while

	glEnd();
/ **/
	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);
#endif
}

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CMesh::assignImage(
	const CMRPTImage& img )
{
	MRPT_TRY_START;

	// Make a copy:
	m_textureImage = img;
	m_enableTransparency = false;

	MRPT_TRY_END;
}
/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CMesh::writeToStream(CStream &out,int *version) const
{

	if (version)
		*version = 1;
	else
	{
		writeToStreamRender(out);

		// Version 0:
		out << m_textureImage;
		out << xMin << xMax << yMin << yMax;
		out << Z << U << V << mask;  // We don't need to serialize C, it's computed
		out << m_enableTransparency;
		out << m_colorFromZ;
		// new in v1
		out << m_isWireFrame;
		out << int16_t(m_colorMap);
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CMesh::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
	case 1:
		{
			readFromStreamRender(in);

			in >> m_textureImage;

			in >> xMin;
			in >> xMax;
			in >> yMin;
			in >> yMax;

			in >> Z >> U >> V >> mask;
			in >> m_enableTransparency;
			in >> m_colorFromZ;

			if (version>=1)
			{
				in >> m_isWireFrame;
				int16_t	i;
				in >> i;
				m_colorMap =  mrpt::vision::TColormap(i);
			}
			else	m_isWireFrame = false;

			m_modified_Z = true;
		}
		break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}


void CMesh::updateColorsMatrix()
{
	if (!m_modified_Z) return;

	const size_t cols = Z.getColCount();
	const size_t rows = Z.getRowCount();

	C.setSize(rows,cols);

	// Compute the "smoothed" height matrix:
	// ------------------------------------------
	CMatrixFloat	MEANS(rows,cols);
	const int W = 20;
	for (size_t i=0;i<rows;i++)
	{
		for (size_t j=0;j<cols;j++)
		{
			int j0 = max(0,int(j)-W);
			int j1 = min(int(cols-1),int(j)+W);

			int i0 = max(0,int(i)-W);
			int i1 = min(int(rows-1),int(i)+W);

			double S = 0;
			int    N = 0;
			for (int ii=i0;ii<=i1;ii++)
			{
				for (int jj=j0;jj<=j1;jj++)
				{
					S+=Z(ii,jj);
					N++;
				}
			}

			if (N)
				MEANS(i,j) = S / N;
		}
	}

	// Color is proportional to difference between height of a cell and
	//  the mean of the nearby cells MEANS:
	C = Z - MEANS;

	// Ignore cells with mask==0
	for (size_t i=0;i<rows;i++)
		for (size_t j=0;j<cols;j++)
			if (!mask(i,j))
				C(i,j) = 0;

	C.normalize(0.01f,0.99f);

	DEBUG_SAVE_MATRIX(C);

	m_modified_Z = false; // Done
}

void CMesh::setZ( const mrpt::math::CMatrixTemplateNumeric<float> &in_Z )
{
	Z=in_Z;
	m_modified_Z = true;
}

void CMesh::setMask( const mrpt::math::CMatrixTemplateNumeric<float> &in_mask )
{
	mask = in_mask;
}

void CMesh::setUV( const mrpt::math::CMatrixTemplateNumeric<float> &in_U, const mrpt::math::CMatrixTemplateNumeric<float> &in_V)
{
	U=in_U;
	V=in_V;
}

CMatrixFloat & CMesh::getZ()
{
	m_modified_Z = true;
	return Z;
}

CMatrixFloat & CMesh::getMask()
{
	return mask;
}
