/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/core.h>

using namespace mrpt;
using namespace std;
using namespace mrpt::gui;
using namespace mrpt::opengl;

const size_t  	N_MASSES = 100;

const double    BOX     = 60;
const double	V0		= 80;
const double    MASS_MIN = log(1000.0), MASS_MAX = log(100000.0);
const double    M2R      = 0.20;
const double  	LARGEST_STEP = 0.001;
const double 	G = 6e+2;
const double    SEPARATION_FACTOR = 10.0;


struct TMass
{
	TMass() : x(0),y(0),z(0),vx(0),vy(0),vz(0), mass(1), obj3d()
	{ }

	double	x,y,z;
	double	vx,vy,vz;
	double  mass;
	opengl::CSpherePtr	obj3d;
};

void simulateGravity( vector<TMass> &objs, double At);

// ------------------------------------------------------
//				GravityDemo
// ------------------------------------------------------
void GravityDemo()
{
	CDisplayWindow3D	win("MRPT example: 3D gravity simulator- JLBC 2008",800,600);


	mrpt::random::Randomize();

	win.setCameraElevationDeg( 50.0f );
	win.setCameraZoom( 1000 );

	COpenGLScenePtr &theScene = win.get3DSceneAndLock();

	// Modify the scene:
	// ------------------------------------------------------
	{
		opengl::CGridPlaneXYPtr obj = opengl::CGridPlaneXY::Create(-2000,2000,-2000,2000,0,100);
		obj->m_color_R = obj->m_color_G = obj->m_color_B = 0.3f;
		theScene->insert( obj );
	}

	// Create the masses:
	// ----------------------------------------------------

	vector<TMass>	masses(N_MASSES);

	// Init at random poses & create opengl objects:
	for (size_t i=0;i<N_MASSES;i++)
	{
		masses[i].x = mrpt::random::RandomNormal(0,BOX);
		masses[i].y = mrpt::random::RandomNormal(0,BOX);
		masses[i].z = mrpt::random::RandomNormal(0,BOX) / 5;

		double a=atan2(masses[i].y,masses[i].x);

		masses[i].vx = -V0*sin(a); //mrpt::random::RandomUni(-V0,V0);
		masses[i].vy =  V0*cos(a); //mrpt::random::RandomUni(-V0,V0);
		masses[i].vz =  0; //mrpt::random::RandomUni(-V0,V0);

		masses[i].mass = exp( mrpt::random::RandomUni(MASS_MIN,MASS_MAX) );
		opengl::CSpherePtr & obj = masses[i].obj3d = opengl::CSphere::Create();

		obj->m_color_R = mrpt::random::RandomUni(0.1,0.9);
		obj->m_color_G = mrpt::random::RandomUni(0.1,0.9);
		obj->m_color_B = mrpt::random::RandomUni(0.1,0.9);
		obj->m_color_A = 0.5;

		obj->m_radius = M2R * pow( masses[i].mass, 1.0/3.0); // Guess why ^(1/3) ;-)
		obj->m_x = masses[i].x;
		obj->m_y = masses[i].y;
		obj->m_z = masses[i].z;
		theScene->insert( obj );
	}

	// IMPORTANT!!! IF NOT UNLOCKED, THE WINDOW WILL NOT BE UPDATED!
	win.unlockAccess3DScene();

	mrpt::utils::CTicTac	tictac;
	tictac.Tic();

	double t0 = tictac.Tac();

	while (!mrpt::system::os::kbhit() && win.isOpen() )
	{
		// Move the scene:
		double t1 = tictac.Tac();
		double At = t1-t0;
		t0 = t1;

		// Simulate a At, possibly in several small steps:
		size_t  n_steps = ceil(At/LARGEST_STEP)+1;
		double At_steps = At / n_steps;
		for (size_t j=0;j<n_steps;j++)
			simulateGravity( masses, At_steps);

		// Update the 3D scene:
		//COpenGLScenePtr &theScene =
		win.get3DSceneAndLock();

		for (size_t i=0;i<N_MASSES;i++)
		{
			opengl::CSpherePtr & obj = masses[i].obj3d;
			obj->m_x = masses[i].x;
			obj->m_y = masses[i].y;
			obj->m_z = masses[i].z;
		}
		// IMPORTANT!!! IF NOT UNLOCKED, THE WINDOW WILL NOT BE UPDATED!
		win.unlockAccess3DScene();

		// Update window:
		win.forceRepaint();
		mrpt::system::sleep(10);
	};
}

void simulateGravity( vector<TMass> &objs, double At)
{
	const size_t N=objs.size();

	typedef vector<double> TForce;
	vector<TForce>	forces(N);


	for (size_t i=0;i<N;i++)
		forces[i].assign(3,0.0);

	for (size_t i=0;i<(N-1);i++)
	{
		// Compute overall gravity force:
		for (size_t j=i+1;i<N;i++)
		{
			double Ax = objs[j].x - objs[i].x;
			double Ay = objs[j].y - objs[i].y;
			double Az = objs[j].z - objs[i].z;
			double D2 = square(Ax)+square(Ay)+square(Az);

			double D = sqrt(D2);

			D = max(D, SEPARATION_FACTOR *(double)( objs[i].obj3d->m_radius + objs[j].obj3d->m_radius) );

			double K = G * objs[i].mass * objs[j].mass / square(D);
			Ax /= D;
			Ay /= D;
			Az /= D;

			forces[i][0] += Ax * K;
			forces[i][1] += Ay * K;
			forces[i][2] += Az * K;

			forces[j][0] -= Ax * K;
			forces[j][1] -= Ay * K;
			forces[j][2] -= Az * K;
		}
	}

	// F = m a
	for (size_t i=0;i<N;i++)
	{
		TForce a = forces[i];
		a[0] /= objs[i].mass;
		a[1] /= objs[i].mass;
		a[2] /= objs[i].mass;

		objs[i].vx += a[0]*At;
		objs[i].vy += a[1]*At;
		objs[i].vz += a[2]*At;

		objs[i].x += objs[i].vx*At;
		objs[i].y += objs[i].vy*At;
		objs[i].z += objs[i].vz*At;
	}
}


// ------------------------------------------------------
//						MAIN
// ------------------------------------------------------
int main()
{
	try
	{
		GravityDemo();
		return 0;
	} catch (std::exception &e)
	{
		std::cout << "MRPT exception caught: " << e.what() << std::endl;
		return -1;
	}
	catch (...)
	{
		printf("Untyped exception!!");
		return -1;
	}
}
